// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Config_h__
#define __Config_h__

// Class Config provides evaluation of configuration 
// information from file or command line.

#include "Puma/Flags.h"
#include "Puma/ConfOption.h"
#include "Puma/ErrorStream.h"
#include "Puma/OptsParser.h"
#include <string>
using namespace std;

namespace Puma {


class Config {
  Flags _flags;
  ErrorStream &_err;
  Array<const ConfOption *> _options;
  
public:
  enum OPTIONS {
    PRE_ASSERT = 1,
    PRE_DEFINE, 
    PRE_UNDEFINE, 
    PRE_LOCK_MACRO,
    PRE_INCLUDE,
    CFG_FILE, 
    PROJ_PATH, 
    PROJ_DESTINATION,
    SET_SUFFIX, 
    SET_EXTENSION, 
    SET_WRITE_PROTECTED,
    SET_OPTION, 
    SET_OPTION_ARG
  };
   
public:
  Config (ErrorStream &);
  Config (const Config &);
  ~Config ();
  
  // read config file
  Config (ErrorStream &, const char *);
  // read command line
  Config (ErrorStream &, int &, char **&);
  // read config file and command line
  Config (ErrorStream &, const char *, int &, char **&);
        
  // join with another configuration
  void Join (const Config &);

  // evaluate file, if file not given read from $PUMA_CONFIG
  // or /etc/puma.config
  void Read (const char * = (const char *)0);
  // evaluate command line
  void Read (int &, char **&);

  // add options to configuration
  void Add (const ConfOption *);
  void Add (const char *);
  void Add (const char *, const char *);
  void Add (const char *, const char *, const char *);

  // get options
  unsigned Options () const;
  const ConfOption *Option (unsigned) const;
  const ConfOption *Option (const char *) const;

  Flags FlagTable () const;
  
private:
  bool Valid (const char *, const char * = (const char *)0) const;

  bool Process (OptsParser&);
  bool ProcessAssert (const string&);
  bool ProcessDefine (const string&);
  bool ProcessLockMacro (const string&);
  bool ProcessUndefine (const string&);
  bool ProcessConfigFile (const string&,const string&);
  bool ProcessArgument (const string&,const string&);
  bool ProcessPathArgument (const string&,const string&);

};

inline Config::Config (ErrorStream &e) : _err (e)
 {}
inline Config::Config (const Config &c) : _err (c._err)
 { Join (c); }
  
inline Config::Config (ErrorStream &e, const char *file) : _err (e)
 { Read (file); }
inline Config::Config (ErrorStream &e, int &argc, char **&argv) : _err (e)
 { Read (argc, argv); }
inline Config::Config (ErrorStream &e, const char *file, int &argc, char **&argv) : _err (e)
 { Read (file); Read (argc, argv); }

inline Config::~Config  ()
 { for (unsigned i = 0; i < Options (); i++)
     delete (ConfOption *) Option (i);
 }

inline void Config::Add (const ConfOption *o) 
 { _options.append (o); }
inline void Config::Add (const char *n)
 { Add (new ConfOption (n)); }
inline void Config::Add (const char *n, const char *a1)
 { Add (new ConfOption (n, a1)); }
inline void Config::Add (const char *n, const char *a1, const char *a2)
 { Add (new ConfOption (n, a1, a2)); }

inline unsigned Config::Options () const
 { return _options.length (); }
inline const ConfOption *Config::Option (unsigned i) const
 { return _options.lookup (i); }

inline Flags Config::FlagTable () const
 { return _flags; }


} // namespace Puma

#endif /* __Config_h__ */
