/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */


#include "environment.h"
#include "globaldata.h"
#include "player.h"
#include "tank.h"
#include "menu.h"
#include "lineseq.h"
#include "files.h"

// When defined draws AI 'planning'
//#define AI_PLANNING_DEBUG	1

PLAYER::PLAYER (GLOBALDATA *global, ENVIRONMENT *env)
{
        int my_colour;

	_global = global;
	_env = env;
	money = 15000;
	score = 0;
	played = 0;
	won = 0;
	tank = NULL;
        team = TEAM_NEUTRAL;
	_turnStage = 0;
	selected = FALSE;
        changed_weapon = false;

	_targetMatrix = new double[global->screenWidth];

        tank_bitmap = 0.0;

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	strncpy (_name, "New", NAME_LENGTH);
	type = HUMAN_PLAYER;

	// 10% of time set to perplay weapon preferences
	preftype = (rand () % 10)?ALWAYS_PREF:PERPLAY_PREF;
	generatePreferences ();
	vengeful = rand () % 100;
	vengeanceThreshold = (double)(rand () % 1000) / 1000.0;
	revenge = NULL;
	selfPreservation = (double)(rand () % 3000) / 1000;
	painSensitivity = (double)(rand () % 3000) / 1000;
	defensive = (double)((rand () % 10000) - 5000) / 5000.0;

	// color = rand () % WHITE;
        my_colour = rand() % 4;
        switch (my_colour)
        {
           case 0: color = makecol(200 + (rand() % 56), rand() % 25, rand() % 25); break;
           case 1: color = makecol( rand() % 25, 200 + (rand() % 56), rand() % 25); break;
           case 2: color = makecol( rand() % 25, rand() % 25, 200 + (rand() % 56) ); break;
           case 3: color = makecol( 200 + (rand() % 56), rand() % 25, 200 + (rand() % 56)); break;
        }
	typeText[0] = "Human";
	typeText[1] = "Useless";
	typeText[2] = "Guesser";
	typeText[3] = "Range Finder";
	typeText[4] = "Targetter";
	typeText[5] = "Deadly";
	preftypeText[0] = "Per Game";
	preftypeText[1] = "Only Once";
        tank_type[0] = "Normal";
        tank_type[1] = "Classic";
        teamText[0] = "Sith";
        teamText[1] = "Neutral";
        teamText[2] = "Jedi";

	initMenuDesc ();
}

PLAYER::PLAYER (GLOBALDATA *global, ENVIRONMENT *env, FILE *file, bool text_file)
{
	_global = global;
	_env = env;
	money = 15000;
	score = 0;
	tank = NULL;
	_turnStage = 0;
	selected = FALSE;
        
        tank_bitmap = 0.0;
        team = TEAM_NEUTRAL;

	_targetMatrix = new double[global->screenWidth];

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

        // generate fresh weapon preferences because I goofed
        // a while back and did not save them all. So to
        // be backward compatible, generate the preferences.
        // If they are saved, then the saved ones will
        // over-write these ones
        generatePreferences();

	if (text_file)
           loadFromFile_Text (file);
        else
           loadFromFile (file);

	type = (int)type;

	typeText[0] = "Human";
	typeText[1] = "Useless";
	typeText[2] = "Guesser";
	typeText[3] = "Range Finder";
	typeText[4] = "Targetter";
	typeText[5] = "Deadly";
	preftypeText[0] = "Per Game";
	preftypeText[1] = "Only Once";
        tank_type[0] = "Normal";
        tank_type[1] = "Classic";
        teamText[0] = "Sith";
        teamText[1] = "Neutral";
        teamText[2] = "Jedi";
	initMenuDesc ();
}

void PLAYER::setComputerValues ()
{
	switch ((int)type) {
		case HUMAN_PLAYER:
			break;
		case USELESS_PLAYER:
			rangeFindAttempts = 1;
			retargetAttempts = 0;
			focusRate = 2.0;
			errorMultiplier = 2.0;
			break;
		case GUESSER_PLAYER:
			rangeFindAttempts = 1;
			retargetAttempts = 0;
			focusRate = 0.6;
			errorMultiplier = 1.0;
			break;
		case RANGEFINDER_PLAYER:
			rangeFindAttempts = 10;
			retargetAttempts = 2;
			focusRate = 0.1;
			errorMultiplier = 0.3;
			break;
		case TARGETTER_PLAYER:
			rangeFindAttempts = 20;
			retargetAttempts = 5;
			focusRate = 1.0;
			errorMultiplier = 0.1;
			break;
		case DEADLY_PLAYER:
                case PART_TIME_BOT:
			rangeFindAttempts = 40;
			retargetAttempts = 10;
			focusRate = 1.0;
			errorMultiplier = 0.02;
			break;
		default:
			rangeFindAttempts = 0;
			retargetAttempts = 0;
			focusRate = 0;
			errorMultiplier = 1.0;
			break;
	}
}

int displayPlayerName (GLOBALDATA *global, ENVIRONMENT *env, int x, int y, void *data);
void PLAYER::initMenuDesc ()
{

	int destroyPlayer (GLOBALDATA *global, ENVIRONMENT *env, void *data);
        int i = 0;
        int height = -68;

	menudesc = new MENUDESC;
	if (!menudesc) {
		perror ("player.cc: Failed allocating memory for menudesc in PLAYER::initMenuDesc");
		exit (1);
	}
	menudesc->title = _name;

	// Name,Color
	menudesc->numEntries = 10;
	menudesc->okayButton = TRUE;
	menudesc->quitButton = FALSE;

	menuopts = new MENUENTRY[menudesc->numEntries];
	if (!menuopts) {
		perror ("player.cc: Failed allocating memory for menuopts in PLAYER::initMenuDesc");
		exit (1);
	}

	// Player name
	menuopts[i].name = "Name";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = color;
	menuopts[i].value = (double*)_name;
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_TEXTTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	// Player colour
	menuopts[i].name = "Colour";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&color;
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_COLORTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	// Player type (human, computer)
	menuopts[i].name = "Type";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&type;
	menuopts[i].min = 0;
	menuopts[i].max = LAST_PLAYER_TYPE - 1;
	menuopts[i].increment = 1;
	menuopts[i].defaultv = 0;
	menuopts[i].format = "%s";
	menuopts[i].specialOpts = typeText;
	menuopts[i].type = OPTION_SPECIALTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
	i++; height += 20;

        menuopts[i].name = "Team";
        menuopts[i].displayFunc = NULL;
        menuopts[i].color = WHITE;
        menuopts[i].value = (double *)&team;
        menuopts[i].min = 0;
        menuopts[i].max = TEAM_JEDI;
        menuopts[i].increment = 1;
        menuopts[i].defaultv = TEAM_NEUTRAL;
        menuopts[i].format = "%s";
        menuopts[i].specialOpts = teamText;
        menuopts[i].type = OPTION_SPECIALTYPE;
        menuopts[i].viewonly = FALSE;
        menuopts[i].x = _global->halfWidth - 3;
        menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	// Player preftype (human, computer)
	menuopts[i].name = "Generate Pref.";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&preftype;
	menuopts[i].min = 0;
	menuopts[i].max = ALWAYS_PREF;
	menuopts[i].increment = 1;
	menuopts[i].defaultv = 0;
	menuopts[i].format = "%s";
	menuopts[i].specialOpts = preftypeText;
	menuopts[i].type = OPTION_SPECIALTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menuopts[i].name = "Played";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&played;
	menuopts[i].format = "%.0f";
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_DOUBLETYPE;
	menuopts[i].viewonly = TRUE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menuopts[i].name = "Won";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&won;
	menuopts[i].format = "%.0f";
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_DOUBLETYPE;
	menuopts[i].viewonly = TRUE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

        menuopts[i].name = "Tank Type";
        menuopts[i].displayFunc = NULL;
        menuopts[i].color = WHITE;
        menuopts[i].value = &tank_bitmap;
        menuopts[i].min = 0;
        menuopts[i].max = 1;
        menuopts[i].increment = 1;
        menuopts[i].defaultv = 0;
        menuopts[i].format = "%1.0f";
        menuopts[i].specialOpts = tank_type;
        menuopts[i].type = OPTION_SPECIALTYPE;
        menuopts[i].viewonly = FALSE;
        menuopts[i].x = _global->halfWidth - 3;
        menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menuopts[i].name = "Delete This Player";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)destroyPlayer;
	menuopts[i].data = (void*)this;
	menuopts[i].type = OPTION_ACTIONTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menudesc->entries = menuopts;
}

PLAYER::~PLAYER ()
{
	if (tank)
		delete tank;

	delete menuopts;
	delete menudesc;
}

// Following are config file definitions
#define	GFILE_LATEST_VERSION	11

#define GFILE_NAME		_name, sizeof (char), 11, file
#define GFILE_NAME_EXTENDED     _name, sizeof (char), NAME_LENGTH, file
#define	GFILE_WEAPONPREFERENCE1	_weaponPreference, sizeof (int), 51, file
#define	GFILE_WEAPONPREFERENCE2	_weaponPreference, sizeof (int), 54, file
#define	GFILE_WEAPONPREFERENCE3	_weaponPreference, sizeof (int), 58, file
#define	GFILE_WEAPONPREFERENCE4	_weaponPreference, sizeof (int), 60, file
#define	GFILE_WEAPONPREFERENCE5	_weaponPreference, sizeof (int), 62, file
#define	GFILE_TYPE		&type, sizeof (double), 1, file
#define	GFILE_VENGEFUL		&vengeful, sizeof (int), 1, file
#define	GFILE_VENGEANCETHRESHOLD &vengeanceThreshold, sizeof (double), 1, file
#define	GFILE_COLOR		&color, sizeof (int), 1, file
#define	GFILE_COLOR2		&color2, sizeof (int), 1, file
#define	GFILE_PLAYED		&played, sizeof (double), 1, file
#define	GFILE_WON		&won, sizeof (double), 1, file
#define GFILE_PREFTYPE		&preftype, sizeof (double), 1, file
#define	GFILE_SELFPRESERVATION	&selfPreservation, sizeof (double), 1, file
#define	GFILE_PAINSENSITIVITY	&painSensitivity, sizeof (double), 1, file
#define	GFILE_DEFENSIVE		&defensive, sizeof (double), 1, file
#define GFILE_TANK_BITMAP       &tank_bitmap, sizeof (double), 1, file
#define GFILE_TURRET_BITMAP    &turret_bitmap, sizeof (double), 1, file

#define	GFILE_VERSION_1(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_WEAPONPREFERENCE1);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_2(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE2);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_3(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE3);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_4(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_5(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_6(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);

#define	GFILE_VERSION_7(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);\
				a (GFILE_SELFPRESERVATION);\
				a (GFILE_PAINSENSITIVITY);

#define	GFILE_VERSION_8(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE5);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);\
				a (GFILE_SELFPRESERVATION);\
				a (GFILE_PAINSENSITIVITY);

#define	GFILE_VERSION_9(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE5);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);\
				a (GFILE_SELFPRESERVATION);\
				a (GFILE_PAINSENSITIVITY);\
				a (GFILE_DEFENSIVE);

#define GFILE_VERSION_10(a)     a (GFILE_NAME_EXTENDED);\
                                a (GFILE_VENGEFUL);\
                                a (GFILE_VENGEANCETHRESHOLD);\
                                a (GFILE_TYPE);\
                                a (GFILE_COLOR);\
                                a (GFILE_COLOR2);\
                                a (GFILE_WEAPONPREFERENCE5);\
                                a (GFILE_PLAYED);\
                                a (GFILE_WON);\
                                a (GFILE_PREFTYPE);\
                                a (GFILE_SELFPRESERVATION);\
                                a (GFILE_PAINSENSITIVITY);\
                                a (GFILE_DEFENSIVE);

#define GFILE_VERSION_11(a)     a (GFILE_NAME_EXTENDED);\
                                a (GFILE_VENGEFUL);\
                                a (GFILE_VENGEANCETHRESHOLD);\
                                a (GFILE_TYPE);\
                                a (GFILE_COLOR);\
                                a (GFILE_COLOR2);\
                                a (GFILE_WEAPONPREFERENCE5);\
                                a (GFILE_PLAYED);\
                                a (GFILE_WON);\
                                a (GFILE_PREFTYPE);\
                                a (GFILE_SELFPRESERVATION);\
                                a (GFILE_PAINSENSITIVITY);\
                                a (GFILE_DEFENSIVE);\
                                a (GFILE_TANK_BITMAP);\
                                a (GFILE_TURRET_BITMAP); // remove this line in version 12



#define	GFILE_WRITE_LATEST	GFILE_VERSION_11(fwrite)
#define	GFILE_READ_VERSION_1	GFILE_VERSION_1(fread)
#define	GFILE_READ_VERSION_2	GFILE_VERSION_2(fread)
#define	GFILE_READ_VERSION_3	GFILE_VERSION_3(fread)
#define	GFILE_READ_VERSION_4	GFILE_VERSION_4(fread)
#define	GFILE_READ_VERSION_5	GFILE_VERSION_5(fread)
#define GFILE_READ_VERSION_6	GFILE_VERSION_6(fread)
#define GFILE_READ_VERSION_7	GFILE_VERSION_7(fread)
#define GFILE_READ_VERSION_8	GFILE_VERSION_8(fread)
#define GFILE_READ_VERSION_9	GFILE_VERSION_9(fread)
#define GFILE_READ_VERSION_10   GFILE_VERSION_10(fread);
#define GFILE_READ_VERSION_11   GFILE_VERSION_11(fread);



/*
Save the player settings in text form. Fields are
formateed as
[name]=[value]\n

Function returns true on success and false on failure.
*/
int PLAYER::saveToFile_Text (FILE *file)
{
   int count;

   if (! file) return FALSE;
   // start section with "*PLAYER*"
   fprintf(file, "*PLAYER*\n");
   fprintf(file, "NAME=%s\n", _name);
   fprintf(file, "VENGEFUL=%d\n", vengeful);
   fprintf(file, "VENGEANCETHRESHOLD=%lf\n", vengeanceThreshold);
   fprintf(file, "TYPE=%lf\n", type);
   fprintf(file, "COLOR=%d\n", color);
   fprintf(file, "COLOR2=%d\n", color2);
   for (count = 0; count < THINGS; count++)
      fprintf(file, "WEAPONPREFERENCES=%d %d\n", count, _weaponPreference[count]);

   fprintf(file, "PLAYED=%lf\n", played);
   fprintf(file, "WON=%lf\n", won);
   fprintf(file, "PREFTYPE=%lf\n", preftype);
   fprintf(file, "SELFPRESERVATION=%lf\n", selfPreservation);
   fprintf(file, "PAINSENSITIVITY=%lf\n", painSensitivity);
   fprintf(file, "DEFENSIVE=%lf\n", defensive);
   fprintf(file, "TANK_BITMAP=%lf\n", tank_bitmap);
   fprintf(file, "TEAM=%lf\n", team);
   fprintf(file, "***\n");
   return TRUE;
}


/*
This function tries to load player data from a text file.
Each line is parsed for "field=value", except WEAPONPREFERENCES
which is parsed "field=index value".
If all goes well TRUE is returned, on error the function returns FALSE.
-- Jesse
*/
int PLAYER::loadFromFile_Text (FILE *file)
{
    char line[MAX_CONFIG_LINE];
    int equal_position, line_length;
    int index, wp_value;
    char field[MAX_CONFIG_LINE], value[MAX_CONFIG_LINE];
    char *result = NULL;
    bool done = false;

    if (! file) return FALSE;

    // read until we hit line "*PLAYER*" or "***" or EOF
    do
    {
      result = fgets(line, MAX_CONFIG_LINE, file);
      if (! result)     // eof
         return FALSE;
      if (! strncmp(line, "***", 3) )     // end of record
         return FALSE;
    } while ( strncmp(line, "*PLAYER*", 8) );     // read until we hit new record

    while ( (result) && (!done) )
    {
       // read a line
       memset(line, '\0', MAX_CONFIG_LINE);
       result = fgets(line, MAX_CONFIG_LINE, file);
       if (result)
       {
          // if we hit end of the record, stop
          if (! strncmp(line, "***", 3) ) return TRUE;
          // find equal sign
          line_length = strlen(line);
          // strip newline character
          if ( line[line_length - 1] == '\n')
          {
               line[line_length - 1] = '\0';
               line_length--;
          }
          equal_position = 1;
          while ( ( equal_position < line_length) && (line[equal_position] != '=') )
              equal_position++;
          // make sure we have valid equal sign
          if (equal_position <= line_length)
          {
             // seperate field from value
             memset(field, '\0', MAX_CONFIG_LINE);
             memset(value, '\0', MAX_CONFIG_LINE);
             strncpy(field, line, equal_position);
             strcpy(value, & (line[equal_position + 1]));
             // check which field we have and process value
             if (! strcasecmp(field, "name") )
                 strncpy(_name, value, NAME_LENGTH);
             else if (! strcasecmp(field, "vengeful") )
                 sscanf(value, "%d", &vengeful);
             else if (! strcasecmp(field, "vengeancethreshold") )
                 sscanf(value, "%lf", &vengeanceThreshold);
             else if (! strcasecmp(field, "type") )
                 sscanf(value, "%lf", &type);
             else if (! strcasecmp(field, "color") )
                 sscanf(value, "%d", &color);
             else if (! strcasecmp(field, "color2") )
                 sscanf(value, "%d", &color2);
             else if (! strcasecmp(field, "played") )
                 sscanf(value, "%lf", &played);
             else if (! strcasecmp(field, "won") )
                 sscanf(value, "%lf", &won);
             else if (! strcasecmp(field, "preftype") )
                 sscanf(value, "%lf", &preftype);
             else if (! strcasecmp(field, "selfpreservation") )
                 sscanf(value, "%lf", &selfPreservation);
             else if (! strcasecmp(field, "painsensititveity") )
                 sscanf(value, "%lf", &painSensitivity);
             else if (! strcasecmp(field, "defensive") )
                 sscanf(value, "%lf", &defensive);
             else if (! strcasecmp(field, "tank_bitmap") )
                 sscanf(value, "%lf", &tank_bitmap);
             else if (! strcasecmp(field, "team") )
                 sscanf(value, "%lf", &team);
             else if (! strcasecmp(field, "weaponpreferences") )
             {
                 sscanf(value, "%d %d", &index, &wp_value);
                 if ( (index < THINGS) && (index >= 0) )
                    _weaponPreference[index] = wp_value;
             }

          }    // end of valid data line
       }      // end of if we read a line properly
    }   // end of while not done

     // make sure previous human players are restored as humans
    if (type == PART_TIME_BOT)
           type = HUMAN_PLAYER;

    return TRUE;
}


int PLAYER::saveToFile (FILE *file)
{
	int version = GFILE_LATEST_VERSION;
        double turret_bitmap;  // quick hack to fix version 11

	if (!file)
		return (-1);

        // make sure part time bots are saved as human
        if (type == PART_TIME_BOT)
           type = HUMAN_PLAYER;

	fwrite (&version, sizeof (int), 1, file);

	GFILE_WRITE_LATEST

	return (ferror (file));
}

int PLAYER::loadFromFile (FILE *file)
{
	int version;
        double turret_bitmap;     // quick hack to fix version 11

	if (!file)
		return (-1);

	fread (&version, sizeof (int), 1, file);

	switch (version) {
		case 1:
			GFILE_READ_VERSION_1;
			break;
		case 2:
			GFILE_READ_VERSION_2;
			break;
		case 3:
			GFILE_READ_VERSION_3;
			break;
		case 4:
			GFILE_READ_VERSION_4;
			break;
		case 5:
			GFILE_READ_VERSION_5;
			break;
		case 6:
			GFILE_READ_VERSION_6;
			break;
		case 7:
			GFILE_READ_VERSION_7;
			break;
		case 8:
			GFILE_READ_VERSION_8;
			break;
		case 9:
			GFILE_READ_VERSION_9;
			break;
                case 10:
                        GFILE_READ_VERSION_10;
                        break;
                case 11:
                        GFILE_READ_VERSION_11;
                        break;
		default:
			break;
	}
	
	if (version < 6) {
		// Apply change between v 5 and 6 of the player config
		// 10% of time set to perplay weapon preferences
		preftype = (rand () % 10)?ALWAYS_PREF:PERPLAY_PREF;
		generatePreferences ();
	}
	if (version < 7) {
		// Apply change between v 6 and 7
		selfPreservation = (double)(rand () % 3000) / 1000;
		painSensitivity = (double)(rand () % 3000) / 1000;
	}
	if (version < 8) {
		// Apply changes between 7 and 8
		generatePreferences ();
	}
	if (version < 9) {
		// Apply changes between 8 and 9
		defensive = (double)((rand () % 10000) - 5000) / 5000.0;
	}

        // make sure previous human players are restored as humans
        if (type == PART_TIME_BOT)
           type = HUMAN_PLAYER;

	return (ferror (file));
}

void PLAYER::exitShop ()
{
	double tmpDM = 0;

	damageMultiplier = 1.0;
	tmpDM += ni[ITEM_INTENSITY_AMP] * item[ITEM_INTENSITY_AMP].vals[0];
	tmpDM += ni[ITEM_VIOLENT_FORCE] * item[ITEM_VIOLENT_FORCE].vals[0];
	if (tmpDM > 0)
		damageMultiplier += pow (tmpDM, 0.6);
}


// run this at the begining of each turn
void PLAYER::newRound ()
{
        // if the player is under computer control, give it back to the player
        if ( type == PART_TIME_BOT )
           type = HUMAN_PLAYER;

	setComputerValues ();
	if (!tank) {
		tank = new TANK(_global, _env);
		if (!tank) {
			perror ("player.cc: Failed allocating memory for tank in PLAYER::newRound");
			exit (1);
		}
		tank->player = this;
	}
	tank->newRound ();

        /*
        This used to make sure money did not over-flow.
        But the check has been moved to the buystuff function
        because interest is applied there. 
        -- Jesse
        if (money > 1000000000)
           money = 1000000000;
        if (money < 0)
           money = 0;
        */
        changed_weapon = false;
}

void PLAYER::initialise ()
{
	long int totalPrefs;
	int rouletteCount;

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	totalPrefs = 0;
	for (int weapCount = 0; weapCount < THINGS; weapCount++)
		totalPrefs += _weaponPreference[weapCount];

	rouletteCount = 0;
	for (int weapCount = 0; weapCount < THINGS; weapCount++) {
		int weapRSpace = (int)((double)_weaponPreference[weapCount] / totalPrefs * NUM_ROULETTE_SLOTS);
		int weapRCount = 0;

		if (weapRSpace < 1)
			weapRSpace = 1;
		while (weapRCount < weapRSpace && rouletteCount + weapRCount < NUM_ROULETTE_SLOTS) {
			_rouletteWheel[rouletteCount + weapRCount] = weapCount;
			weapRCount++;
		}
		rouletteCount += weapRSpace;
	}
	while (rouletteCount < NUM_ROULETTE_SLOTS)
		_rouletteWheel[rouletteCount++] = rand () % THINGS;
}

void PLAYER::generatePreferences ()
{
	for (int weapCount = 0; weapCount < THINGS; weapCount++) {
		_weaponPreference[weapCount] = (rand () % MAX_WEAP_PROBABILITY);
	}
	// Add greater chance of buying parachutes
	_weaponPreference[WEAPONS + ITEM_PARACHUTE] += MAX_WEAP_PROBABILITY;
}

int PLAYER::selectRandomItem ()
{
	return (_rouletteWheel[rand () % NUM_ROULETTE_SLOTS]);
}

void PLAYER::setName (char *name)
{
        // initalize name
        memset(_name, '\0', NAME_LENGTH);
	strncpy (_name, name, NAME_LENGTH - 1);
}

int PLAYER::controlTank (TANK *ctank)
{
	if (key[KEY_F1])
		save_bmp ("scrnshot.bmp", _env->db, NULL);

	if (key_shifts & KB_CTRL_FLAG && ctrlUsedUp) {
		if (key[KEY_LEFT] || key[KEY_RIGHT] ||
			key[KEY_UP] || key[KEY_DOWN] ||
                        key[KEY_PGUP] || key[KEY_PGDN])
			ctrlUsedUp = TRUE;
		else
			ctrlUsedUp = FALSE;
	} else {
		ctrlUsedUp = FALSE;
	}

	if (_global->computerPlayersOnly &&
		((int)_global->skipComputerPlay >= SKIP_HUMANS_DEAD)) {
		if (_env->stage == 3)
			return (-1);
	}

	k = 0;
	if (keypressed () && !fi) {
		k = readkey ();

		if ((_env->stage == 3) &&
			(k >> 8 == KEY_ENTER ||
			 k >> 8 == KEY_ESC ||
			 k >> 8 == KEY_SPACE))
			return (-1);
		if (k >> 8 == KEY_ESC) {
			void clockadd ();
			install_int_ex (clockadd, SECS_TO_TIMER(6000));
			int mm = _env->ingamemenu ();
                        install_int_ex(clockadd, BPS_TO_TIMER(_global->frames_per_second));
			_env->make_update (0, 0, _global->screenWidth, _global->screenHeight);
			_env->make_bgupdate (0, 0, _global->screenWidth, _global->screenHeight);
			
			//Main Menu
			if (mm == 1) 
			{
				_global->command = GLOBAL_COMMAND_MENU;
				return (-1);
			}
			else if (mm == 2)  //Quit game
			{
				_global->command = GLOBAL_COMMAND_QUIT;
				return (-1);
			}
		}
                // check for number key being pressed
                if ( (k >> 8 >= KEY_0) && (k >> 8 <= KEY_9) )
                {
                    int value = (k >> 8) - KEY_0;
                   
                    // make sure the value is within range
                    if (value < _global->numPlayers)
                    {
                        if ( _global->players[value] )
                        {
                            TANK *my_tank = _global->players[value]->tank;
                            if (my_tank)
                            { 
                                sprintf(_global->tank_status, "%s: %d + %d -- Team: ", _global->players[value]->_name,
                                        my_tank->l, my_tank->sh);
                                strcat(_global->tank_status, _global->players[value]->Get_Team_Name());
                                _global->tank_status_colour = _global->players[value]->color;
                                _global->updateMenu = 1;
                            }
                            else
                              _global->tank_status[0] = 0;
                        }
                    }
                       
                }    // end of check status keys
	}
	if ((int)type == HUMAN_PLAYER || !ctank) {
		return (humanControls (ctank));
	} else if (!_env->stage) {
		return (computerControls (ctank));
	}
	return (0);
}

int PLAYER::chooseItemToBuy ()
{
	int currItem = selectRandomItem ();
	int itemNum = currItem - WEAPONS;
	int cumulative;
	int nextMod,curramt,newamt;

        /* if the AI is intelligent and there is gravity,
           they buy parachutes
        */
        if ( ( (type >= RANGEFINDER_PLAYER) && (type <= DEADLY_PLAYER) ) &&
             ( _env->landSlideType > LANDSLIDE_NONE))
        {
            if ( ni[ITEM_PARACHUTE] < 10)
            { 
               // do we have the funds?
               if (money >= item[ITEM_PARACHUTE].cost )
               {
                     money -= item[ITEM_PARACHUTE].cost;
                     ni[ITEM_PARACHUTE] += item[ITEM_PARACHUTE].amt;
                     return ITEM_PARACHUTE;
               }
            }
        }

	//loop through all weapon preferences
	for (int count = 0; count < THINGS; count++) {

		//is item available for purchase or selected first item
		//first item is free
		if ( (!_env->isItemAvailable (currItem)) || (currItem==0))
			continue;

		//determine the likelyhood of purchasing this selection
		//less likely the more of the item is owned		
		//if have zero of item, it is a fifty/fifty chance of purchase	
		if(currItem < WEAPONS) {
		  curramt = nm[currItem];
		  newamt = weapon[currItem].amt;
		} else {
		  curramt = ni[currItem - WEAPONS];
		  newamt = item[currItem - WEAPONS].amt;
		}

		if ((itemNum >= ITEM_INTENSITY_AMP &&
			itemNum <= ITEM_VIOLENT_FORCE) ||
                        (itemNum == ITEM_REPAIRKIT) ||
			(itemNum >= ITEM_ARMOUR &&
			 itemNum <= ITEM_PLASTEEL)) {
			cumulative = TRUE;
		} else {
			cumulative = FALSE;
		}
		  
		if (!cumulative) {
			nextMod = curramt / newamt;
		} else {
			nextMod = 1;
		}
		if (nextMod <= 0) {
			nextMod = 1;
		}

		if (rand () % nextMod) 
			continue;
			
		//weapon
		if (currItem < WEAPONS)
		{
			//don't buy if already maxed out
			if(nm[currItem] >=99) continue;
			//purchase the item
			if (money >= weapon[currItem].cost)
			{
			  money -= weapon[currItem].cost;
			  nm[currItem] += weapon[currItem].amt;
			  //don't allow more than 99
			  if(nm[currItem] > 99) nm[currItem] = 99;
			  return currItem;
			}
		}
		else  //item
		{
			//don't buy if already maxed out
			if(ni[currItem-WEAPONS] >=99) continue;
			//purchase the item
			if (money >= item[currItem - WEAPONS].cost) {
				money -= item[currItem - WEAPONS].cost;
				ni[currItem - WEAPONS] += item[currItem - WEAPONS].amt;
				//don't allow more than 99
				if(ni[currItem - WEAPONS] > 99) ni[currItem - WEAPONS] = 99;
				return (currItem);
			}
		}
	}
	return (-1);
}

char *PLAYER::selectRevengePhrase (double scale)
{
        char *filename;
        char *line;

        filename = (char *) calloc( strlen(DATA_DIR) + 64, sizeof(char) );
        if (! filename) return NULL;

        strcpy(filename, DATA_DIR);
        if (_global->language == LANGUAGE_ENGLISH)
           strcat(filename, "/revenge.txt");
        else if (_global->language == LANGUAGE_PORTUGUESE)
           strcat(filename, "/revenge.pt_BR.txt");

        // static const LINESEQ says( filename, LINESEQ::skip_blanks );
        line = Random_Line(filename);
        free(filename);
        return line;
	// return says.random( );
}

char *PLAYER::selectGloatPhrase (double scale)
{
        char *filename;
        char *line;

        filename = (char *) calloc( strlen(DATA_DIR) + 64, sizeof(char) );
        if (! filename) return NULL;

        strcpy(filename, DATA_DIR);
	if (_global->language == LANGUAGE_ENGLISH)
           strcat(filename, "/gloat.txt");
        else if (_global->language == LANGUAGE_PORTUGUESE)
           strcat(filename, "/gloat.pt_BR.txt");

        // static const LINESEQ says( filename, LINESEQ::skip_blanks );
        line = Random_Line(filename);
        free(filename); 
        return line;
	// return says.random( );
}

char *PLAYER::selectSuicidePhrase (double scale)
{
        char *filename;
        char *line;

        filename = (char *) calloc( strlen(DATA_DIR) + 64, sizeof(char) );
        if (! filename) return NULL;

        strcpy(filename, DATA_DIR);
        if (_global->language == LANGUAGE_ENGLISH)
           strcat(filename, "/suicide.txt");
        else if (_global->language == LANGUAGE_PORTUGUESE)
           strcat(filename, "/suicide.pt_BR.txt");

        // static const LINESEQ says( filename, LINESEQ::skip_blanks );
        line = Random_Line(filename);       
        free(filename);
        return line;
        // return says.random();
}


int PLAYER::traceShellTrajectory (TANK *ctank, int *estimateX, int *estimateY, double targetX, double targetY, double x, double y, double xv, double yv)
{
	TANK *ltank;
	int hitSomething = 0;
	double distance;
	int direction = (char)copysign (1, xv);
	double drag;
	int ticks = 0;
	int maxTicks = 500;
	double startX, startY;

	drag = weapon[ctank->cw].drag;
	if (ni[ITEM_DIMPLEP]) {
		drag *= item[ITEM_DIMPLEP].vals[0];
	} else if (ni[ITEM_SLICKP]) {
		drag *= item[ITEM_SLICKP].vals[0];
	}

	while ((!hitSomething) && (ticks < maxTicks) && (y < MENU ||
		 (getpixel (_env->terrain, (int)x, (int)y) == PINK))) {
		for (int objCount = 0; (ltank = (TANK*)_env->getNextOfClass (TANK_CLASS, &objCount)) && ltank; objCount++) {
			if (ltank->player != this) {
				double xaccel = 0, yaccel = 0;
				ltank->repulse (x, y, &xaccel, &yaccel);
				xv += xaccel;
				yv += yaccel;
			}
		}

		startX = x;
		startY = y;
                // hit side of the screen
		if (x + xv < 1 || x + xv > (_global->screenWidth-1)) {
                      // bounce if wall is rubber
                      if ( _env->current_wallType == WALL_RUBBER )
                      {
		  	  xv = -xv;	//bounce on the border
			  direction = -direction;
                      }
                      else if ( _env->current_wallType == WALL_SPRING )
                      {
                          xv = -xv * SPRING_CHANGE;
                          direction = -direction; 
                      }
                      else if ( _env->current_wallType == WALL_STEEL )  // wall is steel
                      {
                          hitSomething = 1;
                          break;
                      }
		} else {
			// motion - wind affected
			double accel = (_env->wind - xv) / weapon[ctank->cw].mass * drag * _env->viscosity;
			xv += accel;
			x += xv;
		}
		if (y + yv >= _global->screenHeight)
		{
			yv = -yv * 0.5;
			xv *= 0.95;
			if (fabs(xv) + fabs(yv) < 0.8)
				hitSomething = 1;
		}
		y += yv;
		yv += _env->gravity * (100.0 / _global->frames_per_second);

		if (	(y > _global->screenHeight + 100) ||
			(x > _global->screenWidth + 100) ||
			(x < 0 - 100)) {
			hitSomething = 1;
			break;
		}
		if (checkPixelsBetweenTwoPoints (_global, _env, &startX, &startY, x, y)) {
			x = startX;
			y = startY;
			hitSomething = 1;
			break;
		}
#ifdef	AI_PLANNING_DEBUG
		// Plot trajectories for debugging purposes
		circlefill (screen, (int)x, (int)y, 2, color);
#endif
		ticks++;
	}
#ifdef	AI_PLANNING_DEBUG
	// Targetting circle for debugging purposes
	circlefill (screen, (int)x, (int)y, 10, color);
	circlefill (screen, (int)targetX, (int)targetY, 20, color);
#endif

	*estimateX = (int)x;
	*estimateY = (int)y;
	distance = vector_length_f ((float)(x - targetX), (float)(y - targetY), 0.0);
	if (ticks > maxTicks)
		_targetPower /= 2;
	if ((targetX - x < 0) && (direction < 0))
		distance = -distance;
	else if ((targetX - x > 0) && (direction > 0))
		distance = -distance;
	return ((int)distance);
}

int PLAYER::rangeFind (TANK *ctank, int *estimateX, int *estimateY, int maxAttempts)
{
	int findCount;
	int spread = weapon[ctank->cw].spread;
	int overshoot = 1;
	int z;

	findCount = 0;
	while (overshoot && (findCount < maxAttempts)) {
		spread = 1;
		for (z = 0; z < spread; z++) {
			double mxv,myv;
			int ca;

			ca = _targetAngle + ((SPREAD * z) - (SPREAD * (spread - 1) / 2));
			mxv = _global->slope[ca][0] * _targetPower * (100.0 / _global->frames_per_second) / 100;
			myv = _global->slope[ca][1] * _targetPower * (100.0 / _global->frames_per_second) / 100;

			overshoot = (int)(traceShellTrajectory (
				ctank,
				estimateX, estimateY,
				_targetX, _targetY,
				ctank->x + (_global->slope[ca][0] * GUNLENGTH) - mxv,
				ctank->y + (_global->slope[ca][1] * GUNLENGTH) - myv,
					mxv, myv));
			if (overshoot < 0) {
				_targetPower += (int)((rand () % (abs (overshoot) + 10)) * focusRate);
				if (_targetPower > MAX_POWER)
					_targetPower = 2000;
			} else {
				_targetPower -= (int)((rand () % (abs (overshoot) + 10)) * focusRate);
				if (_targetPower < 0)
					_targetPower = 0;
			}
			if (abs (overshoot) < abs (ctank->smallestOvershoot)) {
				ctank->smallestOvershoot = overshoot;
				ctank->bestPower = _targetPower;
				ctank->bestAngle = _targetAngle;
			}
			findCount++;
		}
	}
	return (overshoot);
}

int PLAYER::calculateDirectAngle (int dx, int dy)
{
	double angle;

	angle = atan2 ((double)dx, (double)dy) / PI * 180;
	angle += (rand () % 40 - 20) * errorMultiplier;

	if (angle < 0)
		angle = angle + 360;
	if (angle < 90)
		angle = 90;
	else if (angle > 270)
		angle = 270;

	return ((int)angle);
}

int PLAYER::calculateAngle (int dx, int dy)
{
	double angle;

	angle = atan2 ((double)dx, (double)dy - abs (dx)) / PI * 180;
	angle += (rand () % 40 - 20) * errorMultiplier;

	if (angle < 0)
		angle = angle + 360;
	if (angle < 90)
		angle = 135;
	else if (angle > 270)
		angle = 225;

	return ((int)angle);
}

int PLAYER::calculatePowerGivenAngle (int dx, int dy, int angle)
{
	double airTime, dxTime;
	double requiredPower;
	double slopeX, slopeY;
	double powerMult;
	double minYv, minV;

	slopeX = _global->slope[angle][1];
	slopeY = _global->slope[angle][0];

	/*
		The greater dx, the longer the missile will be in the air.
		Steeper angle, longer air time.
		The greater dy, the shorter air time.
		Double the gravity, double the power.
		Following wind = less power needed.
	*/


	// calculate minimum V to reach height dy
	if (dy > 0)
		minYv = sqrt (dy * _env->gravity * 2 * (100.0 / _global->frames_per_second));
	else
		minYv = 0;
	if ((slopeY != 0) && (minYv != 0))
		minV = fabs (minYv / slopeY);
	else
		minV = 0;
	if (slopeX != 0) {
		dxTime = (dx/fabs(slopeX));
	} else {
		// entirely down to the elements now
		dxTime = (dx / 0.000001);
	}
	//windAccel = (_env->wind / 20 * 0.2 * _env->viscosity);
	//dxTime = dxTime - (0.5 * windAccel * (dxTime*dxTime));
	if (dx < 0) {
		if (dxTime > 0)
			return (-1); // Won't ever get there
	} else {
		if (dxTime < 0)
			return (-1); // Won't ever get there
	}
	// Low target, less power
	// xdistance proportional to sqrt(dy)
	airTime = fabs(dxTime) + ((dy * slopeY) * _env->gravity * (100.0 / _global->frames_per_second)) * 2;

	// Less airTime doesn't necessarily mean less power
	// Horizontal firing means more power needed even though
	//   airTime is minimised
	powerMult = 1;
//	powerMult = (slopeX * slopeY);
//	if (powerMult == 0)
//		powerMult = 0.01;
	requiredPower = (sqrt (airTime * _env->gravity * (100.0 / _global->frames_per_second)) / powerMult) * 100;
	requiredPower += (int)((rand () % 100 - 50) * errorMultiplier);
	return ((int)requiredPower);
}

int PLAYER::adjustAngleGivenClearance (TANK *ctank, int xdist, int ydist, int clearance)
{
	while (!ctank->shootClearance (_targetAngle, clearance)) {
		if (xdist > 0) {
			if (_targetAngle < 270) {
				_targetAngle++;
			} else {
				_targetAngle += (int)((double)(rand () % 40 - 20) / (rangeFindAttempts + 1));
				break;
			}
		} else {
			if (_targetAngle > 90) {
				_targetAngle--;
			} else {
				_targetAngle += (int)((double)(rand () % 40 - 20) / (rangeFindAttempts + 1));
				break;
			}
		}
	}
	if (_targetAngle > 270)
		_targetAngle = 270 - (_targetAngle - 270);
	else if (_targetAngle < 90)
		_targetAngle = 90 + (90 - _targetAngle);

	return (_targetAngle);
}

/*
 * Add the estimated scores for a given tank with a given weapon
 *   to _targetMatrix
 * Take into account how much damage is likely to be done and whether it will
 *   be enough to destroy the tank.
 * Positional correction to account for wind effect on 2nd stage.
 */
int PLAYER::addTankDamageToMatrix (TANK *ctank, TANK *ltank, int weapNum)
{
	int radius = weapon[weapNum].radius;
	double wdamage = calcTotalEffectiveDamage (weapNum) *
					weapon[weapNum].spread;
	int tankX = (int)ltank->x;
	int tankY = (int)ltank->y;
	double windAffect;

	if ((weapNum >= RIOT_BOMB &&
		 weapNum <= HVY_RIOT_BOMB)) {
		// wdamage should probably be renamed to something
		//   more accurate
		wdamage = calcDefenseValue (ctank, ltank) *
			radius * (defensive + 1);
	}

	// Adjust tanks pos according to drag on submunition
	if (weapon[weapNum].submunition >= 0) {
		windAffect = (weapon[weapon[weapNum].submunition].drag * _env->wind / _env->gravity);
	} else {
		windAffect = 0;
	}

	if ( (ltank->player == this) ||
             ( (ltank->player->team == team) &&
               (team != TEAM_NEUTRAL) ) )
        {
		wdamage = -wdamage * painSensitivity;
		radius = (int)(radius * selfPreservation);
	} else if (ltank->player == revenge) {
		wdamage *= 1 + (double)vengeful / 50.0;
	}

	for (int cx = 0; cx < _global->screenWidth; cx++) {
		// Weapon specific calculations...
		int damage = 0;
		double distance = 0.0;   // just to avoid compiler warnings
		int cy = _env->surface[cx];
		int effectiveTankX = (int)(tankX + ((tankY - cy) * windAffect));
		if (weapNum >= RIOT_CHARGE && weapNum <= RIOT_BLAST) {
			// add points only within range and above ctank
			if (	(cy > ctank->y - (radius / 2)) ||
				(fabs (ctank->x - cx) > radius))
				continue;
		} else if (weapNum >= SML_ROLLER && weapNum <= DTH_ROLLER) {
			// Only aim rollers above other tanks
			if (cy > tankY + TANKHEIGHT &&
				fabs (ctank->x - cx) > radius)
				continue;
		} else if (weapNum >= SML_LAZER && weapNum <= LRG_LAZER) {
			// Lazer can only be aimed above horizontal
			if (cy > ctank->y)
				continue;
		} else if (weapNum >= SHAPED_CHARGE && weapNum <= CUTTER) {
			double ydist = abs ((cy - tankY) * 20);
			if (ydist > TANKHEIGHT/2)
				ydist -= TANKHEIGHT/2;
			else
				ydist = 0;
			distance = sqrt (pow ((float)cx - effectiveTankX, 2) + pow (ydist, 2));
		}

		if (weapNum < SHAPED_CHARGE || weapNum > CUTTER) {
			distance = sqrt (pow ((float)cx - effectiveTankX, 2) + pow ((float)cy - tankY, 2));
		}

		// Estimate fall damage
		if (cy - tankY > 0 && abs (cx - effectiveTankX) < radius) {
			if (ltank->player == this)
				damage -= radius - abs (cx - effectiveTankX);
			else
				damage += radius - abs (cx - effectiveTankX);
		}

		if (distance <= (radius + TANKHEIGHT/2)) {

			damage += (int) (wdamage * (1.0 - (distance / (radius + TANKHEIGHT/2)) / 2));
		}

		// Getting rid of a tank is tactically advantageous as 
		//   it is one less to shoot you.
		// This could probably use a personality affector
		//   (extent to which a player would rather finish a tank
		//   off than cause up to x amount of damage total).
		if (ltank->l + ltank->sh < damage) {
			if (ltank->player == this)
				damage -= 50;
			else
				damage += 50;
		}

		_targetMatrix[cx] += damage;
	}
	return (0);
}


/*
 * Calculate the damage matrix for the given weapon
 */
void PLAYER::calcDamageMatrix (TANK *ctank, int weapNum)
{
	int objCount = 0;
	TANK *ltank = NULL;

	for (int cx = 0; cx < _global->screenWidth; cx++)
		_targetMatrix[cx] = 0;

	while (((ltank = (TANK*)_env->getNextOfClass (TANK_CLASS, &objCount))) || ltank) {
		if (ltank->l <= 0)
			continue;
		if (weapNum < WEAPONS) {
			addTankDamageToMatrix (ctank, ltank, weapNum);
		} else {
			int itemNum = weapNum - WEAPONS;
			if (itemNum  >= ITEM_VENGEANCE && itemNum <= ITEM_FATAL_FURY) {
				// add sqrt distances for each tank * potential damage
				long int totalEffectiveDamage = calcTotalEffectiveDamage (itemNum);
				_targetMatrix[(int)ctank->x] += sqrt (fabs (ctank->x - ltank->x)) * totalEffectiveDamage;

			}
		}
		objCount++;
	}
}

double PLAYER::selectTarget ()
{
	int targetXCoord, targetYCoord;

	return (selectTarget (&targetXCoord, &targetYCoord));
}

double PLAYER::selectTarget (int *targetXCoord, int *targetYCoord)
{
	double highestScore = 0;
	int startAtX = rand () % _global->screenWidth;

	*targetXCoord = 0;
	for (int xCount = 0; xCount < _global->screenWidth; xCount++) {
		int xPos = (startAtX + xCount) % _global->screenWidth;
		if (_targetMatrix[xPos] > highestScore) {
			*targetXCoord = xPos;
			highestScore = _targetMatrix[xPos];
		}
	}
	*targetYCoord = _env->surface[*targetXCoord];

#ifdef	AI_PLANNING_DEBUG
	// Plot damage matrix for debugging purposes
	for (int xCount = 0; xCount < _global->screenWidth; xCount++) {
		vline (screen, xCount, _env->surface[xCount], _env->surface[xCount] + 20, makecol ((int)(_targetMatrix[xCount] / highestScore * 200), (int)(_targetMatrix[xCount] / highestScore * 200), (int)(_targetMatrix[xCount] / highestScore * 200)));
	}
#endif

	return (highestScore);
}

/*
 * calculate defense versus offense.
 * This can be based on turns to kill as a ratio (self to enemy).
 * If the ratio is too far in favour of the enemy, be defensive.
 * The ratio will largely be based on the armour of each tank as
 *   it's the only thing we know for sure. Perhaps later we could
 *   use known characteristics for the given player (fave weapon,
 *   defensiveness, accuracy etc.).
 */
double PLAYER::calcDefenseValue (TANK *ctank, TANK *ltank)
{
	// double strengthRatio;
	double armourRatio = 1.0;    // set a default, any default

	if (ctank->l > 0 && ltank->l > 0) {
		if (ctank->l + ctank->sh <= ltank->l + ltank->sh) {
			armourRatio = (double)(ctank->l + ctank->sh) /
					(double)(ltank->l + ltank->sh);
		} else {
			armourRatio = -(double)(ltank->l + ltank->sh) /
					(double)(ctank->l + ctank->sh);
		}
	}

	// strengthRatio = armourRatio;
	// return (strengthRatio);
        return armourRatio;
}

int PLAYER::computerSelectItem (TANK *ctank)
{
	int cw = 0; // Current Weapon

	if (ctank->howBuried () > 135) {
		if (nm[RIOT_BLAST] > 0)
			cw = RIOT_BLAST;
		else if (nm[RIOT_CHARGE] > 0)
			cw = RIOT_CHARGE;
		else if (nm[RIOT_BOMB] > 0)
			cw = RIOT_BOMB;
		else if (nm[HVY_RIOT_BOMB] > 0)
			cw = HVY_RIOT_BOMB;
		else if (ni[ITEM_TELEPORT] > 0)
			cw = WEAPONS + ITEM_TELEPORT;
		else if (nm[SML_LAZER] > 0)
			cw = SML_LAZER;
		else if (nm[MED_LAZER] > 0)
			cw = MED_LAZER;
		else if (nm[LRG_LAZER] > 0)
			cw = LRG_LAZER;
	}
	if (cw == 0) {
		double highestScore = -1;
		int highestScorer = 0;
		int targetXCoord, targetYCoord;
		for (cw = 0; cw < THINGS; cw++) {

			if (cw < WEAPONS) {
				if (!nm[cw]) {
					continue;
				} else {
					double score;
					calcDamageMatrix (ctank, cw);
					score = selectTarget (&targetXCoord,
							&targetYCoord);
					if (weapon[cw].cost > 0)
						score = (score *
							_weaponPreference[cw]) /
							weapon[cw].cost;
					if (score > highestScore) {
						highestScorer = cw;
						highestScore = score;
						_targetX = targetXCoord;
						_targetY = targetYCoord;
					}
				}
			} else {
				double score = 0.0;   // to avoid warnings
				int itemNum = cw - WEAPONS;

				if ((!ni[itemNum]) ||
					(!item[itemNum].selectable))
					continue;

				if (itemNum == ITEM_TELEPORT) {
					if (!ni[ITEM_PARACHUTE])
						continue;
				}

				if ((itemNum >= ITEM_VENGEANCE) &&
					(itemNum <= ITEM_FATAL_FURY)) {
					if (ctank->l + ctank->sh < 20) {
						calcDamageMatrix (ctank, cw);
						score = selectTarget ();
						score /= item[itemNum].cost;
					}
				}
				if (score > highestScore) {
					highestScorer = cw;
					highestScore = score;
				}
			}
			//break;
		}
		cw = highestScorer;
	}

	if ((cw < WEAPONS && !nm[cw]) || (cw >= THINGS) || (!ni[cw - WEAPONS]))
		cw = 0;

	ctank->cw = cw;
	_global->updateMenu = 1;

	return (cw);
}

int PLAYER::computerControls (TANK *ctank)
{
	// At the most basic: select target, select weapon, aim and fire
	ctank->requireUpdate ();
	if (_turnStage == SELECT_WEAPON) {
		computerSelectItem (ctank);
		_turnStage++;

	} else if (_turnStage == SELECT_TARGET) {
		selectTarget ();
		_turnStage++;

	} else if (_turnStage == CALCULATE_ATTACK) {
		int overshoot;
		int lastOvershoot = 0;    // avoid compiler warning
		int estimateX, estimateY;
		int xdist = (int)(_targetX - ctank->x);
		int ydist = (int)(_targetY - ctank->y);

		ctank->smallestOvershoot = _global->screenWidth;
		if (ctank->howBuried () > 135) {
			_targetAngle = 180;
			_targetPower = MAX_POWER / 10;
		} else {
			if (ctank->cw >= SML_LAZER && ctank->cw <= LRG_LAZER) {
				_targetAngle = calculateDirectAngle (xdist, ydist);
			} else {
				_targetAngle = calculateAngle (xdist, ydist);
				_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, 150);
				_targetPower = calculatePowerGivenAngle (xdist, ydist, _targetAngle);
				if (rangeFindAttempts > 0)
					overshoot = rangeFind (ctank, &estimateX, &estimateY, rangeFindAttempts);
				else
					overshoot = 0;
				if (abs (overshoot) > weapon[ctank->cw].radius) {
					for (int retargetCount = 0; (retargetCount < retargetAttempts) && (abs (overshoot) > weapon[ctank->cw].radius); retargetCount++) {
						if (overshoot != 0 && (float)abs (overshoot - lastOvershoot) / abs (overshoot) < 0.01) {
							// There could be a
							//   more useful test?
							_targetAngle = (rand () % 180) + 90;
							_targetPower = MAX_POWER / 2;
						} else {
							/*computerSelectItem (ctank);
							selectTarget (&_targetX, &_targetY);
							xdist = (int)(_targetX - ctank->x);
							ydist = (int)(_targetY - ctank->y);
							*/

							_targetAngle = calculateAngle (xdist, ydist);
							_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, 150);
							_targetPower = calculatePowerGivenAngle (xdist, ydist, _targetAngle);
						}
						if (rangeFindAttempts > 0)
							overshoot = rangeFind (ctank, &estimateX, &estimateY, rangeFindAttempts);
						else
							overshoot = 0;
						lastOvershoot = overshoot;
					}
				}
				if (abs (ctank->smallestOvershoot) > weapon[ctank->cw].radius * 3) {
					_targetAngle = calculateAngle (xdist, ydist);
					_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, (int)(weapon[ctank->cw].radius * 1.5));
					_targetPower = (MAX_POWER / 2);
				} else {
					_targetAngle = ctank->bestAngle;
					_targetPower = ctank->bestPower;
				}
			}
			/*if (_targetAngle == -1) {
				computerSelectItem (ctank);
				selectTarget (&_targetX, &_targetY);
				_turnStage--;
			}*/
		}
		_turnStage++;

	} else if (_turnStage == AIM_WEAPON) {
		if (_targetAngle > ctank->a && ctank->a < 270) {
			// Left
			ctank->a++;
			_global->updateMenu = 1;
		} else if (_targetAngle < ctank->a && ctank->a > 90) {
			// Right
			ctank->a--;
			_global->updateMenu = 1;
		} else if (_targetPower < (ctank->p - 3) && ctank->p > 0) {
			// Reduce power
			ctank->p -= 5;
			_global->updateMenu = 1;
		} else if (_targetPower > (ctank->p + 3) && ctank->p < MAX_POWER) {
			// Increase Power
			ctank->p += 5;
			_global->updateMenu = 1;
		} else {
			_turnStage++;
		}

	} else if (fi) {
		// if (fi) don't do any of the following
	} else if (_turnStage == FIRE_WEAPON) {
		// ctank->activateCurrentSelection ();
                ctank->simActivateCurrentSelection();
                gloating = false;
		_turnStage = 0;
	}
	return (0);
}

int PLAYER::humanControls (TANK *ctank)
{
	if (ctank) {
		if (!_env->mouseclock && mouse_b & 1 && mouse_x >= 250
			&& mouse_x < 378 && mouse_y >= 11 && mouse_y < 19) {
			_global->updateMenu = 1;
			if (ctank->fs) {
				ctank->sht++;
			}
			ctank->fs = 1;
			if (ctank->sht > SHIELDS - 1) {
				ctank->sht = 0;
			}
		}
		if (!_env->mouseclock && mouse_b & 1 && mouse_x >= 250
			&& mouse_x < 378 && mouse_y >= 21
			&& mouse_y < 29 && ctank->player->ni[ctank->sht] > 0 && (ctank->fs || ctank->sh > 0)) {
			_global->updateMenu = 1;
			ctank->ds = ctank->sht;
			ctank->player->ni[ctank->sht]--;
			ctank->sh = (int)item[ctank->sht].vals[SHIELD_ENERGY];
		}
	
		ctank->requireUpdate ();
	}
	
	//Keyboard control
	if (!_env->stage) {
		if (ctank) {
			if (key[KEY_LEFT] && !ctrlUsedUp && ctank->a < 270) {
				ctank->a++;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_RIGHT] && !ctrlUsedUp && ctank->a > 90) {
				ctank->a--;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_DOWN] && !ctrlUsedUp && ctank->p > 0) {
				ctank->p -= 5;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_UP] && !ctrlUsedUp && ctank->p < MAX_POWER) {
				ctank->p += 5;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
                        if ( key[KEY_PGUP] && !ctrlUsedUp && ctank->p < MAX_POWER )
                        {
                            ctank->p += 100;
                            if (ctank->p > MAX_POWER)
                               ctank->p = MAX_POWER;
                            _global->updateMenu = 1;
			    if (key_shifts & KB_CTRL_FLAG)
				ctrlUsedUp = TRUE;
                        }
                        if ( key[KEY_PGDN] && !ctrlUsedUp && ctank->p > 0)
                        {
                            ctank->p -= 100;
                            if (ctank->p < 0)
                               ctank->p = 0;
                            _global->updateMenu = 1;
		            if (key_shifts & KB_CTRL_FLAG)
			      ctrlUsedUp = TRUE;
                        }
		}
	}
	if (k && !fi) {
		if (!_env->stage) {
			if (ctank) {
				if (k >> 8 == KEY_N) {
					ctank->a = 180;
					_global->updateMenu = 1;
				}
				if (k >> 8 == KEY_TAB) {
					_global->updateMenu = 1;
					while (1) {
						ctank->cw++;
						if (ctank->cw >= THINGS)
							ctank->cw = 0;
						if (ctank->cw < WEAPONS) {
							if (ctank->player->nm[ctank->cw])
								break;
						} else {
							if (item[ctank->cw - WEAPONS].selectable && ctank->player->ni[ctank->cw - WEAPONS])
								break;

						}
					}
					//calcDamageMatrix (ctank, ctank->cw);
					//selectTarget ();
                                        changed_weapon = false;
				}
                                if ( k >> 8 == KEY_BACKSPACE) {
                                        _global->updateMenu = 1;
                                        while (1) {
                                                ctank->cw--;
                                                if (ctank->cw < 0)
                                                   ctank->cw = THINGS - 1;
 
                                                if (ctank->cw < WEAPONS) {
                                                        if (ctank->player->nm[ctank->cw])
                                                                break;
                                                } else {
                                                        if (item[ctank->cw - WEAPONS].selectable && ctank->player->ni[ctank->cw - WEAPONS])
                                                                break;

                                                }
                                        }
                                        changed_weapon = false;
                                }

                                // put the tank under computer control
                                if ((k >> 8 == KEY_F10) )
                                {
                                    type = PART_TIME_BOT;
                                    setComputerValues();
                                    return (computerControls (ctank));
                                }

				if ((k >> 8 == KEY_SPACE) &&
					(((ctank->cw < WEAPONS) && (ctank->player->nm[ctank->cw] > 0)) ||
						((ctank->cw < THINGS) && (ctank->player->ni[ctank->cw - WEAPONS] > 0)))) {
				//	ctank->activateCurrentSelection ();
                                ctank->simActivateCurrentSelection();
                                gloating = false;
				}
			}
		}
		if ((_env->stage == 3) && (k >> 8 == KEY_ENTER || k >> 8 == KEY_ESC || k >> 8 == KEY_SPACE))
			return (-1);
	}
	return (0);
}



// returns a static string to the player's team name
char *PLAYER::Get_Team_Name()
{
    char *team_name = "";

    switch ( (int) team)
    {
       case TEAM_JEDI: team_name = "Jedi"; break;
       case TEAM_NEUTRAL: team_name = "Neutral"; break;
       case TEAM_SITH: team_name = "Sith"; break;
    }

    return team_name;
}

