/*
  Segment.C
  Uros Platise, Dec 1998
  
  Even more powerful segments are provided that in earilier uAsm.
  Any segment can now also be ref. as abstract segment or removable.
  
  * Abstract segments are those which are not put into executable file.
  
  * Removable segments allow additional feature on optimizing the size
    of the executable file. If there is no call to removable segment,
    it is simply removed.
*/

#ifdef WIN32
#pragma warning( disable : 4786)
#pragma warning( disable : 4800)
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Global.h"
#include "Lexer.h"
#include "Symbol.h"
#include "Object.h"
#include "Segment.h"
#include "Reports.h"

const int TSegDep::LOG_BITPR = 5; 	/* log of bits per record -- 
                                           this may vary in the future */
const int TSegDep::NAMES_PER_LINE = 4;	/* Used by Print() function */			      

TSegment::TSegment():
  segUsageCnt(1),fitted(false),errorActive(false) {
  strcpy(segRec[SEG_ROOT].name, "[root]");
  segRec[SEG_ROOT].align=1;
  segRec[SEG_ROOT].abs=0;
  close();
}

void TSegment::close(){
  cseg=&segRec[SEG_ROOT];
}

/* Returns true if child is created. */
bool TSegment::set(const char* name){
  TsegRecCI refidx;
  TSegmentRec* cseg_bck=cseg;
  
  /* find segment within refs ... */
  for (refidx=cseg->segRef.begin();refidx!=cseg->segRef.end(); refidx++){
    if (strcmp(name,(*refidx)->name)==0){
      cseg=*refidx;
#ifdef DEBUG      
      reports.Info(TReports::VL_SEG4,"set=%s\n",cseg->name);
#endif      
      return false;
    }
  }  
  /* create child */  
  if (refidx==cseg->segRef.end()){
    if (segUsageCnt==MAX_SEGMENTS){
      throw generic_error("Maximum number of segments is reached.\n"
                        "Increase MAX_SEGMENTS variable in Segment.h file.");
    }  
    cseg->segRef.push_back(&segRec[segUsageCnt]);
    cseg = &segRec[segUsageCnt++];
    cseg->parent = cseg_bck;
    
    /* If parent is removable, child is also removable and so abstract */
    cseg->flags = (TSegmentRec::TFlags)(cseg->flags | cseg_bck->flags);
    
    /* Assign Name and Label */    
    strncpy(cseg->name, name, SEG_MAXNAMELEN);
    if (cseg->name [SEG_MAXNAMELEN-1] != 0){
      throw syntax_error("Segment name too long: ", name);
    }
    
    if ((strlen(cseg->name)+strlen(cseg_bck->label)+1)>=SEG_LABELLEN){
      throw syntax_error("Segment label name too long: ", cseg->label);
    }
    strcat(cseg->label,cseg_bck->label);
    strcat(cseg->label,"_");
    strcat(cseg->label,name);
    
    /* If parent alignment size differs and our is not 1 report error.
       Otherwise if our is 1, take parents one 
    */       
    if (cseg->align==SEG_UNKNOWNALIGN){
      cseg->align=cseg_bck->align;
      
      /* Update alignment parameter of the mirror segment */
      if (cseg->mirrorType){segRec[cseg->mirrorSegNo].align = cseg->align;}
    }
    if (cseg->align!=cseg_bck->align && cseg->align!=1 && 
        cseg_bck->size == SEG_UNKNOWNSIZE && 
	cseg->parent != &segRec[SEG_ROOT]){
      throw syntax_error("\n  Children of not fixed size segments may be"
        " aligned to one byte only.\n  Bad alignment for: ", cseg->name);
    }

    /* create external symbol (to satisfy macro check) */
    symbol.addInternalLabel(cseg->label);
  }
  return true;
}

void TSegment::setPrevious(){
  assert(cseg->parent!=NULL); cseg=cseg->parent;
}

/* Update current segment attributes, size and abs */
void TSegment::update(const TSegmentRec& tmpSeg){
  /* adjust maximum size */
  if (cseg->size < tmpSeg.size){cseg->size=tmpSeg.size;}
  
  cseg->PC += tmpSeg.PC;	  /* add current PC */
  
  /* absolute address */
  if (cseg->abs!=-1 && tmpSeg.abs!=-1 && cseg->abs!=tmpSeg.abs){
    throw syntax_error("Absolute address already declared for: ",cseg->name);
  }
  if (tmpSeg.abs!=-1){cseg->abs=tmpSeg.abs;}

  /* if stored segment is not yet abstract and segTmp is, make cseg too */
  cseg->flags = 
    (TSegmentRec::TFlags)(cseg->flags | (tmpSeg.flags & TSegmentRec::Abstract));
  
  /* if stored segment is removable but this one is not, clear removable flag*/
  cseg->flags = 
    (TSegmentRec::TFlags)(cseg->flags & ((0xff-TSegmentRec::Removable) |
    (tmpSeg.flags & TSegmentRec::Removable)));
  
  /* check align size */
  if (cseg->align!=tmpSeg.align && tmpSeg.align!=SEG_UNKNOWNALIGN){
    reports.Info(TReports::VL_SEG1,"align: current=%d, new=%d\n", 
      cseg->align, tmpSeg.align);
    throw syntax_error("Align size missmatch for segment: ", cseg->name);
  }    
  
  /* Mirror Updates */
  if (cseg != &tmpSeg){
    
    /* mirror redeclaration check if and only 
       if tmpSeg is not existing segment 
    */
    if (cseg->mirrorType && tmpSeg.mirrorType){
      throw syntax_error("Mirror segment was already declared for: ",
	cseg->name);
    }
    
    /* if everything is okay, update mirror fields */
    if (tmpSeg.mirrorType){
      cseg->mirrorType = tmpSeg.mirrorType;
      cseg->mirrorSegNo = tmpSeg.mirrorSegNo;
      cseg->update_mirror_no = tmpSeg.update_mirror_no;
    }
  }
}

/*
  New segment is build upon segRec[segUsageCnt] what was suggested
  by the set function.
*/
bool TSegment::parse(){  
  if (strcmp(lxP->string,"seg")!=0){return false;}

  TSegmentRec& segTemplate = segRec[segUsageCnt];  
  segTemplate.clear();
  
  /* parse flags */
  WHILE_TOKEN{
    if (lxP->type!=TlxData::STRING){
      throw syntax_error("Bad format",lxP->string);}
            
    if (strcmp(lxP->string,"removable")==0){
      segTemplate.flags=
        (TSegmentRec::TFlags)(segTemplate.flags|TSegmentRec::Removable);
    }
    else if (strcmp(lxP->string,"abstract")==0){
      segTemplate.flags=
        (TSegmentRec::TFlags)(segTemplate.flags|TSegmentRec::Abstract);
    }
    else if (strcmp(lxP->string,"abs")==0){segTemplate.abs=parseValue();}
    else if (strcmp(lxP->string,"size")==0){segTemplate.size=parseValue();}
    else if (strcmp(lxP->string,"align")==0){segTemplate.align=parseValue();}
    else if (strcmp(lxP->string,"mirror")==0){
      GET_TOKEN;
      if (strcmp(lxP->string,"=")){
        throw syntax_error("Bad format at ",lxP->string);
      }
      parseSegmentName(true);      
      segTemplate.mirrorType = TSegmentRec::NoCompression;
      segTemplate.mirrorSegNo = getSegNo();
    }
    else{break;}
  }
  PUT_TOKENBACK;
  
  /* parse segment name with its segTemplate */
  parseSegmentName();
  update(segTemplate);
  
  char buf [SEG_LABELLEN];  
  object.outSegment(cseg-segRec);
  object.outPCMarker(getPC(buf));  
  return true;
}

long TSegment::parseValue(){
  /* expecting = */
  GET_TOKEN;
  if (strcmp(lxP->string,"=")){
    throw syntax_error("Bad format at ",lxP->string);
  }
  GET_TOKEN;
  syntax.Parse_GAS(1);
  if (gas.status!=TGAS::Solved){
    throw syntax_error("Value expected at ",lxP->string);
  }
  PUT_TOKENBACK;
  return gas.result();  
}

void TSegment::parseSegmentName(bool mirror){  
  close();			/* go to root first */
  do{
    lexer._gettoken();
    if (lxP->type!=TlxData::STRING){
      throw syntax_error("Segment line without segment name.");
    }
    bool newSeg = set(lxP->string);
    if (mirror && newSeg){
      throw segment_error("Mirror segment is not declared yet: ",lxP->string);
    }
    GET_TOKEN;	/* get delimiter - full stop, which seperates segments ... */
  }while(lxP->type==TlxData::CONTROL && lxP->string[0]=='.');  
  PUT_TOKENBACK;
}

/* 
  It returns current program counter relative to segment
*/
char* TSegment::getPC(char* PC_str){
  if (cseg==NULL || cseg==segRec){
    throw segment_error("Code is present out of segment.");
  }
  sprintf(PC_str,"(%s+0x%lX)",cseg->label,cseg->PC);
  return PC_str;
}

void TSegment::incPC(long relative){
  if (cseg==NULL || cseg==segRec){
    throw segment_error("Code is present out of segment.");
  }
  cseg->PC+=relative;
  if (cseg->size!=-1 && cseg->PC > cseg->size){
    throw segment_error("Segment full: ",cseg->name);
  }  
}

bool TSegment::isEnabled(int segNo){
  return !(segRec[segNo].status & TSegmentRec::Removed);
}

bool TSegment::isAbstract(int segNo){
  return (segRec[segNo].flags & TSegmentRec::Abstract);
}


/*
  Remove unused segments according to the Segment Dependencies.
  Loop until all propagations are done. [slow f. - could be opt]
*/
void TSegment::removeUnused(){
  bool propagate;
  TsegRecCI segRecCI;
  
  /* remove themselves from the dependency list */
  for (int si=1;si<segUsageCnt;si++){
    segRec[si].dep.Remove(si);
  }
  
  /* remove dead segments */
  do{
    propagate=false;
    for (int si=1;si<segUsageCnt;si++){

      if (segRec[si].dep.Empty() && 
          (segRec[si].flags & TSegmentRec::Removable) &&
          !(segRec[si].status & TSegmentRec::Removed)){	  	  
	  
	/* check if children are removed */
        for (segRecCI=segRec[si].segRef.begin(); 
	     segRecCI != segRec[si].segRef.end() && 
	       ((*segRecCI)->status & TSegmentRec::Removed);
	     segRecCI++);
	     
	if (segRecCI!=segRec[si].segRef.end()){continue;}
	  
	segRec[si].status=
	  (TSegmentRec::TStatus)(segRec[si].status|TSegmentRec::Removed);
	  
	symbol.undefine(segRec[si].label);

	/* remove this segment from other segments dep list */
	for(int j=1;j<segUsageCnt;j++){segRec[j].dep.Remove(si);}	
	propagate=true;
      }
    }
  }while(propagate);
}


/* FITTER */

/* segment own size plus size of all childs
   check for badly removed segments! 
   level=0 (ROOT_SEG), level=1 (BASE_SEGMENTS), level=2 (CHILDS of BASE_SEG)
*/
void TSegment::findSum(TSegmentRec* segp, int level){
#ifdef DEBUG
  reports.Info(TReports::VL_SEG4,"findSum()\n");
#endif  

  if (segp->status & TSegmentRec::Removed){segp->sumSize=0;return;}
  
  /* For all childs flash.ch1, flash.ch2, ... do the following:
     if segment is fixed size, add ->size parameter otherwise PC 
  */
  if (segp->size!=SEG_UNKNOWNSIZE && level>=2){
    segp->sumSize+=segp->size;
  } else{
    segp->sumSize += segp->PC;
  }
  
  TsegRecCI segRecCI;
  for (segRecCI=segp->segRef.begin();segRecCI!=segp->segRef.end();segRecCI++){
    findSum(*segRecCI, level+1); segp->sumSize += (*segRecCI)->sumSize;
  }

  /* if segment's maximum size is not defined, let's define it */
  if (segp->size==SEG_UNKNOWNSIZE){segp->size=segp->sumSize;}   
        
  /* check range */
  if (segp->sumSize > segp->size){
    segp->status = (TSegmentRec::TStatus)(segp->status | TSegmentRec::OutofRange);errorActive=true;
    reports.Info(TReports::VL_ALL,"Not enough space for `%s': "
      "req=%ld avail=%ld bytes\n",
      segp->name, segp->sumSize, segp->size);
  }
#ifdef DEBUG
  reports.Info(TReports::VL_SEG4," %s: sum = %ld, size= %ld\n", 
    segp->name, segp->sumSize, segp->size);
#endif    
}

/*
  Before calling findsum function, mirror segment must
  be declared.
*/
void TSegment::CalcMirrors(){
  /* set mirrors */
  for (int i=0; i < segUsageCnt; i++){
    if (segRec[i].mirrorType & TSegmentRec::NoCompression){
      /* get new segment number and set PC */
      
      /* Appropriate Segment Number is defined in the loadSegment() */
      /*
      segRec[i].mirrorSegNo = 
        preproc.csrc->segP->seg[segRec[i].mirrorSegNo].newNo;
      */
      segRec[segRec[i].mirrorSegNo].PC = segRec[i].PC;
      
      /* Set current _MIRROR_ segment active and create children */
      cseg = &segRec[segRec[i].mirrorSegNo];
      CopyMirrorChildren(&segRec[i]);
    }
    if (segRec[i].mirrorType & TSegmentRec::RLE1){
      throw generic_error("RLE1 is not supported yet.\n");}
  }
}

/* Mirror children of the original segment.
*/
void TSegment::CopyMirrorChildren(TSegmentRec* p){
  TsegRecI segRecI;
    
  for (segRecI = p->segRef.begin(); segRecI != p->segRef.end(); segRecI++){
  
    /* Child found. Skip removed segments. */
    if ((*segRecI)->status & TSegmentRec::Removed){continue;}
        
    reports.Info(TReports::VL_SEG4, "Mirroring child: %s\n", (*segRecI)->name);

    /* set data structure for child: copy original attr. to the mirror */
    TSegmentRec& segTemplate = segRec[segUsageCnt];  
    segTemplate.clear();    
    segTemplate.align       = cseg->align;
    segTemplate.mirrorType  = TSegmentRec::NoMirror;
    segTemplate.flags       = cseg->flags;
    segTemplate.abs         = cseg->abs;
    segTemplate.size        = cseg->size;
    segTemplate.PC          = 0;    
    
    /* assign mirror */
    (*segRecI)->mirrorType = ((*segRecI)->parent)->mirrorType;
    (*segRecI)->mirrorSegNo = segUsageCnt;
    
    /* create child and go in */
    if (set((*segRecI)->name)==false){
      throw syntax_error("Cannot mirror segment: ", (*segRecI)->name);
    }
    CopyMirrorChildren(*segRecI);
    setPrevious();
  }  
}


/* Segment Child Sort Function

   - Abstract segments are pushed after non-Abstract segs.
   - and higher absolute value to the bottom
*/
#define RFS_FUNC(abs, ntype, type)	((!(ntype) & (abs)) | (type))

void TSegmentRec::sortRefs(){
  vector< TSegmentRec* >::iterator ri;
  bool update=true;
  bool ntype;
  while(update==true){
    update=false; 
    for (ri=segRef.begin(); (ri+1) < segRef.end(); ri++){
    
      ntype = ((*ri)->flags&TSegmentRec::Abstract) <
	            ((*(ri+1))->flags&TSegmentRec::Abstract);
/*		    
      printf("%s,%lx,%d / %s,%lx,%d  (ntype=%d)\n", 
        (*ri)->name, (*ri)->abs, (*ri)->flags, 
	(*(ri+1))->name, (*(ri+1))->abs, (*(ri+1))->flags, ntype);
*/	
      if (RFS_FUNC( (*ri)->abs > ((*(ri+1))->abs),
      		    ntype,
      		    ((*ri)->flags&TSegmentRec::Abstract) >
	            ((*(ri+1))->flags&TSegmentRec::Abstract) )){
        update=true; 
	TSegmentRec* buf=*ri; *ri=*(ri+1); *(ri+1)=buf;
      }
    }
  }
}

void TSegment::fit(TSegmentRec* segp){
  /* if removed or error active, skip it! */
  if (segp->status & TSegmentRec::Removed ||
      segp->status & TSegmentRec::OutofRange){return;}
#ifdef DEBUG
  reports.Info(TReports::VL_SEG4,"\nfitting `%s'\n", segp->name);
#endif  
  TMicroStack<TSSRec> notPlaced(segp->segRef.size());
  TMicroStack<TSSRec> freeArea(segp->segRef.size()+1);
  TsegRecI sgci;
  
  /* sort already fixed segments by its absolute address */  
  segp->sortRefs();
 
  /* get starting address */
  long stAddr, stEnd;
  if (segp->abs==SEG_NOTPLACED){
    throw syntax_error("Absolute address of the base segment is not defined: ",
      segp->name);
  }
  stAddr = segp->start();
  
  /* get free area and fixed (already placed) segments
     Since refs are sorted by abs, not placed segments will come first 
  */
  for(sgci=segp->segRef.begin(); sgci != segp->segRef.end(); sgci++){  
    if ((*sgci)->abs==SEG_NOTPLACED){
    
      if (!((*sgci)->status & TSegmentRec::Removed)){      
        /* remove zero bytes segments and set their offset to the end
	   of the parent code */
	if ((*sgci)->sumSize==0){
	  (*sgci)->abs = stAddr;
	} else {
          notPlaced.push(TSSRec((*sgci)->sumSize,*sgci));	 
	}
      }
    }	
    else{
      stEnd = (*sgci)->abs;
      if ((stEnd-stAddr) > 0){freeArea.push(TSSRec(stAddr,stEnd));}
      else if ((stEnd-stAddr)<0){
        (*sgci)->status = 
	  (TSegmentRec::TStatus)((*sgci)->status | TSegmentRec::Collision);
	errorActive=true;
      }
      stAddr = (*sgci)->end();
      if ((segp->size-stAddr)<0){
        (*sgci)->status = 
	  (TSegmentRec::TStatus)((*sgci)->status | TSegmentRec::OutofRange);
	errorActive=true;
      }
    }
  }  
  /* add free tail */  
  stEnd = segp->abs + segp->size;
  if ((stEnd-stAddr) > 0){freeArea.push(TSSRec(stAddr,stEnd));}

#ifdef DEBUG    
  /* print free space */
  for(int i=0;i<freeArea.size();i++){
    reports.Info(TReports::VL_SEG4,"free: [$%lx,$%lx]\n",
      freeArea[i].start,freeArea[i].end());
  }      
#endif     
  
  /* fit floating segments - bigger to bigger holes in the front */
  notPlaced.sort();
  TSSRec *freeSeg, *npSeg;	/* ref to: free and notplaced segment */
  while(notPlaced.size()>0){    
    freeArea.sort();
    if (freeArea.size()==0){
      errorActive=true;
      while(notPlaced.size()>0){      
        notPlaced.top().ref->status =
	  (TSegmentRec::TStatus)(notPlaced.top().ref->status | 
	  TSegmentRec::OutofRange);
	  
	notPlaced.pop();
      }
      break;
    }
    freeSeg = &freeArea.top(); npSeg = &notPlaced.top();
        
    /* take top one and put into top free space -
       if operation fails, mark all remaining segments OutofRange 
    */
#ifdef DEBUG       
    reports.Info(TReports::VL_SEG4,"alloc: [$%lx,$%lx] $%lx, $%lx\n", 
      freeSeg->start, freeSeg->end(), freeSeg->size, npSeg->size);
#endif
    /* calculate alignment */
    int align_bytes = npSeg->ref->align;
    align_bytes    -= (freeSeg->start % align_bytes);
    align_bytes    %= npSeg->ref->align;

    reports.Info(TReports::VL_SEG2,
      "Aligning segment `%s' (align=%d) with %d offset byte(s).\n",
      npSeg->ref->name, npSeg->ref->align, align_bytes);

    /* The free memory hole must take in account extra bytes for alignment.
    */
    if (freeSeg->size < (npSeg->size + align_bytes)){
      errorActive=true;
      while(notPlaced.size()>0){
        notPlaced.top().ref->status = 
	  (TSegmentRec::TStatus)(notPlaced.top().ref->status | 
	  TSegmentRec::OutofRange);
	notPlaced.pop();
      }
    } else{
#ifdef DEBUG    
      reports.Info(TReports::VL_SEG4,"setting seg: `%s' with size = $%lx\n", 
                   npSeg->ref->name, npSeg->ref->sumSize);
#endif
      npSeg->ref->abs = freeSeg->start + align_bytes;
      freeSeg->start += npSeg->ref->sumSize + align_bytes;
      freeSeg->size  -= npSeg->ref->sumSize + align_bytes; 
      notPlaced.pop();
    }
  }  
  /* Fit all childs too */
  for(sgci=segp->segRef.begin();sgci!=segp->segRef.end();sgci++){fit(*sgci);}
}

void TSegment::fitter(){  
  removeUnused();
  CalcMirrors();
  findSum(segRec);
  if (!errorActive){
    /* fit every sub-segment of the root segment separately */
    TsegRecCI segRecCI=segRec[SEG_ROOT].segRef.begin();
    for (;segRecCI!=segRec[SEG_ROOT].segRef.end(); segRecCI++){fit(*segRecCI);}
    fitted=true;
  }
  Report();
  if (errorActive==true){
    throw generic_error("Cannot fit all segments."
      " See segment tree for details.");
  }
}

/* Adjust Segment Labels to Current File */
void TSegment::adjustSegments(){
  char labelValue [LX_STRLEN];
  for (int si=1; si<segUsageCnt; si++){
    if (segRec[si].status & TSegmentRec::Removed){continue;}
    sprintf(labelValue,"0x%lX",segRec[si].abs+preproc.csrc->segP->seg[si].rel);
    symbol.modifyValue(segRec[si].label, labelValue);
  }
}

/*
  After loading segments from current object files map
  dep. bitmap. This part could be optimized.
*/
void TSegment::MapSegmentDependencies(){
  for (int si=1; si<segUsageCnt; si++){
    if (!segRec[si].dep.Mapped()){
      segRec[si].dep.MapSegments(*preproc.csrc->segP);
    }
  }
}


#define SAVE_NONE	0
#define SAVE_ROOTINFO	1
#define SAVE_GOBACK	2

void TSegment::saveSegments(){
  if (cseg==NULL){
    reports.Warnning(TReports::Segments,"File does not contain program code.");
    return;
  }
  saveSegmentTree(&segRec[SEG_ROOT]);
} 

void TSegment::saveSegmentTree(TSegmentRec* p){
  if (p!=segRec){
    p->dep.SaveObj();
    object.outOperand(p->PC);
    object.outOperand(p->size);
    object.outOperand(p->abs);
    object.outOperand(p->flags);
    object.outOperand(p->mirrorType);
    object.outOperand(p->mirrorSegNo);
    object.outOperand(p->align);
    object.outOperand(p-segRec); 		/* report segment number! */
    object.outStringOperand(p->name);;
    object.outSegmentData(SAVE_ROOTINFO);	/* root information */
  }
  TsegRecCI segRecCI;
  for (segRecCI=p->segRef.begin(); segRecCI != p->segRef.end(); segRecCI++){
    if (!((*segRecCI)->status & TSegmentRec::Removed)){
      saveSegmentTree(*segRecCI);}
  }
  if (p!=segRec){object.outSegmentData(SAVE_GOBACK);}	/* one back ... */
}

void TSegment::loadSegment(int segNo, bool end_of_load){
  if (!end_of_load){
    TSegmentRec& segTemplate = segRec[segUsageCnt];
    
    /* clear template */
    segTemplate.clear();
    
    if (segNo==SAVE_GOBACK){setPrevious();return;}
    if (segNo!=SAVE_ROOTINFO){
      throw syntax_error("Invalid segment direction: Object is corrupted.");
    }        
    int objSegNo            = object.popOperand();
    segTemplate.align       = (int)object.popOperand();
    segTemplate.mirrorSegNo = (int)object.popOperand();
    segTemplate.mirrorType  = (TSegmentRec::TMirror)object.popOperand();
    segTemplate.flags       = (TSegmentRec::TFlags)object.popOperand();
    segTemplate.abs         = object.popOperand();
    segTemplate.size        = object.popOperand();
    segTemplate.PC          = object.popOperand();
    segTemplate.dep.LoadObj();
  
    /* Does segment need mirror update? */  
    if (segTemplate.mirrorType != TSegmentRec::NoMirror){
      segTemplate.update_mirror_no = true;
    }        

//    printf("%d:", segUsageCnt);

    /* ADD SEGMENT
       update cross-files segment numbers and relative program counter 
    */
    if (set(object.popStringOperand())==false){
      preproc.csrc->segP->seg[cseg-segRec].rel = cseg->PC;
      update(segTemplate);
    }
    
    preproc.csrc->segP->seg[objSegNo].newNo = cseg-segRec;
/*    
    printf("`%s' PC=$%lx, nN=%d, cPx=%d, mirror=%d, seg[mirror]=%d\n", 
      cseg->name, 
      preproc.csrc->segP->seg[objSegNo].rel,
      preproc.csrc->segP->seg[objSegNo].newNo,
      segTemplate.mirrorType, 
      segTemplate.mirrorSegNo,
      preproc.csrc->segP->seg[segTemplate.mirrorSegNo].newNo);
*/
  }
  else{	/* update mirror segments in this file */
    for (int i=1; i<segUsageCnt; i++){
      if (segRec[i].update_mirror_no == true){
/*
        int d = preproc.csrc->segP->seg[segRec[i].mirrorSegNo].newNo;	
	printf("(%s,%s): updating mirror: %d (%s,%s) -> %d (%s,%s)\n", 
	  TellBaseSegment(i), segRec[i].name,
	  segRec[i].mirrorSegNo, 
	  TellBaseSegment(segRec[i].mirrorSegNo),
	  segRec[segRec[i].mirrorSegNo].name,
          d, 
	  TellBaseSegment(d),
	  segRec[d].name);
*/
        segRec[i].update_mirror_no = false;  
	segRec[i].mirrorSegNo =
          preproc.csrc->segP->seg[segRec[i].mirrorSegNo].newNo;
      }
    }  
  }
}

void TSegment::Report(){
  /* Print Segment Tree */
  reports.Info(TReports::VL_SEG1, "\n%-30.30s   %8.8s   %8.8s %8.8s\n", 
    "SEGMENT TREE:", "ABS", "SIZE", "FLAGS");
  ReportSegmentTree(segRec,0);
  reports.Info(TReports::VL_SEG1, "\n");
  
  /* Overall Segment Information */
  if (fitted==false){return;}
  reports.Info(TReports::VL_SEG1, "%-30.30s   %10.10s %10.10s %10.10s\n", 
    "OVERALL SEGMENT INFO:", "SUM SIZE", "MAX SIZE", "USAGE [%]");
  
  TsegRecCI segRecCI;
  for (segRecCI=segRec[SEG_ROOT].segRef.begin(); 
       segRecCI!=segRec[SEG_ROOT].segRef.end(); segRecCI++){
         ReportOverall(*segRecCI);}
  reports.Info(TReports::VL_SEG1, "\n");
}

void TSegment::ReportSegmentTree(TSegmentRec* p, int level){
  if (p!=&segRec[SEG_ROOT]){
    reports.Info(TReports::VL_SEG1, 
      "%*.0s%c %-*.*s", level, "", (level==2)?'+':'*',
      SEG_MAXNAMELEN-level, SEG_MAXNAMELEN-level, p->name);      
      
    if (p->abs==-1 || fitted==false){
      reports.Info(TReports::VL_SEG1, "     -     ");
    }else{
      reports.Info(TReports::VL_SEG1, " $%8.8lx ",p->abs);
    }
    if (p->PC==0){reports.Info(TReports::VL_SEG1, "     .     ");}
    else{reports.Info(TReports::VL_SEG1, " $%8.8lx ",p->PC);}

    if (p->flags&TSegmentRec::Abstract){
      reports.Info(TReports::VL_SEG1, "Abstract ");
    }
    if (p->flags&TSegmentRec::Removable){
      if (p->status & TSegmentRec::Removed){
        reports.Info(TReports::VL_SEG1, "*REMOVED*");
      }
      else{reports.Info(TReports::VL_SEG1, "Removable ");}
    }        
    reports.Info(TReports::VL_SEG1, "\n");
    
    /* Error Report */
    if (p->status!=TSegmentRec::Ok && p->status!=TSegmentRec::Removed){
      reports.Info(TReports::VL_SEG1, "%*.0s  ^ Error: ", level, "");
      if (p->status & TSegmentRec::OutofRange){
        reports.Info(TReports::VL_SEG1, "Out of Range ");
      }
      if (p->status & TSegmentRec::Collision){
        reports.Info(TReports::VL_SEG1, "Collision ");
      }
      reports.Info(TReports::VL_SEG1, "\n");
    }
    
    /* Additional Segment Dep. Report */
    p->dep.Print(level);
  }
  TsegRecCI segRecCI;
  p->sortRefs();
  for (segRecCI=p->segRef.begin(); segRecCI != p->segRef.end(); segRecCI++){
    ReportSegmentTree(*segRecCI, level+2);}
}

void TSegment::ReportOverall(TSegmentRec* p){
  float usage;
  if (p->size==0){usage=0;}
  else{usage=(100.0*(float)p->sumSize)/(float)(p->size);}
  reports.Info(TReports::VL_SEG1, "  + %-30.30s $%8.8lx  $%8.8lx   %.1f\n", 
    p->name, p->sumSize, p->size, usage);
}


void TSegment::SetSeg(int seg_no){
  assert(avasta.Operation()==TAVAStatus::ASSEMBLING_PASS2);
  assert(seg_no<segUsageCnt && seg_no>=0);
  cseg = &segRec[seg_no];
}

void TSegment::AddDep(int seg_no, int current_seg_no){  
  if (current_seg_no==SEGNUMBER_UNDEFINED ||
      current_seg_no==seg_no){return;}	/* opt: see RemoveUnused */      
      
  //printf("TSegment::AddDep(%d,%d)\n", seg_no, current_seg_no);      
  assert(seg_no<segUsageCnt && seg_no>=0);
  assert(current_seg_no<segUsageCnt && current_seg_no>=0);
  segRec[current_seg_no].dep.Add(seg_no);
}

void TSegment::MergeDep(int seg_no, const TSegDep& src){
  assert(seg_no>=0 && seg_no<segUsageCnt);
  if (seg_no==0){return;}
  segRec[seg_no].dep.Merge(src);
}

/* Returns base segment of any segment in a tree */
const char* TSegment::TellBaseSegment(int segNo) const {
  assert(segNo>0 && segNo<segUsageCnt);
  const TSegmentRec* p = &segRec[segNo];  
  while(p->parent != segRec){p=p->parent;} /*go up and find parent=segrec*/
  return p->name;
}

/* List root's base segments */
int TSegment::GetRootsSegments(unsigned child) const {
  if (segRec[SEG_ROOT].segRef.size()<=child){return -1;}
  return segRec[SEG_ROOT].segRef[child] - segRec;
}

const char* TSegment::TellSegmentName(int segNo) const {
  assert(segNo>=0 && segNo<segUsageCnt);
  return segRec[segNo].name;
}

int TSegment::TellNoSegments() const {
  return segUsageCnt;
}

int TSegment::TellMirror(int segNo) const {
  assert(segNo>0 && segNo<segUsageCnt);
  if (segRec[segNo].mirrorType==TSegmentRec::NoMirror){return -1;}
  return segRec[segNo].mirrorSegNo;
}

int TSegment::TellAlign(int segNo) const {
  assert(segNo>0 && segNo<segUsageCnt);
  return segRec[segNo].align;
}


/* Segment Dependencies */

void TSegDep::Add(int seg_no){  
  assert(seg_no<max_segs);
  int byte_offset = seg_no>>LOG_BITPR;
  int bit = 1<<(seg_no - (byte_offset<<LOG_BITPR));
  if (!(segbitmap[byte_offset] & bit)){count++;}
  segbitmap[byte_offset] |= bit;
  if (last_byte < byte_offset){last_byte = byte_offset;}
}

void TSegDep::Remove(int seg_no){  
  assert(seg_no<max_segs);
  int byte_offset = seg_no>>LOG_BITPR;
  int bit = 1<<(seg_no - (byte_offset<<LOG_BITPR));
  if (segbitmap[byte_offset] & bit){count--;}
  segbitmap[byte_offset] &= ~bit;
  assert(count>=0);  
}

void TSegDep::Print(int level) const{
  if (count==0){return;}
  reports.Info(TReports::VL_SEG2, 
          "%*.0s  Referenced from %d other segment(s):",
          level, "", count);	
  int npl=0;
  for(int i=0; i<max_segs;i++){
    if (segbitmap[i/(1<<LOG_BITPR)] & (1<<(i%(1<<LOG_BITPR)))){
      if (npl==0){
        reports.Info(TReports::VL_SEG2, "\n%*.0s    %s", level, "",
          segment.TellSegmentName(i));
	npl=NAMES_PER_LINE-1;
      } else {
        reports.Info(TReports::VL_SEG2, ", %s", segment.TellSegmentName(i));
	npl--;
      }
    }
  }
  reports.Info(TReports::VL_SEG2, "\n\n");
}

void TSegDep::Merge(const TSegDep &src){
  int no_bytes = max(src.last_byte, last_byte);
  TSegBitMap tmp1,tmp2;
  for(int i=0;i<=no_bytes;i++){
    tmp1 = segbitmap[i]; segbitmap[i] |= src.segbitmap[i]; tmp2 = segbitmap[i];
    for(int j=0;j<(1<<LOG_BITPR);j++){
      if ((tmp1 & 1) ^ (tmp2 & 1)){count++;}
      tmp1>>=1; tmp2>>=1;
    }
  }
}

void TSegDep::MapSegments(const TSegTable& segtbl){
  TSegBitMap bmbyte;
  TSegBitMap* new_sbm = new TSegBitMap[max_byte];
  int byte_offset, bit;
  int seg_cnt = 0, tran_seg_no; 
  
  {for (int i=0;i<max_byte;i++){new_sbm[i]=0;}}	/* clear array */
  for (int i=0;i<=last_byte;i++){
    bmbyte = segbitmap[i];
    for(int j=0;j<(1<<LOG_BITPR);j++){  
      if (bmbyte & 1){
        tran_seg_no = segtbl.seg[seg_cnt].newNo;
        byte_offset = tran_seg_no>>LOG_BITPR;
        bit = 1<<(tran_seg_no - (byte_offset<<LOG_BITPR));      
        new_sbm[byte_offset] |= bit;
      }
      seg_cnt++;
      bmbyte>>=1;
    }    
  }  
  /* after remapping, find _new_ last byte */
  for (last_byte = max_byte-1; segbitmap[last_byte]==0; last_byte--);
  delete[] segbitmap;
  segbitmap = new_sbm;
  mapped = true;
}

/* Save Function writes length of used array of used array in 
   reversed (stacked) order. Load does reverse process.
*/   
void TSegDep::SaveObj(){
  for(int i=last_byte;i>=0;i--){object.outOperand(segbitmap[i]);}
  object.outOperand(count);	/* count is saved due to internal self-check */
  object.outOperand(last_byte);
}

void TSegDep::LoadObj(){
  last_byte = (int)object.popOperand();
  count     = (int)object.popOperand();
  for(int i=0;i<=last_byte;i++){segbitmap[i]=(TSegBitMap)object.popOperand();}
}

void TSegDep::LoadObj(const TSegTable& segtbl){
  TSegBitMap bmbyte;
  int byte_offset, bit;
  int seg_cnt = 0, tran_seg_no;
  last_byte   = (int)object.popOperand();
  count       = (int)object.popOperand();
  
  for(int i=0;i<=last_byte;i++){
    bmbyte = (TSegBitMap)object.popOperand();
    for(int j=0;j<(1<<LOG_BITPR);j++){  
      if (bmbyte & 1){
        tran_seg_no = segtbl.seg[seg_cnt].newNo;
        byte_offset = tran_seg_no>>LOG_BITPR;
        bit = 1<<(tran_seg_no - (byte_offset<<LOG_BITPR));      
        segbitmap[byte_offset] |= bit;	
      }
      seg_cnt++;
      bmbyte>>=1;
    }    
  }  
  /* after remapping, find _new_ last byte */
  for (last_byte = max_byte-1; segbitmap[last_byte]==0; last_byte--);
}
