/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
/** 
 * This is testing the package comments section for nz.net.catalyst.*
 */
package nz.net.catalyst;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

/**
 * Application entry point.  Reads application configuration and
 * starts the application.  <p>
 *
 * Should be sub-classed for each different program required.
 */

public class Main
{
  public static final String CONFIG_SUFFIX = ".config";
  private static Main instance = new Main();

  protected String args[] = new String[0];
  protected Class application = Main.class;
  protected String[] startClassNames = null;
  protected Method[] StartList = null;

  public static void main(String args[])
  {
    new Main().init(args);
  }

  public static Main getInstance()
  {
    return instance;
  }

  public void init(String args[])
  {
    instance = this;
    setApplication(this.getClass());
    setArgs(args);

    loadConfig();
    initLogging();
    loadStartList();
    start();
  }

  protected void setArgs(String args[])
  {
    this.args = args == null ? new String[0] : args;
  }

  protected void setApplication(Class application)
  {
    this.application = application == null ? Main.class : application;
  }

  public Class getApplication()
  {
    return application;
  }

  public String[] getArgs()
  {
    return (String[])args.clone();
  }

  protected void initLogging()
  {
    Log.init();
  }

  protected void loadConfig()
  {
    String configProp = getApplication().getName() + CONFIG_SUFFIX;
    String configFileName = System.getProperty(configProp);
    boolean configPropMissing = configFileName == null;

    if (configPropMissing)
      configFileName = getBaseName() + ".config";

    File configFile = new File(configFileName);
    if (configFile.exists())
    {
      try
      {
        Config.load(configFile);
        if (configPropMissing)
          System.setProperty(configProp, configFileName);
      }
      catch (IOException e)
      {
        throw new RuntimeException("Error loading config file: " + configFile.getPath(), e);
      }
    }
  }

  protected String getBaseName()
  {
    String className = getApplication().getName();
    int lastDot = className.lastIndexOf('.');
    return className.substring(lastDot + 1);
  }

  protected void loadStartList()
  {
    String baseClass = getApplication().getName();
    String startProp = baseClass + ".start";
    String startListProp = System.getProperty(baseClass + ".start");
    Log.debug("Getting start list: " + startProp + " = " + startListProp);
    startClassNames = Util.getCsv(startListProp);

    if (startClassNames == null || startClassNames.length == 0)
      startClassNames = new String[]{baseClass};

    resolveStartClasses();
  }


  protected void resolveStartClasses()
  {
    resolveStartMethods("Start", new Class[] {String[].class});
  }

  protected void resolveStartMethods(String methodName, Class[] argListTypes)
  {

    List starters = new ArrayList(startClassNames.length);

    for (int i = 0; i < startClassNames.length; ++i)
    {
      Class starter;

      try
      {
        Log.debug("Trying to load class: " + startClassNames[i]);
        starter = Class.forName(startClassNames[i]);
        Log.debug("Loaded class: " + starter.getName());

      }
      catch (ClassNotFoundException e)
      {
        throw new RuntimeException(
          "Error loading class \"" + startClassNames[i] + '"',
          e);
      }

      try
      {
        Method startMethod = starter.getDeclaredMethod(
          methodName, argListTypes);

        starters.add(startMethod);
      }
      catch (NoSuchMethodException e)
      {
        StringBuffer sb = new StringBuffer("(");
        if (argListTypes != null)
        {
          for (int arg = 0; arg < argListTypes.length; ++arg)
          {
            if (arg > 0)
              sb.append(", ");
            StringBuffer array = new StringBuffer();
            Class cl = argListTypes[arg];

            while (cl.isArray())
            {
              array.append("[]");
              cl = cl.getComponentType();
            }

            sb.append(cl.getName());
            sb.append(array);
          }
        }
        sb.append(')');

        throw new RuntimeException(
          "Error calling method \"" + methodName + sb + " in " + startClassNames[i],
          e);
      }
    }
    Method[] array = new Method[starters.size()];
    starters.toArray(array);
    setStartMethods(array);
  }

  protected void setStartMethods(Method[] list)
  {
    StartList = list == null ? new Method[0] : list;
  }

  protected Method[] getStartMethods()
  {
    return StartList;
  }

  protected void start()
  {
    Method[] starters = getStartMethods();
    if (starters == null)
      return;

    for (int i = 0; i < starters.length; ++i)
    {
      try
      {
        Log.debug("About to invoke " + starters[i].getDeclaringClass().getName() + "." + starters[i].getName());
        starters[i].invoke(null, new Object[]{getArgs()});
        Log.debug("Returned from " + starters[i].getDeclaringClass().getName() + "." + starters[i].getName());
      }
      catch (IllegalAccessException e)
      {
        throw new RuntimeException("Error activating " + starters[i].getName() + "() method in " +
                              starters[i].getDeclaringClass().getName(), e);
      }
      catch (InvocationTargetException e)
      {
        throw new RuntimeException("Exception in " + starters[i].getName() + "() method in " +
                              starters[i].getDeclaringClass().getName(), e);
      }
    }
  }
}
