package nz.net.catalyst.lucene.testclient;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Lucene" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Lucene", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

/**
 * This is adapted from the standard Lucene demo to use the Lucene
 * Server interface.
 */
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.net.Socket;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;
import nz.net.catalyst.Main;
import nz.net.catalyst.StringPair;
import nz.net.catalyst.lucene.server.Constants;
import nz.net.catalyst.lucene.server.ECommand;
import nz.net.catalyst.lucene.server.Server;
import nz.net.catalyst.lucene.server.Transmission;

public class IndexFiles extends Main
{
  private static final IndexFiles instance = new IndexFiles();
  private static final String PORT_PROPERTY = "LuceneServerPort";
  private static final String HOST_PROPERTY = "LuceneServerHost";
  private static final String APPLICATION_PROPERTY = "Application";

  private static int port;
  private static String host;
  private static String application;

  private static final Map sums = new HashMap();

  private int count = 0;
  private long sum = 0;

  public static void main(String[] args)
  {
    Log.debug("In main");

    instance.init(args);
  }

  public static void Start(String[] args) throws Exception
  {
    Log.debug("Starting");
    port = Integer.getInteger(PORT_PROPERTY, Server.LISTENER_PORT).intValue();
    host = System.getProperty(HOST_PROPERTY, "localhost");
    application = System.getProperty(APPLICATION_PROPERTY, "test");

    Socket socket = new Socket(host, port);
    PrintWriter out = new PrintWriter(
      new BufferedWriter(
        new OutputStreamWriter(
          socket.getOutputStream(), "UTF-8"), 1048576), false);
    Date start = new Date();
    BufferedReader in = new BufferedReader(
      new InputStreamReader(socket.getInputStream(), "UTF-8"));

    for (int i = 0; i < args.length; ++i)
      indexDocs(out, in, new File(args[i]));

    if (args.length == 0)
    {
      // No file names on command line: Read files names from System.in
      BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
      String line;
      while (null != (line = stdin.readLine()))
        indexDocs(out, in, new File(line.trim()));
    }

    socket.close();

    Date end = new Date();

    for (Iterator it = sums.entrySet().iterator(); it.hasNext(); )
    {
      Entry e = (Entry)it.next();
      String key = (String)e.getKey();
      IndexFiles stat = (IndexFiles)e.getValue();
      System.out.println(key + ": Count: " + stat.count + ", Total: " +
                         stat.sum / 1000 + " secs, Average: " +
                         stat.sum / stat.count + " ms");
    }

    System.out.print(end.getTime() - start.getTime());
    Log.log(ELog.INFO, " total milliseconds");
  }

  public static void indexDocs(PrintWriter out, BufferedReader in, File file)
    throws Exception
  {
    if (file.isDirectory())
    {
      String[] files = file.list();
      for (int i = 0; i < files.length; i++)
          indexDocs(out, in, new File(file, files[i]));
    }
    else
    {
      Log.log(ELog.INFO, "adding " + file);
      long beforeTransmission = System.currentTimeMillis();
      Transmission command = new Transmission(ECommand.INDEX);
      command.add(Constants.SERIAL, file.getPath());
      command.add(Constants.ID, file.getPath());
      command.add(Constants.APPLICATION, application);
      command.add("Published", String.valueOf(file.lastModified() / 1000L));
      command.add(Constants.BODY, "Text");

      long longFileSize = file.length();
      if (longFileSize > Integer.MAX_VALUE)
        longFileSize = Integer.MAX_VALUE;

      int fileSize = (int)longFileSize;
      long beforeReadingBody = System.currentTimeMillis();
      Reader isr =new InputStreamReader(new FileInputStream(file),"UTF-8");
      char[] buffer = new char[fileSize];
      fileSize = isr.read(buffer);
      command.setBody(new String(buffer));
      long beforeSending = System.currentTimeMillis();
      command.write(out);
      long beforeReceiving = System.currentTimeMillis();
      Transmission response = Transmission.create(in);
      long finished = System.currentTimeMillis();
      if (response == null)
      {
        System.err.println("Input closed while we were waiting for a response!");
      }
      else
      {
        response.write(new PrintWriter(System.out));
        response.add("Response Time", String.valueOf(finished- beforeReceiving));

        for (Iterator it = response.getHeadersView().iterator(); it.hasNext();)
        {
          StringPair entry = (StringPair)it.next();
          try
          {
            long value = Long.parseLong(entry.getValueString());
            String key = entry.getKeyString();
            IndexFiles stat = (IndexFiles)sums.get(key);
            if (stat == null)
            {
              stat = new IndexFiles();
              sums.put(key,stat);
            }
            stat.count++;
            stat.sum += value;
          }
          catch (NumberFormatException e)
          {
            // Ignore
          }
        }
      }
      System.err.println("Creating transmission: " +
                         (beforeReadingBody - beforeTransmission) + " ms");
      System.err.println("Reading Body: " +
                         (beforeSending - beforeReadingBody) + " ms");
      System.err.println("Sending command: " +
                         (beforeReceiving - beforeSending) + " ms");
      System.err.println("Response time: " +
                         (finished- beforeReceiving) + " ms");
    }
  }
}
