#!/bin/bash
# Upgrade/patch a POSTGRESQL database for an Axyl website.

# NOTE: This is not normally run standalone. The main DB upgrade
# script 'install/upgrade-axyl-databases.sh normally runs this.
#
# THIS SCRIPT REQUIRES THE AXYL COMMON FUNCTIONS TO BE DEFINED
# To do this source ${AXYL_HOME}/install/axyl-common.sh from a
# containing shell script, which then sources this one.
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
DBTYPE="postgresql"
NAME="upgrade-db.sh"

# INCOMING PARAMETERS
#  $1  - DB_NAME
#  $2  - DB_USER
#  $3  - DB_PASSWD ('none' means a blank password)
#  $4  - DB_HOST ('direct' means a local database server)
#  $5  - DB_PORT
#  $6  - AXYL_VERSION (Axyl version, '' or 'n.n.n'
#  $7  - MODE ('upgrade' or 'stamp-only')
#  $8  - DB_DETECT_MODE ('interactive' (default) or 'auto')
DB_NAME=""
DB_USER=""
DB_PASSWD=""
DB_HOST=""
DB_PORT=5432
AXYL_VERSION="unknown"
MODE=upgrade
DB_DETECT_MODE=interactive

# Echo and log messages. We use the common Axyl function 'logit()' to send
# our log messages to this globally defined $LOGFILE. It also logs to
# syslog as well.
LOGFILE=${AXYL_LOGS}/upgrade-db.log
[ ! -f $LOGFILE ] && >$LOGFILE

POS=0
while [ $# -gt 0 ] ; do
  POS=`expr $POS + 1`
  case $POS in
    1)  DB_NAME=$1
        ;;
    2)  DB_USER=$1
        ;;
    3)  DB_PASSWD=$1
        ;;
    4)  DB_HOST=$1
        ;;
    5)  DB_PORT=$1
        ;;
    6)  AXYL_VERSION=$1
        ;;
    7)  MODE=$1
        ;;
    8)  DB_DETECT_MODE=$1
        ;;
  esac
  shift
done
logit "${DBTYPE} ${NAME}: db=$DBNAME user=$DBUSER host=$DBHOST port=$DBPORT axver=$AXYL_VERSION mode=$MODE detectmode=$DB_DETECT_MODE"

# Cater for standalone running of this script. Normally we are called
# with AXYL_HOME et al already defined..
if [ -z $AXYL_HOME ] ; then
	CONFDIR=/etc/axyl
	CONF=${CONFDIR}/axyl.conf
	if [ ! -f $CONF ] ; then
	  echo "Axyl configuration file $CONF not found!"
	  exit 2
	else
	  . $CONF
	  if [ ! -d $AXYL_HOME ] ; then
	    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
	    exit 6
	  fi
	  . ${AXYL_HOME}/install/axyl-common.sh
	fi
fi

# We require Postgres to be locally installed, at least as postgresql-client
# even if no servers are created locally. This is based on the standard
# Debian location, with a few likely fallbacks.

# Detect database, and set up database vars. This set up the following
# variables:
# PG_MULTI_CLUSTER     # Eg. '8.1/main' Postgres version and cluster
# PG_VERSION           # Version of the database eg. '8.1'
# PG_VERSION_SUFFIX    # Version suffix eg. '-8.1'
# PG_BIN               # Path to the Postgres binary files
# PG_CONF              # Path to the Postgre configuration files

. ${AXYL_HOME}/db/postgres/detect-db.sh $DB_DETECT_MODE

# Now set paths to our executables
PSQL=${PG_BIN}/psql

# Upgrade the database contents (tables and data)
if [ "$PG_BIN" != "" ] ; then

    if [ -x $PSQL ] ; then
		# Optional host settings for remotely accessed databases..
		HOSTOPTS=""
		[ "$DB_HOST" != "direct" ] && HOSTOPTS="--host $DB_HOST --port $DB_PORT"
		
		# Make sure our database connection user is present..
		USER=`$PSQL --tuples-only --username postgres --dbname template1 $HOSTOPTS --command "SELECT usename FROM pg_user WHERE usename='$DB_USER'" | tr -d ' '`
		if [ "$USER" != "$DB_USER" ] ; then
			logit "${DBTYPE} ${NAME}: creating database user $DB_USER"
			$PSQL --username postgres --dbname template1 $HOSTOPTS --command "CREATE USER $DB_USER WITH PASSWORD '$DB_PASSWD' CREATEDB"
		fi

		# Acquire application version from ax_control table which must exist
		# or else we simply skip the rest of the process.
		AXYL_DB_VERSION=
		AX_CONTROL=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT relname FROM pg_class WHERE relname='ax_control'" | tr -d ' '`
		if [ "$AX_CONTROL" = "ax_control" ] ; then
			AXYL_DB_VERSION=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT site_axyl_version FROM ax_control" | tr -d ' '`
			if [ "$MODE" != "stamp-only" -a "$AXYL_DB_VERSION" != "" ] ; then
				if [ "$AXYL_DB_VERSION" = "$AXYL_VERSION" ] ; then
					logit "${DBTYPE} ${NAME}: database ${DB_NAME} is up to date."
				else
					# Ok, we know the registered version, and we know the Axyl installation
					# version, so let's see if we have a patch file series for this upgrade.
					# No patch files, no upgrade.
					UPGRADEROOT=${AXYL_HOME}/db/postgres/upgrade
					cd ${UPGRADEROOT}
					patchfiles=`ls ${UPGRADEROOT}`
					if [ "$patchfiles" != "" ] ; then
						patchSQL=`tempfile --prefix dbupgrade` 
						patchver=$AXYL_DB_VERSION
						gotpatches=0
						for patchfile in $patchfiles ; do
							if [ "$patchver" = "$AXYL_VERSION" ] ; then
								break
							fi
							patchnos=`echo $patchfile | sed "s;.sql;;"`
							if [ "$patchnos" != "" ] ; then
								fromver=`echo $patchnos | cut -d'_' -f2`
								nextver=`echo $patchnos | cut -d'_' -f4`
								if [ "$patchver" = "$fromver" ] ; then
									cat $patchfile >> $patchSQL
									patchver=$nextver
									gotpatches=1
								fi
							fi
						done
						if [ $gotpatches -eq 1 ] ; then
							logit "${DBTYPE} ${NAME}: patching ${DB_NAME} ${AXYL_DB_VERSION} --> ${patchver}"
							$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --file $patchSQL >>$LOGFILE 2>&1
							if [ $? -ne 0 ] ; then
								logit "${DBTYPE} ${NAME}: errors occurred during the patch process"
							fi
						fi
					else
						logit "${DBTYPE} ${NAME}: no patches were provided for ${DB_NAME} ${AXYL_DB_VERSION}"
					fi
					# Leave the application at the correct version for next time..
					$PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control set site_axyl_version='$AXYL_VERSION'" >>$LOGFILE 2>&1
					# Remove temporary patch file..
					rm -f $patchSQL
                fi
            else
                # All we do in this case is assume everything is up to date..
                if [ "$MODE" = "stamp-only" ] ; then
                  logit "${DBTYPE} ${NAME}: stamping database $DB_NAME with version $AXYL_VERSION"
                fi
                N=`$PSQL --tuples-only --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "SELECT COUNT(*) from ax_control" | tr -d ' '`
                if [ $N -eq 0 ] ; then
                  $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "INSERT INTO ax_control (site_axyl_version) VALUES('$AXYL_VERSION')" >>$LOGFILE 2>&1
                else
                  $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "UPDATE ax_control set site_axyl_version='$AXYL_VERSION'" >>$LOGFILE 2>&1
                fi
            fi
        else
            # No ax_control..
            $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "CREATE TABLE ax_control (app_prefix text,site_axyl_version text,site_docroot text)" >>$LOGFILE 2>&1
            logit "${DBTYPE} ${NAME}: stamping database with version $AXYL_VERSION"
            $PSQL --username $DB_USER --dbname $DB_NAME $HOSTOPTS --command "INSERT INTO ax_control (site_axyl_version) VALUES('$AXYL_VERSION')" >>$LOGFILE 2>&1
        fi
    else
        logit "${DBTYPE} ${NAME}: postgres client binaries (psql) not found."
        exit 0
    fi
	
else
    logit "${DBTYPE} ${NAME}: could not determine postgresql bin location."
    exit 0
fi
	
# END