<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    pim-defs.php                                            */
/* Author:      Paul Waite/Mark Kessell                                 */
/* Description: Definitions for manipulating personal information such  */
/*              as organisations, addresses, emails, phone numbers,     */
/*              urls, etc.                                              */
/*                                                                      */
/* ******************************************************************** */
/** @package pim */

// ......................................................................
// DEFINITIONS
/** Create new organisation ID value */
define("NEW_ORGANISATION",  -1);
/** Create new contact info ID value */
define("NEW_CONTACTINFO",   -1);

// ----------------------------------------------------------------------
/**
* The organisation class manages a single set of information related to a
* "organisation". This is an identity distinct from the system-oriented
* uuser record, and is intended to be a generic mechanism for storing
* data on human beings.
* @package pim
*/
class organisation extends RenderableObject {
  /** Organisation ID */
  var $orgid = NEW_ORGANISATION;
  /** Organisation name */
  var $org_name = "";
  /** Description */
  var $description = "";
  /** Contact person name */
  var $org_contact = "";
  /** Contact info object */
  var $contactinfo;
  /** Whether organisation is enabled */
  var $enabled = true;
  /** Whether organisation exists in database */
  var $exists = false;
  /**
  * Constructor
  * Create a new organisation object.
  * @param string $id The unique identity of the organisation.
  */
  function organisation($id=NEW_ORGANISATION) {
    $this->get($id);
  } // organisation
  // ....................................................................
  /**
  * Get the organisation.
  * Retrieves the specified organisation from database.
  * @param string $id The unique integer identity of the organisation to get.
  */
  function get($id="") {
    debug_trace($this);
    if ($id != "") {
      $this->orgid = $id;
    }
    $this->exists = false;
    if ($this->orgid != NEW_ORGANISATION) {
      // Try and read in existing organisation info..
      $q  = "SELECT * FROM ax_organisation";
      $q .= " WHERE org_id=$this->orgid";
      $orgQ = dbrecordset($q);
      if ($orgQ->hasdata) {
        $this->org_name    = $orgQ->field("org_name");
        $this->description = $orgQ->field("org_desc");
        $this->org_contact = $orgQ->field("org_contact");
        $this->enabled     = $orgQ->istrue("enabled");
        $cinfoid           = $orgQ->field("contactinfo_id");
        if ($cinfoid != "") {
          $this->contactinfo = new contactinfo($cinfoid);
        }
        else {
          $this->contactinfo = new contactinfo();
        }
        $this->exists = true;
      }
    }
    else {
      // Initialise the NEW object values..
      $this->initialise();
    }
    debug_trace();
    // Return true if at least the organisation exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save the organisation.
  * Save this organisation to the database. Create a new one if it
  * doesn't already exist.
  */
  function save() {
    debug_trace($this);
    // Deal with brand new organisation..
    if (!$this->exists) {
      // If we are in need of a new ID, then get one..
      if ($this->orgid == NEW_ORGANISATION) {
        $this->orgid = get_next_sequencevalue("seq_org_id", "ax_organisation", "org_id");
        $this->contactinfo->save();
      }
      $savQ = new dbinsert("ax_organisation");
      $savQ->set("org_id", $this->orgid);
    }
    else {
      $savQ = new dbupdate("ax_organisation");
      $savQ->where("org_id=$this->orgid");
    }
    $savQ->set("org_name",    $this->org_name);
    $savQ->set("org_desc",    $this->description);
    $savQ->set("org_contact", $this->org_contact);
    $savQ->set("enabled",     $this->enabled);
    if (isset($this->contactinfo)) {
      $savQ->set("contactinfo_id", $this->contactinfo->contactinfoid);
    }
    $this->exists = $savQ->execute();
    debug_trace();

  } // save
  // ....................................................................
  /**
  * Delete the organisation.
  * Delete this organisation from the database.
  */
  function delete() {
    debug_trace($this);
    if ($this->exists) {
      start_transaction();
      if (isset($this->contactinfo)) {
        $this->contactinfo->delete();
      }
      if (dbcommand("DELETE FROM ax_organisation WHERE org_id=$this->orgid")) {
        $this->initialise();
        $this->exists = false;
      }
      commit();
    }
    debug_trace();
  } // delete
  // ....................................................................
  /** Clear the object variables to default values. */
  function initialise() {
    $this->orgid = NEW_ORGANISATION;
    $this->org_name = "";
    $this->description = "";
    $this->org_contact = "";
    $this->contactinfo = new contactinfo();
    $this->enabled = true;
    $this->exists = false;
  } // initialise
  // ....................................................................
  /**
  * Process a possible POST of a form with new contact info data. We
  * only do anything here if the special var $_save_x is defined (ie. this
  * would be from user clicking button named "_save"), and also the organisation
  * ID $_ORG_org_id. We return true if POST was processed.
  * @return boolean True if POSTed contact info was processed, else false
  */
  function POSTprocess() {
    $processed = false;
    debug_trace($this);
    global $_save_x, $_ORG_org_id;
    if (isset($_save_x) && isset($_ORG_org_id)) {
      // Only process org data meant for us..
      if ($this->orgid == $_ORG_org_id) {
        // Process/save any POSTed contact info..
        if (isset($this->contactinfo)) {
          $this->contactinfo->POSTprocess();
        }
        // Access POSTed information..
        global $_ORG_org_name, $_ORG_org_name, $_ORG_description;
        global $_ORG_org_contact, $_ORG_enabled;
        // Stash the new data..
        $this->org_name    = $_ORG_org_name;
        $this->description = $_ORG_description;
        $this->org_contact = $_ORG_org_contact;
        $this->enabled     = isset($_ORG_enabled);
        // Now save these changes..
        $this->save();
        $processed = true;
      }
    }
    if (isset($_delete_x) && isset($_ORG_org_id)) {
      $this->delete();
      $processed = true;
    }
    debug_trace();
    return $processed;
  } // POSTprocess
  // ....................................................................
  /** Display organisation subform via the render() method.
  * @param string $id_mode Whether to hide the ID or not
  */
  function html($idmode="hide_id") {
    debug_trace($this);

    // FIELD - org_id
    if ($idmode == "hide_id") {
      $orgid_F = new form_hiddenfield("_ORG_org_id", $this->orgid);
    }
    else {
      $orgid_F = new form_labelfield("Org ID", $this->orgid);
    }

    // FIELD - First name
    $org_name_F = new form_textfield("_ORG_org_name", "Organisation name", $this->org_name);
    $org_name_F->setstyle("width: 345px");

    // FIELD - Description
    $description_F = new form_textfield("_ORG_description", "Description", $this->description);
    $description_F->setstyle("width: 345px");

    // FIELD - Organisation contact person
    $org_contact_F = new form_textfield("_ORG_org_contact", "Contact person", $this->org_contact);
    $org_contact_F->setstyle("width: 345px");

    // FIELD - Enabled
    $enabled_F = new form_checkbox("_ORG_enabled", "Enabled");
    $enabled_F->checked = $this->enabled;

    // SUB-FORMS..
    $sform = new subform();
    $sform->add($orgid_F);
    $sform->add($org_name_F);
    $sform->add($description_F);
    $sform->add($org_contact_F);
    $sform->add($enabled_F);
    $forms = $sform->render();
    if (isset($this->contactinfo)) {
      $forms .= $this->contactinfo->render();
    }

    // Table containing the form..
    $Tdetail = new table("organisation");
    $Tdetail->setpadding(2);
    $Tdetail->tr("background-color:#333333;");
    $Tdetail->td("<b>ORGANISATION DETAILS</b>", "color:white");
    $Tdetail->tr();
    $Tdetail->td( $forms );
    $Tdetail->td_alignment("right", "top");

    debug_trace();
    return $Tdetail->render();
  } // html
} // organisation class

// ----------------------------------------------------------------------
/**
* The contactinfo class manages a single set of contact information
* which might be associated with a person or thing.
* @package pim
*/
class contactinfo extends RenderableObject {
  // Public
  /** contactinfo ID */
  var $contactinfoid;
  /** addr_postal */
  var $addr_postal = "";
  /** addr_street */
  var $addr_street = "";
  /** addr_suburb */
  var $addr_suburb = "";
  /** addr_city */
  var $addr_city = "";
  /** addr_country */
  var $addr_country = "";
  /** addr_code */
  var $addr_code = "";
  /** addr_other */
  var $addr_other = "";
  /** emails */
  var $emails = "";
  /** phone */
  var $phone = "";
  /** phone_fax */
  var $phone_fax = "";
  /** phone_work */
  var $phone_work = "";
  /** phone_mobile */
  var $phone_mobile = "";
  /** urls */
  var $urls = "";

  // Private
  /** Whether contactinfo exists in database
      @access private */
  var $exists = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new contactinfo object.
  * @param string $id The unique identity of the contactinfo.
  */
  function contactinfo($id=NEW_CONTACTINFO) {
    $this->get($id);
  } // contactinfo
  // ....................................................................
  /**
  * Retrieves the specified contactinfo from database.
  * @param string $id The unique integer identity of the contactinfo to get.
  */
  function get($id="") {
    debug_trace($this);
    if ($id != "") {
      $this->contactinfoid = $id;
    }
    $this->exists = false;
    if ($this->contactinfoid != NEW_CONTACTINFO) {
      // Try and read in existing contactinfo info..
      $q  = "SELECT * FROM ax_contactinfo";
      $q .= " WHERE contactinfo_id=$this->contactinfoid";
      $savQ = dbrecordset($q);
      if ($savQ->hasdata) {
        $this->addr_postal  = $savQ->field("addr_postal");
        $this->addr_street  = $savQ->field("addr_street");
        $this->addr_suburb  = $savQ->field("addr_suburb");
        $this->addr_city    = $savQ->field("addr_city");
        $this->addr_country = $savQ->field("addr_country");
        $this->addr_code    = $savQ->field("addr_code");
        $this->addr_other   = $savQ->field("addr_other");
        $this->emails       = $savQ->field("emails");
        $this->phone        = $savQ->field("phone");
        $this->phone_fax    = $savQ->field("phone_fax");
        $this->phone_work   = $savQ->field("phone_work");
        $this->phone_mobile = $savQ->field("phone_mobile");
        $this->urls         = $savQ->field("urls");
        $this->exists = true;
      }
    }
    debug_trace();
    // Return true if at least the contactinfo exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save the contactinfo.
  * Save this contactinfo to the database. Create a new one if it
  * doesn't already exist.
  */
  function save() {
    debug_trace($this);
    // Deal with brand new contactinfo..
    if (!$this->exists) {
      // If we are in need of a new ID, then get one..
      if ($this->contactinfoid == NEW_CONTACTINFO) {
        $this->contactinfoid = get_next_sequencevalue("seq_contactinfo_id", "ax_organisation", "org_id");
      }
      $savQ = new dbinsert("ax_contactinfo");
      $savQ->set("contactinfo_id", $this->contactinfoid);
    }
    else {
      $savQ = new dbupdate("ax_contactinfo");
      $savQ->where("contactinfo_id=$this->contactinfoid");
    }
    $savQ->set("addr_postal",  $this->addr_postal);
    $savQ->set("addr_street",  $this->addr_street);
    $savQ->set("addr_suburb",  $this->addr_suburb);
    $savQ->set("addr_city",    $this->addr_city);
    $savQ->set("addr_country", $this->addr_country);
    $savQ->set("addr_code",    $this->addr_code);
    $savQ->set("addr_other",   $this->addr_other);
    $savQ->set("emails",       $this->emails);
    $savQ->set("phone",        $this->phone);
    $savQ->set("phone_fax",    $this->phone_fax);
    $savQ->set("phone_work",   $this->phone_work);
    $savQ->set("phone_mobile", $this->phone_mobile);
    $savQ->set("urls",         $this->urls);
    $this->exists = $savQ->execute();
    debug_trace();
  } // save
  // ....................................................................
  /**
  * Delete the contactinfo.
  * Delete this contactinfo from the database.
  */
  function delete() {
    debug_trace($this);
    if ($this->exists) {
      dbcommand("DELETE FROM ax_contactinfo WHERE contactinfo_id=$this->contactinfoid");
    }
    debug_trace();
  } // delete
  // ....................................................................
  /**
  * Process a possible POST of a form with new contact info data. We
  * only do anything here if the special var $_save_x is defined (ie. this
  * would be from user clicking button named "_save"), and also the contact
  * info IF $_CI_contactinfo_id. We return true if POST was processed.
  * @return boolean True if POSTed contact info was processed, else false
  */
  function POSTprocess() {
    debug_trace($this);
    global $_save_x, $_CI_contactinfo_id;
    if (isset($_save_x) && isset($_CI_contactinfo_id)) {
      debugbr("CINF POSTprocess: processing submitted form data id=$_CI_contactinfo_id");

      // Bring our ID into sync with submitted one..
      $this->contactinfoid = $_CI_contactinfo_id;
      $this->exists = ($this->contactinfoid != NEW_CONTACTINFO);

      // Access POSTed information..
      global $_CI_postal,  $_CI_street,    $_CI_suburb,     $_CI_city;
      global $_CI_country, $_CI_code,      $_CI_other,      $_CI_emails;
      global $_CI_phone,   $_CI_phone_fax, $_CI_phone_work, $_CI_phone_mobile;
      global $_CI_urls;
      // Stash the new data..
      $this->addr_postal  = $_CI_postal;
      $this->addr_street  = $_CI_street;
      $this->addr_suburb  = $_CI_suburb;
      $this->addr_city    = $_CI_city;
      $this->addr_country = $_CI_country;
      $this->addr_code    = $_CI_code;
      $this->addr_other   = $_CI_other;
      $this->emails       = $_CI_emails;
      $this->phone        = $_CI_phone;
      $this->phone_fax    = $_CI_phone_fax;
      $this->phone_work   = $_CI_phone_work;
      $this->phone_mobile = $_CI_phone_mobile;
      $this->urls         = $_CI_urls;
      // Now save these changes..
      $this->save();
      $processed = true;
    }
    debug_trace();
  } // POSTprocess
  // ....................................................................
  /**
  * Return a standard contact details set.
  * This is designed for in-line contact details such as might be
  * included as-is in an email confirmation of order for example. Only
  * fields which are not nullstring are included.
  */
  function contact_details() {
    $a = array();
    if ($this->addr_postal != "")  $a[] = $this->addr_postal;
    if ($this->addr_street != "")  $a[] = $this->addr_street;
    if ($this->addr_suburb != "")  $a[] = $this->addr_suburb;
    if ($this->addr_city != "")    $a[] = $this->addr_city;
    if ($this->addr_code != "")    $a[] = $this->addr_code;
    if ($this->addr_country != "") $a[] = $this->addr_country;
    if ($this->addr_other != "")   $a[] = $this->addr_other;
    if ($this->phone != "")        $a[] = "Tel: $this->phone";
    if ($this->phone_fax != "")    $a[] = "Fax: $this->phone_fax";
    if ($this->phone_work != "")   $a[] = "Work: $this->phone_work";
    if ($this->phone_mobile != "") $a[] = "Mob: $this->phone_mobile";
    if ($this->urls != "")         $a[] = "WWW: $this->urls";
    return implode(",\n", $a);
  } // contact_details
  // ....................................................................
  /** A diagnostic dump of the class variables for debugging purposes. */
  function dump() {
    debug_trace($this);
    debugbr("CONTACT INFO DUMP (id=$this->contactinfoid)");
    debugbr("addr_postal: "  . $this->addr_postal,  DBG_DUMP);
    debugbr("addr_street: "  . $this->addr_street,  DBG_DUMP);
    debugbr("addr_suburb: "  . $this->addr_suburb,  DBG_DUMP);
    debugbr("addr_city: "    . $this->addr_city,    DBG_DUMP);
    debugbr("addr_country: " . $this->addr_country, DBG_DUMP);
    debugbr("addr_code: "    . $this->addr_code,    DBG_DUMP);
    debugbr("addr_other: "   . $this->addr_other,   DBG_DUMP);
    debugbr("emails: "       . $this->emails,       DBG_DUMP);
    debugbr("phone: "        . $this->phone,        DBG_DUMP);
    debugbr("phone_fax: "    . $this->phone_fax,    DBG_DUMP);
    debugbr("phone_work: "   . $this->phone_work,   DBG_DUMP);
    debugbr("phone_mobile: " . $this->phone_mobile, DBG_DUMP);
    debugbr("urls: "         . $this->urls,         DBG_DUMP);
    debug_trace();
  } // debug_cinfo
  // ....................................................................
  /** Display Contactinfo subform via the render() method. */
  function html() {
    debug_trace($this);

    // FIELD - contactinfo_id
    $contact_F = new form_hiddenfield("_CI_contactinfo_id", $this->contactinfoid);

    // FIELD - Addr_postal
    $postal_F = new form_textfield("_CI_postal", "Address - Postal", $this->addr_postal);
    $postal_F->setstyle("width: 345px");

    // FIELD - Addr_Street
    $street_F = new form_textfield("_CI_street", "Address - Street", $this->addr_street);
    $street_F->setstyle("width: 345px");

    // FIELD - Addr_Suburb
    $suburb_F = new form_textfield("_CI_suburb", "Address - Suburb", $this->addr_suburb);
    $suburb_F->setstyle("width: 345px");

    // FIELD - Addr_City
    $city_F = new form_textfield("_CI_city", "Address - City", $this->addr_city);
    $city_F->setstyle("width: 345px");

    // FIELD - Addr_Country
    $country_F = new form_textfield("_CI_country", "Address - Country", $this->addr_country);
    $country_F->setstyle("width: 345px");

    // FIELD - Addr_Code
    $code_F = new form_textfield("_CI_code", "Address - Code", $this->addr_code);
    $code_F->setstyle("width: 345px");

    // FIELD - Addr_Other
    $other_F = new form_textfield("_CI_other", "Address - Other", $this->addr_other);
    $other_F->setstyle("width: 345px");

    // FIELD - Email
    $this->emails = preg_replace("/[ |]/", "\n", $this->emails);
    $emails_F = new form_memofield("_CI_emails", "Email Address", $this->emails);
    $emails_F->setstyle("width: 345px");
    $emails_caption = new form_labelfield("", "<small><i>Enter one e-mail address per line</i></small>");

    // FIELD - Phone
    $phone_home_F = new form_textfield("_CI_phone", "Home Phone", $this->phone);
    $phone_home_F->setstyle("width: 345px");

    // FIELD - Fax
    $phone_fax_F = new form_textfield("_CI_phone_fax", "Fax", $this->phone_fax);
    $phone_fax_F->setstyle("width: 345px");

    // FIELD - phone_work
    $phone_work_F = new form_textfield("_CI_phone_work", "Work Phone", $this->phone_work);
    $phone_work_F->setstyle("width: 345px");

    // FIELD - phone_mobile
    $phone_mobile_F = new form_textfield("_CI_phone_mobile", "Mobile Phone", $this->phone_mobile);
    $phone_mobile_F->setstyle("width: 345px");

    // FIELD - urls
    $this->urls = preg_replace("/[ |]/", "\n", $this->urls);
    $urls_F = new form_memofield("_CI_urls", "URL's", $this->urls);
    $urls_F->setstyle("width: 345px");
    $urls_caption = new form_labelfield("", "<small><i>Enter one URL per line</i></small>");

    // FORM..
    $sform = new subform();
    $sform->add($contact_F);
    $sform->add($postal_F);
    $sform->add($street_F);
    $sform->add($suburb_F);
    $sform->add($city_F);
    $sform->add($country_F);
    $sform->add($code_F);
    $sform->add($other_F);
    $sform->add($emails_F);
    $sform->add($emails_caption);
    $sform->add($phone_home_F);
    $sform->add($phone_fax_F);
    $sform->add($phone_work_F);
    $sform->add($phone_mobile_F);
    $sform->add($urls_F);
    $sform->add($urls_caption);

    $Tdetail = new table("contactinfo");
    $Tdetail->setpadding(2);
    $Tdetail->tr("background-color:#333333;");
    $Tdetail->td("<b>Contact Information</b>", "color:white");
    $Tdetail->tr();
    $Tdetail->td( $sform->render() );
    $Tdetail->td_alignment("right", "top");
    $s = $Tdetail->render();

    debug_trace();
    return $s;
  } // html
} // contactinfo class

// ----------------------------------------------------------------------
?>