/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef SHAREDPTR_H_
#define SHAREDPTR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AtomicCounter.h"
#include <memory>

template<typename T, typename SYNCH>
class SharedPtr
{
	template<typename OT, typename OSYNCH> friend class SharedPtr;
public:
	SharedPtr() : m_pObj(0), m_pRefCounter(0) {}
	
	explicit
	SharedPtr(T* obj);
	
	SharedPtr(SharedPtr const& other);
	
	template<typename OT>
	SharedPtr(SharedPtr<OT, SYNCH> const& other);
	
	~SharedPtr();
	
	SharedPtr& operator=(SharedPtr const& rhs);
	
	template<typename OT>
	SharedPtr& operator=(SharedPtr<OT, SYNCH> const& rhs);
	
	T& operator*() const { return *m_pObj; }
	
	T* operator->() const { return m_pObj; }
	
	operator T*() const { return m_pObj; }
	
	T* get() const { return m_pObj; }
	
	void reset(T* obj);
	
	void swap(SharedPtr& other);
private:
	T* m_pObj;
	AtomicCounter<SYNCH>* m_pRefCounter;
};


template<typename T, typename SYNCH>
SharedPtr<T, SYNCH>::SharedPtr(T* obj)
{
	// auto_ptr improves exception safety here.
	std::auto_ptr<T> obj_ptr(obj);
	m_pRefCounter = obj ? new AtomicCounter<SYNCH>(1) : 0;
	m_pObj = obj_ptr.release();
}

template<typename T, typename SYNCH>
SharedPtr<T, SYNCH>::SharedPtr(SharedPtr const& other)
:	m_pObj(other.m_pObj),
	m_pRefCounter(other.m_pRefCounter)
{
	if (m_pRefCounter)
		++*m_pRefCounter;
}

template<typename T, typename SYNCH>
template<typename OT>
SharedPtr<T, SYNCH>::SharedPtr(SharedPtr<OT, SYNCH> const& other)
:	m_pObj(other.m_pObj),
	m_pRefCounter(other.m_pRefCounter)
{
	if (m_pRefCounter)
		++*m_pRefCounter;
}

template<typename T, typename SYNCH>
SharedPtr<T, SYNCH>::~SharedPtr()
{
	if (m_pRefCounter) {
		if (--*m_pRefCounter == 0) {
			delete m_pObj;
			delete m_pRefCounter;
		}
	}
}

template<typename T, typename SYNCH>
inline SharedPtr<T, SYNCH>&
SharedPtr<T, SYNCH>::operator=(SharedPtr const& rhs)
{
	SharedPtr(rhs).swap(*this);
	return *this;
}

template<typename T, typename SYNCH>
template<typename OT>
inline SharedPtr<T, SYNCH>&
SharedPtr<T, SYNCH>::operator=(SharedPtr<OT, SYNCH> const& rhs)
{
	SharedPtr(rhs).swap(*this);
	return *this;
}

template<typename T, typename SYNCH>
inline void
SharedPtr<T, SYNCH>::reset(T* obj)
{
	SharedPtr(obj).swap(*this);
}

template<typename T, typename SYNCH>
void
SharedPtr<T, SYNCH>::swap(SharedPtr& other)
{
	T* obj = other.m_pObj;
	AtomicCounter<SYNCH>* cnt = other.m_pRefCounter;
	other.m_pObj = m_pObj;
	other.m_pRefCounter = m_pRefCounter;
	m_pObj = obj;
	m_pRefCounter = cnt;
}

template<typename T, typename SYNCH>
inline void swap(SharedPtr<T, SYNCH>& o1, SharedPtr<T, SYNCH>& o2)
{
	o1.swap(o2);
}

#endif
