/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef STRONGPTR_H_
#define STRONGPTR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "WeakPtr.h"
#include "IntrusivePtr.h"
#include <memory>

template<typename SYNCH, typename T, typename TP = std::auto_ptr<T> >
class StrongPtr : public WeakPtr<SYNCH, T, TP>
{
private:
	typedef typename WeakPtr<SYNCH, T, TP>::WeakRef WRef;
	
	class StrongRef : public WRef
	{
	public:
		StrongRef(T* obj) : WRef(obj) {}
		
		void ref() { ++m_strongCounter; }
		
		void unref() { if (!--m_strongCounter) this->clear(); }
	private:
		AtomicCounter<SYNCH> m_strongCounter;
	};
public:
	StrongPtr() {}
	
	StrongPtr(TP obj);
	
	StrongPtr(StrongPtr const& other);
	
	~StrongPtr();
	
	StrongPtr& operator=(StrongPtr const& rhs);
	
	void reset(T* obj);
	
	void swap(StrongPtr& other);
private:
	IntrusivePtr<StrongRef> m_ptrStrongRef;
};


template<typename SYNCH, typename T, typename TP>
StrongPtr<SYNCH, T, TP>::StrongPtr(TP obj)
{
	// TP is passed instead of T* to improve exception safety.
	m_ptrStrongRef.reset(new StrongRef(obj));
	WeakPtr<SYNCH, T, TP>::set(m_ptrStrongRef);
}

template<typename SYNCH, typename T, typename TP>
StrongPtr<SYNCH, T, TP>::StrongPtr(StrongPtr const& other)
:	WeakPtr<SYNCH, T, TP>(other),
	m_ptrStrongRef(other.m_ptrStrongRef)
{
}

template<typename SYNCH, typename T, typename TP>
StrongPtr<SYNCH, T, TP>::~StrongPtr()
{
}

template<typename SYNCH, typename T, typename TP>
inline void
StrongPtr<SYNCH, T, TP>::reset(T* obj)
{
	StrongPtr(obj).swap(*this);
}

template<typename SYNCH, typename T, typename TP>
inline StrongPtr<SYNCH, T, TP>&
StrongPtr<SYNCH, T, TP>::operator=(StrongPtr const& rhs)
{
	StrongPtr(rhs).swap(*this);
	return *this;
}

template<typename SYNCH, typename T, typename TP>
void
StrongPtr<SYNCH, T, TP>::swap(StrongPtr& other)
{
	WeakPtr<SYNCH, T, TP>::swap(other);
	m_ptrStrongRef.swap(other.m_ptrStrongRef);
}

template<typename SYNCH, typename T, typename TP>
inline void swap(StrongPtr<SYNCH, T, TP>& o1, StrongPtr<SYNCH, T, TP>& o2)
{
	o1.swap(o2);
}

#endif
