/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CompressorResponseFilter.h"
#include "GzipCompressor.h"
#include "ResponseFilterChain.h"
#include "RequestStatus.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestMetadata.h"
#include "HttpStatusLine.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "ErrorDescriptor.h"
#include "ErrorFactory.h"
#include "ErrorCodes.h"
#include "StringUtils.h"
#include "IntrusivePtr.h"

using namespace std;

CompressorResponseFilter::CompressorResponseFilter(ResponseFilterChain& chain)
:	ResponseFilterBase(chain),
	m_isEOF(false)
{
}

CompressorResponseFilter::~CompressorResponseFilter()
{
}

void
CompressorResponseFilter::processMetadata(
	RequestStatus& status, std::auto_ptr<HttpResponseMetadata> metadata)
{
	BString const cenc_str("Content-Encoding");
	BString const gzip_str("gzip");
	HttpHeader* hdr = metadata->headers().getHeaderPtr(cenc_str);
	if (!hdr) {
		metadata->headers().setHeader(HttpHeader(cenc_str, gzip_str));
	} else {
		HttpHeaderStructure structure(*hdr);
		structure.appendElement(HttpHeaderElement(gzip_str));
		structure.commitChanges(*hdr);
	}
	metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_UNKNOWN);
	outputMetadata(status, metadata);
}

void
CompressorResponseFilter::processBodyData(
	RequestStatus& status, SplittableBuffer& data, bool eof)
{
	m_isEOF |= eof;
	m_compressor.consume(data, m_isEOF);
	SplittableBuffer output;
	size_t written = 0;
	
	do {
		written = m_compressor.retrieveOutput(output, OUTPUT_LIMIT);
		if (written != 0) {
			outputBodyData(status, output, false);
		}
		if (m_compressor.isError() && !status.isCancelled()) {
			string const problem("error compressing response");
			status.cancel(
				ErrorFactory::errGenericError(
					ErrorCodes::ERROR_COMPRESSING_RESPONSE, problem,
					getRequest().requestLine().getURI(), problem,
					HttpStatusLine::SC_INTERNAL_SERVER_ERROR
				)
			);
			return;
		}
	} while (written != 0 && !status.isCancelled());
	
	if (m_isEOF) {
		outputBodyData(status, output, m_isEOF);
	}
}
