/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef CONNECTIONROUTE_H_
#define CONNECTIONROUTE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SymbolicInetAddr.h"
#include "ProxyDescriptor.h"
#include <vector>

/**
 * \brief Describes a connection through a (possibly empty) proxy chain.
 *
 * \anchor proxy_chain
 * Note that the proxy chain does not directly correspond to what you get
 * from Forwarding::Resolver::resolve().  The reason is that connections
 * don't go beyond an HTTP proxy that may be present in the chain (as a last
 * element).  In this case the HTTP proxy becomes the destination, and is
 * removed from the proxy chain.  It's client's responsibility to make this
 * adjustment before constructing a ConnectionRoute object.
 */
class ConnectionRoute
{
public:
	// Member-wise copying is OK.
	
	/**
	 * \brief Direct connection (without proxies)
	 */
	ConnectionRoute(SymbolicInetAddr const& destination);
	
	/**
	 * \brief Connection through a proxy chain.
	 *
	 * \note Proxy chain must not contain HTTP proxies.
	 *       \ref proxy_chain "More info"
	 */
	ConnectionRoute(
		std::vector<ProxyDescriptor> const& proxies,
		SymbolicInetAddr const& destination);
	
	/**
	 * \brief Connection through a proxy chain.
	 *
	 * Proxy chain is represented by a pair of iterators pointing to
	 * ProxyDescriptor objects.
	 *
	 * \note Proxy chain must not contain HTTP proxies.
	 *       \ref proxy_chain "More info"
	 */
	template<typename Iter>
	ConnectionRoute(
		Iter begin, Iter end, SymbolicInetAddr const& destination);
	
	~ConnectionRoute();
	
	std::vector<ProxyDescriptor> const& getProxies() const { return m_proxies; }
	
	SymbolicInetAddr const& getDestination() const { return m_destination; }
	
	/**
	 * \brief Get the address of the host we are connected to.
	 *
	 * That would be the first proxy in a chain,
	 * or destination address in case of a direct connecion.
	 */
	SymbolicInetAddr const& getPhysicalPeerAddr() const;
	
	/**
	 * \brief Get the address of the host we are communicating with.
	 *
	 * That would be destination address.
	 */
	SymbolicInetAddr const& getLogicalPeerAddr() const { return m_destination; }
	
	void swap(ConnectionRoute& other);
private:
	std::vector<ProxyDescriptor> m_proxies;
	SymbolicInetAddr m_destination;
};


template<typename Iter>
ConnectionRoute::ConnectionRoute(
	Iter begin, Iter end, SymbolicInetAddr const& destination)
:	m_proxies(begin, end),
	m_destination(destination)
{
}

inline void swap(ConnectionRoute& o1, ConnectionRoute& o2)
{
	o1.swap(o2);
}

bool operator==(ConnectionRoute const& lhs, ConnectionRoute const& rhs);

inline bool operator!=(ConnectionRoute const& lhs, ConnectionRoute const& rhs) {
	return !(lhs == rhs);
}

bool operator<(ConnectionRoute const& lhs, ConnectionRoute const& rhs);

#endif
