/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestLog.h"
#include "RequestTag.h"
#include "AbstractCommand.h"
#include "AbstractRequestLogHandler.h"
#include "InterthreadCommandQueue.h"
#include "DownloadProgress.h"
#include "URI.h"
#include <string>

using namespace std;

/*
Note that we dont't send RequestTag's because they reference data that
is not thread-safe.  Instead we send a RequestId constructed from RequestTag
together with any other useful data extracted from it.
*/

class RequestLog::RequestCommand : public AbstractCommand
{
public:
	RequestCommand(
		RequestId const& req_id,
		std::string const& uri, RequestType type);
	
	virtual ~RequestCommand();
	
	virtual void operator()();
private:
	RequestId m_reqId;
	std::string m_uri;
	RequestType m_type;
};


class RequestLog::ResponseBeginCommand : public AbstractCommand
{
public:
	ResponseBeginCommand(
		RequestId const& req_id, int status_code,
		DownloadProgress const& progress);
	
	virtual ~ResponseBeginCommand();
	
	virtual void operator()();
private:
	DownloadProgress m_progress;
	RequestId m_reqId;
	int m_statusCode;
};


class RequestLog::ResponseEndCommand : public AbstractCommand
{
public:
	ResponseEndCommand(
		RequestId const& req_id, int req_flags,
		DownloadProgress const& progress, bool error);
	
	virtual ~ResponseEndCommand();
	
	virtual void operator()();
private:
	DownloadProgress m_progress;
	RequestId m_reqId;
	int m_reqFlags;
	bool m_error;
};


class RequestLog::RequestCancelCommand : public AbstractCommand
{
public:
	RequestCancelCommand(
		RequestId const& req_id, DownloadProgress const& progress);
	
	virtual ~RequestCancelCommand();
	
	virtual void operator()();
private:
	DownloadProgress m_progress;
	RequestId m_reqId;
};


AbstractRequestLogHandler* RequestLog::m_spHandler = 0;

void
RequestLog::logRequest(
	RequestTag const& tag, URI const& uri, RequestType type)
{
	if (!m_spHandler) {
		return;
	}
	
	m_spHandler->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new RequestCommand(tag, uri.toString(), type)
		)
	);
}

void
RequestLog::logResponseBegin(RequestTag const& tag, int status_code)
{
	if (!m_spHandler) {
		return;
	}
	
	m_spHandler->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new ResponseBeginCommand(
				tag, status_code, tag->downloadProgress()
			)
		)
	);
}

void
RequestLog::doLogResponseEnd(RequestTag const& tag, bool error)
{
	if (!m_spHandler) {
		return;
	}
	
	m_spHandler->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new ResponseEndCommand(
				tag, tag->flags().intValue(),
				tag->downloadProgress(), error
			)
		)
	);
}

void
RequestLog::logRequestCancel(RequestTag const& tag)
{
	if (!m_spHandler) {
		return;
	}
	
	m_spHandler->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new RequestCancelCommand(tag, tag->downloadProgress())
		)
	);
}


/*====================== RequestLog::RequestCommand ======================*/

RequestLog::RequestCommand::RequestCommand(
	RequestId const& req_id, std::string const& uri, RequestType type)
:	m_reqId(req_id),
	m_uri(uri),
	m_type(type)
{
}

RequestLog::RequestCommand::~RequestCommand()
{
}

void
RequestLog::RequestCommand::operator()()
{
	getHandler()->processRequest(m_reqId, m_uri, m_type);
}


/*=================== RequestLog::ResponseBeginCommand ===================*/

RequestLog::ResponseBeginCommand::ResponseBeginCommand(
	RequestId const& req_id, int status_code,
	DownloadProgress const& progress)
:	m_progress(progress),
	m_reqId(req_id),
	m_statusCode(status_code)
{
}

RequestLog::ResponseBeginCommand::~ResponseBeginCommand()
{
}

void
RequestLog::ResponseBeginCommand::operator()()
{
	getHandler()->processResponseBegin(m_reqId, m_statusCode, m_progress);
}


/*==================== RequestLog::ResponseEndCommand ====================*/

RequestLog::ResponseEndCommand::ResponseEndCommand(
	RequestId const& req_id, int req_flags,
	DownloadProgress const& progress, bool error)
:	m_progress(progress),
	m_reqId(req_id),
	m_reqFlags(req_flags),
	m_error(error)
{
}

RequestLog::ResponseEndCommand::~ResponseEndCommand()
{
}

void
RequestLog::ResponseEndCommand::operator()()
{
	getHandler()->processResponseEnd(m_reqId, m_reqFlags, m_progress, m_error);
}


/*=================== RequestLog::ResponseCancelCommand ===================*/

RequestLog::RequestCancelCommand::RequestCancelCommand(
	RequestId const& req_id, DownloadProgress const& progress)
:	m_progress(progress),
	m_reqId(req_id)
{
}

RequestLog::RequestCancelCommand::~RequestCancelCommand()
{
}

void
RequestLog::RequestCancelCommand::operator()()
{
	getHandler()->processRequestCancel(m_reqId, m_progress);
}
