/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Server.h"
#include "ResponseFilterChain.h"
#include "SharedPtr.h"
#include "ServerConnection.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestLine.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "FilterTryList.h"
#include "ImmediateResponse.h"
#include "CraftedResponse.h"
#include "ErrorDescriptor.h"
#include "ErrorFactory.h"
#include "ErrorCodes.h"
#include "LoggingResponseHandler.h"
#include "ServerConnectionPool.h"
#include "BString.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include "ScopedIncDec.h"
#include "AbstractDataConsumer.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "HttpStreamWriter.h"
#include "HttpMessageShaper.h"
#include "GlobalState.h"
#include "Conf.h"
#include "Forwarding.h"
#include "ServiceContext.h"
#include "SymbolicInetAddr.h"
#include "ProxyDescriptor.h"
#include "ConnectionRoute.h"
#include "DownloadProgress.h"
#include "URI.h"
#include "TimeDelta.h"
#include "StringUtils.h"
#include "InsensitiveEqual.h"
#include "NetworkActivityReporter.h"
#include "RequestLog.h"
#include "RequestTag.h"
#include "AsyncConnectorError.h"
#include "SocksError.h"
#include "Debug.h"
#include <algorithm>
#include <list>
#include <vector>
#include <cassert>

using namespace std;

class Server::Request :
	public AbstractRequestHandler,
	public AbstractDataConsumer
{
	DECLARE_NON_COPYABLE(Request)
public:
	Request(ServiceContext& context, ServerWeakPtr const& server_ptr,
		ConstRequestPtr const& request_metadata,
		RequestTag const& request_tag,
		IntrusivePtr<AbstractResponseHandler> const& response_handler);
	
	virtual ~Request();
	
	AbstractResponseHandler& responseHandler() { return *m_ptrResponseHandler; }
	
	bool hasData() const { return !m_data.empty() && !m_ptrErrDesc.get(); }
	
	SplittableBuffer& data() { return m_data; }
	
	SplittableBuffer const& data() const { return m_data; }
	
	bool isEOF() const { return m_isEOF; }
	
	bool isRestartable() const;
	
	bool isConsumingStarted() const { return m_isConsumingStarted; }
	
	void setConsumingStarted(bool val = true) { m_isConsumingStarted = val; }
	
	bool isFullyConsumed() const { return data().empty() && isEOF(); }
	
	bool restart();
	
	std::auto_ptr<ErrorDescriptor> const& errorDescriptor() {
		return m_ptrErrDesc;
	}
	
	std::auto_ptr<ErrorDescriptor> retrieveErrorDescriptor() {
		return m_ptrErrDesc;
	}
	
	ConstRequestPtr const& getOrigMetadata() const {
		return m_ptrOrigMetadata;
	}
	
	HttpRequestLine const& getRequestLine() const {
		return m_ptrOrigMetadata->requestLine();
	}
	
	URI const& getRequestURI() const {
		return getRequestLine().getURI();
	}
	
	HttpHeadersCollection const& getRequestHeaders() const {
		return m_ptrOrigMetadata->headers();
	}
	
	HttpRequestMetadata const& getOutgoingMetadata() const {
		return *m_ptrOutgoingMetadata;
	}
	
	bool isConnectRequest() const {
		return getRequestLine().getMethod() == BString("CONNECT");
	}
	
	RequestTag const& getTag() const { return m_tag; }
	
	ConnectionRoute const& getConnectionRoute() const { return m_connRoute; }
	
	bool isGoingThroughHttpProxy() const { return m_connDestIsHttpProxy; }
	
	void updateRoute();
	
	virtual void cancel(std::auto_ptr<ErrorDescriptor> edesc);
	
	virtual bool isCancelled() const;
private:
	virtual void appendBodyData(SplittableBuffer& data, bool eof);
	
	virtual void processNewData(SplittableBuffer& data, bool eof);
	
	void maybeReportClientIP(Config const& conf);
	
	void updateRoute(Forwarding::Resolver const& resolver);
	
	ServiceContext& m_rContext;
	ServerWeakPtr m_ptrServer;
	ConstRequestPtr m_ptrOrigMetadata;
	std::auto_ptr<HttpRequestMetadata> m_ptrOutgoingMetadata;
	RequestTag m_tag;
	IntrusivePtr<AbstractResponseHandler> m_ptrResponseHandler;
	ConnectionRoute m_connRoute;
	bool m_connDestIsHttpProxy;
	bool m_isCancelled;
	std::auto_ptr<ErrorDescriptor> m_ptrErrDesc;
	SplittableBuffer m_data;
	bool m_isEOF;
	bool m_isConsumingStarted;
	bool m_isConstructionDone;
	HttpStreamWriter m_streamWriter;
};


class Server::PhantomRequest : public AbstractRequestHandler
{
	DECLARE_NON_COPYABLE(PhantomRequest)
public:
	PhantomRequest(
		ConstRequestPtr const& response_metadata,
		RequestTag const& request_tag,
		IntrusivePtr<AbstractResponseHandler> const& response_handler,
		std::auto_ptr<ImmediateResponse> response);
	
	~PhantomRequest();
	
	AbstractResponseHandler& responseHandler() {
		return *m_ptrResponseHandler;
	}
	
	ImmediateResponse& response() { return *m_ptrResponse; }
	
	ConstRequestPtr const& getRequestPtr() const {
		return m_ptrOrigMetadata;
	}
	
	HttpRequestMetadata const& getRequestMetadata() const {
		return *m_ptrOrigMetadata;
	}
	
	HttpRequestLine const& getRequestLine() const {
		return m_ptrOrigMetadata->requestLine();
	}
	
	URI const& getRequestURI() const {
		return getRequestLine().getURI();
	}
	
	RequestTag const& getTag() const { return m_tag; }
	
	std::auto_ptr<ErrorDescriptor> const& errorDescriptor() {
		return m_ptrErrDesc;
	}
	
	std::auto_ptr<ErrorDescriptor> retrieveErrorDescriptor() {
		return m_ptrErrDesc;
	}
private:
	virtual void appendBodyData(SplittableBuffer& data, bool eof);
	
	virtual void cancel(std::auto_ptr<ErrorDescriptor> edesc);
	
	virtual bool isCancelled() const;
	
	ConstRequestPtr m_ptrOrigMetadata;
	RequestTag m_tag;
	IntrusivePtr<AbstractResponseHandler> m_ptrResponseHandler;
	std::auto_ptr<ImmediateResponse> m_ptrResponse;
	std::auto_ptr<ErrorDescriptor> m_ptrErrDesc;
	bool m_isCancelled;
};


class Server::ScopedSuspender
{
public:
	ScopedSuspender(Server& owner) : m_rOwner(owner) {
		m_rOwner.suspend();
	}
	
	~ScopedSuspender() {
		m_rOwner.resume();
	}
private:
	Server& m_rOwner;
};


class Server::RequestCanceller
{
public:
	void operator()(IntrusivePtr<Request> const& req) {
		RequestLog::logRequestCancel(req->getTag());
	}
	
	void operator()(IntrusivePtr<PhantomRequest> const& req) {
		if (req) {
			RequestLog::logRequestCancel(req->getTag());
		}
	}
};


Server::Server(ServiceContext& context, ServerTimeouts const& timeouts)
:	m_rContext(context),
	m_timeouts(timeouts),
	m_pipelineSize(0),
	m_lastResponseVersion(0, 0),
	m_readTimer(
		sigc::mem_fun(*this, &Server::onReadTimeout),
		m_timeouts.getReadTimeout(),
		&context.reactor()
	),
	m_writeTimer(
		sigc::mem_fun(*this, &Server::onWriteTimeout),
		m_timeouts.getWriteTimeout(),
		&context.reactor()
	),
	m_delayedActionTimer(
		sigc::mem_fun(*this, &Server::onDelayedAction),
		TimeDelta::zero(),
		&context.reactor()
	),
	m_eventMask(Reactor::NO_EVENTS),
	m_flags(0),
	m_isWaitingForWrite(0),
	m_ptrSelf(this)
{
	m_rContext.reactor().beforeSleepSignal().connect(
		sigc::mem_fun(*this, &Server::updateState)
	);
}

Server::~Server()
{
}

IntrusivePtr<AbstractRequestHandler>
Server::submitRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	IntrusivePtr<Request> request(new Request(
		m_rContext, m_ptrSelf, request_metadata,
		request_tag, handler
	));
	validateRequest(*request);
	enqueueRequest(request);
	updatePipeline();
	return request;
}

IntrusivePtr<AbstractRequestHandler>
Server::submitRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler,
	std::auto_ptr<ImmediateResponse> response_generator)
{
	IntrusivePtr<PhantomRequest> request(new PhantomRequest(
		request_metadata, request_tag,
		handler, response_generator
	));
	enqueuePhantomRequest(request);
	return request;
}

bool
Server::isIdle() const
{
	assert(m_phantomRequestQueue.size() >= m_requestQueue.size());
	return m_phantomRequestQueue.empty();
}

bool
Server::isOverloaded() const
{
	assert(m_phantomRequestQueue.size() >= m_requestQueue.size());
	return m_phantomRequestQueue.size() > MAX_PIPELINE_SIZE;
}

void
Server::reset()
{
	storeConnection();
	m_connector.abort();
	assert(!m_ptrConnection.get());
	assert(!m_handlerId);
	assert(!m_readTimer.isActive());
	assert(!m_writeTimer.isActive());
	
	RequestCanceller rc;
	std::for_each(m_requestQueue.begin(), m_requestQueue.end(), rc);
	std::for_each(m_phantomRequestQueue.begin(), m_phantomRequestQueue.end(), rc);
	
	m_requestQueue.clear();
	m_phantomRequestQueue.clear();
	m_pipelineSize = 0;
	
	m_inputData.clear();
	m_lastResponseVersion.set(0, 0);
	m_delayedActionTimer.deactivate();
	m_flags &= ~WRITE_BUFFER_FULL;
}

void
Server::enqueueRequest(IntrusivePtr<Request> const& req)
{
	m_requestQueue.push_back(req);
	m_phantomRequestQueue.push_back(IntrusivePtr<PhantomRequest>());
	
	RequestLog::RequestType type = RequestLog::NORMAL_REQUEST;
	if (m_rContext.isScriptServer(this)) {
		type = RequestLog::INTERNAL_REQUEST;
	} else if (req->getTag()->flags().isSet(RequestTag::REQUEST_ANALYZE)) {
		type = RequestLog::ANALYZE_REQUEST;
	}
	RequestLog::logRequest(req->getTag(), req->getRequestURI(), type);
}

void
Server::enqueuePhantomRequest(IntrusivePtr<PhantomRequest> const& req)
{
	m_phantomRequestQueue.push_back(req);
	
	RequestLog::RequestType type = RequestLog::NORMAL_REQUEST;
	if (req->getTag()->flags().isSet(RequestTag::REQUEST_AD_SUBST)) {
		type = RequestLog::SUBST_REQUEST;
	}
	RequestLog::logRequest(req->getTag(), req->getRequestURI(), type);
}

void
Server::updatePipeline()
{	
	assert(m_requestQueue.size() >= m_pipelineSize);
	
	do {
		/*
		DEBUGLOG("queue size = " << m_requestQueue.size()
			<< ", phantom queue size = " << m_phantomRequestQueue.size()
			<< ", pipeline size = " << m_pipelineSize);
		*/
		if (m_requestQueue.size() == m_pipelineSize) {
			//DEBUGLOG("nothing to do");
			if (m_pipelineSize == 0) {
				//DEBUGLOG("storing connection");
				storeConnection();
			}
			break;
		}
		
		Request& next_req = *m_requestQueue[m_pipelineSize];
		
		if (next_req.isCancelled()) {
			break;
		}
		
		if (m_pipelineSize > 0) {
			Request const& prev_req = *m_requestQueue[m_pipelineSize - 1];
			
			if (!m_ptrConnection.get()) {
				//DEBUGLOG("We don't have a connection yet");
				break;
			}
			if (m_ptrConnection->getNumResponses() == 0) {
				//DEBUGLOG("No responses received on this connection");
				break;
			}
			if (m_ptrConnection->getMaxResponseVersion() < HttpVersion::HTTP_1_1) {
				//DEBUGLOG("Pipelining is only useful for servers that support HTTP/1.1");
				break;
			}
			if (!prev_req.isFullyConsumed()) {
				// This will also catch the case of a CONNECT request.
				//DEBUGLOG("The previous request is not complete");
				break;
			}
			if (m_ptrConnection->getRoute() != next_req.getConnectionRoute()) {
				//DEBUGLOG("Request to a different address or through a different proxy chain");
				break;
			}
			if (!m_ptrConnection->isExpectedToPersist()) {
				//DEBUGLOG("Connection is not expected to persist");
				break;
			}
			if (m_pipelineSize >= MAX_PIPELINE_SIZE) {
				//DEBUGLOG("MAX_PIPELINE_SIZE reached");
				break;
			}
			if (!prev_req.isRestartable() || !next_req.isRestartable()) {
				// only requests without a body may be pipelined
				//DEBUGLOG("Either this or the previous request is not restartable");
				break;
			}
		}
		
		//DEBUGLOG("extending pipeline");
		
		++m_pipelineSize;
		
		if (m_pipelineSize == 1 && isNewConnectionRequiredFor(next_req)) {
			//DEBUGLOG("new connection is required");
			storeConnection();
		}
	} while (false);
}

bool
Server::validateRequest(Request& req)
{
	if (!validateRequestProtocol(req)) {
		return false;
	}
	if (!validateRequestExpectations(req)) {
		return false;
	}
	if (!validateTunnelPort(req)) {
		return false;
	}
	return true;
}

bool
Server::validateRequestProtocol(Request& req)
{
	if (req.isConnectRequest()) {
		return true;
	}
	
	BString const http("http");
	URI const& uri = req.getRequestURI();
	BString const& scheme = uri.getScheme();
	if (!StringUtils::ciEqual(scheme.begin(), scheme.end(), http.begin(), http.end())) {
		req.cancel(
			ErrorFactory::errUnsupportedProtocol(
				ErrorCodes::UNSUPPORTED_PROTOCOL,
				"unsupported protocol",
				uri, uri.getScheme().toStdString()
			)
		);
		return false;
	}
	
	return true;
}

bool
Server::validateRequestExpectations(Request& req)
{
	URI const& uri = req.getRequestURI();
	HttpHeadersCollection const& headers = req.getRequestHeaders();
	HttpHeader const* expect_hdr = headers.getHeaderPtr(BString("Expect"));
	if (expect_hdr) {
		HttpHeaderStructure structure(*expect_hdr);
		list<HttpHeaderElement>::iterator it = structure.elements().begin();
		list<HttpHeaderElement>::iterator const end = structure.elements().end();
		for (; it != end; ++it) {
			if (!InsensitiveEqual()(it->getName(), BString("100-continue"))) {
				req.cancel(
					ErrorFactory::errExpectationFailed(
						ErrorCodes::EXPECTATION_FAILED,
						"expectation failed", uri,
						it->getName().toStdString()
					)
				);
				return false;
			}
		}
	}
	
	return true;
}

bool
Server::validateTunnelPort(Request& req)
{
	if (!req.isConnectRequest()) {
		return true;
	}
	
	URI const& uri = req.getRequestURI();
	unsigned const port = uri.guessPort();
	if (!GlobalState::ReadAccessor()->config().isTunnelPortAllowed(port)) {
		req.cancel(
			ErrorFactory::errUrlForbidden(
				ErrorCodes::TUNNEL_PORT_NOT_ALLOWED,
				"tunnel port not allowed", uri
			)
		);
		return false;
	}
	
	return true;
}

bool
Server::isNewConnectionRequiredFor(Request const& req) const
{
	if (!m_ptrConnection.get()) {
		return true;
	}
	if (!m_ptrConnection->isExpectedToPersist()) {
		return true;
	}
	if (m_ptrConnection->getRoute() != req.getConnectionRoute()) {
		return true;
	}
	if (!req.isRestartable()) {
		return true;
	}
	return false;
}

void
Server::requestConnectionFor(Request const& req)
{
	assert(!m_ptrConnection.get());
	if (req.isRestartable()) {
		// we always use a fresh connection for unrestartable requests
		// (requests with bodies)
		auto_ptr<ServerConnection> conn(
			ServerConnectionPool::instance()->retrieveConnection(
				req.getConnectionRoute()
			)
		);
		if (conn.get()) {
			//DEBUGLOG("Reusing an existing server connection");
			processNewHttpConnection(conn);
			return;
		}
	}
	
	//DEBUGLOG("Initiating a new connection");
	m_connector.initiate(
		*this, m_rContext.reactor(), req.getConnectionRoute(),
		&m_timeouts.getConnectTimeout() // FIXME: need a different timeout
	);
}

bool
Server::isOutputReady() const
{
	return m_pipelineSize > 0 && m_requestQueue[m_pipelineSize - 1]->hasData();
}

size_t
Server::getPendingOutputSize() const
{
	size_t size = 0;
	typedef deque<IntrusivePtr<Request> >::const_iterator Iter;
	Iter it = m_requestQueue.begin();
	Iter const end = m_requestQueue.end();
	for (; it != end; ++it) {
		size += (*it)->data().size();
	}
	return size;
}

void
Server::onNewOutput()
{
	if (getPendingOutputSize() > MAX_WRITE_BUF_SIZE) {
		ScopedIncDec<int> flag_manager(m_isWaitingForWrite);
		while (isOutputReady()) {
			if (m_rContext.reactor().handleEvents() != Reactor::SUCCESS) {
				break;
			}
		}
	}
}

void
Server::processOutputData()
{
	while (m_pipelineSize > 0 && m_ptrConnection.get()) {
		if (writeRequestData(*m_requestQueue[m_pipelineSize - 1]) == 0) {
			break;
		}
	}
}

size_t
Server::writeRequestData(Request& req)
{
	if (!req.hasData()) {
		return 0;
	}

#if defined(ACE_WIN32)
	/*
	There are certain buggy software firewalls that have problems with
	multi-fragment writes. We workaround it by coalescing fragments and
	doing a single write. Fortunately, we only have to do it for requests
	(where it has almost no effect on performance), as I haven't
	encountered a firewall that has problems with fragmented responses.
	*/
	BString buf(req.data().toBString());
	ssize_t sent = m_ptrConnection->peer().send(buf.begin(), buf.size());
	buf.clear(); // just a small optimization of memory usage
#else
	iovec iov[ACE_IOV_MAX];
	int iov_size = 0;
	for (SplittableBuffer::SpanIterator iter(req.data().spanBegin());
	     iov_size < int(sizeof(iov)/sizeof(*iov)) && !iter.isAtRightBorder();
	     ++iov_size, ++iter) {
		iov[iov_size].iov_base = (char*)iter->begin();
		iov[iov_size].iov_len = iter->size();
	}
	ssize_t sent = m_ptrConnection->peer().sendv(
		iov, iov_size
		// the timeout is infinite, but the socket is in nonblocking mode
	);
#endif
	if (sent < 0 && errno == EWOULDBLOCK) {
		m_flags |= WRITE_BUFFER_FULL;
	} else if (sent > 0) {
		m_flags &= ~WRITE_BUFFER_FULL;
		m_writeTimer.postpone();
		NetworkActivityReporter::instance()->reportActivity();
#ifdef DEBUG
		bool const is_script_server = m_rContext.isScriptServer(this);
		if (!req.isConsumingStarted()) {
			AbstractDebugAgent::HttpMessageType msg_type =
				is_script_server
				? AbstractDebugAgent::SCRIPT_REQUEST
				: AbstractDebugAgent::OUTGOING_REQUEST;
			DBG_HTTP_MESSAGE_BEGIN(req.getOutgoingMetadata(), req.getTag(), msg_type);
		}
		
		SplittableBuffer traf;
		req.data().splitFrontAppendBack(sent, traf);
		if (is_script_server) {
			DBG_TRAFFIC_TO_SCRIPT_SERVER(traf);
		} else {
			DBG_TRAFFIC_TO_SERVER(traf);
		}
#else
		req.data().trimFront(sent);
#endif
		
		req.setConsumingStarted();
		if (req.isFullyConsumed()) {
			//DEBUGLOG("Request " << req.getRequestURI() << " has been fully consumed");
#ifdef DEBUG
			dbgReportOutgoingRequestSent();
#endif
			updatePipeline();
		}
		return sent;
	} else {
		//DEBUGLOG("Error writing data to server");
		/*
		We don't drop the connection here because there may still be some pending data
		in the socket buffer, which may even complete some of the pending requests.
		There is also no need to unregister the WRITE_MASK, as sockets in error
		state don't generate write events.
		*/
	}
	return 0;
}

void
Server::processInputData(bool eof)
{
	while (true) {
		if (m_inputData.empty() && !eof) {
			break;
		}
		
		if (m_pipelineSize == 0) {
			assert(m_requestQueue.empty());
			DEBUGLOG("Unexpected data (or EOF) received from server");
			// we don't need to call restartPendingRequests() because m_pipelineSize == 0
			dropConnection();
			m_inputData.clear();
			updatePipeline();
			break;
		}
		
		assert(!m_requestQueue.empty());
		IntrusivePtr<Request> req(m_requestQueue.front());
		
		if (isInactiveState()) {
			if (req->isConnectRequest() && !req->isGoingThroughHttpProxy()) {
				switchToUnsizedFlatBody();
			} else {
				switchToResponseStart(req->getOrigMetadata());
			}
		}
		
		SplittableBuffer body_data;
		bool body_eof = false;
		bool const empty_input = m_inputData.empty();
		Status const status = processNewData(
			m_inputData, eof, body_data, body_eof
		);
		
		req->getTag()->downloadProgress().received() += body_data.size();
		
		if (!req->isCancelled()) {
			if (status == MESSAGE_ENDS) {
#ifdef DEBUG
				AbstractDebugAgent::HttpMessageType msg_type =
					m_rContext.isScriptServer(this)
					? AbstractDebugAgent::SCRIPT_RESPONSE
					: AbstractDebugAgent::INCOMING_RESPONSE;
				DBG_HTTP_MESSAGE_END(msg_type);
#endif
			}
			
			if (!body_data.empty() || body_eof) {
				ScopedSuspender suspender(*this);
				
				// This may result in a recursive invocation
				// of reactor event loop.
				req->responseHandler().processBodyData(
					*req, body_data, body_eof
				);
			}
		}
		
		if (m_pipelineSize == 0 || m_requestQueue.front() != req) {
			// Removed from the pipeline by recursive invocation
			// of reactor event loop
		} else if (req->errorDescriptor().get()) {
			assert(req->isCancelled());
			processImmediateResponses();
			// This will drop the server connection if neccessary.
			// Dropping the connection also causes switchToInactive() to be called.
		} else if (status == MESSAGE_ENDS) {
			DEBUGLOG("Response finishes: " << req->getRequestURI());
			assert(isInactiveState()); // MESSAGE_ENDS implies that
			if (m_ptrConnection.get()) {
				m_ptrConnection->registerResponse(m_lastResponseVersion);
			}
			shiftRequestQueue();
			if (eof || !m_ptrConnection.get() ||
			    !m_ptrConnection->isExpectedToPersist()) {
				restartPendingRequests();
				dropConnection();
				// m_inputData may still contain some useful data
			}
			processImmediateResponses();
			updatePipeline();
		} else if (status == ERROR_STATE) {
			// keep in mind: an eof while in ResponseStart state generates an error
			/*
			DEBUGLOG("ERROR parsing response: "
				<< (errorDescriptor().get()
				? errorDescriptor()->getDebugMessage()
				: errorMessage()));
			DEBUGLOG("empty_input = " << empty_input
				<< ", have_connection = " << (m_ptrConnection.get() ? true : false)
				<< ", num_responses = " << (m_ptrConnection.get() ? m_ptrConnection->getNumResponses() : 0)
				<< ", request_restartable = " << m_requestQueue.front()->isRestartable());
			*/
			/*
			This could be an OK situation:
			You have two requests in queue, the server respondes on the first one
			and immediately closes the connection. An EOF while in ResponseStart state
			generates an error.
			*/
			if (empty_input && (m_ptrConnection.get()
			    ? m_ptrConnection->getNumResponses() > 0 : true)
			    && m_requestQueue.front()->isRestartable()) {
				/*
				Q: Why do we check for empty_input?
				A: The only case when an error is generated after processing an empty input
				is in the ResponseStart (or RequestStart) state.
				Q: Why do we check for getNumResponses() > 0?
				A: Because otherwise we could get an infinite loop if the server always
				closes the connection instead of answering the response.
				Q: Why do we allow the lack of connection?
				A: Because if the connection that we've dropped had zero responses,
				we'd already generated the appropriate response and moved on to the next request.
				*/
			} else {
				/*
				DEBUGLOG("Request URI: " << m_requestQueue.front()->getRequestURI());
				DEBUGLOG("empty_input = " << empty_input << ", got_responses = "
					<< (m_ptrConnection->getNumResponses() > 0));
				*/
#ifdef DEBUG
				AbstractDebugAgent::HttpMessageType msg_type =
					m_rContext.isScriptServer(this)
					? AbstractDebugAgent::SCRIPT_RESPONSE
					: AbstractDebugAgent::INCOMING_RESPONSE;
				DBG_HTTP_MESSAGE_ERROR(msg_type);
#endif
				if (errorDescriptor().get()) {
					answerWithError(retrieveErrorDescriptor());
				} else {
					answerWithError(
						ErrorFactory::errParsingResponse(
							ErrorCodes::ERROR_PARSING_RESPONSE,
							"error parsing response: "+errorMessage(),
							m_requestQueue.front()->getRequestURI(),
							errorMessage()
						)
					);
				}
			}
			restartPendingRequests();
			dropConnection();
			m_inputData.clear();
			updatePipeline();
			break;
		}
		
		if (status == NEED_MORE_DATA) {
			// Can happen for example if input ends in the middle of a header.
			break;
		}
		
		if (eof && m_inputData.empty()) {
			// We have to break here, or the eof will be interpreted
			// as a response to the next queued request (if any).
			break;
		}
	}
	if (eof) {
		dropConnection();
	}
}

void
Server::dbgReportOutgoingRequestSent()
{
#ifdef DEBUG
	AbstractDebugAgent::HttpMessageType msg_type =
		m_rContext.isScriptServer(this)
		? AbstractDebugAgent::SCRIPT_REQUEST
		: AbstractDebugAgent::OUTGOING_REQUEST;
	DBG_HTTP_MESSAGE_END(msg_type);
#endif
}

void
Server::shiftRequestQueue()
{
	assert(!m_requestQueue.empty());
	assert(!m_phantomRequestQueue.empty());
	m_requestQueue.pop_front();
	m_phantomRequestQueue.pop_front();
	if (m_pipelineSize > 0) {
		--m_pipelineSize;
	}
}

bool
Server::connectionMayBeStored() const
{
	if (!m_ptrConnection.get()) {
		return false;
	}
	if (!m_ptrConnection->isExpectedToPersist()) {
		return false;
	}
	if (m_ptrConnection->getNumResponses() == 0) {
		return false;
	}
	if (m_pipelineSize != 0 && m_requestQueue.front()->isConsumingStarted()) {
		return false;
	}
	return true;
}

void
Server::storeConnection()
{
	if (m_ptrConnection.get()) {
		if (!connectionMayBeStored()) {
			dropConnection();
		} else {
			auto_ptr<ServerConnection> conn(retrieveConnection());
			assert(conn.get());
			ServerConnectionPool::instance()->storeConnection(conn);
			//DEBUGLOG("connection stored");
		}
	}
}

void
Server::dropConnection()
{
	retrieveConnection();
}

std::auto_ptr<ServerConnection>
Server::retrieveConnection()
{
	auto_ptr<ServerConnection> conn;
	
	if (m_ptrConnection.get()) {
		m_readTimer.deactivate();
		m_writeTimer.deactivate();
		
		if (m_handlerId) {
			m_rContext.reactor().unregisterHandler(m_handlerId);
			m_handlerId = ReactorHandlerId();
		}
		
		conn = m_ptrConnection;
		m_eventMask = Reactor::NO_EVENTS;
		resetFlags();
		switchToInactive();
	}
	
	return conn;
}

void
Server::restartPendingRequests()
{
	if (m_pipelineSize > 0) {
		for (size_t i = 0; i < m_pipelineSize; ++i) {
			m_requestQueue[i]->restart();
		}
		m_pipelineSize = 0;
		m_connector.abort();
	}
}

void
Server::answerWithError(std::auto_ptr<ErrorDescriptor> edesc)
{
	assert(!m_requestQueue.empty());
	
	Request& req = *m_requestQueue.front();
	
	{
		ScopedSuspender suspender(*this);
		req.responseHandler().processError(req, edesc);
	}
	
	shiftRequestQueue();
}

bool
Server::isRequestAtPipelineStart(Request const* req) const
{
	return (m_pipelineSize > 0 && &*m_requestQueue.front() == req);
}

void
Server::onReadTimeout()
{
	DEBUGLOG("server read timeout");
	onTimeout();
}

void
Server::onWriteTimeout()
{
	DEBUGLOG("server write timeout");
	onTimeout();
}

void
Server::onTimeout()
{
	URI const& uri = m_requestQueue.front()->getRequestURI();
	answerWithError(
		ErrorFactory::errServerTimeout(
			ErrorCodes::SERVER_TIMEOUT,
			"server timeout", uri
		)
	);
	restartPendingRequests();
	dropConnection();
	updatePipeline();
}

void
Server::updateState()
{
	/*
	upadateState() MUST NOT generate any output!!!
	Server::updateState() and Client::updateState() are called just before
	going to sleep to update their event masks. If one of them generates
	output for the other, that other one would need to update its event
	mask, but it may not have a chance to do so, as it may have already run
	in the current reactor dispatch.
	
	Functions that can generate output include processArtificialResponses()
	and requestConnectionFor().
	*/
	
	if (!(m_flags & READING_SUSPENDED)) {
		if (isImmediateResponseRequired()) {
			m_delayedActionTimer.ensureActive();
		} else if (m_pipelineSize > 0 && !m_ptrConnection.get() &&
		           !m_connector.isInProgress()) {
			assert(m_pipelineSize == 1);
			Request& req = *m_requestQueue.front();
			if (req.hasData() || req.isConnectRequest()) {
				m_delayedActionTimer.ensureActive();
			}
		}
	}
	
	if (!m_ptrConnection.get()) {
		return; // events and timers are already unregistered / deactivated
	}
	
	if (!(m_flags & WRITE_BUFFER_FULL) && isOutputReady()) {
		// Necessary on the edge-triggered WFMOReactor.
		processOutputData();
	}
	
	
	Reactor::IOEvents mask = m_eventMask;
	
	if (m_flags & READING_SUSPENDED) {
		mask &= ~Reactor::READ;
	} else {
		mask |= Reactor::READ;
	}
	if (isOutputReady()) {
		mask |= Reactor::WRITE;
	} else {
		mask &= ~Reactor::WRITE;
		if (m_isWaitingForWrite) {
			m_rContext.reactor().wakeup();
		}
		
		m_flags &= ~WRITE_BUFFER_FULL;
		// It may actually be full, but we don't want to miss
		// an edge-triggered (in case of WFMOReactor) write event.
	}
	
	if ((mask & Reactor::READ) && m_pipelineSize > 0 &&
	    m_requestQueue.front()->isFullyConsumed()) {
		m_readTimer.ensureActive();
	} else {
		m_readTimer.deactivate();
	}
	if (mask & Reactor::WRITE) {
		m_writeTimer.ensureActive();
	} else {
		m_writeTimer.deactivate();
	}
	
	if (mask != m_eventMask) {
		m_rContext.reactor().setEvents(m_handlerId, mask);
		m_eventMask = mask;
	}
}

bool
Server::processLeadingPhantomResponse()
{
	if (!m_phantomRequestQueue.empty()) {
		if (PhantomRequest* req = m_phantomRequestQueue.front().get()) {
			ScopedSuspender suspender(*this);
			
			req->response().output(
				req->responseHandler(),
				req->getRequestPtr(), *req, req->getTag()
			);
			
			if (req->errorDescriptor().get()) {
				req->responseHandler().processError(
					*req, req->retrieveErrorDescriptor()
				);
			}
			
			m_phantomRequestQueue.pop_front();
			return true;
		}
	}
	return false;
}

bool
Server::processLeadingErrorResponse()
{
	if (!m_requestQueue.empty()) {
		Request& req = *m_requestQueue.front();
		if (req.isCancelled()) {
			assert(req.errorDescriptor().get());
			if (!req.isConsumingStarted()) {
				answerWithError(req.retrieveErrorDescriptor());
			} else {
#ifdef DEBUG
				if (!req.isFullyConsumed()) {
					AbstractDebugAgent::HttpMessageType msg_type =
						m_rContext.isScriptServer(this)
						? AbstractDebugAgent::SCRIPT_REQUEST
						: AbstractDebugAgent::OUTGOING_REQUEST;
					DBG_HTTP_MESSAGE_ERROR(msg_type);
				}
#endif
				answerWithError(req.retrieveErrorDescriptor());
				restartPendingRequests();
				dropConnection();
				m_inputData.clear();
			}
			// Note: we have answerWithError() in both branches, which
			// calls shiftRequestQueue(), so the request is removed
			// from the queue.
			return true;
		}
	}
	return false;
}

void
Server::processImmediateResponses()
{
	bool need_update_pipeline = false;
	while (true) {
		bool phantom_processed = processLeadingPhantomResponse();
		bool error_processed = processLeadingErrorResponse();
		if (phantom_processed || error_processed) {
			need_update_pipeline |= error_processed;
			continue;
		}
		break;
	};
	if (need_update_pipeline) {
		updatePipeline();
	}
}

bool
Server::isImmediateResponseRequired() const
{
	if (!m_phantomRequestQueue.empty() && m_phantomRequestQueue.front().get()) {
		return true;
	}
	if (!m_requestQueue.empty() && m_requestQueue.front()->isCancelled()) {
		return true;
	}
	return false;
}

void
Server::eventGotProvisionalResponse(std::auto_ptr<HttpResponseMetadata> metadata)
{
	Request& req = *m_requestQueue.front();
	ScopedSuspender suspender(*this);
	req.responseHandler().processProvisionalResponse(req, metadata);
}

void
Server::eventGotMetadata(
	std::auto_ptr<HttpResponseMetadata> metadata, bool is_persistent)
{
	m_lastResponseVersion = metadata->statusLine().getHttpVersion();
	Request& req = *m_requestQueue.front();
	
#ifdef DEBUG
	DEBUGLOG("Response begins: " << req.getRequestURI());
	
	AbstractDebugAgent::HttpMessageType msg_type =
		m_rContext.isScriptServer(this)
		? AbstractDebugAgent::SCRIPT_RESPONSE
		: AbstractDebugAgent::INCOMING_RESPONSE;
	DBG_HTTP_MESSAGE_BEGIN(*metadata, req.getTag(), msg_type);
#endif
	
	HttpMessageShaper::prepareForForwarding(*metadata);
	
	if (!is_persistent && m_ptrConnection.get()) {
		m_ptrConnection->toBeClosed();
	}
	
	DownloadProgress& dp = req.getTag()->downloadProgress();
	dp.received() = 0;
	if (metadata->getBodyStatus() == HttpResponseMetadata::BODY_SIZE_UNKNOWN) {
		dp.total() = DownloadProgress::UNDEFINED;
	} else {
		dp.total() = metadata->getBodySize();
	}
	
	ScopedSuspender suspender(*this);
	req.responseHandler().processResponseMetadata(req, metadata);
}

void
Server::onConnectionEstablished(AutoClosingSAP<ACE_SOCK_Stream>& conn)
{
	//DEBUGLOG("Connection established");
	processNewHttpConnection(conn);
}

void
Server::onConnectionFailed(
	AsyncConnectorError const& err, SymbolicInetAddr const& addr)
{
	DEBUGLOG("Connection failed");
	assert(m_pipelineSize > 0);
	assert(m_requestQueue.size() >= m_pipelineSize);
	
	m_ptrConnection.reset(0); // Do we need this?
	Request const& req = *m_requestQueue.front();
	URI const& uri = req.getRequestURI();
	
	switch (err.getCode()) {
		case AsyncConnectorError::NAME_RESOLUTION_FAILED: {
			answerWithError(
				ErrorFactory::errHostNameResolution(
					ErrorCodes::DNS_ERROR, err.toString(),
					uri, addr.getHost()
				)
			);
			break;
		}
		case AsyncConnectorError::CONNECTION_TIMEOUT: {
			answerWithError(
				ErrorFactory::errConnectionTimeout(
					ErrorCodes::CONNECT_TIMEOUT, err.toString(),
					uri, addr.getHost(), addr.getPort()
				)
			);
			break;
		}
		case AsyncConnectorError::CONNECTION_REFUSED:
		case AsyncConnectorError::DESTINATION_UNREACHABLE:
		case AsyncConnectorError::GENERIC_ERROR:
		default: {
			answerWithError(
				ErrorFactory::errConnectionFailed(
					ErrorCodes::CONNECT_FAILED,
					"connection failed", uri,
					addr.getHost(), addr.getPort(),
					err.toString()
				)
			);
			break;
		}
	}
	updatePipeline();
}

void
Server::onConnectionFailed(SocksError const& err, ProxyDescriptor const& proxy)
{
	DEBUGLOG("Forwarding through a socks proxy failed");
	handleForwardingProblem(err.toString(), proxy);
}

void
Server::onConnectionFailed(std::string const& err_msg, ProxyDescriptor const& proxy)
{
	DEBUGLOG("Forwarding through a proxy failed");
	handleForwardingProblem(err_msg, proxy);
}

void
Server::handleForwardingProblem(
	std::string const& problem, ProxyDescriptor const& proxy)
{
	assert(m_pipelineSize > 0);
	assert(m_requestQueue.size() >= m_pipelineSize);
	
	m_ptrConnection.reset(0); // Do we need this?
	Request const& req = *m_requestQueue.front();
	URI const& uri = req.getRequestURI();
	
	answerWithError(
		ErrorFactory::errForwardingThroughProxy(
			ErrorCodes::ERROR_FORWARDING_THROUGH_PROXY, problem,
			uri, problem, proxy.getTypeAsString(),
			proxy.getAddr().getHost(), proxy.getAddr().getPort()
		)
	);
	
	updatePipeline();
}

void
Server::handleRead(ACE_HANDLE)
{
	if (m_flags & READING_SUSPENDED) {
		// this may happen because the event mask is updated after
		// all of the triggered events are processed
		return;
	}
	
	processImmediateResponses(); // it may drop the connection
	if (!m_ptrConnection.get()) {
		return;
	}
	
	auto_ptr<DataChunk> chunk = DataChunk::create(READ_BUF_SIZE);
	ssize_t received = m_ptrConnection->peer().recv(
		chunk->getDataAddr(), chunk->getDataSize()
		// the timeout is infinite, but the socket is in nonblocking mode
	);
	if (received > 0) {
		NetworkActivityReporter::instance()->reportActivity();
		chunk = DataChunk::resize(chunk, received);
#ifdef DEBUG
		SplittableBuffer traf;
		traf.append(chunk);
		if (m_rContext.isScriptServer(this)) {
			DBG_TRAFFIC_FROM_SCRIPT_SERVER(traf);
		} else {
			DBG_TRAFFIC_FROM_SERVER(traf);
		}
		m_inputData.appendDestructive(traf);
#else
		m_inputData.append(chunk);
#endif
	} else if (received < 0 && errno == EWOULDBLOCK) {
		// not a problem
		return;
	}
	m_readTimer.postpone();
	bool eof = (received <= 0);
	processInputData(eof);
}

void
Server::handleWrite(ACE_HANDLE)
{
	m_flags &= ~WRITE_BUFFER_FULL;
	processOutputData();
}

void
Server::processNewHttpConnection(AutoClosingSAP<ACE_SOCK_Stream>& strm)
{
	assert(m_pipelineSize > 0);
	assert(m_requestQueue.size() >= m_pipelineSize);
	
	Request const& req = *m_requestQueue.front();
	auto_ptr<ServerConnection> conn(
		new ServerConnection(req.getConnectionRoute(), strm)
	);
	processNewHttpConnection(conn);
}

void
Server::processNewHttpConnection(std::auto_ptr<ServerConnection> conn)
{
	assert(conn.get());
	assert(m_pipelineSize > 0);
	assert(m_requestQueue.size() >= m_pipelineSize);
	
	Request const& req = *m_requestQueue.front();
	m_ptrConnection = conn;
	m_ptrConnection->setTcpNoDelay();
	m_ptrConnection->peer().enable(ACE_NONBLOCK);
	ACE_HANDLE handle = m_ptrConnection->peer().get_handle();
	m_flags &= ~WRITE_BUFFER_FULL;
	m_eventMask = Reactor::READ;
	if (req.hasData()) {
		m_eventMask |= Reactor::WRITE;
	}
	
	try {
		m_handlerId = m_rContext.reactor().registerHandler(
			handle, IntrusivePtr<EventHandlerBase>(this),
			m_eventMask
		);
	} catch (Reactor::Exception& e) {
		string problem("Could not register a socket in the reactor: "+string(e.what()));
		DEBUGLOG2(problem);
		m_ptrConnection.reset(0);
		answerWithError(
			ErrorFactory::errGenericError(
				ErrorCodes::UNKNOWN_ERROR, problem,
				req.getRequestURI(), problem,
				HttpStatusLine::SC_INTERNAL_SERVER_ERROR
			)
		);
		return;
	}
	
	if (req.isConnectRequest() && !req.isGoingThroughHttpProxy()) {
		auto_ptr<HttpResponseMetadata> metadata(
			createConnectionEstablishedResponse(req)
		);
		eventGotMetadata(metadata, /* is_persistent = */ false);
	}
}

std::auto_ptr<HttpResponseMetadata>
Server::createConnectionEstablishedResponse(Request const& req)
{
	auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(
			HttpStatusLine(
				req.getOrigMetadata()->requestLine().getHttpVersion(),
				200, BString("Connection established")
			)
		)
	);
	metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_UNKNOWN);
	return metadata;
}

void
Server::onDelayedAction()
{
	if (m_flags & READING_SUSPENDED) {
		/*
		The intent of READING_SUSPENDED is to prevent the server
		from providing more data to the response handler which
		is blocked in a recursive handleEvents().
		*/
		return;
	}
	
	if (m_pipelineSize > 0 && !m_ptrConnection.get() && !m_connector.isInProgress()) {
		assert(m_pipelineSize == 1);
		m_requestQueue.front()->updateRoute();
		// This may trigger a proxy configuration error, which
		// will be handled by processArtificialResponses() below.
	}
	
	if (isImmediateResponseRequired()) {
		processImmediateResponses();
		// This may result in a recursive main loop invocation.
	}
	
	assert(!isImmediateResponseRequired());
	
	if (m_pipelineSize > 0 && !m_ptrConnection.get() && !m_connector.isInProgress()) {
		assert(m_pipelineSize == 1);
		Request& req = *m_requestQueue.front();
		if (req.hasData() || req.isConnectRequest()) {
			requestConnectionFor(req);
		}
	}
}


/*=========================== Server::Request ==============================*/

Server::Request::Request(
	ServiceContext& context, ServerWeakPtr const& server_ptr,
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& response_handler)
:	m_rContext(context),
	m_ptrServer(server_ptr),
	m_ptrOrigMetadata(request_metadata),
	m_ptrOutgoingMetadata(new HttpRequestMetadata(*request_metadata)),
	m_tag(request_tag),
	m_ptrResponseHandler(
		new LoggingResponseHandler(response_handler, request_tag)
	),
	m_connRoute(SymbolicInetAddr(std::string(), -1)),
	m_connDestIsHttpProxy(false),
	m_isCancelled(false),
	m_isEOF(false),
	m_isConsumingStarted(false),
	m_isConstructionDone(false),
	m_streamWriter(*this)
{
	{
		GlobalState::ReadAccessor global_state;
		maybeReportClientIP(global_state->config());
		updateRoute(global_state->forwardingResolver());
	}
	
	HttpStreamWriter::HttpProxy const http_proxy = (
		isGoingThroughHttpProxy()
		? HttpStreamWriter::THROUGH_HTTP_PROXY
		: HttpStreamWriter::NO_HTTP_PROXY
	);
	m_streamWriter.startRequest(
		*m_ptrOutgoingMetadata, HttpStreamWriter::CONN_KEEP_ALIVE,
		 http_proxy, HttpStreamWriter::CHUNKED_DONT_USE
	);
	
	m_isConstructionDone = true;
}

Server::Request::~Request()
{
}

void
Server::Request::updateRoute()
{
	updateRoute(GlobalState::ReadAccessor()->forwardingResolver());
}

void
Server::Request::updateRoute(Forwarding::Resolver const& resolver)
{
	Forwarding::ProxyChainConstPtr proxy_chain;
	URI const& uri = getRequestURI();
	
	try {
		proxy_chain = resolver.resolve(uri);
	} catch (Forwarding::ResolverException const& e) {
		std::string const problem(e.what());
		cancel(
			ErrorFactory::errRemoteProxyConfig(
				ErrorCodes::REMOTE_PROXY_CONFIG_PROBLEM,
				problem, uri, problem
			)
		);
		return;
	}
	
	if (!proxy_chain->empty()
	    && proxy_chain->back().getType() == ProxyDescriptor::HTTP) {
		ConnectionRoute(
			proxy_chain->begin(), --proxy_chain->end(),
			proxy_chain->back().getAddr()
		).swap(m_connRoute);
		m_connDestIsHttpProxy = true;
	} else {
		SymbolicInetAddr const dest(
			uri.getHost().toStdString(),
			uri.guessPort()
		);
		ConnectionRoute(*proxy_chain, dest).swap(m_connRoute);
		m_connDestIsHttpProxy = false;
	}
}

void
Server::Request::cancel(std::auto_ptr<ErrorDescriptor> edesc)
{
	if (!m_isCancelled) {
		m_ptrErrDesc = edesc;
		m_isCancelled = true;
	}
}

bool
Server::Request::isCancelled() const
{
	return m_isCancelled;
}

void
Server::Request::appendBodyData(SplittableBuffer& data, bool eof)
{
	if (m_isCancelled) {
		return;
	}
	
	m_streamWriter.appendBodyData(data, eof);
	if (m_streamWriter.getBufferedBodyDataSize() > MAX_BUFFERED_REQUEST_BODY) {
		string const problem("request body too big");
		cancel(
			ErrorFactory::errParsingRequest(
				ErrorCodes::REQUEST_BODY_TOO_BIG, problem,
				getRequestURI(), problem
			)
		);
		return;
	}
}

void
Server::Request::processNewData(SplittableBuffer& data, bool eof)
{
	bool const was_eof = m_isEOF;
	m_data.appendDestructive(data);
	m_isEOF |= eof;
	
	if (!m_isConstructionDone) {
		return;
	}
	
	if (Server* server = m_ptrServer.get()) {
#ifdef DEBUG
		if (!was_eof && m_isEOF && m_data.empty()) {
			server->dbgReportOutgoingRequestSent();
		}
#endif
		server->onNewOutput();
	} else {
		// the server is being (or have already been) destroyed
		m_data.clear();
	}
}

bool
Server::Request::isRestartable() const
{
	return !m_ptrOutgoingMetadata->hasBody();
}

bool
Server::Request::restart()
{
	if (!isRestartable()) {
		return false;
	}
	m_data.clear();
	m_ptrErrDesc.reset();
	m_isEOF = false;
	m_isCancelled = false;
	m_isConsumingStarted = false;
	updateRoute(); // user settings might have changed
	
	HttpStreamWriter::HttpProxy const http_proxy = (
		isGoingThroughHttpProxy()
		? HttpStreamWriter::THROUGH_HTTP_PROXY
		: HttpStreamWriter::NO_HTTP_PROXY
	);
	m_streamWriter.startRequest(
		*m_ptrOutgoingMetadata, HttpStreamWriter::CONN_KEEP_ALIVE,
		 http_proxy, HttpStreamWriter::CHUNKED_DONT_USE
	);
	
	SplittableBuffer body_data;
	m_streamWriter.appendBodyData(body_data, true);
	return true;
}

void
Server::Request::maybeReportClientIP(Config const& conf)
{
	Config::ReportClientIP report = conf.getReportClientIP();
	if (report == Config::REPORT_IP_FIXED) {
		m_ptrOutgoingMetadata->headers().addHeader(
			BString("X-Forwarded-For"),
			BString(conf.getFixedClientIP())
		);
	} else if (report == Config::REPORT_IP_ON) {
		enum { MAX_IP_LEN = 46 }; // 46 is the maximum length for IPV6
		auto_ptr<DataChunk> chunk = DataChunk::create(MAX_IP_LEN);
		if (m_rContext.clientAddr().get_host_addr(chunk->getDataAddr(), MAX_IP_LEN)) {
			size_t const len = strlen(chunk->getDataAddr());
			m_ptrOutgoingMetadata->headers().addHeader(
				BString("X-Forwarded-For"),
				BString(DataChunk::resize(chunk, len))
			);
		}
	}
}


/*======================== Server::PhantomRequest ============================*/

Server::PhantomRequest::PhantomRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& response_handler,
	std::auto_ptr<ImmediateResponse> response)
:	m_ptrOrigMetadata(request_metadata),
	m_tag(request_tag),
	m_ptrResponseHandler(
		new LoggingResponseHandler(response_handler, request_tag)
	),
	m_ptrResponse(response),
	m_isCancelled(false)
{
}

Server::PhantomRequest::~PhantomRequest()
{
}

void
Server::PhantomRequest::appendBodyData(SplittableBuffer&, bool)
{
}

void
Server::PhantomRequest::cancel(std::auto_ptr<ErrorDescriptor> edesc)
{
	if (!m_isCancelled) {
		m_ptrErrDesc = edesc;
		m_isCancelled = true;
	}
}

bool
Server::PhantomRequest::isCancelled() const
{
	return m_isCancelled;
}
