/*=====================================================================*/
/*    .../prgm/project/bigloo/api/pthread/src/Posix/bglpthread.c       */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Fri Feb 22 12:12:04 2002                          */
/*    Last change :  Sat Jan 28 05:40:17 2006 (serrano)                */
/*    Copyright   :  2002-06 Manuel Serrano                            */
/*    -------------------------------------------------------------    */
/*    C utilities for native Bigloo fair threads implementation.       */
/*=====================================================================*/
#include <pthread.h>
#include <sched.h>
#include <stdlib.h>
#include <string.h>

#define GC_PRIVATE_H
#include <gc.h>
#include <bglpthread.h>

/*---------------------------------------------------------------------*/
/*    Imports                                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DECL void bgl_multithread_dynamic_denv_register();

/*---------------------------------------------------------------------*/
/*    pthread_key_t                                                    */
/*    bgldenv_key ...                                                  */
/*---------------------------------------------------------------------*/
static pthread_key_t bgldenv_key;

/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglpth_single_thread_denv ...                                    */
/*---------------------------------------------------------------------*/
static bgldenv_t bglpth_single_thread_denv = 0L;

/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglpth_dynamic_env ...                                           */
/*---------------------------------------------------------------------*/
bgldenv_t
bglpth_dynamic_env() {
   return (bgldenv_t)pthread_getspecific( bgldenv_key );
}
 
/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglpth_dynamic_env_set ...                                       */
/*---------------------------------------------------------------------*/
static bgldenv_t
bglpth_dynamic_env_set( bgldenv_t env ) {
   pthread_setspecific( bgldenv_key, env );
   
   return env;
}

/*---------------------------------------------------------------------*/
/*    bglpthread_t                                                     */
/*    bglpth_thread_new ...                                            */
/*---------------------------------------------------------------------*/
bglpthread_t
bglpth_thread_new( obj_t thunk ) {
   bglpthread_t t = (bglpthread_t)GC_MALLOC( sizeof( struct bglpthread ) );

   pthread_mutex_init( &(t->mutex), 0L );
   pthread_cond_init( &(t->condvar), 0L );

   t->thunk = thunk;
   t->specific = BUNSPEC;
   t->cleanup = BUNSPEC;
   t->status = 0;
   
   return t;
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    bglpth_thread_cleanup ...                                        */
/*---------------------------------------------------------------------*/
static void
bglpth_thread_cleanup( void *arg ) {
   bglpthread_t self = (bglpthread_t)arg;
   obj_t cleanup = self->cleanup;

   /* lock the internal state of the thread */
   pthread_mutex_lock( &(self->mutex) );
   
   /* mark the thread terminated */
   self->status = 2;
   
   /* unlock all locked mutexes */
   bglpth_mutexes_unlock( self );
   
   /* unlock the internal state of the thread */
   pthread_mutex_unlock( &(self->mutex) );
   
   /* invoke user cleanup */
   if( PROCEDUREP( cleanup ) ) {
      PROCEDURE_ENTRY( cleanup )( cleanup, self->bglthread, BEOA );
   }
}

/*---------------------------------------------------------------------*/
/*    static void *                                                    */
/*    bglpth_thread_run ...                                            */
/*---------------------------------------------------------------------*/
static void *
bglpth_thread_run( void *arg ) {
   bglpthread_t self = (bglpthread_t)arg;
   obj_t thunk = self->thunk;

   /* The environment is stored in a specific variable for dynamic   */
   /* access but it is pointed to by the thread structure for the GC */
   bglpth_dynamic_env_set( self->env );

   self->env->stack_bottom = (char *)&arg;
   self->env->current_thread = self;
   
   bgl_init_trace();

   pthread_setcanceltype( PTHREAD_CANCEL_ASYNCHRONOUS, 0 );
   
   pthread_cleanup_push( bglpth_thread_cleanup, arg );
   
   /* mark the thread started */
   pthread_mutex_lock( &(self->mutex) );
   self->status = 1;
   pthread_cond_broadcast( &(self->condvar) );
   pthread_mutex_unlock( &(self->mutex) );

   /* enter the user code */
   PROCEDURE_ENTRY( thunk )( thunk, BEOA );
   pthread_cleanup_pop( 1 );

   /* returns self so the GC is unable to collect self (and the */
   /* thread specific dynamic env) until the thread completes   */
   return (void *)self;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_thread_start ...                                          */
/*---------------------------------------------------------------------*/
void
bglpth_thread_start( bglpthread_t thread, obj_t bglthread, bool_t dt ) {
   pthread_attr_t a;

   pthread_attr_init( &a );

   if( dt ) pthread_attr_setdetachstate( &a, PTHREAD_CREATE_DETACHED );

   thread->bglthread = bglthread;
   thread->env = bgl_dup_dynamic_env( BGL_DYNAMIC_ENV() );

   if( pthread_create( &(thread->pthread), &a, bglpth_thread_run, thread ) )
      FAILURE( string_to_bstring( "thread-start!" ),
	       string_to_bstring( "Cannot start thread" ),
	       string_to_bstring( strerror( errno ) ) );
}

/*---------------------------------------------------------------------*/
/*    bglpthread_t                                                     */
/*    bglpth_current_pthread ...                                       */
/*---------------------------------------------------------------------*/
bglpthread_t
bglpth_current_pthread() {
   return BGL_DYNAMIC_ENV()->current_thread;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglpth_current_thread ...                                        */
/*---------------------------------------------------------------------*/
obj_t
bglpth_current_thread() {
   bglpthread_t cur = bglpth_current_pthread();

   return cur ? cur->bglthread : BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_thread_join ...                                           */
/*---------------------------------------------------------------------*/
void
bglpth_thread_join( bglpthread_t t ) {
   pthread_mutex_lock( &(t->mutex) );
   if( !t->status ) {
      pthread_cond_wait( &(t->condvar), &(t->mutex) );
   }
   pthread_mutex_unlock( &(t->mutex) );

   if( pthread_join( t->pthread, 0L ) ) {
      FAILURE( string_to_bstring( "thread-join!" ),
	       string_to_bstring( "Cannot join thread" ),
	       string_to_bstring( strerror( errno ) ) );
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglpth_thread_terminate ...                                      */
/*---------------------------------------------------------------------*/
bool_t
bglpth_thread_terminate( bglpthread_t t ) {
   pthread_mutex_lock( &(t->mutex) );
   if( t->status != 2 ) {
      pthread_cancel( t->pthread );
      pthread_mutex_unlock( &(t->mutex) );
      return 1;
   } else {
      pthread_mutex_unlock( &(t->mutex) );
      return 0;
   }
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_setup_thread ...                                          */
/*---------------------------------------------------------------------*/
void
bglpth_setup_thread() {
#if HAVE_SIGACTION
   struct sigaction sigact;
   sigemptyset( &(sigact.sa_mask) );
   sigact.sa_handler = SIG_IGN;
   sigact.sa_flags = SA_RESTART;
   sigaction( SIGPIPE, &sigact, NULL );
#else
   signal( SIGPIPE, SIG_IGN );
#endif

   /* main dynamic env init */
   bgl_init_dynamic_env();

   /* Keep the environment in a global for the GC */
   bglpth_single_thread_denv = single_thread_denv;

   /* Store it in a thread variable */
   pthread_key_create( &bgldenv_key, 0L );
   bglpth_dynamic_env_set( single_thread_denv );

   /* mark the main environment as multithreaded */
   single_thread_denv = 0;
   
   bgl_multithread_dynamic_denv_register( &bglpth_dynamic_env );
}
