/***************************************************************************
                  mozilla.cpp  -  Mozilla Class Implementation
                             -------------------
    begin                : Sat Oct 05 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <ctype.h>
#include <stack>
#include <vector>

#include "mozilla.h"
#include "browserlist.h"
#include "xmlparser.h"

#include <QFile>
#include <QFileDialog>
#include <QDir>

#ifdef _WIN32
#include <shlobj.h>
#endif

/***************************************************************************
 *                                                                         *
 *   Constants                                                             *
 *                                                                         *
 ***************************************************************************/

#define CHARS_GROWBY			100

#define SAXSTATE_A				(SAXSTATE_USER+1)
#define SAXSTATE_BODY			(SAXSTATE_USER+2)
#define SAXSTATE_DL				(SAXSTATE_USER+3)
#define SAXSTATE_DT				(SAXSTATE_USER+4)
#define SAXSTATE_H3				(SAXSTATE_USER+5)
#define SAXSTATE_HTML			(SAXSTATE_USER+6)
#define SAXSTATE_P				(SAXSTATE_USER+7)
#define SAXSTATE_DD				(SAXSTATE_USER+8)
#define SAXSTATE_HR				(SAXSTATE_USER+9)

/***************************************************************************
 *                                                                         *
 *   SAX2 Parser Call-back Functions                                       *
 *                                                                         *
 ***************************************************************************/

static	void	htmlStartDocument(void *);

static	void	htmlEndA(void *, const xmlChar *, const xmlChar *, short, short);
static	void	htmlEndDL(void *, const xmlChar *, const xmlChar *, short, short);
static	void	htmlEndH3(void *, const xmlChar *, const xmlChar *, short, short);
static	bool	htmlStartA(void *, const xmlChar *, const xmlChar **, short);
static	bool	htmlStartDL(void *, const xmlChar *, const xmlChar **, short);
static	bool	htmlStartDT(void *, const xmlChar *, const xmlChar **, short);
static	bool	htmlStartH3(void *, const xmlChar *, const xmlChar **, short);
static	bool	htmlStartHR(void *, const xmlChar *, const xmlChar **, short);

/***************************************************************************
 *                                                                         *
 *   Structure Definitions                                                 *
 *                                                                         *
 ***************************************************************************/

typedef struct {
	std::stack<BkFolder *>	folders;
	std::vector<BkAttr> attrsH3;
	BkFolder	*root;
	BkBookmark	*bookmark;
	QString		lastH3;			// string from most recent H3
	QString		lastDD;			// string from most recent DD
	BRWSNUM		browserOrd;
} SAXDATA;

static ELEMHANDLER htmlHandlers[] = {
	{
		SAXSTATE_A,
		"a",
		htmlStartA,
		htmlEndA,
	},{
		SAXSTATE_BODY,
		"body",
		NULL,
		NULL,
	},{
		SAXSTATE_DD,
		"dd",
		NULL,
		NULL,
	},{
		SAXSTATE_DL,
		"dl",
		htmlStartDL,
		htmlEndDL,
	},{
		SAXSTATE_DT,
		"dt",
		htmlStartDT,
		NULL,
	},{
		SAXSTATE_H3,
		"h3",
		htmlStartH3,
		htmlEndH3,
	},{
		SAXSTATE_HTML,
		"html",
		NULL,
		NULL,
	},{
		SAXSTATE_P,
		"p",
		NULL,
		NULL,
	},{
		SAXSTATE_HR,
		"hr",
		htmlStartHR,
		NULL,
	},
	// this element marks the end of the list
	{ 0, NULL, NULL, NULL }
};

/***************************************************************************
 *                                                                         *
 *   htmlStartDocument()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the SAX2 parsing engine before any      *
 *      others.  It uses the opportunity to initialize state data.  It     *
 *      also pushes the current state on to a state stack.  This stack is  *
 *      pushed when a new XML element is encountered and popped at the end *
 *      of the element.  This allows the callback functions to always      *
 *      know where they are in the XML document.                           *
 *                                                                         *
 ***************************************************************************/

static void htmlStartDocument(void *ctx)
{
	SAXDATA *data = (SAXDATA *)ctx;

	data->bookmark = NULL;
	data->lastH3 = QString::null;
}

/***************************************************************************
 *                                                                         *
 *   htmlStartA()                                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar **atts                                               *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      true on error, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called when the xml parser encounters an HTML <A> *
 *      element.                                                           *
 *                                                                         *
 ***************************************************************************/

static bool htmlStartA(void *ctx, const xmlChar *, const xmlChar **atts, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	BkBookmark	bookmark;
	int order =
		data->folders.top()->folders() +
		data->folders.top()->bookmarks() +
		data->folders.top()->separators() + 1;
	data->bookmark = &data->folders.top()->addBookmark(bookmark);
	data->bookmark->setOrder(data->browserOrd, order);

	// this bookmark belongs to Mozilla

	data->bookmark->setBrowserFound(data->browserOrd);
	data->bookmark->setBrowserSaved(data->browserOrd);

	// save some attributes in the element

	if(atts)
	{
		int	i;

		for(i=0; atts[i]; i+=2)
		{
			if(!xmlStrcasecmp(atts[i], (const xmlChar *)"href"))
			{
				data->bookmark->setUrl(QString::fromUtf8((const char *)atts[i+1]));
				data->bookmark->setValidField(BKVALID_URL);
			}
			else
				data->bookmark->setAttr(
				QString::fromUtf8((const char *)atts[i]),
				QString::fromUtf8((const char *)atts[i+1]),
					data->browserOrd);
		}
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   htmlStartDL()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *         (not used)                                 *
 *      const xmlChar **atts    (not used)                                 *
 *      short                   (not used)                                 *
 *   Return:                                                               *
 *      true on error, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called by the xmlparser when it encounters an     *
 *      HTML <DL> element.                                                 *
 *                                                                         *
 ***************************************************************************/

static bool htmlStartDL(void *ctx, const xmlChar *, const xmlChar **, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	BkFolder folder, *child;

	// if this is the root folder, push a pointer to the root folder
	// on to the stack.

	if(data->folders.size() == 0)
		child = data->root;
	else
	{
		int order =
			data->folders.top()->folders() +
			data->folders.top()->bookmarks() +
			data->folders.top()->separators() + 1;
		child = &data->folders.top()->addFolder(folder);
		child->setOrder(data->browserOrd, order);
	}
	data->folders.push(child);
	if(data->lastH3 != QString::null)
	{
		data->folders.top()->setTitle(data->lastH3);
		data->folders.top()->setValidField(BKVALID_TITLE);
	}

	// There might be a folder description to save

	if(data->lastDD != QString::null)
	{
		data->folders.top()->setDesc(data->lastDD.simplified());
		data->folders.top()->setValidField(BKVALID_DESC);
		data->lastDD = QString::null;
	}

	// If the H3 segment had any attributes, save them

	for(int iAttr=0; iAttr<(int)data->attrsH3.size(); iAttr++)
	{
		data->folders.top()->setAttr(
			data->attrsH3[iAttr].m_name,
			data->attrsH3[iAttr].m_value,
			data->attrsH3[iAttr].m_browser);
	}
	data->attrsH3.clear();

	// this folder belongs to mozilla

	data->folders.top()->setBrowserFound(data->browserOrd);
	data->folders.top()->setBrowserSaved(data->browserOrd);

	return false;
}

/***************************************************************************
 *                                                                         *
 *   htmlStartDT()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *          (not used)                                *
 *      const xmlChar **         (not used)                                *
 *      short                    (not used)                                *
 *   Return:                                                               *
 *      true on error, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called by the xml parser when it encounters an    *
 *      HTML <DT> element.                                                 *
 *                                                                         *
 ***************************************************************************/

static bool htmlStartDT(void *ctx, const xmlChar *, const xmlChar **, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	if(	data->lastDD != QString::null
	 &&	data->bookmark != NULL)
	{
		data->bookmark->setDesc(data->lastDD.simplified());
		data->bookmark->setValidField(BKVALID_DESC);
	}
	data->lastDD = QString::null;
	data->bookmark = NULL;

	return false;
}

/***************************************************************************
 *                                                                         *
 *   htmlStartH3()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *          (not used)                                *
 *      const xmlChar **atts                                               *
 *      short                    (not used)                                *
 *   Return:                                                               *
 *      true on error, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called by the xml parser when it encounters an    *
 *      HTML <H3> element.                                                 *
 *                                                                         *
 ***************************************************************************/

static bool htmlStartH3(void *ctx, const xmlChar *, const xmlChar **atts, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	Q_ASSERT(data->folders.size() > 0);

	// save some attributes in the element

	if(atts)
	{
		for(int i=0; atts[i]; i+=2)
		{
			BkAttr	attr;

			attr.m_browser = data->browserOrd;
			attr.m_name = QString::fromUtf8((const char *)atts[i]);
			attr.m_value = QString::fromUtf8((const char *)atts[i+1]);

			data->attrsH3.push_back(attr);
		}
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   htmlStartHR()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *          (not used)                                *
 *      const xmlChar **         (not used)                                *
 *      short                    (not used)                                *
 *   Return:                                                               *
 *      true on error, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called by the xml parser when it encounters an    *
 *      HTML <HR> element.                                                 *
 *                                                                         *
 ***************************************************************************/

static bool htmlStartHR(void *ctx, const xmlChar *, const xmlChar **, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	BkSeparator	separator;
	int order =
		data->folders.top()->folders() +
		data->folders.top()->bookmarks() + 
		data->folders.top()->separators() + 1;
	separator.setOrder(data->browserOrd, order);

	// this separator belongs to Mozilla

	separator.setBrowserFound(data->browserOrd);
	separator.setBrowserSaved(data->browserOrd);

	data->folders.top()->addSeparator(separator);

	return false;
}

/***************************************************************************
 *                                                                         *
 *   htmlEndA()                                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *       (not used)                                   *
 *      const xmlChar *pChars                                              *
 *      short                 (not used)                                   *
 *      short                 (not used)                                   *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the xml parsing engine at the end of    *
 *      the A HTML element.                                                *
 *                                                                         *
 ***************************************************************************/

static void htmlEndA(void *ctx, const xmlChar *, const xmlChar *pChars, short, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	Q_ASSERT(data->bookmark != NULL);
	if(pChars)
	{
		data->bookmark->setTitle(QString::fromUtf8((const char *)pChars));
		data->bookmark->setValidField(BKVALID_TITLE);
	}
}

/***************************************************************************
 *                                                                         *
 *   htmlEndDL()                                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *       (not used)                                   *
 *      const xmlChar *       (not used)                                   *
 *      short                 (not used)                                   *
 *      short                 (not used)                                   *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the xml parsing engine at the end of    *
 *      the DL HTML element.                                               *
 *                                                                         *
 ***************************************************************************/

static void htmlEndDL(void *ctx, const xmlChar *, const xmlChar *, short, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	if(	data->lastDD != QString::null
	 &&	data->bookmark != NULL)
	{
		data->bookmark->setDesc(data->lastDD.simplified());
		data->bookmark->setValidField(BKVALID_DESC);
	}
	data->bookmark = NULL;
	data->lastDD = QString::null;

	Q_ASSERT(data->folders.size() > 0);
	data->folders.pop();
}

/***************************************************************************
 *                                                                         *
 *   htmlEndH3()                                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *       (not used)                                   *
 *      const xmlChar *pChars                                              *
 *      short                 (not used)                                   *
 *      short                 (not used)                                   *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the xml parsing engine at the end of    *
 *      the H3 HTML element.                                               *
 *                                                                         *
 ***************************************************************************/

static void htmlEndH3(void *ctx, const xmlChar *, const xmlChar *pChars, short, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	Q_ASSERT(data->folders.size() > 0);
	if(pChars)
		data->lastH3 = QString::fromUtf8((const char *)pChars);
	else
		data->lastH3 = "";
}

/***************************************************************************
 *                                                                         *
 *   Mozilla::AreBookmarksValid()                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &bookmarks                                           *
 *   Return:                                                               *
 *      true if valid, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called to verify the filename given in the first  *
 *      first parameter is most likely a Mozilla/Netscape bookmark file.   *
 *      It first attempts to open the file and then reads the first line   *
 *      and matches it against what it thinks the first should be.         *
 *                                                                         *
 ***************************************************************************/

bool Mozilla::AreBookmarksValid(const QString &bookmarks)
{
	QFile   file(bookmarks);

	if(file.open(QIODevice::ReadOnly) == false)
		return false;
	QTextStream stream(&file);

	QString line = stream.readLine();
	return line.startsWith("<!DOCTYPE NETSCAPE-Bookmark-file-1>");
}

/***************************************************************************
 *                                                                         *
 *   Mozilla::BrowseForBookmarks()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QString &bookmarks                                                 *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      true if the user did not select a bookmark file, false otherwise   *
 *   Description:                                                          *
 *      This function is called to present the user with a file dialog box *
 *      allowing them to select the location of their bookmark file.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Mozilla::BrowseForBookmarks(const BridgeCfg &cfg, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = cfg.m_AppDataDir + "/Mozilla/Profiles/Default";
	bookmarks =
		QFileDialog::getOpenFileName(parent, "Select Your Bookmarks File",
									 bookmarks, "Bookmark File (bookmarks.html)" );

	return (bookmarks == QString::null);
}
#elif defined(Q_WS_X11)
bool Mozilla::BrowseForBookmarks(const BridgeCfg &, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = QDir::homePath() + "/.mozilla/default";
	bookmarks = QFileDialog::getOpenFileName(parent,
		"Select Your Bookmarks File",
		bookmarks, "Bookmark File (bookmarks.html)");

	return (bookmarks == QString::null);
}
#else
#error "Must define platform-dependent Mozilla::BrowseForBookmarks() Function"
#endif

/***************************************************************************
 *                                                                         *
 *   Mozilla::classFactory()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      the newly allocated class                                          *
 *   Description:                                                          *
 *      This is a static member function of the Mozilla class.  Its job is *
 *      to allocate an instance of the Mozilla class for the caller.       *
 *                                                                         *
 ***************************************************************************/

BrowserBk *Mozilla::classFactory(void)
{
	return new Mozilla;
}

/***************************************************************************
 *                                                                         *
 *   Mozilla::DetectBrowser()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QStringList &path                                                  *
 *   Return:                                                               *
 *      true if an installation of Mozilla was found, false otherwise      *
 *   Description:                                                          *
 *      This function attempts to locate Mozilla's bookmark database on    *
 *      system.  It will only work if the user has just one profile.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Mozilla::DetectBrowser(const BridgeCfg &cfg, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.mozilla/default
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Mozilla\default

	basePath = cfg.m_AppDataDir + "/Mozilla/Profiles/Default";

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	// Now the fun part.  For security reasons, Mozilla obfuscates the
	// profile directory.  It always ends in ".slt", so we'll just search
	// for anything matching this pattern.

	QStringList profileList = base.entryList(QStringList("*.slt"), QDir::Dirs);

	// Did we find any?

	if(profileList.count() < 1)
		return false;

	// Was there only one?

	if(profileList.count() == 1)
	{
		QString	path;
		path = basePath + "/" + profileList[0] + "/bookmarks.html";

		if(QFile::exists(path))
		{
			paths.append(path);
			return true;
		}
	}

	// There was more than one match.  Maybe someone can write some code
	// to list the profiles, allowing the user to pick one.

	return false;
}
#elif defined(Q_WS_X11)
bool Mozilla::DetectBrowser(const BridgeCfg &, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.mozilla/default
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Mozilla\default

	basePath = QDir::homePath() + "/.mozilla/default";

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	// Now the fun part.  For security reasons, Mozilla obfuscates the
	// profile directory.  It always ends in ".slt", so we'll just search
	// for anything matching this pattern.

	QStringList profileList = base.entryList(QStringList("*.slt"), QDir::Dirs);

	// Did we find any?

	if(profileList.count() < 1)
		return false;

	// Was there only one?

	if(profileList.count() == 1)
	{
		QString	path;
		path = basePath + "/" + profileList[0] + "/bookmarks.html";

		if(QFile::exists(path))
		{
			paths.append(path);
			return true;
		}
	}

	// There was more than one match.  Maybe someone can write some code
	// to list the profiles, allowing the user to pick one.

	return false;
}
#else
#error "Must define platform-dependent Mozilla Detection Code"
#endif

/***************************************************************************
 *                                                                         *
 *   Mozilla::IsBrowserRunning()                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      true if browser is running, false otherwise                        *
 *   Description:                                                          *
 *      This function attempts to detect if Mozilla is currently running.  *
 *      While Mozilla is running, it caches a copy of the bookmarks in     *
 *      memory then saves it when the user exits, potentially overwriting  *
 *      changes made by BookmarkBridge.                                    *
 *                                                                         *
 ***************************************************************************/

#ifdef _WIN32
static bool MozRunning;

static BOOL CALLBACK EnumWndProc(HWND hWnd, LPARAM)
{
	char	caption[24];

	if(GetWindowTextA(hWnd, caption, sizeof(caption)) > 0)
	{
		if(strnicmp(caption, "mozilla", 7) == 0)
			MozRunning = true;
	}

	return TRUE;
}

bool Mozilla::IsBrowserRunning(void)
{
	MozRunning = false;
	if(EnumWindows(EnumWndProc, 0) == 0)
		return true;

	return MozRunning;
}
#else
bool Mozilla::IsBrowserRunning(void)
{
	return false;
}
#endif

/***************************************************************************
 *                                                                         *
 *   Mozilla::readBookmarks()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &filename                                            *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to read Mozilla's bookmark file using the   *
 *      HTML parser included with libxml.                                  *
 *                                                                         *
 ***************************************************************************/
static void charHandler(void *ctx, short state, const xmlChar *ch, int len)
{
	SAXDATA *data = (SAXDATA *)ctx;

	if(state == SAXSTATE_DD)
	{
		xmlChar *charData = new xmlChar[len+1];
		if(charData == NULL)
			return;
		memcpy(charData, ch, len);
		charData[len] = 0;
		data->lastDD += QString::fromUtf8((const char *)charData);
		delete [] charData;
	}
}
 
void Mozilla::readBookmarks(const QString &filename, BRWSNUM browserOrd)
				throw(BkException)
{
	SAXDATA		data;

	// make sure the file exists before starting

	if(!QFile::exists(filename))
		BKEXCEPT("The Mozilla bookmarks could not be found.");

	// parse the file

	data.root = &m_Root;
	data.browserOrd = browserOrd;

	ParseHtmlDocument(
		filename,
		&data,
		htmlHandlers,
		htmlStartDocument,
		NULL,
		charHandler);

	Q_ASSERT(data.folders.size() == 0);
}

/***************************************************************************
 *                                                                         *
 *   Mozilla::saveBookmarks()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &filename                                            *
 *      BkFolder &root                                                     *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function begins the process of saving the bookmark tree,      *
 *      specified in the second parameter to the file, specified in the    *
 *      first parameter.                                                   *
 *                                                                         *
 ***************************************************************************/

void Mozilla::saveBookmarks(const QString &filename, BkFolder &root, BRWSNUM browserOrd)
		throw(BkException)
{
	QFile	out;

	// open the file for writing

	out.setFileName(filename);
	if(out.open(QIODevice::WriteOnly | QIODevice::Text) == false)
		BKEXCEPT("Unable to Open Mozilla Bookmark File for Output");

	// set up a text stream

	QTextStream t(&out);
	t.setCodec("UTF-8");

	// write out Mozilla's boilerplate header

	t
		<<	"<!DOCTYPE NETSCAPE-Bookmark-file-1>" << endl
		<< "<!-- This is an automatically generated file." << endl
		<< "It will be read and overwritten." << endl
		<< "Do Not Edit! -->" << endl
		<< "<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=UTF-8\">" << endl
		<< "<TITLE>Bookmarks</TITLE>" << endl
		<< "<H1>Bookmarks</H1>" << endl
		<< endl
		<< "<DL><p>" << endl;

	saveFolder(root, t, 0, browserOrd);

	// finish up

	t << "</DL><p>\n";
}

/***************************************************************************
 *                                                                         *
 *   Mozilla::saveBookmark()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkBookmark &bkmark                                           *
 *      QTextStream &t                                                     *
 *      int levels                                                         *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to save a bookmark, specified in the first  *
 *      parameter to the file, specified in the second parameter.  The     *
 *      third parameter is used for indention in the bookmark file.  It    *
 *      specified how many folders deep we are in the bookmark tree.       *
 *                                                                         *
 ***************************************************************************/

void Mozilla::saveBookmark(
	const BkBookmark &bkmark, QTextStream &t, int levels, BRWSNUM browserOrd)
	throw(BkException)
{
	int		i;
	QString	temp;

	for(i=0; i<levels; i++)
		t << "    ";

	t
		<< "<DT><A HREF=\""
		<< bkmark.url()
		<< "\"";

	for(i=0; i<bkmark.nAttrs(); i++)
	{
		QString	name, value;
		BRWSNUM browser;

		bkmark.attr(i, name, value, browser);

		if(browser != browserOrd)
			continue;

		t
			<< " "
			<< name
			<< "=\""
			<< value
			<< "\"";
	}

	EncodeString(bkmark.title(), temp);
	t
		<< ">"
		<< temp
		<< "</A>\n";

	if(bkmark.isFieldValid(BKVALID_DESC))
	{
		EncodeString(bkmark.desc(), temp);
		t
			<< "<DD>"
			<< temp
			<< endl;
	}
}

/***************************************************************************
 *                                                                         *
 *   Mozilla::saveFolder()                                                 *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &folder                                                   *
 *      QTextStream &t                                                     *
 *      int levels                                                         *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function saves a folder, specified in the first parameter to  *
 *      the bookmark file, specified in the second parameter.  The third   *
 *      parameter is used for indention.  It specifies how many levels     *
 *      deep it is within the bookmark tree.                               *
 *                                                                         *
 ***************************************************************************/

void Mozilla::saveFolder(
	BkFolder &folder, QTextStream &t, int levels, BRWSNUM browserOrd)
	throw(BkException)
{
	int		i;
	QString	temp;
	std::vector<SortedNodeList> nodes;
	std::vector<SortedNodeList>::iterator it;
	
	SortNodes(folder, nodes, browserOrd);
	
	for(it=nodes.begin(); it!=nodes.end(); ++it)
	{
		if(it->m_type == NODETYPE_FOLDER)
		{
			// if this node has been deleted from one of the
			// browsers, skip it...
		
			if(it->m_fit->isDeleted())
				continue;
				
			// Has the user requested to ignore this node?
			// If so, we should skip it if it's not already stored in Mozilla
		
			if(it->m_fit->ignore() && !it->m_fit->browserFound(browserOrd))
				continue;
			
			// save the folder
		
			// write out the title and other attributes

			for(i=0; i<levels+1; i++)
				t << "    ";
			t << "<DT><H3";
			for(i=0; i<it->m_fit->nAttrs(); i++)
			{
				QString	name, value;
				BRWSNUM browser;

				it->m_fit->attr(i, name, value, browser);

				if(browser != browserOrd)
					continue;
	
				t
					<< " "
					<< name
					<< "=\""
					<< value
					<< "\"";
			}
			EncodeString(it->m_fit->title(), temp);
			t
				<< ">"
				<< temp
				<< "</H3>"
				<< endl;
			if(it->m_fit->isFieldValid(BKVALID_DESC))
			{
				EncodeString(it->m_fit->desc(), temp);
				t
					<< "<DD>"
					<< temp
					<< endl;
			}
			for(i=0; i<levels+1; i++)
				t << "    ";
			t << "<DL><p>" << endl;

			saveFolder(*(it->m_fit), t, levels+1, browserOrd);
		
			for(i=0; i<levels+1; i++)
				t << "    ";
			t << "</DL><p>" << endl;
			
			it->m_fit->setBrowserFound(browserOrd);
			it->m_fit->setBrowserSaved(browserOrd);
		}
		else if(it->m_type == NODETYPE_BOOKMARK)
		{
			// if this node has been deleted from one of the
			// browsers, skip it...
		
			if(it->m_bit->isDeleted())
				continue;
		
			// Has the user requested to ignore this node?
			// If so, we should skip it if it's not already stored in Mozilla
		
			if(it->m_bit->ignore() && !it->m_bit->browserFound(browserOrd))
				continue;
			
			// store the bookmark
		
			saveBookmark(*(it->m_bit), t, levels+1, browserOrd);
			
			it->m_bit->setBrowserFound(browserOrd);
			it->m_bit->setBrowserSaved(browserOrd);
		}
		else if(it->m_type == NODETYPE_SEPARATOR)
		{
			// if this separator did not come from Mozilla, then ignore it
			
			if(!it->m_sit->browserFound(browserOrd))
				continue;
				
			for(i=0; i<levels+1; i++)
				t << "    ";
			t << "<HR>" << endl;
		}
		else
			BKEXCEPT("Internal Error: Unrecognized Bookmark Type");
	}
}
