/*  CamStream: a collection of GUI webcam tools
    Copyright (C) 2002-2005 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/


#include <qcombobox.h>
#include <qlcdnumber.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qslider.h>

#include "ChannelEditorDlg.h"



/**
  \class CChannelEditorDlg
  \brief Complete dialog that handles setting up the channels for a TV tuner card

*/



CChannelEditorDlg::CChannelEditorDlg(QWidget *parent, const char *name)
       : CChannelEditor(parent, name, true)
{
   m_pTVChannels = 0;
   m_CurrentFrequencySystem = TVChannel::FrequencyGrid_MAX;
   m_CurrentIsNew = false;
   m_UpdatingGUI = false;
   m_PresetList->setSorting(-1); // do not allow sorting
}

// private


/**
  Does several things:
  # sets m_TVChannels.current
  # calls SetDialogControls
  # sets PresetList
  # emits NewFrequency (if possible)
  # emits NewCurrentChannel
*/
void CChannelEditorDlg::SetNewCurrent(TVChannel *channel)
{
qDebug(">> CChannelEditorDlg::SetNewCurrent(%p)", channel);
   QListViewItem *lvi = 0;

   if (channel != 0) {
     if (m_pTVChannels->find(channel) < 0) {
       qWarning("CChannelEditorDlg::SetNewCurrent: channel not found.");
     }
   }

   SetDialogControls(channel);
   lvi = m_Channel2ListView[channel];
   if (lvi != 0)
     m_PresetList->setSelected(lvi, true);

   if (channel != 0) {
     emit NewFrequency(channel->Frequency() * 1e6);
   }
   emit NewCurrentChannel();
qDebug("<< CChannelEditorDlg::SetNewCurrent");
}

/**
  \brief Update various edit boxes to the current TVChannel struct

  This will update the name, channel, fine-tuning and color-system boxes
  to the current TVChannel struct. If that happens to be a null pointer, the
  elements in the dialog are blanked and disabled.

*/
void CChannelEditorDlg::SetDialogControls(const TVChannel *channel)
{
   QString s;

   m_UpdatingGUI = true;
   if (channel == 0) {
     m_NameLineEdit->setText("");
     SetFrequencySystem(TVChannel::FrequencyGrid_MAX);
     m_SystemChannels->setCurrentItem(-1);
     m_FinetuningSlider->setValue(0);
     m_FrequencyLCD->display("0.00");
   }
   else {
     m_NameLineEdit->setText(channel->Name);
     SetFrequencySystem(channel->FrequencySystem);
     m_SystemChannels->setCurrentItem(channel->Channel);
     m_FinetuningSlider->setValue(channel->Finetuning);
     switch(channel->ColorSystem)
     {
       case TVChannel::PAL_BG:     m_ColorSystemDropdown->setCurrentItem(0); break;
       case TVChannel::NTSC:       m_ColorSystemDropdown->setCurrentItem(1); break;
       case TVChannel::SECAM:      m_ColorSystemDropdown->setCurrentItem(2); break;
       case TVChannel::PAL_NC:     m_ColorSystemDropdown->setCurrentItem(3); break;
       case TVChannel::PAL_M:      m_ColorSystemDropdown->setCurrentItem(4); break;
       case TVChannel::PAL_N:      m_ColorSystemDropdown->setCurrentItem(5); break;
       case TVChannel::NTSC_JAPAN: m_ColorSystemDropdown->setCurrentItem(6); break;
     }
     m_FrequencyLCD->display(s.sprintf("%5.2f", channel->Frequency()));
   }

   m_NameLineEdit->setEnabled(channel != 0);
   m_FrequencySystemDropdown->setEnabled(channel != 0);
   m_SystemChannels->setEnabled(channel != 0);
   m_FinetuningSlider->setEnabled(channel != 0);
   m_ColorSystemDropdown->setEnabled(channel != 0);
   m_UpdatingGUI = false;
}


/**
  \brief Update columns in ListView
  \param channel TVChannel struct

  Sets the values in the colums in the Presets listview at the
  left hand side of the dialog for the given \b channel
*/

void CChannelEditorDlg::SetListEntry(const TVChannel *channel)
{
  QString s;
  QListViewItem *lvi;

  if (channel == 0)
    return;
  lvi = m_Channel2ListView[channel];
  if (lvi == 0)
    return;

  lvi->setText(0, channel->Name);
  switch(channel->FrequencySystem)
  {
    case TVChannel::American: s = tr("American"); break;
    case TVChannel::European: s = tr("European"); break;
    case TVChannel::Japanese: s = tr("Japanese"); break;
  }
  lvi->setText(1, s);
  lvi->setText(2, s.sprintf("%5.2f", channel->Frequency()));
  switch(channel->ColorSystem)
  {  // Not translated on purpose
    case TVChannel::PAL_BG  : s = "PAL-BG"; break;
    case TVChannel::NTSC    : s = "NTSC";   break;
    case TVChannel::SECAM   : s = "SECAM";  break;
    case TVChannel::PAL_NC  : s = "PAL-NC"; break;
    case TVChannel::PAL_M   : s = "PAL-M";  break;
    case TVChannel::PAL_N   : s = "PAL-N";  break;
    case TVChannel::NTSC_JAPAN: s = "NTSC-Japan"; break;
  }
  lvi->setText(3, s);
}


// protected

/**
  \brief New frequency table
  \param table The new system (see \ref FrequencySystem)

  Different countries (continents) have different frequency ranges
  available for their terrestial TV broadcasts. Selecting a new
  table will fill the m_SystemChannels listbox with the channel names.

  If the system is already selected, nothing happens.

*/
void CChannelEditorDlg::SetFrequencySystem(TVChannel::FrequencyGrid system)
{
   int i, j;
   QString fname;

   if (m_CurrentFrequencySystem == system)
     return;
   if (system < 0 || system >= TVChannel::FrequencyGrid_MAX)
     return;

   // Remember...
   m_CurrentFrequencySystem = system;
   // Prevent loops
   m_FrequencySystemDropdown->blockSignals(true);

   // Set GUI
   switch (system) {
     case TVChannel::American:
       m_FrequencySystemDropdown->setCurrentItem(0);
       break;

     case TVChannel::European:  // Europe, including cable
       m_FrequencySystemDropdown->setCurrentItem(1);
       m_FinetuningSlider->setMinValue(-400); //  -4 MHz
       m_FinetuningSlider->setMaxValue(400);
       m_FinetuningSlider->setPageStep(40);
       break;

     case TVChannel::Japanese:
       m_FrequencySystemDropdown->setCurrentItem(2);
       break;

     case TVChannel::FrequencyGrid_MAX:
       m_FrequencySystemDropdown->setCurrentItem(-1);
       m_FrequencySystemDropdown->setEditText(QString::null);
       break;
   } // ..switch

   // Copy names to m_SystemChannels listview box
   m_SystemChannels->clear();
   j = TVChannel::GetNumberOfChannels(system);
   for (i = 0; i < j; i++) {
      m_SystemChannels->insertItem(TVChannel::ChannelName(system, i), i);
   }

   m_FrequencySystemDropdown->blockSignals(false);
}

/**
  \brief Remove channel

  This function removes the channel both from the channel list and the
  display. May update the current pointer.
*/
void CChannelEditorDlg::RemoveChannel(TVChannel *channel)
{
   QListViewItem *lvi = 0;

   if (m_pTVChannels == 0)
     return;
   if (channel == 0)
     return;

   lvi = m_Channel2ListView[channel];
   // Remember, m_pTVChannels is auto-delete
   if (!m_pTVChannels->remove(channel)) {
      qWarning("CChannelEditorDlg::RemoveChannel() Could not remove channel from list!");
   }
   else {
     if (lvi != 0) {
       delete lvi;
     }
     SetNewCurrent(m_pTVChannels->current());
   }
}


/**
   \brief Add channel at specified position

   \param channel The new TV Channel
   \param index Location in array; the default -1 means at end of list (append)

   This function adds the channel to the list

*/
void CChannelEditorDlg::AddChannel(TVChannel *channel, int index)
{
   QListViewItem *lvi = 0;

   if (m_pTVChannels == 0)
     return;

qDebug("AddChannel: index = %d", index);
   if (index == 0) {
qDebug("Added at front");
     lvi = 0; // added at front
   }
   else if (index < 0 || index >= m_pTVChannels->count()) { // added at end
qDebug("Added at end");
     lvi = m_Channel2ListView[m_pTVChannels->getLast()];
   }
   else {
qDebug("Added in between");
     lvi = m_Channel2ListView[m_pTVChannels->at(index - 1)]; // changes current ptr!
   }

qDebug("lvi = %p", lvi);
   if (lvi == 0) {
     lvi = new QListViewItem(m_PresetList);
   }
   else {
     lvi = new QListViewItem(m_PresetList, lvi);
   }

   if (index < 0 || index >= m_pTVChannels->count()) {
     m_pTVChannels->append(channel);
   }
   else {
     m_pTVChannels->insert(index, channel);
   }

   if (lvi == 0) {
     qWarning("CChannelEditorDlg::AddChannel() lvi = 0");
   }
   else {
     m_ListView2Channel[lvi] = channel;
     m_Channel2ListView[channel] = lvi;
   }
}


// protected slots

/**
  \brief Helper slot for the Name linedit box

  Called whenever the user types something in the Name box (which
  contains the name of the channel)
*/
void CChannelEditorDlg::NameChanged(const QString &new_name)
{
   if (m_UpdatingGUI) // prevent loops
     return;
   if (m_pTVChannels->current() != 0)
   {
     m_pTVChannels->current()->Name = new_name;
   }
   SetDialogControls(m_pTVChannels->current());
   SetListEntry(m_pTVChannels->current());
}

/**
  \brief Helper slot for the Frequency system dropdown box

*/
void CChannelEditorDlg::FrequencySystemChanged(int table)
{
   if (m_UpdatingGUI) // prevent loops
     return;
   if (m_pTVChannels->current() != 0) {
     switch (table) {
       case 0: m_pTVChannels->current()->FrequencySystem = TVChannel::American; break;
       case 1: m_pTVChannels->current()->FrequencySystem = TVChannel::European; break;
       case 2: m_pTVChannels->current()->FrequencySystem = TVChannel::Japanese; break;
     }
     m_pTVChannels->current()->Channel = 0;
     m_pTVChannels->current()->Finetuning = 0;
   }
   SetDialogControls(m_pTVChannels->current());
   SetListEntry(m_pTVChannels->current());
   if (m_pTVChannels->current() != 0) {
     emit NewFrequency(m_pTVChannels->current()->Frequency() * 1e6);
   }
}

/**
  \brief Helper slot for the system channel listview box

  Called when the user selects one of the entries in the list of system
  channels systembox; this are the pre-programmed channels in the current
  frequency grid (not to be confused with the Preset list, which is
  the list of TV channels that the user programs).

  Also emits new frequency.
*/
void CChannelEditorDlg::SystemChannelClicked(int channel)
{
   if (m_UpdatingGUI) // prevent loops
     return;
   if (m_pTVChannels->current() != 0) {
     m_pTVChannels->current()->Channel = channel;
   }
   SetListEntry(m_pTVChannels->current());
   SetNewCurrent(m_pTVChannels->current());
}

/**
  \brief Helper slot for the finetuning slider

  For finetuning the frequency of the current TV channel (preset)

  Also emits new frequency.
*/
void CChannelEditorDlg::FinetuningMoved(int offset)
{
   if (m_UpdatingGUI) // prevent loops
     return;
   if (m_pTVChannels->current() != 0)
   {
     m_pTVChannels->current()->Finetuning = offset;
   }
   SetDialogControls(m_pTVChannels->current());
   SetListEntry(m_pTVChannels->current());
   if (m_pTVChannels->current() != 0) {
     emit NewFrequency(m_pTVChannels->current()->Frequency() * 1e6);
   }
}

/**
  \brief Helper slot for Color System dropdown box

  When the user select a new color system (a.k.a. 'norm') from the drop down
  box, this will set that norm in the current channel (if any). It also
  emits a NewColorSystem signal.

*/
void CChannelEditorDlg::ColorSystemChanged(int color)
{
   if (m_UpdatingGUI) // prevent loops
     return;
  if (m_pTVChannels->current() != 0)
  {
    switch (color) {
      case 0: m_pTVChannels->current()->ColorSystem = TVChannel::PAL_BG; break;
      case 1: m_pTVChannels->current()->ColorSystem = TVChannel::NTSC; break;
      case 2: m_pTVChannels->current()->ColorSystem = TVChannel::SECAM; break;
      case 3: m_pTVChannels->current()->ColorSystem = TVChannel::PAL_NC; break;
      case 4: m_pTVChannels->current()->ColorSystem = TVChannel::PAL_M; break;
      case 5: m_pTVChannels->current()->ColorSystem = TVChannel::PAL_N; break;
      case 6: m_pTVChannels->current()->ColorSystem = TVChannel::NTSC_JAPAN; break;
    } // ..switch
  }
  SetDialogControls(m_pTVChannels->current());
  SetListEntry(m_pTVChannels->current());
  if (m_pTVChannels->current() != 0) {
    emit NewColorSystem(m_pTVChannels->current()->ColorSystem);
  }
}


/**
  \brief Helper slot for "Delete" button

  Removes the current entry and selects the next (or previous in case of
  end-of-list) entry, or 0 if it was the last one.
*/
void CChannelEditorDlg::DeleteClicked()
{
   TVChannel *pOldChannel = 0;

   // Check pointers
   if (m_pTVChannels == 0)
     return;

   // Already empty? Nothing to do...
   if (m_pTVChannels->count() == 0)
     return;

   RemoveChannel(m_pTVChannels->current());
}


/**
  \brief Move channel 'up' (earlier in list)

*/
void CChannelEditorDlg::UpClicked()
{
   int At;
   TVChannel *Copy, *Old;
   QListViewItem *lvi = 0, *OldLV = 0, *PrevLV;

   if (m_pTVChannels == 0)
     return;
   At = m_pTVChannels->at(); // current position in list
   if (At <= 0) // already at top/empty
     return;

   // okay... now create copy, remove current, insert copy before previous. Can't do swaps :(
   Old = m_pTVChannels->current();
   OldLV = m_Channel2ListView[Old];
   Copy = new TVChannel(*Old);

   AddChannel(Copy, At - 1);
   RemoveChannel(Old);
   SetListEntry(Copy);
   SetNewCurrent(Copy);

#ifdef DEBUG
   QListIterator<TVChannel> it(*m_pTVChannels);
   while (it.current() != 0)
   {
      qDebug("channel: " + it.current()->Name);
      ++it;
   }
#endif
}

/**
  \brief Move channel 'down' (later in list)

*/
void CChannelEditorDlg::DownClicked()
{
   int At;
   TVChannel *Copy, *Old;
   QListViewItem *lvi = 0, *OldLV = 0, *PrevLV;

   if (m_pTVChannels == 0)
     return;
   At = m_pTVChannels->at(); // current position in list
   if ((At + 1) >= m_pTVChannels->count()) // already at bottom/empty
     return;

   // okay... now create copy, remove current, insert copy after new current.
   Old = m_pTVChannels->current();
   OldLV = m_Channel2ListView[Old];
   Copy = new TVChannel(*Old);

   RemoveChannel(Old);
   AddChannel(Copy, At + 1);
   SetListEntry(Copy);
   SetNewCurrent(Copy);
#ifdef DEBUG
   QListIterator<TVChannel> it(*m_pTVChannels);
   while (it.current() != 0)
   {
      qDebug("channel: " + it.current()->Name);
      ++it;
   }
#endif
}


/**
   \brief Slot for when user clicked on "New..."

   This will create a new entry and append it to the list of channel presets.
   A new entry will be based on the current channel (if any)
*/
void CChannelEditorDlg::NewClicked()
{
   TVChannel *v = 0;

   if (m_pTVChannels == 0)
     return;

   if (m_pTVChannels->current() != 0) {
     v = new TVChannel(*(m_pTVChannels->current()));
     v->Name = "New";
   }
   else {
     v = new TVChannel();
   }

   if (v != 0) {
     AddChannel(v);
     SetListEntry(v);
     SetNewCurrent(v);

     m_NameLineEdit->setFocus();
     buttonApply->setEnabled(true);
     buttonCancel->setEnabled(true);
     m_CurrentIsNew = true;
   }
   else {
     qWarning("CChannelEditorDlg::NewClicked(): could not allocated TVChannel struct.");
   }
}


/**
  \brief Helper slot for Preset list view

  Called when an item in the Preset listview is clicked. It will
  set the current TVChannel pointer belonging to the entry.
*/
void CChannelEditorDlg::PresetClicked(QListViewItem *lvi)
{
qDebug(">> CChannelEditorDlg::PresetClicked(%p)", lvi);
   TVChannel *Search = 0;

   Search = m_ListView2Channel[lvi];
   SetNewCurrent(Search);
qDebug("<< CChannelEditorDlg::PresetClicked()");
}



// public

void CChannelEditorDlg::SetTVChannels(QList<TVChannel> *tv_channels)
{
   QListViewItem *after = 0;

   // clear existing arrays/widgets
   m_Channel2ListView.clear();
   m_ListView2Channel.clear();
   m_PresetList->clear();

   m_pTVChannels = tv_channels;
   if (m_pTVChannels == 0)
     return;

   // fill listview with channels
   QListIterator<TVChannel> it(*m_pTVChannels);
   while (it.current() != 0)
   {
     QListViewItem *lvi = new QListViewItem(m_PresetList, after);
     m_Channel2ListView[it.current()] = lvi;
     m_ListView2Channel[lvi] = it.current();
     SetListEntry(it.current());
     ++it;
     after = lvi;
   }

   SetDialogControls(m_pTVChannels->current());
}

