/**
  \class CVideoDeviceInput

  Every video device has one or more inputs (called channels in the
  Video4Linux API). Each input refers to one of the physical inputs
  on the card/chip, like Tuner, Composite, S-video, etc. This class represents
  such an input.

  In addition, each input can have 0 or more tuners attached to it (to keep
  matters simple...). A tuner does the actual frequency and norm setting,
  and is represented by a \ref CVideoDeviceTuner class.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>

#include "VideoDeviceInput.h"
#include "VideoDeviceLinux.h"

/**
  \brief Constructor

  In the constructor the channel information is queried and stored. It
  will also create any Tuner objects that may be needed.
*/

CVideoDeviceInput::CVideoDeviceInput(CVideoDeviceLinux *video, int channel)
        : QObject(video, "video device input")
{
   char buf[33];
   CVideoDeviceTuner *tuner = 0;

   m_pVideo = video;

   memset(&m_VChan, 0, sizeof(m_VChan));
   m_CurrentTuner = -1;

   m_Tuners.setAutoDelete(TRUE);
   // Query data
   m_VChan.channel = channel;
   if (ioctl(m_pVideo->m_CamFD, VIDIOCGCHAN, &m_VChan) == 0) {
     strncpy(buf, m_VChan.name, 32);
     buf[32] = '\0';
     m_Name = buf;

     if (m_VChan.flags & VIDEO_VC_TUNER) {
       m_Tuners.resize(m_VChan.tuners);
qDebug("%s has %d tuners", buf, m_VChan.tuners);
       for (int i = 0; i < m_VChan.tuners; i++) {
          tuner = new CVideoDeviceTuner(m_pVideo, i);
          m_Tuners.insert(i, tuner);
          connect(tuner, SIGNAL(Selected(int)), this, SLOT(VideoTunerSwitched(int)));
       }
     }
     else
       m_Tuners.resize(0);
   }
   else
     qWarning("CVideoDeviceInput: Warning: no channel info available.");
}

// private slots

void CVideoDeviceInput::VideoTunerSwitched(int n)
{
   m_CurrentTuner = n;
}


// public

/**
  \brief Return channel number.
*/
int CVideoDeviceInput::GetNumber() const
{
   return m_VChan.channel;
}

/**
  \brief Return symbolic name for input.
*/
QString CVideoDeviceInput::GetName() const
{
   return m_Name;
}

/**
  \brief Return type for this input.

  Returns a value from the \ref InputTypes enum, either TV or Camera
*/
int CVideoDeviceInput::GetType() const
{
   int type = Unknown;

    switch(m_VChan.type) {
      case VIDEO_TYPE_TV:     type = TV;     break;
      case VIDEO_TYPE_CAMERA: type = Camera; break;
   }
   return type;
}

/**
  \brief Returns whether this input has audio settings associated with it.
*/
bool CVideoDeviceInput::HasAudio() const
{
   if (m_Flags & VIDEO_VC_AUDIO)
     return true;
   return false;
}


/**
  \brief Make this input the current one.

*/
bool CVideoDeviceInput::Select()
{
   if (ioctl(m_pVideo->GetDescriptor(), VIDIOCSCHAN, &m_VChan) < 0)
   {
     qDebug("CVideoDeviceInput::Select() failed.");
     return false;
   }
   else
     emit Selected(m_VChan.channel);
   return true;
}


/**
  \brief Return number of tuners.

  Most inputs don't have a tuner, or at most 1. Multiple tuners could
  be used for multi-norm cards (each norm having a separate tuner), but
  this hasn't happened sofar.
*/
int CVideoDeviceInput::GetNumberOfTuners() const
{
   return (int)m_Tuners.count();
}

/**
  \brief Get a Tuner object
  \param number The desired input (0 to GetTuners() - 1); -1 for default (current) one
  \return A pointer to a CVideoDeviceTuner object

  This function will return 0 if the current tuner is unknown or \b number is out of range.
*/
CVideoDeviceTuner *CVideoDeviceInput::GetTuner(int number) const
{
   if (number == -1) {
     number = m_CurrentTuner;
   }

   if (number < 0 || number >= (int)m_Tuners.count())
     return 0;
   return m_Tuners.at((uint) number);
}

/**
  \brief Return current tuner number
  \return Tuner number, or -1 if the current tuner is unknown

  Since there is no way to query the current selected tuner from the device,
  this function returns -1 until a tuner has been selected.
*/
int CVideoDeviceInput::GetCurrentTuner() const
{
   return m_CurrentTuner;
}

