/* ActivePack.c 					9/99 */

/* CAUTION: 5/18/00 I changed the definition of p_light in
   cp_types.h */

/* This code is taken from Repack.c; idea is to control various
features of repack computations, flow, repack strategies, status 
reports, saving results, etc. It is to be driven command-line;
optionally it will be forked, with commands coming from
a Java interface window.

cc -o ActivePack ActivePack.c -I/usr/openwin/include cp_remote_library.c hyp_math.c eucl_math.c sph_math.c complex_math.c -g -lm 

execute as "ActivePack java" to start the java interface.
*/

#undef _KERNEL
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/resource.h>
#include "cp_types.h"
#include "cp_proto.h"

#include <xview/tty.h>

/* ===================================================================
   structure for keeping track of grains 
=================================================================== */

struct Grains
 {
   struct p_light *pl;      /* the p_light structure */
   int seed;                /* vert used as seed for this grain */
 };

/* ==================================================================== 
   global declarations
 ==================================================================== */

/* basic packings */
struct p_data *packdata;
/*struct K_data *pK_ptr;
  struct R_data *pR_ptr;*/
struct Grains *grains;   /* collection of pack_light structures */
int grain_count;
int **face_org;

/* utility buffers. */
char packname[256];
char emsgbuf[BUFSIZE],buf[BUFSIZE],data[2*BUFSIZE],time_buf[64],*msgbuf;
char param_name[BUFSIZE],param_option[BUFSIZE]; 

/* repacking parameters */
int max_passes=200;	/* default: number of repack cycles;
			   normally, one "pass" means going
			   through all verts subject to change
			   and changing any that meet criteria. */
int report_cycle=10;	/* default: interval between status reports */
int iterates=2;		/* parameter of current packing routines */
int num_digits=8;           /* default: digits of accurracy */
double recip;            /* used as threshhold for circles to repack */
double toler=.000000001,okerr=.00000000001; /* threshholds/zero */
double m2pi=2.0*M_PI;

/* persistent storage of info for routines */
int *vert_indices,aimnum; /* storage/count of verts subject to adjustment */
double *R0;              /* storage for superstep routines */
int residual_count;
struct Vertlist *residual_list,*residual_ptr;

/* control parameters */
int repack_method=1;  /* 1=standard_iteration */
                      /* 2=superstep_method */
                      /* 3=perron_up_only */
                      /* 4=perron_down_only */
int rad_comp_method=1; /* 1=standard_bisection */
                      /* 2=uniform (equal nghbs) */
                      /* 3=newton_iteration */
int status_flags=1;   /* specified bitwise: */
                      /*   1: use report_cycle */
int data_flags=0;     /* specified bitwise: */
                      /*   1: record ups/downs */
int debug_method=0;   /* default: none */

/* monitoring matrices */
int *count_ups=NULL,*count_downs=NULL,global_passes=0; 
                      /* how often each vert increases or decreases */

/* timing stuff */
int elapse_hr,elapse_min,elapse_sec;
long time_in=0.0,time_out=0.0,elapse_time=0.0,global_elapsed=0.0;

/* file pointers */
FILE *to_java,*from_java;
FILE *data_fp, *packfp=NULL,*cmdfp;

/* procedure prototypes */
int sort_a_cmd(char *inmsg);
int parse_a_cmd(char *command);
int e_riffle_super(struct p_data *p,int passes);
int e_riffle_standard(struct p_data *p,int passes);
double erad_calc(struct p_data *p,int i,double r,double aim,int *chgflag);
int h_riffle_super(struct p_data *p,int passes);
int h_riffle_standard(struct p_data *p,int passes);
int Rewrite(FILE *fp);
int report_status(double cycount,int flag);
int log_pack_info();
int log_param_info();
int read_param_set(char *datastr);
int java_msg(char *datastr);
int check_interrupt();
int open_ups_downs();
int e_perron(int passes);
int granulator(struct p_data *p,int n);
struct RedList *build_grain_redlist(struct p_data *p,int seed,
	       int *vec_seed,struct Vertlist *tether,int gen);
int simplify_grain_redchain(struct p_data *p,struct RedList **redlist,
			    struct Vertlist *tether);
struct p_light *extract_light_pack(struct p_data *p,int seed,
		struct RedList **redlist,int **green_list,
		int *okay,int n,int nmbr);
struct p_light *extract_residual(struct p_data *p,int **green_list,int *okay);
struct p_light *extract_negotiate(struct p_data *p,int *okay);
int find_far_seed(struct p_data *p,int *base_verts,
		  struct Vertlist *tether,int max);
struct Vertlist *create_tether(struct p_data *p,int *green_list);
int emsg();
int msg();
int repack_activity_msg(char *msg);
int grain_free();
extern int e_pack_light_uniform(), h_pack_light_uniform();

/* ==================================================================== 
   main routine
 ==================================================================== */

int main(int argc,char *argv[])
{
  int i,start_java=0;
  int pipeto[2],pipefr[2];
  char inmsg[BUFSIZE],*arg[5];
  pid_t pid;

/* ---------- initialization --------------- */
  
  grain_count=0;
  strcpy(emsgbuf,"ERROR: ");
  msgbuf=emsgbuf+7;
  packname[0]='\0';
  residual_list=residual_ptr=NULL;
  residual_count=0;

/* ----- interpret command-line arguments -------- */
  if (argc > 1) /* eventually may need command line arguments */
    {
      if (!strcmp(argv[1],"java")) start_java=1;
    }

  if ((packdata=(struct p_data *)
       calloc((size_t)1,sizeof(struct p_data)))==NULL ||
      !alloc_pack_space(packdata,10000,0))
    {
      printf("Insufficient memory.");
      exit(0);
    }

/* ----- fork, set up pipes, child fires off Java interface tool -- */
  if (start_java)
    {

	if (pipe(pipeto) || ( pipe(pipefr) 
		&& (close(pipeto[0]),close(pipeto[1]),1) ) )
	 {
		sprintf(msgbuf,"Pipe opening error.  ");
		emsg(msgbuf);
		exit(1);
	 } 

	if ((pid = fork())) /* in parent */
	 {
        	if (pid == -1)
		 {
		   sprintf(msgbuf,"Failed to fork for process.  ");
		   emsg(msgbuf);
		   close(pipeto[0]);close(pipeto[1]);
		   close(pipefr[0]);close(pipefr[1]);
		   exit(2);
		 }
        	(void)close(pipeto[0]);
        	(void)close(pipefr[1]);
        	to_java=fdopen(pipeto[1], "w");
        	from_java=fdopen(pipefr[0], "r");
	 }
	else /* in child */
	 {
	   arg[0]=(char *)calloc((size_t)128,sizeof(char));
	   strcpy(arg[0],"java");
	   arg[1]=(char *)calloc((size_t)128,sizeof(char));
	   strcpy(arg[1],"-classpath");
	   arg[2]=(char *)calloc((size_t)128,sizeof(char));
	   strcpy(arg[2], getenv("HOME") );
	   strcat(arg[2], "/java/attempts/ActivePackControl" );
	   arg[3]=(char *)calloc((size_t)128,sizeof(char));
	   strcpy(arg[3],"ActivePackControl");
	   arg[4]=(char *)NULL;

        	(void)close(pipeto[1]);
        	(void)close(pipefr[0]);
	      	(void)dup2(pipeto[0], 0);
        	(void)dup2(pipefr[1], 1);
			/* in child, close out all notifier stuff. */
		for (i=getdtablesize();i>2;i--) 
			(void)close(i);
		for (i=0;i<NSIG;i++)
			(void)signal(i,SIG_DFL);
		execlp("java",arg[0],arg[1],arg[2],arg[3],arg[4]);
      		_exit(1);
	 }
    }

/* --- mini-notifier loop: continually check for input from java (or stdin)*/

  log_param_info();
  if (!start_java) printf("cmd: ");
  while ((from_java && fgets(inmsg,BUFSIZE,from_java)) 
	 || fgets(inmsg,BUFSIZE,stdin))
    {
      sort_a_cmd(inmsg);
    }
  exit(100);
} /* end of main */


/* ==================================================================== 
   parsing incoming commands
 ==================================================================== */

int sort_a_cmd(char *inmsg)
/* pick of individual commands in input stream, separated by ';' .*/
{
  int flag=1,n;
  char *command,local_stream[BUFSIZE],*token_ptr;

  strcpy(local_stream,inmsg);
  local_stream[BUFSIZE-1]='\0';
  token_ptr=local_stream;
  while (*(token_ptr)==';') token_ptr++;
  stripsp(token_ptr);
  while (flag && (n=strcspn(token_ptr,";"))>0)
    {
      flag=0;
      command=token_ptr;
      if (*(token_ptr+n)!='\0') flag=1;
      *(token_ptr+n)='\0';
      token_ptr=token_ptr+n+1;
      stripsp(token_ptr);
      while (*(token_ptr)==';') token_ptr++;
      flag=parse_a_cmd(command); /* stops if parse returns 0 */
    } /* end of while */
  printf("cmd: ");
  return flag;
} /* sort_a_cmd */

int parse_a_cmd(char *command)
/* Handle the commands passed from sort_a_cmd */
{
  int i,g,j,mxu,mxd,n;
  long elapsetime;
  char *nextpoint,*dpoint,cmd[BUFSIZE],newpack[256];
  struct p_light *pl;
  struct R_data *pR_ptr=packdata->packR_ptr;
  FILE *packfp=NULL,*outfp;

  nextpoint=command;
  if (!grab_next(&nextpoint,cmd)) return 0; /* get command */
  strcpy(data,nextpoint); /* rest is data */
  dpoint=data;
  switch (cmd[0])
    {
    case 'b':
      {
	if (!strncmp(cmd,"bye",3)) /* =========== exit */
	  {
	    printf("bye");
	    exit(5);
	  }
	break;
      }
    case 'g':
      {
	if (!strncmp(cmd,"granulate",9)) /* =========== granulate */
	  {
	    if (!sscanf(dpoint,"%d",&n) || n<3) return 0;
	    grain_count=granulator(packdata,n);
	  }
	break;
      }
    case 'p': 
      {
	if (!strncmp(cmd,"pack_l",6)
	  && sscanf(dpoint,"%d",&g)
	  && grains && g>0 && g<=grain_count) /* ====== pack one p_light */
	  {
	    if (packdata->hes<0) /* hyperbolic */
	      {
		e_pack_light_uniform(grains[g].pl,max_passes,toler,
		  repack_method,&elapsetime);
	      }
	    else if (packdata->hes==0) /* euclidean */
	      {
		h_pack_light_uniform(grains[g].pl,max_passes,toler,
		  repack_method,&elapsetime);
	      }
	    else
	      {
		printf("Packing is spherical, no algorithm yet available.");
		return 0;
	      }
	  }
	else if (!strncmp(cmd,"pack",4))  /* ========= start packing comp  */
	  {
	    if (!packdata->status)
	      {
		printf("No packing has been loaded.");
		return 0;
	      }
	    if (packdata->hes>okerr)
	      {
		printf("Packing is spherical, no algorithm yet available.");
		return 0;
	      }
	    if (packdata->hes<-okerr) switch(repack_method) /* hyperbolic */
	      {
	      case 2: 
		{
		  h_riffle_super(packdata,max_passes);
		  break;
		}
	      default: /* standard iteration */
		{
		  h_riffle_standard(packdata,max_passes);
		  break;
		}
	      }
	    else switch(repack_method) /* euclidean */
	      {
	      case 2: 
		{
		  e_riffle_super(packdata,max_passes);
		  break;
		}
	      default: /* standard iteration */
		{
		  if (repack_method==3 || repack_method==4) 
		    e_perron(max_passes);
		  else e_riffle_standard(packdata,max_passes);
		  break;
		}
	      }
	  }
	break;
      }
    case 'r': 
      { 
	if (!strncmp(cmd,"read",4)) /* ========== read packing */
	  {
	    if (!sscanf(dpoint,"%s",newpack) 
		|| (packfp=fopen(newpack,"r"))==NULL
		|| !readpack(packfp,packdata) )
	      {
		printf("Read of file %s has failed.\n",newpack);
		if (packfp) fclose(packfp);
		return 0;
	      }
	    strncpy(packname,newpack,255);
	    if (packfp) fclose(packfp);
	    packdata->status=1;
	    pR_ptr=packdata->packR_ptr;
	    log_pack_info();
	    open_ups_downs();
	    for (i=1;i<=packdata->nodecount;i++) /* set bdry flags */
	      {
		if (packdata->packK_ptr[i].flower[0]==
		    packdata->packK_ptr[i].flower[packdata->packK_ptr[i].num])
		  packdata->packK_ptr[i].bdry_flag=0;
		else packdata->packK_ptr[i].bdry_flag=1;
	      }
	    /* create storage, identify vertices subject to 
	       adjustment once and for all. */
	    vert_indices=(int *)
	      calloc((size_t)(packdata->nodecount+1),sizeof(int));
	    aimnum=0;
	    for (i=1;i<=packdata->nodecount;i++)
	      if (pR_ptr[i].aim>0)
		{
		  vert_indices[aimnum]=i;
		  aimnum++;
		}
	    if (aimnum==0) 
	      {
		/* fixup: report error */
	      }
	    recip=.333333/aimnum;
	    R0 = (double *)calloc((size_t)(packdata->nodecount+1),
				  sizeof(double));
	    fillcurves(packdata);
	    return 1;
	  }
	else if (!strncmp(cmd,"reset_clock",11)) 
	  /* ======== reset global clock */
	  {
	    global_elapsed=0;
	  }
	else if (!strncmp(cmd,"reset_counts",12) && count_ups) 
	  /* ============ reset the counting data */
	  {
	    for (i=1;i<=packdata->nodecount;i++) 
	      {count_ups[i]=0; count_downs[i]=0;}
	  }
	else if (!strcmp(cmd,"reset_global"))
	  /* ============ reset accumulating data */
	  {
	    global_elapsed=0.0;
	    global_passes=0;
	  }
	break;
      }	
    case 's': 
      { 
	if (!strcmp(cmd,"save_counts") && count_ups 
	    && grab_next(&dpoint,param_name)
	    && (data_fp=fopen(param_name,"w")) )
	  /* ====== store counts data */
	  {
	    mxu=mxd=0;
	    for (i=1;i<=packdata->nodecount;i++)
	      {
		mxu=(count_ups[i]>mxu) ? count_ups[i] : mxu;
		mxd=(count_downs[i]>mxd) ? count_downs[i] : mxd;
	      }
	    fprintf(data_fp,"Data on packing %s.\n  MAX_UPS: %d"
		    "\n MAX_DOWNS: %d\n\n",packname,mxu,mxd);
	    fprintf(data_fp,"  (vert: count_ups  count_downs)\n");
	    for (i=1;i<=packdata->nodecount;i++)
	      {
		fprintf(data_fp,"   %d : %d  %d\n",
			i,count_ups[i],count_downs[i]);
	      }
	    fprintf(data_fp,"\nEND\n");
	    fclose(data_fp);
	    printf("Save of counts to %s is complete.\n",param_name);
	  }

	/* ============= set parameters ================*/
	else if (!strncmp(cmd,"set",3) && read_param_set(dpoint)) 
	  {
	    if (!strncmp(param_name,"num_digits",10)) /* === set accuracy */
	      {
		if (!sscanf(param_option,"%d",&i) || i<2 || i>15)
		  {
		    printf("error: 2 <= num_digits <= 15");
		    break;
		  }
		else num_digits=i;
		toler=0.1;
		for (j=1;j<=i;j++) toler *= 0.1;
		okerr=(0.01)*toler;
	      }
	    else if (!strcmp(param_name,"iterates")) 
	      /* == iterates for bisection/Newton */
	      {
		if (!sscanf(param_option,"%d",&iterates) 
		    || iterates <1 || iterates > 10)
		  iterates=2;
	      }
	    else if (!strcmp(param_name,"repack_method"))
	      {
		if (!strcmp(param_option,"standard_iteration"))
		  repack_method=1;
		else if (!strcmp(param_option,"superstep_method"))
		  repack_method=2;
		else if (!strcmp(param_option,"perron_up_only"))
		  repack_method=3;
		else if (!strcmp(param_option,"perron_down_only"))
		  repack_method=4;
	      }
	    else if (!strncmp(param_name,"rad_comp_method",16))
	      {
		if (!strcmp(param_option,"standard_bisection"))
		  rad_comp_method=1;
		else if (!strcmp(param_option,"uniform"))
		  rad_comp_method=2;
	      }
	    else if (!strcmp(param_name,"data_flags"))
	      {
		if (!strcmp(param_option,"none"))
		  data_flags=0;
		if (!strcmp(param_option,"ups_downs"))
		  {
		    open_ups_downs();
		    data_flags |= 1;
		  }
	      }
	    else if (!strcmp(param_name,"status_flags"))
	      {
		if (!strcmp(param_option,"none"))
		  status_flags=0;
		if (!strcmp(param_option,"report_cycle"))
		  status_flags |= 1;
	      }
	    else if (!strcmp(param_name,"debug_method"))
	      {
		if (!strcmp(param_option,"none"))
		  debug_method=0;
	      }
	    else if (!strcmp(param_name,"report_cycle"))
	      {
		if (!sscanf(param_option,"%d",&report_cycle)
		    || report_cycle<0)
		  report_cycle=1000;
	      }
	    else if (!strcmp(param_name,"max_passes"))
	      {
		if (!sscanf(param_option,"%d",&max_passes)
		    || max_passes<0)
		  max_passes=2000;
	      }
	  } /* end of 'set' routines */
	else if (!strcmp(cmd,"status_report")) 
	  {
	    report_status(0.0,1);
	  }
	break;
      }
    case 'w':
      {
	if (!strcmp(cmd,"write"))  /* ======== write packing */
	  {
	    if (!packdata->status)
	      {
		printf("Write failed: no packing is currently loaded.\n");
		return 0;
	      }
	    if (sscanf(dpoint,"%s",packname)==EOF)
	      strcat(packname,"p");
	    if ((outfp=fopen(packname,"w"))==NULL || !Rewrite(outfp))
	      {
		printf("Attempt to write %s failed.\n",packname);
		if (outfp) fclose(outfp);
		return 0;
	      }
	    printf("Stored packing in file %s.\n",packname);
	    if (outfp) fclose(outfp);
	    return 1;
	  }
	else if (!strncmp(cmd,"write_g",7) /* ===== write grains */
	  && grains)
	  {
	    for (g=1;g<=grain_count;g++)
	      {
		pl=grains[g].pl;
		sprintf(buf,"%s_%d.p",packname,g);
		if (!(packfp=fopen(buf,"w"))) break;
		write_light(packfp,pl,0);
		fclose(packfp);
	      }
	  }
	else if (!strncmp(cmd,"write_cmd",9) /* === write cmd */
	  && grains)
	  {
	    sprintf(buf,"%s.cmd",packname);
	    if (!(cmdfp=fopen(buf,"w"))) break;
	    fprintf(cmdfp,"[]:= act 0;read %s;set_screen -a;disp -w -c;\n",
		    packname);
	    for (g=1;g<=grain_count;g++)
	      {
		pl=grains[g].pl;
		fprintf(cmdfp,"[]:= read -p1 %s_%d.p;map -p1 b;"
			"disp -p0 -cffg %d Vlist\n",
			packname,grains[g].seed,grains[g].seed);
	      }
	    fclose(cmdfp);
	  }
	break;
      }
    case '?':
      {
	if (!strncmp(cmd,"?param",6)) /* ======== show parameters */
	  {
	    log_param_info();
	    return 1;
	  }
	if (!strncmp(cmd,"?pack",5)) /* ======== show pack info */
	  {
	    log_pack_info();
	    return 1;
	  }
	if (!strncmp(cmd,"?cmd",4)) /* ======== show commands */
	  {
	    fprintf(stdout,"Commands: pack  pack -seed  read  "
		    "write ?param set num_digits\n  set repack_method "
		    "(standard_iteration, superstep_method, "
		    "perron_up_only, perron_down_only, "
		    "perron_up_then_down, perron_down_then_up)\n");
	    fprintf(stdout,"set rad_comp_method (standard_bisection, "
		    "uniform)\n  set status_flags (none, n)   "
		    "set data_flags (none, ups_downs)  set "
		    "debug_method (none)\n  reset_clock   "
		    "reset_counts   save_counts <filename>    "
		    "status_report\n  max_passes <n>   "
		    "report_cycle <n>\n");
	    fprintf(stdout,"granulate <n>,  write_granules <root-name>\n");
	  }
	break;
      }
    } /* end of cmd switch */
  return 0;
} /* parse_a_cmd */

/*=================== repacking routines =======================*/

/* =============================================================
BASICS: Vertices subject to adjustment recorded in 'vert_indices'. 
Threshholds are set based on accumulated pack info and tolerance settings. 
Each repacking cycle involves:

- choose next vertex to adjust based on threshhold criteria
- choose repack_method: standard/superstep
- choose radius adjustment method: bisection/Newton/uniform-neighbor,
  verifying limits on adjustments.
- record status info if requested
- record new radius, (update angle sum? may not be worth it)
- check stop criteria
- check for user interrupt

Report results. (This is modified for superstep method, since it's a 
two stage process.) 
NOTE: lot's of inefficiencies here which can be cleared up after testing.
============================================================= */

int e_riffle_super(struct p_data *p,int passes)
/* CRC - modified 5/28/97 */
/* uses the latest model, with super steps and safety checks */
/* adjust eucl radii to meet curvature targets in 'aim'.
A target less than zero means that vertex is free - no adjustments made to it.
Bdry radii adjusted only if their aim >= 0. */
{
  int i,j,count=0,report_count=0,key=0,key0,N,dummy=0,stop_flag=0;
  double r, r2, fbest, faim, del, bet;
  double ttoler, cut=100, fact=-1, cut0, fact0;
  double lmax, rat,rr,ftol=.05;
  struct K_data *pK_ptr=p->packK_ptr;
  struct R_data *pR_ptr=p->packR_ptr;

  time_in=get_time();
  if (global_elapsed==0.0) global_elapsed=time_in;
  ttoler = 3*aimnum*toler; /* adjust tolerance */

/* ++++++++++ main loop */

  while (!stop_flag)

 {
/* save values */
	cut0 = cut;key0 = key;fact0 = fact; 
	for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
/* do update using uniform label model */
	cut = 0;
	for (j=0;j<aimnum;j++) 
	 {
		i = vert_indices[j]; 		/* point to active node */
		faim = pR_ptr[i].aim;	/* get target sum */
		r = pR_ptr[i].rad;	/* get present label */
		e_anglesum_overlap(p,i,r,&fbest,&dummy);  /* compute sum */
/* use the model to predict the next value */
		N = 2*pK_ptr[i].num;
		del = sin(faim/N);
		bet = sin(fbest/N);
		r2 = r*bet*(1-del)/(del*(1-bet));
/* store as new radius label */
		pR_ptr[i].rad = r2; 
		pR_ptr[i].curv = fbest;	/* store new angle sum */
		fbest -= faim;
		cut += fbest*fbest;	/* accum abs error */
	 }
	cut = sqrt(cut);
/* do super step? */
	key = 1;
	if (key0==1)
	 {
		fact = cut/cut0;
		if (fabs(fact-fact0)<ftol) fact = fact/(1-fact);
		lmax = 1000;
		for (j=0;j<aimnum;j++)	 /* find max step */
		 {
		   i = vert_indices[j];
		   r = pR_ptr[i].rad;
		   rat = r - R0[i];
		   if (rat<0)
			lmax = (lmax < (rr=(-r/rat))) ? lmax : rr;
		 }
		fact = (fact < 0.5*lmax) ? fact : 0.5*lmax; 
			/* compute new step */
		for (j=0;j<aimnum;j++)     /* interpolate to new radius */
		 {
		   i = vert_indices[j];
		   pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
		 }
		key = 0;
	 }
	report_count++;

/* ++++++++++ report abbreviated status info */
	if (report_count>report_cycle)
	  {
	    count += report_count;
	    if (status_flags & 1) report_status((double)count,0);
	    report_count=0;
	  }

/* ++++++++++ stop criteria */
	
	if (cut< ttoler) stop_flag=2;
	else if (count>passes) stop_flag=3;

/* ++++++++++ check for user interrupt */

	if (check_interrupt()) stop_flag=4;

 } /* end of while */

/* ++++++++++ return, with reason */
  if (stop_flag==2) printf("Packing stopped because cut=% .6e < toler=% .6e\n",
			   cut,toler);
  else if (stop_flag==3) printf("Packing stopped by limit %d on passes\n",passes);
  else if (stop_flag==4) printf("Packing interrupted by the user\n");
  report_status(count,1); /* final report */

  return count;
} /* e_riffle_super */


/* -----------------------------------------------------------------------
            pre 1996 riffle routine (slow, but dependable)
----------------------------------------------------------------------- */

int e_riffle_standard(struct p_data *p,int passes)
/* (Could use some reworking.) */
{
  int i,j,vert_passes,count=0,stop_flag=0,dummy;
  int status_passes,ca,report_count=0;
  double accum=0,vert_err,err,cut,temprad;
  struct R_data *pR_ptr=p->packR_ptr;

  time_in=get_time();
  if (global_elapsed==0.0) global_elapsed=time_in;
/* Set up counter limits
   fixup: need to settle on what 'count' counts */
  vert_passes=aimnum*passes;
  status_passes=aimnum*report_cycle;

/* ++++++++++ set threshholds */

  for (j=0;j<aimnum;j++)
    {
      i=vert_indices[j];
      e_anglesum_overlap(packdata,i,pR_ptr[i].rad,
	&pR_ptr[i].curv,&dummy);
      err=pR_ptr[i].curv-pR_ptr[i].aim;
      accum += (err<0) ? (-err) : err;
    }

  cut=accum*recip;
  if (cut<toler || count>vert_passes) stop_flag=1;

/* ++++++++++ main loop */

  while (!stop_flag)
    {
      for (j=0;j<aimnum;j++)
	{
	  i=vert_indices[j];
	  e_anglesum_overlap(packdata,i,pR_ptr[i].rad,
	    &pR_ptr[i].curv,&dummy);
	  vert_err=pR_ptr[i].curv-pR_ptr[i].aim;

/* ++++++++++ choose vert */


	  if (fabs(vert_err)>cut) 
	    {
/* ++++++++++ compute new radius */

	      temprad=erad_calc(packdata,i,pR_ptr[i].rad,pR_ptr[i].aim,&dummy);

/* ++++++++++ update individual vert status info */

	      if (data_flags & 1)
		{
		  if (temprad>pR_ptr[i].rad) count_ups[i]++;
		  else if (temprad<pR_ptr[i].rad) count_downs[i]++;
		}
	      pR_ptr[i].rad=temprad;

	      stop_flag++;
	    }
	} /* end of loop through verts */

/* ++++++++++ update angle sums, accum error after a few adjustments */

      accum=0;
      for (j=0;j<aimnum;j++)
	{
	  i=vert_indices[j];
	  e_anglesum_overlap(packdata,i,pR_ptr[i].rad,
			     &pR_ptr[i].curv,&dummy);
	  err=pR_ptr[i].curv-pR_ptr[i].aim;
	  accum += (err<0) ? (-err) : err;
	}
      cut=accum*recip;
      report_count += stop_flag;

/* ++++++++++ report abbreviated status info */
      if (report_count>vert_passes || report_count>status_passes) 
	{
	  count += report_count;
	  if (status_flags & 1) report_status((double)count/aimnum,0);
	  report_count=0;
	}

/* ++++++++++ stop criteria */

      if (!stop_flag) stop_flag=1; 
      else if (cut<toler) stop_flag=2;
      else if (count>vert_passes) stop_flag=3;
      else stop_flag=0; /* continue another loop */

/* ++++++++++ check for user interrupt */

      if (check_interrupt()) stop_flag=4;

    } /* end of main while */

/* ++++++++++ return, with reason */

  ca=(int)(count/aimnum);
  if (stop_flag==1) printf("Packing stopped because no circles adjusted.\n");
  else if (stop_flag==2) printf("Packing stopped because cut=% .6e < toler=% .6e\n",cut,toler);
  else if (stop_flag==3) printf("Packing stopped by limit %d on passes\n",ca);
  else if (stop_flag==4) printf("Packing interrupted by the user\n");

  global_passes += ca;
  report_status(ca,1); /* final report */
  return ca; 

} /* e_riffle_standard */

double erad_calc(struct p_data *p,int i,double r,double aim,int *chgflag)
/* Compute new radius in euclidean packing, method depending on 
rad_comp_method:
  1: standard_bisection
  2: uniform neighbors shortcut 
  3: newton_iteration (not yet reimplemented)
Initial guess = r. */
{
  int n,dummy=0,N;
  double bestcurv,lower=0.5,upper=0.5,upcurv,lowcurv,factor=0.5;
  double del,bet;
  struct K_data *pK_ptr=p->packK_ptr;

  e_anglesum_overlap(p,i,r,&bestcurv,&dummy); /* orig curv */

  if (rad_comp_method==1) /* standard_bisection (with bounds) */
    {
      if (bestcurv>(aim+okerr))
	{
	  upper=r/factor;
	  e_anglesum_overlap(p,i,upper,&upcurv,&dummy);
	  (*chgflag)++;
	  if (upcurv>aim) return upper;
	}
      else if (bestcurv<(aim-okerr))
	{
	  lower=r*factor;
	  e_anglesum_overlap(p,i,lower,&lowcurv,&dummy);
	  (*chgflag)++;
	  if (lowcurv<aim) return lower;
	}
      else return r;
      for (n=1;n<=iterates;n++)
	{
	  if (bestcurv>(aim+okerr)) 
	    {
	      lower=r;
	      lowcurv=bestcurv;
	      r += (aim-bestcurv)*(upper-r)/(upcurv-bestcurv);
	    }
	  else if (bestcurv<(aim-okerr))
	    {
	      upper=r;
	      upcurv=bestcurv;
	      r -= (bestcurv-aim)*(lower-r)/(lowcurv-bestcurv);
	    }
	  else return r;
	  e_anglesum_overlap(p,i,r,&bestcurv,&dummy);
	}
      return r;
    }
  else /* use uniform nghbs as default */
    {
      N = 2*pK_ptr[i].num;
      del = sin(aim/N);
      bet = sin(bestcurv/N);
      return (r*bet*(1-del)/(del*(1-bet)));
    }
  return 1;
} /* erad_calc */

int h_riffle_super(struct p_data *p,int passes)
/* CRC - modified 5/28/97 */
/* uses the best model, with super steps and safety checks 
   also has the anglesum calculation inlined.
   NOTE: overlap routines could use work. */
/* adjust radii to meet curvature targets in 'aim'.
A target less than zero means that vertex is free - no adjustments made.
Bdry radii adjusted only if their aim >= 0. */
{
	int i,j,k,j1,j2,count=0,key=0,key0,N,flag=0;
	double r, r1,r2,r3,fbest,faim,del,bet;
        double ttoler,cut=100,fact=-1,cut0,fact0;
        double ftol=0.05,lmax,rat,twor;
        double m2,m3,sr,t1,t2,t3,tr,o1,o2,o3;
	struct K_data *pK_ptr=p->packK_ptr;
	struct R_data *pR_ptr=p->packR_ptr;

   /* list radii to be adjusted, store squared radii */
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0 
		   && pR_ptr[i].aim<.001)
			pR_ptr[i].rad = (-.2);
		else if (pR_ptr[i].aim>0)
		 {
			if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001)
				pR_ptr[i].rad = .01;
		 }
		if (pR_ptr[i].rad>0)
		pR_ptr[i].rad = pR_ptr[i].rad*pR_ptr[i].rad;
	 }
	ttoler = 3*sqrt((double)aimnum)*toler;		/* adjust tolerance */
	cut = ttoler+1;
/*	cut =3*aimnum*10;*/
while (cut >ttoler && count<passes)         /* Begin Main Loop */
 {
	cut0 = cut;key0 = key;fact0 = fact;	/* save prev values */
	for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
	cut = 0;				    
/* Begin Iteration Loop */
	for (j=0;j<aimnum;j++) 
	 {
	/* anglesum calc */
		fbest = 0;
		i = vert_indices[j];
		if ((r = pR_ptr[i].rad)<0) r = 0;
		sr = sqrt(r);
		N = pK_ptr[i].num;
		if (!p->overlap_status)
		 { 			/* no overlaps */
               		twor = 2*r; 
               		r2 = pR_ptr[pK_ptr[i].flower[0]].rad;
               		m2 = (r2>0) ? (1-r2)/(1-r*r2) : 1;
               		for (k=1;k<=N;k++)	/* loop through petals */
               		 {
               			r3 = pR_ptr[pK_ptr[i].flower[k]].rad;
                   		m3 = (r3>0) ? (1-r3)/(1-r*r3) : 1;
                   		fbest += acos(1-twor*m2*m3); /* angle calc */
                   		m2 = m3;
               		 } 
               	 }
		else /* with overlaps, old routine */
		 {
			j2 = pK_ptr[i].flower[0];
		 	((r2 = pR_ptr[j2].rad) > 0) ? r2=sqrt(r2) : 0;
			o2=pK_ptr[i].overlaps[0];
			for (k=1;k<=N;k++)
			 {
				r1=r2;
				o1=o2;
				j1=j2;
				j2=pK_ptr[i].flower[k];
				((r2 = pR_ptr[j2].rad) >0) ? r2=sqrt(r2) : 0;
				o2=pK_ptr[i].overlaps[k];
				o3=pK_ptr[j1].overlaps[nghb(p,j1,j2)];
				fbest += 
				 acos(h_cos_overlap(sr,r1,r2,o3,o2,o1,&flag));
			 }
		 }
		faim = pR_ptr[i].aim;	/* get target sum */
	/* set up for model */
		N = 2*N;
		del = sin(faim/N);
		bet = sin(fbest/N);
		r2 = (bet-sr)/(bet*r-sr);	/* reference radius */
		if (r2>0) 			/* calc new label */
		 {
			t1 = 1 - r2;
			t2 = 2*del;
			t3 = t2/(sqrt(t1*t1+t2*t2*r2)+t1);
			r2 = t3*t3; 
		 }
		else
			r2 = del*del;		/* use lower limit */
		pR_ptr[i].rad = r2;		/* store new label */ 
		pR_ptr[i].curv = fbest;		/* store new anglesum */
		fbest = fbest-faim;
		cut += fbest*fbest;		/* accumulate error */
	 }
/* End Iteration Loop */

	cut = sqrt(cut);	
	key = 1;
	fact = cut/cut0;
	if (key0==1 && fact < 1.0)		/* try to extrapolate */
	 {
		cut *= fact;
		if (fabs(fact-fact0)<ftol)	/* use a super step */
			fact = fact/(1-fact);
		lmax = 1000;
		for (j=0;j<aimnum;j++)		/* find max allowable step*/
		 {
		   i = vert_indices[j];
		   r = pR_ptr[i].rad;
		   rat = r - R0[i];
		   if (rat>0)			    
			lmax = (lmax < (tr=((1-r)/rat))) ? lmax : tr;    
						/* to keep R<1 */
		   else if (rat < 0)
			lmax = (lmax < (tr= (-r/rat))) ? lmax : tr;      
						/* to keep R>0 */
		 }
/* compute new step */
		fact = (fact < 0.5*lmax) ? fact : 0.5*lmax;
/* interpolate new labels */						
		for (j=0;j<aimnum;j++)
		 {
		   i = vert_indices[j];
		   pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
		 }
		key = 0;
	 }
		
	count++;
 } /* end of main while loop */

	for (i=1;i<=p->nodecount;i++)               /* reset labels */
		if (pR_ptr[i].rad>0)
			pR_ptr[i].rad = sqrt(pR_ptr[i].rad);
	return count;
} /* h_riffle_super */

	/* --------- iteration riffle routine (slow, but dependable) -- */

int h_riffle_standard(struct p_data *p,int passes)
/* (Could use some reworking.) */
{
	int i,j,count=0,dummy,vert_passes;
	double accum,verr,err,cut;
	struct K_data *pK_ptr=p->packK_ptr;
	struct R_data *pR_ptr=p->packR_ptr;

	accum=0;
	for (i=1;i<=p->nodecount;i++)
	 {
		if (pK_ptr[i].bdry_flag &&
			  pR_ptr[i].aim>=0  && pR_ptr[i].aim<.001)
			pR_ptr[i].rad=(-.2);
		else if (pR_ptr[i].aim>0)
		 {
			if (pR_ptr[i].rad<=0 && pR_ptr[i].aim>.00001) 
				pR_ptr[i].rad=.01;
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
	 }
	vert_passes=aimnum*passes; /* to account for different way
				      of incrementing 'count' */
	cut=accum*recip;
	while (cut >toler && count<vert_passes)
	 {
		for (j=0;j<aimnum;j++)
		 {
			i=vert_indices[j];
			h_anglesum_overlap(p,i,pR_ptr[i].rad,
				&pR_ptr[i].curv,&dummy);
			verr=pR_ptr[i].curv-pR_ptr[i].aim;
			if (fabs(verr)>cut)
			 {
			   pR_ptr[i].rad=h_radcalc(p,i,
				pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
			   count++;
			 }
		 }
		accum=0;
		for (j=0;j<aimnum;j++)
		 {
			i=vert_indices[j];
			err=pR_ptr[i].curv-pR_ptr[i].aim;
			accum += (err<0) ? (-err) : err;
		 }
		cut=accum*recip;

	 } /* end of while */
	return (int)(count/aimnum);
} /* h_riffle_standard */

int Rewrite(FILE *fp)
/* write specially for ActivePack */
{
  int i,j,k,n,count,flag=0,jj;
  double angle;
  char format_buf[64];
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;
  
  pK_ptr=packdata->packK_ptr;
  pR_ptr=packdata->packR_ptr;
  count=packdata->nodecount;

  fprintf(fp,"NODECOUNT:   %d\n\n",count);
  fprintf(fp,"ALPHA/BETA/GAMMA:   %d  %d  %d\n\n",
	  packdata->alpha,packdata->beta,packdata->gamma);
  if (packdata->hes<(-1.0)*okerr) sprintf(buf,"hyperbolic\n");
  else if (packdata->hes>okerr) sprintf(buf,"spherical\n");
  else sprintf(buf,"euclidean\n");
  fprintf(fp,"GEOMETRY:   %s\n",buf);
  
  fprintf(fp,"FLOWERS: \n");
  for (n=1;n<=count;n++)
    {
      fprintf(fp,"\n%d %d   ",n,pK_ptr[n].num);
      for (i=0;i<=pK_ptr[n].num;i++) 
	fprintf(fp,"%d ",pK_ptr[n].flower[i]);
    }
  fprintf(fp,"\n\n");
  if (packdata->overlap_status)
    {
      fprintf(fp,"INV_DISTANCES:\n");
      for (i=1;i<=count;i++)
	for (j=0;j<(pK_ptr[i].num+pK_ptr[i].bdry_flag);j++)
	  if (i<(k=pK_ptr[i].flower[j])
	      && fabs((angle=pK_ptr[i].overlaps[j])-1.0)>okerr)
	    fprintf(fp,"\n%d %d  %.6e ",
		    i,k,angle);
      fprintf(fp,"\n  (done)\n\n");
    }
  flag=0;
  for (i=1;i<=count;i++)
    {
      if (pK_ptr[i].bdry_flag 
	  && pR_ptr[i].aim>=0.0) flag++;
      else if ( !pK_ptr[i].bdry_flag 
		&& fabs(pR_ptr[i].aim-m2pi)>(10.0)*toler ) flag++;
    }
  if (flag) /* at least one out of toler */
    {
      jj=j=0;
      for (i=1;i<=count && jj==0;i++) 
	{
	  if ( (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0) 
	       || (!pK_ptr[i].bdry_flag 
		   && (pR_ptr[i].aim<(m2pi+okerr)
		       || pR_ptr[i].aim>(m2pi-okerr))) )
	    jj++;
	}
      if (jj>0) /* at least one non-default aim */
	{
	  fprintf(fp,"ANGLE-AIMS:\n");
	  for (i=1;i<=count;i++)
	    if ( (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0) 
		 || (!pK_ptr[i].bdry_flag 
		     && (pR_ptr[i].aim<(m2pi-okerr)
			 || pR_ptr[i].aim>(m2pi+okerr))) )
	      {
		fprintf(fp," %d % .10e  ",i,pR_ptr[i].aim);
		j++;
		if ((j % 3)==0) fprintf(fp,"\n");
	      }
	}
      fprintf(fp,"\n  (done)\n\n");
    }
  fprintf(fp,"RADII: \n");
  sprintf(format_buf,"%% .%de ",num_digits);
  for (i=1;i<=count;i++)
    {
      fprintf(fp,format_buf,radius(packdata,i));
      if ((i % 4)==0) fprintf(fp,"\n");
    }
  fprintf(fp,"\n\n");
  fprintf(fp,"END\n");fflush(fp);
  return 1;
} /* Rewrite */

int report_status(double cycount,int flag)
/* report status of packing comps. flag=0 means 'short' report, 
tailored to repack method, flag=1 means fuller report.*/
/* fixup: need to look for incompatibilities if overlaps specified,
   report of inv_dist errors. */
{
  int i,j,v_over=0,v_under=0,over_count=0,under_count=0;
  double accum_over=0.0,accum_under=0.0,worst_over=0.0,worst_under=0.0;
  double accum_small=0.0,hold,e_time,off_time;
  struct R_data *pR_ptr=packdata->packR_ptr;

  e_time=get_time();

/* error feedback */

  fillcurves(packdata);
  for (j=0;j<aimnum;j++)
    {
      i=vert_indices[j];
      hold=pR_ptr[i].curv-packdata->packR_ptr[i].aim;
/* perron up info */
      if (hold>okerr 
	  && (flag || repack_method<4)) 
	{
	  accum_over += hold;
	  if (hold>worst_over)
	    {
	      worst_over=hold;
	      v_over=i;
	    }
	  over_count++;
	}
/* perron down info */
      else if ((hold *= -1.0)>okerr 
	       && (flag || repack_method<3 || repack_method==4))
	{
	  accum_under += hold;
	  if (hold>worst_under)
	    {
	      worst_under=hold;
	      v_under=i;
	    }
	  under_count++;
	}
/* remaining tiny stuff */
      else if (flag) accum_small += fabs(hold); /* small errors */
    }

  if (!flag) /* data for local report only */
    {
      time_to_hms(e_time-time_in,&elapse_hr,&elapse_min,&elapse_sec); 
      /* set elapse time string */
      printf("Local status:      pack cycles %d,   elapsed time %s\n",
	     (int)cycount,time_buf);
    }
  else /* data for global report */
    {
      time_to_hms(e_time-global_elapsed,&elapse_hr,&elapse_min,&elapse_sec);
      printf("\nGlobal status: %d verts, %d changeable, %d over, %d under\n Packing cycles %d,   elapsed time %s\n",
	     packdata->nodecount, aimnum, over_count, under_count,
	     global_passes, time_buf);
    }
  if (repack_method<4 || flag) /* perron up */
    {
      printf(" Perron up: %d verts with overage; worst=% .6e at v=%d;\n",
	     over_count,worst_over,v_over);
      printf("                                 Accumulated overage  = % .6e\n",
	     accum_over);
    }
  if (repack_method<3 || repack_method==4 || flag) /* perron down */
    {
      printf(" Perron down: %d verts with underage; worst=% .6e at v=%d;\n",
	     under_count,worst_under,v_under);
      printf("                                 Accumulated underage = % .6e\n",
	     accum_under);
    }
  if (flag)
    printf("Total angle sum error remaining: % .6e\n",
	   accum_over+accum_under+accum_small);
  printf("\n");
  /* adjust clocks for time spent in this routine */
  off_time=get_time()-e_time;
  global_elapsed -= off_time;
  time_in -= off_time; 
  return 1;
} /* report_status */

int log_pack_info() 
{
  printf("Loaded packing: name=%s; nodecount=%d; ",
	  packname,packdata->nodecount);
  if (packdata->hes<-okerr) printf("geom=hyperbolic; ");
  else if (packdata->hes<okerr) printf("geom=euclidean; ");
  else printf("geom=spherical; ");
  if (packdata->overlap_status) printf("Overlap packing.\n\n");
  else printf("Tangency packing.\n");
  return 1;
} /* log_pack_info */

int log_param_info()
{
  printf("\nParameter settings:\n  max_passes=%d; report_cycle=%d; num_digits=%d; iterates=%d;\n  okerr=%f; toler=%f\n",
	 max_passes,report_cycle,num_digits,iterates,okerr,toler);
  sprintf(buf,"  repack_method = ");
  switch(repack_method)
    {
    case 2: {strcat(buf,"superstep_method");break;}
    case 3: {strcat(buf,"Perron_up_only");break;}
    case 4: {strcat(buf,"Perron_down_only");break;}
    default: {strcat(buf,"standard_iteration");break;}
    }
  strcat(buf,"\n  rad_comp_method = ");
  switch(rad_comp_method)
    {
    case 2: {strcat(buf,"uniform");break;}
    default: {strcat(buf,"standard");break;}
    }
  strcat(buf,"\n  status_method = ");
  if (status_flags & 1) strcat(buf,"report_cycle");
  else strcat(buf,"no status reports");
  strcat(buf,"\n  data_flags = ");
  if (data_flags & 1) strcat(buf,"ups_downs");
  else strcat(buf,"none");
  strcat(buf,"\n  debug_method = ");
  switch(debug_method)
    {
    default: {strcat(buf,"none");break;}
    }
  strcat(buf,"\n\n");
  printf(buf);
  return 1;
} /* log_param_info */

int read_param_set(char *datastr)
/* parse string setting some parameter; 
look for "param_name param_value", where value is number or
string. return 0 if malformed. */
{
  char *token_ptr;

  token_ptr=datastr;
  if (!grab_next(&token_ptr,param_name)) return 0;
  if (!grab_next(&token_ptr,param_option)) return 0; 
  return 1;
} /* read_param_set */

int java_msg(char *datastr)
/* send message to java control program */
{
  int n;

  if (!to_java || (n=strlen(datastr))<1) return 0;
  if (datastr[n-1]!='\n') 
    fprintf(to_java,"%s\n",datastr);
  else fprintf(to_java,"%s",datastr);
  return 1;
} /* java_msg */

int check_interrupt()
/* check interrupt channel for interrupt signal */
{
  int retval;
  char inmsg[128];
  struct timeval tv;
  fd_set rfds;

      FD_ZERO(&rfds);
      FD_SET(0, &rfds);
      tv.tv_sec = 0;
      tv.tv_usec = 0;
      retval = select(1, &rfds, NULL, NULL, &tv); /* check stdin */
      if (retval && scanf("127%s",inmsg) && !strcmp(inmsg,"interrupt"))
	return 1;
      return 0;
} /* check_interrupt */

int open_ups_downs()
{
  int n=7;

  if (packdata->status) n=packdata->nodecount;
  if (count_ups) free(count_ups);
  count_ups=(int *)
    calloc((size_t)(n+1),sizeof(int));
  if (count_downs) free(count_downs);
  count_downs=(int *)
    calloc((size_t)(n+1),sizeof(int));
  return 1;
} /* open_ups_downs */

int e_perron(int passes)
{
  int i,j,vert_passes,count=0,stop_flag=0,dummy;
  int status_passes,ca,report_count=0;
  double vert_err,temprad;
  struct R_data *pR_ptr=packdata->packR_ptr;

  time_in=get_time();
  if (global_elapsed==0.0) global_elapsed=time_in;
/* Set up counter limits
   fixup: need to settle on what 'count' counts */
  vert_passes=aimnum*passes;
  status_passes=aimnum*report_cycle;

/* ++++++++++ set threshholds */

  /* toler is the threshhold here */

/* ++++++++++ main loop */

  while (!stop_flag)
    {
      for (j=0;j<aimnum;j++)
	{
	  i=vert_indices[j];
	  e_anglesum_overlap(packdata,i,pR_ptr[i].rad,
	    &pR_ptr[i].curv,&dummy);
	  vert_err=pR_ptr[i].curv-pR_ptr[i].aim;

/* ++++++++++ choose vert */

	  if ((repack_method==3 && vert_err>toler)
	      || (repack_method==4 && vert_err<-toler))
	    {
/* ++++++++++ compute new radius */

	      temprad=erad_calc(packdata,i,pR_ptr[i].rad,pR_ptr[i].aim,&dummy);

/* ++++++++++ update individual vert status info */

	      if (data_flags & 1)
		{
		  if (temprad>pR_ptr[i].rad) count_ups[i]++;
		  else if (temprad<pR_ptr[i].rad) count_downs[i]++;
		}
	      pR_ptr[i].rad=temprad;

	      stop_flag++;
	    }
	} /* end of loop through verts */

/* ++++++++++ update angle sums, accum error after a few adjustments */

      report_count += stop_flag;

/* ++++++++++ report abbreviated status info */
      if (report_count>vert_passes || report_count>status_passes) 
	{
	  count += report_count;
	  if (status_flags & 1) report_status((double)count/aimnum,0);
	  report_count=0;
	}

/* ++++++++++ stop criteria */

      if (count>vert_passes) stop_flag=3;
      else if (!stop_flag) stop_flag=1;
      else stop_flag=0; /* continue another loop */

/* ++++++++++ check for user interrupt */

      if (check_interrupt()) stop_flag=4;

    } /* end of main while */

/* ++++++++++ return */
  /* fixup: report reason for return */

  ca=(int)(count/aimnum);
  if (stop_flag==1) printf("Packing stopped because no circles adjusted.\n");
  else if (stop_flag==3) printf("Packing stopped by limit %d on passes\n",ca);
  else if (stop_flag==4) printf("Packing interrupted by the user\n");

  global_passes += ca;
  report_status(ca,1); /* final report */
  return ca; 
} /* e_perron */

/* ================================================================
=
                    granulation stuff 
	       (many pieces derived from cp files)
=
=
=================================================================== */

/* Idea is to take given complex and break it into simply connected
sub-complexes, approx size determined by granularity n. Should be 
easier to pack and communicate data on these sub-complexes.
We build a linked list of pack_light structures constructed as we 
go along. It's all rather complicated. 


Main Routines:
   granulator
     -- master control routine, creates/cleans up temp data
   find_far_seed
     -- find a seed for a new grain which is far removed from 
        grains already produced.
   build_grain_redlist
     -- use build_redchain and simplify_grain_redchain, as modified
        for special grain properties.
   label_seed_generations
     -- label generations measured from vec_seed; returns vector
        with info
   build_redchain
     -- Build redchain around seed avoiding poison verts (labeled in 
        util_flag)
   simplify_grain_redchain
     -- cut out inclusions formed by red chain
   extract_light_pack
     -- use redlist to build grain data structures.
   extract_residual
     -- gather left over vert.

We need various temporary information. Status of vertices is kept 
in 'green_list': vert v in a grain has green_list[v]=-nmbr, where 
'nmbr' is the number of that grain. Note, in calling 
'find_far_seed', green_list values changed to positive for 
earliest grain number which is not yet surrounded to identify 
'base_verts' used in search for next seed.

(Careful, util_flag needs to be set before certain
subroutines; in most cases it's used for local info.)
*/

int granulator(struct p_data *p,int n)
/* granulate p with grain size n. There remains a problem: 
A grain might entirely engulf one or more previous grains. I 
think previous grains will form connected union, hence should be 
able to go around preliminary bdry of new grain to see that it's 
engulfing former ones. However, how do we fix it? prevent it, 
as it cleans up repeats on its own bdry, from completing the engulfing? 
Return count of grains, set grains pointer. */
{
  int i,j,g,*green_list=NULL,npl=0,nb=0,na=0,max,size,okay,grain_num=1;
  int new_seed,verts_left,cur_target=1;
  struct RedList *redlist=NULL;
  struct p_light *pltrace;
  struct Vertlist *trace,*tether=NULL,*strace;
  struct p_light *tmptrace;

  int debg=0;

  if (!p->status || n<3) return 0;
  grain_free();
  max=2*((int)(p->nodecount/((double)(n*n)))+1); /* max number of
						    grains */
  /* set up 'grain' to keep track of grains generated, 
     green_list to keep track of vertices handled, and
     residual_list for those destined to residual grain. */

  grains=(struct Grains *)
    calloc((size_t)(max+1),sizeof(struct Grains));
  grain_count=0;
  green_list=(int *)calloc((size_t)(p->nodecount+1),sizeof(int));
  if (residual_list) vert_free(&residual_list);
  residual_count=0;
  catalog_faces(p);

  /* do central granule starting at alpha */

  if (!(redlist=build_grain_redlist(p,p->alpha,green_list,tether,n))
      || !(pltrace=extract_light_pack(p,p->alpha,
         &redlist,&green_list,&okay,n,grain_count+1)))
    {
      printf("Failed to form central grain.\n");
      goto KILLIT;
    }
  grain_count++;
  grains[grain_count].pl=pltrace;
  grains[grain_count].seed=p->alpha;

  /* create 'tether', list of verts from bdry to first grain */
  tether=create_tether(p,green_list);

  npl++;
  na=pltrace->counts[0];
  nb=pltrace->counts[0]-pltrace->counts[2];
  cur_target=grain_count; /* cur_target points to earliest grain not yet
			 surrounded -- hence we try to find seed for
			 new grain abutting it. */

  /* loop, adding granules until every vert up to small residual
     set is accounted for. 
     Note: when looking for vert to seed next grain, we try to
     find verts far from the earliest grain in linked list still 
     not totally surrounded. Pass info in green_list: entry > 0 
     means it's poison, but is in the earliest grain, < 0 means it
     is poison because it's in an already surrounded grain. */

  size=(int)(3.0*n/4.0);
  if (size<3) size=3;
  verts_left=p->nodecount-na;

  while (verts_left>n*n)
    {
      new_seed=0;

/* find seed for next new grain */
      grain_num=cur_target;
      while(grain_num<max && grains[grain_num].pl && !new_seed)
	{
	  tmptrace=grains[grain_num].pl;
	  for (j=1;j<=tmptrace->counts[0];j++)
	    /* need temporary change to positive sign in green_list
	       for verts from cur_target grain; these then serve
	       as starting points for search for seed. */
	    if (green_list[(i=tmptrace->orig_indices[j])]==(-grain_num))
	      green_list[i]=grain_num;
	  new_seed=find_far_seed(p,green_list,tether,size);
	  for (j=1;j<=tmptrace->counts[0];j++) /* revert signs */
	    if (green_list[(i=tmptrace->orig_indices[j])]==grain_num)
	      green_list[i]=-grain_num;
	  if (!new_seed) 
	      tmptrace=grains[++grain_num].pl;
	}
      if (!grains[grain_num].pl) /* must have captured all vertices, success */
	goto FINISHUP;

/* create next grain */
      cur_target=grain_num;
      if (redlist) free_redfaces(&redlist);
      if (!(redlist=build_grain_redlist(p,new_seed,green_list,tether,n)))
	goto KILLIT;
      if (!(pltrace=extract_light_pack(p,new_seed,
	 &redlist,&green_list,&okay,n,grain_count+1)))
	{
	  if (okay) goto FINISHUP;
	  else goto KILLIT;    /* ran into some problem */
	}
      grain_count++;
      grains[grain_count].pl=pltrace;
      grains[grain_count].seed=new_seed;

/* maif (debg) goto KILLIT;

y be able to shorten tether */
      if (tether && green_list[tether->v])
	{
	  vert_free(&tether);
	  tether=NULL;
	}
      else if (tether)
	{
	  trace=tether->next;
	  while (trace && trace->next) 
	    {
	      if (!green_list[trace->v])
		trace=trace->next;
	      else /* rest of tether not needed */
		{
		  vert_free(&(trace->next)); 
		  trace->next=NULL;
		}
	    }
	}

/* update counts */
      na += pltrace->counts[0];
      nb += pltrace->counts[0]-pltrace->counts[2];

/* count verts not yet in some grain */
      verts_left=0;
      for (i=1;i<=p->nodecount;i++) if (!green_list[i]) verts_left++;

    } /* end of while */

 FINISHUP:
  destroy_catalog(p->nodecount);
  if (tether) vert_free(&tether);
  if (redlist) free_redfaces(&redlist);

  /* announce results */
  printf("Granulate: %d sub-complexes, avg area %d, avg circum %d.\n"
	 "   Seeds are: ",grain_count,(int)(na/grain_count),
	 (int)(nb/grain_count));
  for (g=1;g<=grain_count;g++)
    printf("Grain %d: area %d, interiors %d\n",
	   g,(grains[g].pl)->counts[0],(grains[g].pl)->counts[2]);


  /* take care of remaining 'residual' and 'negotiation' vertices. */

if (debg) goto KILLIT;


  if ((pltrace=extract_residual(p,&green_list,&okay)) 
      && okay>0)
    {
      grain_count++;
      grains[grain_count].seed=okay;
      grains[grain_count].pl=pltrace;
      printf("'Residual' grain %d: area %d, interiors %d\n",
	     grain_count,(grains[grain_count].pl)->counts[0],
	     (grains[grain_count].pl)->counts[2]);
    }
  else if (!okay) printf("Some error extracting the 'residual' grain.");
  else printf("No residual grain.");

if (debg) goto KILLIT;


  if ((pltrace=extract_negotiate(p,&okay)) 
      && okay>0)
    {
      grain_count++;
      grains[grain_count].seed=okay;
      grains[grain_count].pl=pltrace;
      printf("'Negotiate' grain %d: area %d, interiors %d\n",
	     grain_count,(grains[grain_count].pl)->counts[0],
	     (grains[grain_count].pl)->counts[2]);
    }
  else printf("Some error extracting the 'negotiate' grain.");

if (debg) goto KILLIT;

  if (green_list) {free(green_list);green_list=NULL;}
  /*  vert_free(&residual_list); */
  strace=residual_list;
  while (strace)
    {
      residual_list=residual_list->next;
      free(strace);
      strace=residual_list;
    }

  return grain_count;

 KILLIT:

if (debg) return 0;

  destroy_catalog(p->nodecount);
  if (tether) vert_free(&tether);
  if (redlist) free_redfaces(&redlist);
  vert_free(&residual_list);
  if (green_list) {free(green_list);green_list=NULL;}
  grain_free();
  printf("granulate failed for some reason.\n");
  return 0;
} /* granulator */


/* Given vertex seed v0 and verts designated as 'poison' want the
smallest simply connected complex containing v0 and avoiding poison
verts. E.g., for n>1 generations, make larger gen verts poison.
Main difficulty: need to identify and fill "holes" formed by 
poison verts -- inclusions. 

  Main approach: verts with util_flag < 0 will be off-limits.
First, make up red chain in some way, eg. by generation. Also
have to find "persistent" entry in red chain, one which is kept
in red chain throughout: we use it as we modify the red chain to
determine which part to keep. Next, walk around red chain; when
outside vert is discovered which is repeated elsewhere on the
outside of the red chain, cut out the intervening chain which
forms an inclusion.*/

struct RedList *build_grain_redlist(struct p_data *p,int seed,
	       int *vec_seed,struct Vertlist *tether,int gen)
/* Given seed and number gen>1, find red chain enclosing smallest 
simply connected sub-complex containing vertices up to and including 
generation gen from seed, but stopping at any 'green' v (ie, 
vec_seed[v]<0). Note vec_seed and tether unchanged. */
{
  int i,j,k,num,o_gen=0,*gen_list,dum;
  int debg=0,out_vert=1,in_vert=0,tick=0,wflag=0;
  struct Vertlist *dum_fdo=NULL;
  struct RedList *redlist=NULL,*rtrace,*keep_ptr=NULL;
  struct K_data *pK_ptr;

  if (!(gen_list=
	label_seed_generations(p,seed,vec_seed,gen-1,&out_vert,&dum))) 
    return NULL; 
  pK_ptr=p->packK_ptr;

/* set generation gen and above verts and green verts to poison */
  for (i=1;i<=p->nodecount;i++) 
    {
      if (gen_list[i]<=0 || gen_list[i]>=gen) 
	pK_ptr[i].util_flag=-1;
      else pK_ptr[i].util_flag=0;
    }
/* remove any isolated poison verts */
  for (i=1;i<=p->nodecount;i++)
    if (pK_ptr[i].util_flag==-1)
      {
	k=0;
	for (j=0;j<=pK_ptr[i].num;j++) 
	  k+= pK_ptr[pK_ptr[i].flower[j]].util_flag;
	if (!k)
	  {
	    pK_ptr[i].util_flag=0;
	    tick++;
	  }
      }

/* build the redchain and set its data */
  dum_fdo=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  if (p->rwb_flags) free(p->rwb_flags);
  p->rwb_flags=(int *)calloc((size_t)(p->facecount+1),sizeof(int));
  if (!(redlist=build_redchain(p,seed,dum_fdo)))
    {
      fprintf(stderr,"Failed to build redchain.");
      free(p->rwb_flags);
      p->rwb_flags=NULL;
      goto BOMB;
    }

if (debg) record_redlist(redlist,p->facecount);

/* define a persistent redlist entry (so we don't discard the
   wrong portion of redlist) */

/* if out_vert isn't green, it's a bdry vert and we need to work 
   our way back in generation towards gen. */
  if (gen_list[out_vert]>0)
    while ((o_gen=gen_list[out_vert])>gen)
      {
	num=pK_ptr[out_vert].num;
	for (j=0;j<=num;j++)
	  if (gen_list[pK_ptr[out_vert].flower[j]]>0 
	      && gen_list[pK_ptr[out_vert].flower[j]]<o_gen)
	    {
	      out_vert=pK_ptr[out_vert].flower[j];
	      j=num+1;
	    }
      }
  /* out_vert should now be in edge of desired red chain; get
     in_vert, one earlier generation nghb. */
      
  num=pK_ptr[out_vert].num;
  tick=0;
  for (j=0;j<=num;j++)
    if ((gen_list[out_vert]>0 && gen_list[(i=pK_ptr[out_vert].flower[j])]>0
	 && gen_list[i]<o_gen)
	|| (gen_list[out_vert]<=0 && gen_list[pK_ptr[out_vert].flower[j]]>0))
      {
	in_vert=pK_ptr[out_vert].flower[j];
	j=num+1;
	tick++;
      }
  if (!tick) 
    {
      fprintf(stderr,"Didn't find in_vert.");
      goto BOMB;
    }

  /* identify a redlist entry whose face contains in/out_vert's */

  rtrace=redlist;
  while (rtrace!=redlist || !(wflag++))
    {
      if (check_face(p,rtrace->face,in_vert,out_vert)>=0
	  || check_face(p,rtrace->face,out_vert,in_vert)>=0)
	{
	  keep_ptr=rtrace;
	  rtrace=redlist;
	}
      else rtrace=rtrace->next;
    }
  if (!keep_ptr) 
    {
      fprintf(stderr,"Didn't find persistent face.");
      goto BOMB;
    } 

  /* set redlist to start at persistent entry, then process red 
     chain to remove inclusions */

  redlist=keep_ptr;
  if (!simplify_grain_redchain(p,&redlist,tether)) 
    {
      fprintf(stderr,"Failed to simplify properly.");
      goto BOMB;
    }
  return redlist;

 BOMB:
  if (gen_list) free(gen_list);
  if (redlist) free_redfaces(&redlist);
  if (dum_fdo) vert_free(&dum_fdo);
  return NULL;
} /* build_grain_redlist */

int simplify_grain_redchain(struct p_data *p,struct RedList **redlist,
			    struct Vertlist *tether)
/* Given a red chain, cut 'inclusions' to form new red chain. 
E.g., if p is simply connected, can cookie this to get smallest 
simply connected sub-complex containing the original red chain. 
Note that the first entry of redlist is considered 'persistent'; 
that is, it must remain in the red chain, hence it helps define 
which parts of chain are kept and which removed as inclusions 
during processing. Tether is chain of verts which shouldn't be 
crossed: it prevents previous grains from being treated as an 
inclusion to be filled in. Return 0 on error. */
{
  int i,n,nn,*face_ptr,findex,bindex,eindex,indx,nindex,num;
  int trip_flag,stop_flag=0,cface,vert,switch_flag,hits,w,u,v,f;
  int debg=0,last_vert,wflag=0,prev_vert=0,next_vert,dont_cut;
  struct RedList *rtrace,*upstream,*stop_ptr,*backtrace,*foretrace;
  struct RedList *check_ptr,*keep_ptr,*newface;
  struct RedList *cut_begin,*cut_end,*new_red,*hold;
  struct Vertlist *ttrace;
  struct K_data *pK_ptr;

  if (!p->status || (*redlist)==NULL) return 0;
  keep_ptr=*redlist; /* first entry of incoming redlist must be
			the chosen 'persistent' entry; this is the
			responsibility of the calling routine. */
  pK_ptr=p->packK_ptr;

if (debg) record_redlist(*redlist,p->facecount);

/* Modify red chain to fill 'inclusions'. Method: 
   walk around outer edge verts, and when one is repeated, 
   discard the intervening red chain (insuring that keep_ptr
   is kept). Repeat as needed. Resulting red chain should define 
   simply connected sub-complex (assuming p is simply connected).
*/
   
  while (!stop_flag) /* main loop: go until pruning done */
    {
      stop_flag=1;
      trip_flag=0;
      stop_ptr=*redlist;
      last_vert=0;
      while (!trip_flag) /* go through whole red chain */
	{
	  upstream=*redlist;
	  cface=(*redlist)->face;
	  indx=nghb_tri(p,(*redlist)->next->face,cface);
	  vert=p->faces[cface].vert[indx];
	  /* if blue, take care of isolated vert first */
	  if ((*redlist)->prev->face==(*redlist)->next->face
	      && p->faces[cface].vert[(indx+2) % 3]!=last_vert)
	    last_vert=vert=p->faces[cface].vert[(indx+2) % 3];
	  else last_vert=0;
	  face_ptr=face_org[vert]+1;
	  num=pK_ptr[vert].num;
/* find index of cface */
	  findex=0;
	  while (*(face_ptr+findex)!=cface && findex<(num-1)) findex++;
/* find index of downstream face sharing vert; note, redlist gets 
   shifted at least one position downstream. */
	  i=findex;
	  while ( (*redlist)->next->face==*(face_ptr+((i+num-1) % num))
		  && (*redlist)->next->face!=cface ) 
	    {
	      *redlist=(*redlist)->next;
	      if ((*redlist)==stop_ptr) 
		trip_flag++;
	      i--;
	    }
	  if ((*redlist)->next->face==cface 
	      && (*redlist)->prev->face!=cface) /* red wraps single vert */
	    return 0;
	  eindex= (num+i) % num;

/* find index of upstream face */
	  i=findex;
	  while ( upstream->prev->face==*(face_ptr+((i+1) % num))
		  && upstream->prev->face!=cface )
	    {
	      upstream=upstream->prev;
	      i++;
	    }
	  if (upstream->prev->face==cface 
	      && upstream->next->face!=cface) 
	    return 0; 
	  bindex=(i % num);
/* go around outer verts to see if we come across vert again */
	  rtrace=*redlist;
	  while (rtrace!=upstream
		 && p->faces[rtrace->face].vert[nghb_tri(p,
		    rtrace->next->face,rtrace->face)]!=vert)
	    rtrace=rtrace->next;
	  if (rtrace!=upstream) /* got a hit */
	    {
	      /* save beginning/end of red chain segment to kill */
	      cut_begin=(*redlist)->next;
	      cut_end=rtrace->prev;
	      /* get index of this face */
	      nindex=0;
	      while (*(face_ptr+nindex)!=rtrace->face && nindex<(num-1))
		nindex++;
	      /* decide which portion of chain to toss */
	      if (pK_ptr[vert].bdry_flag)
		{
		  if (eindex>nindex) switch_flag=1;
		  else switch_flag=0;
		}
	      else
		{
		  switch_flag=0;
		  check_ptr=cut_begin;
		  while (check_ptr!=rtrace)
		    {
		      if (check_ptr==keep_ptr)
			{
			  switch_flag=1;
			  check_ptr=rtrace;
			}
		      else check_ptr=check_ptr->next;
		    }
		}

	      if (switch_flag) /* switch portion to be tossed */
		{
		  eindex=nindex;
		  while (rtrace->next->face
			 ==*(face_ptr+((eindex+num-1) % num)))
		    {
		      rtrace=rtrace->next;
		      eindex--;
		    }
		  cut_begin=rtrace->next;
		  cut_end=upstream->prev;
		  *redlist=rtrace;
		  nindex=bindex;
		}
	      n=(eindex+num-nindex-1) % num;
	      indx=(eindex+num-1) % num;

	      /* don't cut if new segment would cross tether */
	      ttrace=tether;
	      if (ttrace) prev_vert=ttrace->v;
	      hits=next_vert=0;
	      while (ttrace && !hits)
		{
		  if (ttrace->v==vert) 
		    {
		      hits=1;
		      if (ttrace->next) next_vert=ttrace->next->v;
		    }
		  else
		    {
		      prev_vert=ttrace->v;
		      ttrace=ttrace->next;
		    }
		}
	      dont_cut=0;
	      if (hits) /* would we cross tether? */
		for (i=nindex;i<=n;i++)
		  if ((nn=pK_ptr[vert].flower[i % num])==prev_vert
		      || nn==next_vert)
		    {
		      dont_cut=1;
		      i=n;
		    }
	      if (!dont_cut)
		{     
		  /* add new section of red chain */
		  while (n>0)
		    {
		      (*redlist)->next=new_red=(struct RedList *)
			calloc((size_t)1,sizeof(struct RedList));
		      new_red->prev=*redlist;
		      new_red->face=*(face_ptr+indx);
		      *redlist=new_red;
		      indx=(indx+num-1) % num;
		      n--;
		    }
		  rtrace->prev=*redlist;
		  stop_ptr=*redlist=(*redlist)->next=rtrace;
		  stop_flag=0; 
		  /* destroy cutoff segment */
		  cut_end->next=NULL;
		  rtrace=cut_begin;
		  while (rtrace!=NULL)
		    {
		      hold=rtrace->next;
		      free(rtrace);
		      rtrace=hold;
		    }
		  last_vert=0;
		}
	    } /* took care of hit */
	  if (!last_vert) *redlist=(*redlist)->next; /* last_vert true would 
						    mean face was blue,
						    nothing happened, but
						    now need to handle 
						    its second vert */
	  if ((*redlist)==stop_ptr) trip_flag++;
	} /* end of while (go to new 'vert') */
    } /* end of main loop */

  /* One final processing step for red chain: catch instances
     at vert v (int in parent) with red chain wrapping clockwise
     around v and getting all but one face in star of v. We will
     make v interior to sub-complex by shorting red chain across
     omitted face. One adjustment can lead to others,
     so keep cycling through list until no more adjustments.
     (Note: this adjustment can't cause complex to cross tether.) */

  hits=1;
  while (hits)
    {
      hits=0;
      rtrace=*redlist;
      wflag=0;
      while (rtrace!=(*redlist) || !(wflag++))
	{
	  indx=nghb_tri(p,rtrace->next->face,rtrace->face);
	  v=p->faces[rtrace->face].vert[indx];
	  if (!pK_ptr[v].bdry_flag)
	    {
	      backtrace=rtrace;
	      while (p->faces[backtrace->prev->face].vert[
		nghb_tri(p,backtrace->face,backtrace->prev->face)]==v)
		backtrace=backtrace->prev;
	      foretrace=rtrace->next;
	      while (p->faces[foretrace->face].vert[
		nghb_tri(p,foretrace->next->face,foretrace->face)]==v)
		foretrace=foretrace->next;
	      w=p->faces[backtrace->face].vert[
		(nghb_tri(p,backtrace->next->face,backtrace->face)+2) % 3];
	      u=p->faces[foretrace->face].vert[
		(nghb_tri(p,foretrace->prev->face,foretrace->face)+2) % 3];
	      if ((f=what_face(p,u,v,w))) /* <u,v,w> form face? do shortcut */
		{
		  newface=(struct RedList *)
		    calloc((size_t)1,sizeof(struct RedList));
		  newface->face=f;
		  newface->prev=backtrace;
		  newface->next=backtrace=backtrace->next;
		  /* cut out part of redchain going clockwise about v */
		  while (newface->next!=foretrace)
		    {
		      newface->next=backtrace->next;
		      if (backtrace==(*redlist)) *redlist=newface;
		      free(backtrace);
		      backtrace=newface->next;
		    }
		  newface->prev->next=newface;
		  newface->next->prev=newface;
		  rtrace=newface->prev;
		  wflag=0;
		  hits=1;
		}
	    }
	  rtrace=rtrace->next;
	} /* end of inner while */
    } /* end of outer while */

  return (1); /* success */
} /* simplify_grain_redchain */


struct p_light *extract_residual(struct p_data *p,int **green_list,int *okay)
/* Form p_light structure with remaining vertices, ie. green_list[v]==0 
or in rsidual_list -- even if they don't form a legit complex,  
okay=1 means no error, even if we return a NULL. */
{
  int i,j,k,m,v,tick=1,w,seed=0,vcount,icount=0;
  struct Vertlist *strace=NULL;
  struct p_light *pl;
  struct K_data *pK_ptr;

  int debg=0;
  
  pK_ptr=p->packK_ptr;
  if (!(*green_list) && !residual_list)
    {*okay=-1;return NULL;}

  /* indicate residual verts which are interior to parent packing */
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;
  strace=residual_list;
  while (strace) /* from accumulated residual list */
    {
      v=strace->v;
      if (!pK_ptr[v].bdry_flag) pK_ptr[v].util_flag=1;
      icount++;
      strace=strace->next;
    }
  for (i=p->nodecount;i>0;i--) /* other verts not picked up */
    if (!((*green_list)[i]) && !pK_ptr[i].bdry_flag) 
      {
	pK_ptr[i].util_flag=1;
	icount++;
	seed=i;
      }
   if (!seed) /* no interior verts */
    {
      *okay=-1;
      return NULL;
    }

   /* bdry are just nghb's of the relative interiors; ie,
      neglect isolated bdry. */
   vcount=icount;

   for (i=0;i<=p->nodecount;i++) 
     if (pK_ptr[i].util_flag>0)
       for (j=0;j<=pK_ptr[i].num;j++)
	 if (!pK_ptr[(k=pK_ptr[i].flower[j])].util_flag)
	   {
	     vcount++;
	     pK_ptr[k].util_flag=-vcount;
	   }

  *okay=0;
  pl=(struct p_light *)calloc((size_t)1,sizeof(struct p_light));
  pl->counts=(int *)calloc((size_t)5,sizeof(int));
  pl->counts[0]=vcount;
  pl->counts[1]=(int)(p->hes);
  pl->counts[2]=icount;
  pl->orig_indices=(int *)calloc((size_t)(pl->counts[0]+1),sizeof(int));
  pl->radii=(double *)
    calloc((size_t)(pl->counts[0]+1),sizeof(double));

if (debg) return NULL;

  /* set util_flag to new index and find size of var_indices block */

  for (i=1;i<=p->nodecount;i++)
    if (pK_ptr[i].util_flag>0) /* interior */
      {
	icount++;
	pK_ptr[i].util_flag=icount;
	pl->counts[3] += pK_ptr[i].num+1;
      }

if (debg) return NULL;

  /* store interior block info, orig_indices, and radii */
  pl->var_indices=(int *)calloc((size_t)(pl->counts[3]+1),sizeof(int));
  for (i=1;i<=p->nodecount;i++)
    {
      k=pK_ptr[i].util_flag;
      if (k>0) /* interior */
	{
	  pl->var_indices[tick++]=pK_ptr[i].num;
	  /* create flower list. Note: every flower should be
	     interior, so don't record the redundant last petal index. */
	  for (j=0;j<pK_ptr[i].num;j++)
	    {
	      w=pK_ptr[i].flower[j];
	      if (!(m=pK_ptr[w].util_flag)) 
		goto SCREWUP2; /* every petal should be an interior or 
				 bdry, hence >0 or <0 util_flag, resp. */
	      m = (m<0) ? -m : m;
	      pl->var_indices[tick++] = m;
	    }
	  pl->radii[k]=p->packR_ptr[i].rad;
	  pl->orig_indices[k]=i;
	}
      else if (k<0) /* bdry */
	{
	  pl->radii[-k]=p->packR_ptr[i].rad;
	  pl->orig_indices[-k]=i;
	}
    }

if (debg) return NULL;

  /* success */
   for (i=p->nodecount;i>0;i--) 
     if (pK_ptr[i].util_flag) (*green_list)[i]=-(grain_count+1);
  pl->counts[4]=p->nodecount;
  *okay=seed;
  return pl; 

 SCREWUP2:
  free_p_light(&pl);
  *okay=0;
  return NULL;
} /* extract_residual */

struct p_light *extract_negotiate(struct p_data *p,int *okay)
     /* Grain bdry's intersect and we need a p_light structure 
to pack these. Thus we identify all interior verts which lie in
the bdry of some grain and create p_light. (Experience may show
we should include this in the residual pack.) okay=1 means
no packing, but no error. */
{
  int i,j,k=0,m,tick=1,w,seed=0,vcount,icount=0;
  struct p_light *pl;
  struct K_data *pK_ptr;
  
  pK_ptr=p->packK_ptr;
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;

  for (i=1;i<=grain_count;i++)
    {
      pl=grains[i].pl;
      for (j=pl->counts[2]+1;j<=pl->counts[0];j++)
	if (!pK_ptr[(k=pl->orig_indices[j])].bdry_flag
	    && !pK_ptr[k].util_flag)
	  pK_ptr[k].util_flag=++icount; /* interior of negotiate packing */
    }
  seed=k; /* treat last indexed interior as 'seed' */
  if (!icount) /* must be something wrong unless there is only
		  one grain = whole original packing. */
    {
      if (grain_count==1) *okay=1; 
      else *okay=0;
      return NULL;
    }

   /* bdry are just nghb's of the relative interiors; ie,
      neglect isolated bdry. */
   vcount=icount;

   for (i=p->nodecount;i>0;i--) 
     if (pK_ptr[i].util_flag>0)
       for (j=0;j<=pK_ptr[i].num;j++)
	 if (!pK_ptr[(k=pK_ptr[i].flower[j])].util_flag)
	   {
	     vcount++;
	     pK_ptr[k].util_flag=-vcount;
	   }

  *okay=0;
  pl=(struct p_light *)calloc((size_t)1,sizeof(struct p_light));
  pl->counts=(int *)calloc((size_t)5,sizeof(int));
  pl->counts[0]=vcount;
  pl->counts[1]=(int)p->hes;
  pl->counts[2]=icount;
  pl->orig_indices=(int *)calloc((size_t)(pl->counts[0]+1),sizeof(int));
  pl->radii=(double *)
    calloc((size_t)(pl->counts[0]+1),sizeof(double));

  /* set util_flag to new index and find size of var_indices block */

  icount=0;
  for (i=1;i<=p->nodecount;i++)
    if (pK_ptr[i].util_flag>0) /* interior */
      {
	icount++;
	pl->counts[3] += pK_ptr[i].num+1;
	pK_ptr[i].util_flag=icount;
      }

  /* store interior block info, orig_indices, and radii */
  pl->var_indices=(int *)calloc((size_t)(pl->counts[3]+1),sizeof(int));
  for (i=1;i<=p->nodecount;i++)
    {
      k=pK_ptr[i].util_flag;
      if (k>0) /* interior */
	{
	  pl->var_indices[tick++]=pK_ptr[i].num;
	  /* create flower list. Note: every flower should be
	     interior, so don't record the redundant last petal index. */
	  for (j=0;j<pK_ptr[i].num;j++)
	    {
	      w=pK_ptr[i].flower[j];
	      if (!(m=pK_ptr[w].util_flag)) 
		goto SCREWUP3; /* every petal should be an interior or 
				 bdry, hence >0 or <0 util_flag, resp. */
	      m = (m<0) ? -m : m;
	      pl->var_indices[tick++] = m;
	    }
	  pl->radii[k]=p->packR_ptr[i].rad;
	  pl->orig_indices[k]=i;
	}
      else if (k<0) /* bdry */
	{
	  pl->radii[-k]=p->packR_ptr[i].rad;
	  pl->orig_indices[-k]=i;
	}
    }
  /* success */
  pl->counts[4]=p->nodecount;
  *okay=seed;
  return pl; 

 SCREWUP3:
  free_p_light(&pl);
  *okay=0;
  return NULL;
} /* extract_negotiate */

int find_far_seed(struct p_data *p,int *base_verts,
		  struct Vertlist *tether,int max) 
/* Return index of vert v as far (up to max) generations from 
'base_verts' v (ie, base_verts[v]>0) as possible, staying away 
from 'poison' w (base_verts[w]<0) as far as possible, and trying
(a little) to stay away from bdry verts. 

Status is kept in util_flag: 0=still free, >0 marks generation
from base_verts, -1=poison or 'tether' vert, -2 fender layers to
keep away from poison; current best choice is 'return_vert'. 

Idea is to use dynamic linked lists: targens is list of next 
generation, poigens is list of latest generation fending off 
from the poison verts. Each pass through, (a) label free verts 
next to current targens verts and put them in new targens list.
(b) pass through poigens and label as -2 any free vert ngbh's.
In step (a) also relabel any -2 verts with their generation,
but only add to targens if 'strategy' flag is set, meaning we
ran out of free verts and are forced to eat into the layers 
built up around the poison verts to try to reach max.

Return 0 if no nonpoison verts are found in first generation 
next to base_verts or if non is found in generation <= max. */
{
  int i,j,v,gen_count=2,hits,return_vert=0;
  int strategy=0;
  struct K_data *pK_ptr;
  struct Vertlist *targens=NULL,*poigens=NULL,*ttrace,*ptrace;
  struct Vertlist *hold_targens=NULL,*vertlist=NULL,*vtrace;

  if (!p->status || !base_verts) return 0;
  pK_ptr=p->packK_ptr;
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;
  if (max<=0) max=p->nodecount; /* no bound on generations */

  /* set up target/poison lists */
  targens=ttrace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  poigens=ptrace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  hits=0;
  for (i=1;i<=p->nodecount;i++)
    {
      if (base_verts[i]>0)
	{
	  ttrace=ttrace->next=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
	  ttrace->v=i;
	  pK_ptr[i].util_flag=1;
	  hits++;
	}
      else if (base_verts[i]<0 || pK_ptr[i].bdry_flag)
	{
	  ptrace=ptrace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
	  ptrace->v=i;
	  if (base_verts[i]<0) pK_ptr[i].util_flag=-1;
	  else pK_ptr[i].util_flag=-2;
	}
    }
  /* 'tether' is a linked list of verts from first grain out to
     some bdry vert; make these poison to prevent subsequent grains
     from wrapping around existing ones. */
  vtrace=tether;
  while (vtrace)
    {
      if (pK_ptr[i=vtrace->v].util_flag==1)
	{
	  pK_ptr[i].util_flag=-1;
	  hits--;
	}
      else pK_ptr[i].util_flag=-1; 
      vtrace=vtrace->next;
    }
  if (!hits) /* base_verts verts have no free neighbors */ 
    {
      return_vert=0;
      goto OUTAHERE;
    }
  ttrace=targens;
  targens=targens->next; /* first location wasn't used */
  free(ttrace);
  ptrace=poigens;
  poigens=poigens->next; /* first location wasn't used */
  free(ptrace);

  /* main loop */
  hits=0;
  while (!(hits++) && gen_count<=max 
	 && (targens || (strategy && hold_targens)))
    {
      /* use or throw out previous target list */
      if (!targens) targens=hold_targens; 
      else vert_free(&hold_targens);
      vertlist=hold_targens=targens; /* process old target list */
      targens=ttrace=(struct Vertlist *)  /* create next target list */
	calloc((size_t)1,sizeof(struct Vertlist));
      /* label any unlabeled nghb's of base_verts, or if strategy==1,
	 any nghb's labeled <= -2 (note we don't relabel poison) */
      do
	{
	  v=vertlist->v;
	  for (i=0;i<=pK_ptr[v].num;i++)
	    {
	      if (!pK_ptr[(j=pK_ptr[v].flower[i])].util_flag)
		{
		  strategy=0;
		  return_vert=j;
		  pK_ptr[j].util_flag=gen_count;
		  ttrace=ttrace->next=(struct Vertlist *)
		    calloc((size_t)1,sizeof(struct Vertlist));
		  ttrace->v=j;
		  hits=0;
		}
	      else if (pK_ptr[j].util_flag <= -2)
		{
		  if (strategy) /* no free verts, so add this to list */
		    {
		      return_vert=j;
		      pK_ptr[j].util_flag=gen_count;
		      ttrace=ttrace->next=(struct Vertlist *)
			calloc((size_t)1,sizeof(struct Vertlist));
		      ttrace->v=j;
		      hits=0;
		    }	      
		  else /* don't add to list, but label generation */
		    pK_ptr[j].util_flag=gen_count;
		}
	    }
	  vertlist=vertlist->next;
	} while (vertlist);
      ttrace=targens;
      targens=targens->next; /* first position was empty */
      free(ttrace);

      /* what's status? */
      if (targens) /* found verts in new generation, so throw
		      old list out, continue as normal */
	vert_free(&hold_targens);
      else if (gen_count==2) /* first pass and didn't find anything */
	{
	  return_vert=0;
	  goto OUTAHERE;
	}
      else  /* ran out of candidates; shift strategy to allow moving 
	       closer to poison */
	{
	  strategy=1;
	  gen_count--;
	}

      /* now go through poison, mark nghb's with -2 */

      if (poigens)
	{
	  vertlist=poigens; /* process old poison list */
	  poigens=ptrace=   /* create next poison list */
	    (struct Vertlist *)calloc((size_t)1,sizeof(struct Vertlist));
	  do
	    {
	      v=vertlist->v;
	      for (i=0;i<=pK_ptr[v].num;i++)
		if (!pK_ptr[(j=pK_ptr[v].flower[i])].util_flag)
		  {
		    pK_ptr[j].util_flag = -2;
		    ptrace=ptrace->next=(struct Vertlist *)
		      calloc((size_t)1,sizeof(struct Vertlist));
		    ptrace->v=j;
		  }
	      vertlist=vertlist->next;
	    } while (vertlist);
	  ptrace=poigens;
	  poigens=poigens->next; /* first position was empty */
	  free(ptrace);
	}

      gen_count++;
    } /* end of outer while */

    OUTAHERE:
      vert_free(&targens);
      vert_free(&poigens);
      return return_vert;
} /* find_far_seed */

struct Vertlist *create_tether(struct p_data *p,int *green_list)
/* list of verts running from bdry to verts set in green_list. */
{
  int hits=0,gen_count=1,v,vv=0,i,j;
  struct Vertlist *genlist=NULL,*final_list=NULL,*vertlist=NULL;
  struct Vertlist *trace,*gtrace;
  struct K_data *pK_ptr;

  if (!p->status || !p->num_bdry_comp || !green_list) 
    return final_list;
  pK_ptr=p->packK_ptr;
  genlist=trace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  for (i=1;i<=p->nodecount;i++) 
    {
      pK_ptr[i].util_flag=0;
      if (green_list[i])
	{
	  if (pK_ptr[i].bdry_flag)  /* green_list has a bdry vert */
	    {
	      vert_free(&genlist);
	      return final_list; 
	    }
	  pK_ptr[i].util_flag=gen_count;
	  trace=trace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
	  trace->v=i;
	}
    }
  trace=genlist;
  genlist=genlist->next; /* first spot empty */
  free(trace);

  /* mark generations out to first bdry vert */
  while (!(hits++))
    {
      gen_count++;
      vertlist=genlist; /* use up old list */
      genlist=gtrace=(struct Vertlist *) /* create new list */
	calloc((size_t)1,sizeof(struct Vertlist));
      do
	{
	  v=vertlist->v;
	  for (i=0;i<=pK_ptr[v].num;i++)
	    if (!pK_ptr[(j=pK_ptr[v].flower[i])].util_flag)
	      {
		pK_ptr[j].util_flag=gen_count;
		if (pK_ptr[j].bdry_flag) goto GOTONE;
		gtrace=gtrace->next=(struct Vertlist *)
		  calloc((size_t)1,sizeof(struct Vertlist));
		gtrace->v=j;
		hits=0;
	      }
	  trace=vertlist;
	  vertlist=vertlist->next;
	  free(trace);
	} while (vertlist);
      gtrace=genlist;
      genlist=genlist->next;
      free(gtrace);
    }
  /* reaching here means some error; didn't find a bdry vert */  
  goto BUMBLE;

  /* create the list */
 GOTONE:
  final_list=trace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  trace->v=j;
  while (gen_count>1)
    {
      v=trace->v;
      for (i=0;i<=pK_ptr[v].num;i++)
	if (pK_ptr[(vv=pK_ptr[v].flower[i])].util_flag < gen_count
	    && pK_ptr[vv].util_flag>0)
	  {
	    i=-1;
	    break;
	  }
      if (i!=-1) goto BUMBLE; /* something wrong */
      trace=trace->next=(struct Vertlist *)
	calloc((size_t)1,sizeof(struct Vertlist));
      trace->v=vv;
      gen_count--;
    }
  if (gen_count<1) goto BUMBLE; /* something wrong */
  trace=final_list;
  final_list=final_list->next; /* first spot empty */
  free(trace);
  vert_free(&genlist);
  return final_list;
  
 BUMBLE:
  vert_free(&genlist);
  vert_free(&vertlist);
  vert_free(&final_list);
  return NULL;
} /* create_tether */
  
int emsg()
{
  printf(emsgbuf);return 1;
}

int msg()
{
  printf(msgbuf);return 1;
}

int repack_activity_msg(char *msg)
{
  printf(msg);return 1;
}

int grain_free()
{
  int i;

  if (!grains) return 1;
  for (i=1;i<=grain_count;i++) 
      free_p_light(&grains[i].pl);
  free(grains);
  grains=NULL;
  grain_count=0;
  return 1;
}





