package Language::INTERCAL::Rcfile;

# Configuration files for sick and intercalc

# This file is part of CLC-INTERCAL

# Copyright (c) 2007 Claudio Calvelli, all rights reserved.

# CLC-INTERCAL is copyrighted software. However, permission to use, modify,
# and distribute it is granted provided that the conditions set out in the
# licence agreement are met. See files README and COPYING in the distribution.

use strict;
use Carp;
use File::Spec::Functions;

use Language::INTERCAL::Exporter '1.-94.-4';
use Language::INTERCAL::GenericIO '1.-94.-3';

use vars qw($PERVERSION);
$PERVERSION = "CLC-INTERCAL INTERCAL/Rcfile.pm 1.-94.-3";

my %rcdefs = (
    'WRITE' => [ \&_rc_array, [] ],
    'UNDERSTAND' => [ \&_rc_understand, [] ],
    'UNDERSTAND ANYWHERE' => [ \&_rc_anywhere, [] ],
    'SPEAK' => [ \&_rc_array, [] ],
    'PRODUCE' => [ \&_rc_scalar, '' ],
);

sub new {
    @_ == 1 or croak "Usage: new Language::INTERCAL::Rcfile";
    my ($class) = @_;
    my %data = ();
    for my $k (keys %rcdefs) {
	$data{$k} = ref $rcdefs{$k}[1] ? [] : '';
    }
    my @include =
	grep { -d $_ }
	     map { catdir($_, qw(Language INTERCAL Include)) }
		 @INC;
    bless {
	options => {
	    rcfile => [],
	    include => \@include,
	    nouserrc => 0,
	},
	userinc => 0,
	rccmd => [],
	data => \%data,
	prog => {},
    }, $class;
}

sub setoption {
    @_ == 3 or croak "Usage: RCFILE->setoption(NAME, VALUE)";
    my ($rc, $name, $value) = @_;
    exists $rc->{options}{$name}
	or die "Unknown option $name\n";
    if (ref $rc->{options}{$name}) {
	if ($name eq 'include') {
	    my $userinc = $rc->{userinc}++;
	    splice(@{$rc->{options}{$name}}, $userinc, 0, $value);
	} else {
	    push @{$rc->{options}{$name}}, $value;
	}
    } else {
	$rc->{options}{$name} = $value;
    }
    $rc;
}

sub getoption {
    @_ == 2 or croak "Usage: RCFILE->getoption(NAME)";
    my ($rc, $name) = @_;
    exists $rc->{options}{$name}
	or die "Unknown option $name\n";
    $rc->{options}{$name};
}

sub getitem {
    @_ == 2 or croak "Usage: RCFILE->getitem(NAME)";
    my ($rc, $name) = @_;
    exists $rc->{data}{$name}
	or die "Unknown item $name\n";
    ref $rc->{data}{$name} or return $rc->{data}{$name};
    @{$rc->{data}{$name}};
}

sub program_options {
    @_ == 2 or croak "Usage: RCFILE->program_options(PROGRAM)";
    my ($rc, $program) = @_;
    $program = uc($program);
    exists $rc->{prog}{$program} or return ();
    %{$rc->{prog}{$program}};
}

sub _rc_array {
    my ($rc, $mode, $ln, $file) = @_;
    die "Missing value for $mode\n" if $ln eq '';
    push @{$rc->{data}{$mode}}, $ln;
}

sub _rc_scalar {
    my ($rc, $mode, $ln, $file) = @_;
    die "Missing value for $mode\n" if $ln eq '';
    $rc->{data}{$mode} = $ln;
}

sub _rc_understand {
    my ($rc, $mode, $ln, $file) = @_;
    my $suffix;
    if ($ln =~ s/^(['"])(.*?)\1\s*//) {
	$suffix = $2;
    } elsif ($ln =~ s/^(\S+)\s*//) {
	$suffix = $1;
    } else {
	die "$file\: Invalid $mode\: missing SUFFIX\n";
    }
    if ($ln =~ s/^ANYWHERE\s*//i) {
	$mode .= ' ANYWHERE';
	$ln = $suffix . ' ' . $ln;
	return &{$rcdefs{$mode}[0]}($rc, $mode, $ln, $file);
    }
    $ln =~ s/^AS\s*//i or die "$file\: Invalid $mode\: missing AS\n";
    my $name;
    if ($ln =~ s/^(['"])(.*?)\1\s*//) {
	$name = $2;
    } elsif ($ln =~ s/^(\w+)\s*//) {
	$name = $1;
    } else {
	die "$file\: Invalid $mode\: missing NAME\n";
    }
    my %map = ( '' => [] );
    while ($ln ne '') {
	if ($ln =~ s/^WITH\s*//i) {
	    while (1) {
		my $maybe = '';
		$maybe = $1 if $ln =~ s/^(\?)//;
		my $preload;
		if ($ln =~ s/^(['"])(.*?)\1\s*//) {
		    $preload = $2;
		} elsif ($ln =~ s/^(\w+)\s*//) {
		    $preload = $1;
		} else {
		    die "$file\: Invalid $mode\: missing PRELOAD\n";
		}
		push @{$map{''}}, $maybe . $preload;
		$ln =~ s/^\+\s*// or last;
	    }
	    next;
	}
	if ($ln =~ s/^DISCARDING\s*//) {
	    my $option;
	    if ($ln =~ s/^(['"])(.*?)\1\s*//) {
		$option = $2;
	    } elsif ($ln =~ s/^(\S+)\s*//) {
		$option = $1;
	    } else {
		die "$file\: Invalid $mode\: missing DISCARDING\n";
	    }
	    $map{$option} = [[], ''];
	    next;
	}
	die "$file\: Invalid $mode\: $ln\n";
    }
    if ($suffix =~ s/^\.\.([^\.]+)\.//) {
	# special item used for program configuration - this avoids changing
	# the syntax of .sickrc again
	$rc->{prog}{$1}{$suffix} = [$name, \%map];
    } else {
	push @{$rc->{data}{$mode}}, [$suffix, $name, \%map];
    }
}

sub _rc_anywhere {
    my ($rc, $mode, $ln, $file) = @_;
    my $suffix;
    if ($ln =~ s/^(['"])(.*?)\1\s*//) {
	$suffix = $2;
    } elsif ($ln =~ s/^(\S+)\s*//) {
	$suffix = $1;
    } else {
	die "$file\: Invalid $mode\: missing SUFFIX\n";
    }
    $ln =~ s/^AS\s*//i or die "$file\: Invalid $mode\: missing AS\n";
    my $name;
    if ($ln =~ s/^(['"])(.*?)\1\s*//) {
	$name = $2;
    } elsif ($ln =~ s/^(\w+)\s*//) {
	$name = $1;
    } else {
	die "$file\: Invalid $mode\: missing NAME\n";
    }
    $ln =~ s/^WITH\s*//i or die "$file\: Invalid $mode\: missing WITH\n";
    my @preload = ();
    while (1) {
	my $maybe = '';
	$maybe = $1 if $ln =~ s/^(\?)//;
	my $preload;
	if ($ln =~ s/^(['"])(.*?)\1\s*//) {
	    $preload = $2;
	} elsif ($ln =~ s/^(\S+)\s*//) {
	    $preload = $1;
	} else {
	    die "$file\: Invalid $mode\: missing PRELOAD\n";
	}
	push @preload, $maybe . $preload;
	$ln =~ s/^\+\s*// or last;
    }
    die "$file\: Invalid $mode\: extra data at end ($ln)\n"
	if $ln ne '';
    push @{$rc->{data}{$mode}}, [$suffix, \@preload, $name];
}

sub load {
    @_ == 1 or croak "Usage: RCFILE->load";
    my ($rc) = @_;
    unless (@{$rc->{options}{rcfile}}) {
	# TODO - make the following portable (is there such a thing?)
	my @home = ();
	if ($ENV{HOME}) {
	    @home = ($ENV{HOME});
	} else {
	    my $name = getlogin;
	    if (! $name || ! getpwnam($name)) {
		$name = getpwuid($<);
	    }
	    if ($name && getpwnam($name)) {
		@home = (getpwnam($name))[7];
	    }
	}
	my $u = $rc->{options}{nouserrc};
	$rc->{options}{rcfile} = [
	    map {canonpath($_)}
		grep { -f $_ }
		     map { catfile($_, "system.sickrc"),
			   ($u ? () : catfile($_, ".sickrc")),
			 }
			 (@{$rc->{options}{include}}, @home, '.')
	];
	$rc->{options}{rcfile} = [$rc->{options}{rcfile}[0]]
	    if $u && @{$rc->{options}{rcfile}} > 1;
    }
    for my $rcfile (@{$rc->{options}{rcfile}}) {
	my $fh = Language::INTERCAL::GenericIO->new('FILE', 'w', $rcfile)
	    or die "$rcfile: $!\n";
	my $mode = undef;
	my $text = '';
	my $mln = '';
	my $no = 0;
	while ('' ne (my $ln = $fh->write_text())) {
	    chomp $ln;
	    $ln =~ s/^\s*//;
	    $no++;
	    next if $ln eq '';
	    my $rn = "$rcfile\:$no";
	    if ($ln =~ s/^(?:DO|PLEASE)\s*NOTE\s*//i) {
		&{$rcdefs{$mode}[0]}($rc, $mode, $text, $mln)
		    if defined $mode && $mode ne '';
		$mode = '';
	    } elsif ($ln =~ s/^I\s*DO\s*N[O']T\s*(\S+)//i) {
		&{$rcdefs{$mode}[0]}($rc, $mode, $text, $mln)
		    if defined $mode && $mode ne '';
		$mode = uc($1);
		die "No such action \"$1\": $rn\n"
		    unless exists $rcdefs{$mode};
		die "Invalid declaration \"$_\": $rn\n"
		    unless ref $rc->{data}{$mode};
		$rc->{data}{$mode} = [];
		$mode = undef;
	    } elsif ($ln =~ s/^I\s*CAN\s*(\S+)\s*//i) {
		&{$rcdefs{$mode}[0]}($rc, $mode, $text, $mln)
		    if defined $mode && $mode ne '';
		$mode = uc($1);
		exists $rcdefs{$mode}
		    or die "No such action \"$1\": $rn\n";
		$text = $ln;
		$mln = $rn;
	    } elsif (defined $mode) {
		$text .= ' ' . $ln;
	    } else {
		die "Syntax error: $rn\n";
	    }
	}
	&{$rcdefs{$mode}[0]}($rc, $mode, $text, $mln)
	    if defined $mode && $mode ne '';
    }
    $rc;
}

sub run {
    @_ == 2 or croak "Usage: RCFILE->run(UI)";
    my ($rc, $ui) = @_;
    for my $cmdline (@{$rc->{rccmd}}) {
	$ui->do($cmdline);
    }
    $rc;
}

1;
