/* ui_plugins.c - user interface plugin handling code
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* External variables and functions */
extern plugin_data *plugin_list;
extern plugin_data *current_plugin;
extern char *text;
extern GtkWidget *textarea;
extern GtkWidget *sourcearea;
extern GtkWidget *main_vbox;

/* Global variables */
GtkWidget *current_plugin_widget; /* Widget belonging to the current mode plugin */
GtkWidget *infoarea;              /* Area in plugin-viewer containing a description of the plugin */

static const int description_col = 4;

void cb_display_information (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer user_data);

/* Apply plugin transformation and display it in the text area */
void display_text(void) {
    char *newtext;
    int length;

    newtext = current_plugin->transform(text);
    length = strlen(newtext);
    
    gtk_text_set_point(GTK_TEXT(textarea), 0);
    gtk_text_forward_delete(GTK_TEXT(textarea), gtk_text_get_length(GTK_TEXT(textarea)));
    gtk_text_insert(GTK_TEXT(textarea), NULL, NULL, NULL, newtext, length);

    free(newtext);
}

/* Activate the plugin and remove an old one if necessary */
void activate_plugin(plugin_data *plugin) {
    GtkWidget *new_plugin_widget;
    if (!plugin) {
	g_warning("Error: Null plugin attempted to be activated.");
	return;
    }

    /* Fitness plugins can't be activated */
    if (plugin->plugin_type == PLUGIN_FITNESS)
	return;

    if (plugin->plugin_type == PLUGIN_FLOATING) {
	new_plugin_widget = plugin->make_widget(text);

	if (new_plugin_widget)
	    gtk_widget_show(new_plugin_widget); /* This will cause the floasting plugin display to be created */
	return;
    }

    /* Otherwise it's a mode plugin */
    if (current_plugin)
	current_plugin->deactivate_plugin();
    if (current_plugin_widget)
	gtk_container_remove(GTK_CONTAINER(main_vbox), current_plugin_widget);
    
    plugin->init_plugin();
    
    /* Move new plugin into position */
    new_plugin_widget = plugin->make_widget(text);
	
    gtk_box_pack_start(GTK_BOX(main_vbox), new_plugin_widget, FALSE, FALSE, 0);
    gtk_box_reorder_child(GTK_BOX(main_vbox), new_plugin_widget, 3);
    gtk_widget_show(new_plugin_widget);
    current_plugin = plugin;
    current_plugin_widget = new_plugin_widget;

    display_text();
}

/* Callback for menu plugins - gdata contains the number in the list*/
/* of the plugin to activate */
void cb_call_plugin(GtkWidget *widget, gpointer gdata) {
    int i;
    plugin_data *plugin_ptr = plugin_list;
    for (i = 0; i < GPOINTER_TO_INT(gdata); i++)
	plugin_ptr = plugin_ptr->next;
    activate_plugin(plugin_ptr);
}

/* Create the plugin viewer */

static char *plugin_viewer_titles[] = {"Name", "Version", "Author", "Type", "Description"};

GtkWidget *make_plugin_viewer(void) {
    int column;
    GtkWidget *clist, *dialog, *button, *label_description;
    char *row_data[5], name[BUFFER_SIZE], version[BUFFER_SIZE], author[BUFFER_SIZE], type[BUFFER_SIZE], description[BUFFER_SIZE];
    plugin_data *current_plugin = plugin_list;    

    row_data[0] = name; row_data[1] = version; row_data[2] = author; row_data[3] = type; row_data[4] = description;

    dialog  = gtk_dialog_new();
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_widget_set_usize(dialog, -1, -1);
    gtk_window_set_title(GTK_WINDOW(dialog), "Plugins Loaded");
    
    button = gtk_button_new_with_label("Dismiss");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    clist = gtk_clist_new_with_titles(5, plugin_viewer_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);

    while (current_plugin) {
	sprintf(name, "%s", current_plugin->name);
	sprintf(version, "%s", current_plugin->version);
	sprintf(author, "%s", current_plugin->author);
	sprintf(type, "%s", (current_plugin->plugin_type == PLUGIN_MODE) ? "Mode" : ((current_plugin->plugin_type == PLUGIN_FLOATING) ? "Floating" : "Fitness"));
	sprintf(description, "%s", current_plugin->description);
	gtk_clist_append(GTK_CLIST(clist), row_data);
	current_plugin = current_plugin->next;
    }
    
    for (column = 0; column <= 4; column++)
	gtk_clist_set_column_width(GTK_CLIST(clist), column, gtk_clist_optimal_column_width(GTK_CLIST(clist), column));

    gtk_signal_connect(GTK_OBJECT(clist), "select-row", GTK_SIGNAL_FUNC(cb_display_information), NULL);
    gtk_clist_set_column_visibility(GTK_CLIST(clist), description_col, FALSE);
    gtk_widget_show(clist);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), clist, TRUE, TRUE, 0);

    label_description = gtk_label_new("Description");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label_description, FALSE, FALSE, 0);
    gtk_widget_show(label_description);

    infoarea = gtk_text_new(NULL, NULL);
    gtk_text_set_editable(GTK_TEXT(infoarea), FALSE);    
    gtk_text_set_word_wrap(GTK_TEXT(infoarea), TRUE);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), infoarea, TRUE, TRUE, 1);
    gtk_widget_show(infoarea);

    return dialog;
}

void cb_start_plugin_viewer(GtkWidget *widget, gpointer gdata) {
    gtk_widget_show(make_plugin_viewer());
}

void cb_display_information (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer user_data) {
    char **text;
    char buf[BUFFER_SIZE];
    text = (char **) &buf;
    gtk_clist_get_text(clist, row, description_col, text);
    gtk_text_set_point(GTK_TEXT(infoarea), 0);
    gtk_text_forward_delete(GTK_TEXT(infoarea), gtk_text_get_length(GTK_TEXT(infoarea)));
    gtk_text_insert(GTK_TEXT(infoarea), NULL, NULL, NULL, *text, strlen(*text));
}
