/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#ifndef Attribute_hh
#define Attribute_hh

#include <string>
#include <vector>

namespace Cryptonit {

/** Arbitrary types for Attribute values.
 *  In all cases the values are stored in string format.
 */
typedef enum {
    AttributeInteger,
    AttributeLong,
    AttributeFloat,
    AttributeString,
    AttributeBinary,
    AttributeBase64
} AttributeType;


/** This class represents an attribute's value list.
 *  Values are stored in string format, disregarding the
 *  specified type.
 *  If you want to store values containing '\0', like binary
 *  datas, use methods with the length parameter.
 */
class Attribute 
{
private:

    std::vector<std::string> values;

    /** Arbitrary type of the values. All values of one attribute are
     *  in the same type.
     *  Infact, the values are stored in string format, it's just a
     *  hint.
     */
    AttributeType type;

public:
    
    /** Instantiates a new attribute
     */
    Attribute( );


    /** Copy constructor.
     *
     *  @param src : source
     */
    Attribute( const Attribute& src );


    /** Copy constructor
     *
     *  @param attr : source object
     */
    Attribute( Attribute* attr );


    /** Instantiates a new attribute of type 't'.
     *
     * @param t : attribute type (AttributeType)
     */
    Attribute( AttributeType t );

    
    /** Instantiates a new attribute with value 'val' of type AttributeString.
     *
     *  @param val : attribute value
     */
    Attribute( const std::string &val );


    /** Instantiates a new attribute with value 'val' of type 't'.
     *
     *  @param val : attribute value
     *  @param t : attribute type (AttributeType)
     */
    Attribute( const std::string &val, AttributeType t );


    /** Instantiates a new attribute with value 'val' of size 'len' with
     *	specified type 't'.
     *  Use this method if you want to add binary values. The value length could
     *  be retrieved with the size() method.
     *
     *  @param val : attribute value
     *  @param len : attribute value length
     *  @param t : attribute type (AttributeType)
     */
    Attribute( const char* val, unsigned long len, AttributeType t );


    /** Instantiates a new attribute with all values from a given vector.
     *
     *  @param val : vector containing values to be added.
     */
    Attribute( const std::vector<std::string>& val );


    /** Instantiates a new attribute with all values from a given vector, of
     *  a specified type.
     *
     *  @param val : vector containing values to be added.
     *  @param t : attribute type (AttributeType)
     */
    Attribute( const std::vector<std::string>& val, AttributeType t );


    /** Destroy a Attribute object, erasing all data 
     *  contained in the value list.
     */
    virtual ~Attribute();


    /** Returns current attribute type.
     */
    virtual AttributeType getType() const;
    

    /** Appends a new value of type AttributeString to the values list.
     *  Returns true if operation succeed.
     *
     *  @param val : new value
     */
    virtual bool append( const std::string &val );

    
    /** Appends a new value of the specified type to the values list.
     *  Returns true if operation succeed.
     *
     *  @param val : new value
     *  @param t : value type
     */
    virtual bool append( const std::string &val, AttributeType t );


    /** Appends a new value of the specified type to the values list.
     *  Use this method if you want to add binary values.
     *  Returns true if operation succeed.
     *
     *  @param val : new value
     *  @param len : value size
     *  @param t : value type
     */
    virtual bool append( const char* val, unsigned long len, AttributeType t );

    
    /** Erases the specified value of the values list.
     *
     *  @param val : value to be erased
     */
    virtual bool erase( const std::string &val );


    /** Erases all values stored in the list.
     */
    virtual bool clear();

    
    /** Returns a vector with all attribute's values.
     */
    virtual std::vector<std::string> getValues();


    /** Returns the first value of the values list.
     */
    virtual const std::string& getFirstValue() const;


    /** Returns the last value of the values list.
     */
    virtual const std::string& getLastValue() const;

    /** Returns the total number of values.
     */
    virtual unsigned int getNbValues() const;




    class iterator;
    friend class iterator;

    /** Nested class for iterating inside the values.
     */
    class iterator {

	/** Reference on the current Attribute object.
	 */
	Attribute& ds;

	/** Current iterator on the values vector.
	 */
	std::vector<std::string>::iterator itr;


    public:

	/** Creates the iterator.
	 */
	iterator(class Attribute& d) : ds(d), itr( ds.values.begin() ) {}

	/** Creates the "end sentinel" iterator
	 */
	iterator(class Attribute& d, bool) : ds(d), itr( ds.values.end() ) {}

	/** Returns the current value, use it for iterator operation.
	 */
	std::vector<std::string>::iterator current() const { return itr; }

	/** Prefix incrementation.
	 */
	std::vector<std::string>::iterator operator++() { return ++itr; }

	/** Postfix incrementation.
	 */
	std::vector<std::string>::iterator operator++( int ) { return itr++; }

	/** Comparaison equality operator.
	 */
	bool operator==(const iterator& rv) const {
	    return itr == rv.itr;
	}

	/** Comparaison difference operator.
	 */
	bool operator!=(const iterator& rv) const {
	    return itr != rv.itr;
	}

	friend std::ostream& operator<<(std::ostream& os, const iterator& it) {
	    return os << *(it.current());
	}

    };
    
    /** Returns an iterator on the begin of the values vector.
     */
    iterator begin() { return iterator(*this); }

    /** Returns an iterator on the end of the values vector.
     */
    iterator end() { return iterator(*this, true);}

};

} // Namespace
#endif
