# mdb.rb : a backend that can extract electrochemical data stored within
# MS Access databases created by the PAR PowerSuite utilities. I doubt
# it will be useful to many people, but, well no one knows !

# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA


require 'SciYAG/Backends/backend.rb'
require 'Dobjects/Dvector'
require 'Dobjects/Function'

module SciYAG

  class MDBBackend < Backend

    include Dobjects
    
    describe 'mdb', 'PAR PowerSuite MDB files', <<EOD, false
Reads the MS Access files produced by the PAR electrochemistry suite.
Abstract Backend, needs to be redefined by children.
EOD


    
    # The table of the file where we want to look for data.
    attr_accessor :data_set_table_name

    # The name of the X, Y and Z columns in the table.
    attr_accessor :data_set_x_col, :data_set_y_col, :data_set_z_col

    # If this attribute is set, the sets are further split
    # for the different values of that column
    attr_accessor :data_set_name_col


    # A small class to hold a data set
    class DataSet
      include Dobjects
      
      # The name of the dataset
      attr_accessor :name

      # The DataSetID property
      attr_accessor :data_set_id

      # Used internally to create a new dvector
      def new_dvector
        return Dvector.new(1000).resize(0)
      end

      # Returns the keys
      def subsets
#         return @x_data.keys.compact # removes the nil key...
        return @subsets.compact
      end
      
      def initialize(name, data_set_id)
        @name = name
        @data_set_id = data_set_id

        @subsets = []

        @x_data = {}            # A hash indexed on the set name
        @y_data = {}            # A hash indexed on the set name
        @z_data = {}            # A hash indexed on the set name
      end

      # Returns the x_data for the given set
      def x_data(set = nil)
        return @x_data[set]
      end

      # Returns the y_data for the given set
      def y_data(set = nil)
        return @y_data[set]
      end

      # Returns the z_data for the given set
      def z_data(set = nil)
        return @z_data[set]
      end

      # Adds a point to the set
      def add_point(x,y,z,set = nil)
        if not @x_data[set]
          @subsets << set
          @x_data[set] = new_dvector
          @y_data[set] = new_dvector
          @z_data[set] = new_dvector
        end
        @x_data[set] << x
        @y_data[set] << y
        @z_data[set] << z
      end

    end

    def initialize
      super()
      @data_set_table_name = nil           # has to be redefined by children
      # A hash holding dataset_name -> DataSet object
      @data_sets = {}
      @data_set_id = []         # The same as upper,
      # but with the set number

      # The column for different sets
      @data_set_name_col = "Series"

      # The current database
      @current_database = ""
    end

    # The separator for columns when reading the file.
    COL_SEP = "###"
    # The command to run mdb-export
    MDB_EXPORT = "mdb-export"

    # Reads a table and returns the table of lines, optionnally beginning
    # with the header
    def mdb_export_table(file, table, header = false)
      cmd_line = "#{MDB_EXPORT} -Q -d '#{COL_SEP}' " + 
                     if header
                       " "
                     else
                       "-H "
                     end +
                     "'#{file}' '#{table}' "  +
        "| sort -n"             # we sort numerically with the program sort,
                                # decently faster
      mdb = IO.popen(cmd_line)
      l = mdb.readlines
      return l
    end

    # Reads a table and turns it into a list of hashes
    def mdb_table_to_hash(file, table) 
      pipe = IO.popen("#{MDB_EXPORT} -Q -d '#{COL_SEP}' -R '&&&&&' " +
                      "#{file} #{table}")
      header = pipe.readline.chomp.split(COL_SEP)
      raw_entries = pipe.read.split('&&&&&').map {|l| l.split(COL_SEP)}
      entries = []
      for raw_entry in raw_entries
        entry = {}
        header.length.times do |i|
          entry[header[i]] = raw_entry[i]
        end
        entries << entry
      end
      return entries
    end

    # Turns the first line returned by mdb_export_table into a nice header
    # hash
    def turn_header_into_hash(lines)
      header = lines.shift
      header = header.split(COL_SEP)
      header_hash = {}
      i = 0
      for name in header
        header_hash[name] = i
        i+=1
      end
      return header_hash
    end

    # Small helper function
    def safe_float(x)
      begin
        Float(x)
      rescue
        0.0/0.0
      end
    end

    # Inherit the baseline stuff.
    inherit_parameters :base_line

    attr_reader :current_database
    param :read_mdb_file, :current_database, "readdb", "Read Database", 
    {:type => :file, :filters => "Databases (*.mdb)"}, 
    "Reads a database into memory"

    # Reads the contents of a MDB file according to different defaults.
    def read_mdb_file(file)
      # First, make out the datasets from the file
      if @current_database == file
        return                  # We already read it !!
      end
      @current_database = file
      
      data_sets_properties =  mdb_table_to_hash(file, 'DataSetProperties')

      prospective_data_sets = []
      for set in data_sets_properties
        prospective_data_sets << DataSet.new(set["DataSetName"], 
                                             set["DataSetID"])
      end

      # We turn it into another array indexed on DataSetID for faster lookup
      @data_sets_ids = []
      for ds in prospective_data_sets
        @data_sets_ids[ds.data_set_id.to_i] = ds
      end
      
      # Now, we need to get data from the files
      raise "@data_set_table_name has to be set" unless @data_set_table_name
      raise "@data_set_y_col has to be set" unless @data_set_y_col
      raise "@data_set_x_col has to be set" unless @data_set_x_col
      raise "@data_set_z_col has to be set" unless @data_set_z_col
      
      l = mdb_export_table(file, @data_set_table_name, true)
      h = turn_header_into_hash(l)
      if h.key?(@data_set_x_col) and 
          h.key?(@data_set_y_col) and 
          h.key?(@data_set_z_col)
        # We first sort the data respective to the first column
#         l.sort! do |a,b|
#           i_a = a[0,a.index(COL_SEP)].to_i
#           i_b = b[0,b.index(COL_SEP)].to_i
#           i_a <=> i_b
#         end
        x_col = h[@data_set_x_col]
        y_col = h[@data_set_y_col]
        z_col = h[@data_set_z_col]
        set_col = if @data_set_name_col
                    h[@data_set_name_col]
                  else 
                    false
                  end
        id = h["DataSetID"]
        

        l.each do |l|
          a = l.split(COL_SEP)
          ds = @data_sets_ids[a[id].to_i]
          x = safe_float(a[x_col])
          y = safe_float(a[y_col])
          z = safe_float(a[z_col])
          if data_set_name_col
            ds.add_point(x,y,z,a[set_col])
          else
            ds.add_point(x,y,z)
          end
        end
        
      else
        raise "It looks like #{@data_set_x_col} or #{@data_set_y_col} is " +
          "missing from the table #{@data_set_table_name} of file #{file}"
      end

      # Now, we populate the data_sets hash
      for ds in prospective_data_sets 
        @data_sets[ds.name] = ds
      end
    end

    # Baseline stuff: we inherit it from the parents.


    # This RE tells if a set looks like it might contain a subset.
    # Slurps anything until the last @.
    SUBSET_RE = /(.+)@(.+)/

    
    # An internal function saying how to make a X,Y dataset from
    # a x,y,z dataset. Can depend on many things. We just output
    # X and Y cols here. Better be redefined by children.
    def internal_to_external(x,y,z)
      return Function.new(x,y)
    end

    # Internal function to get the data corresponding to one set.
    def get_data(set)
      if set =~ SUBSET_RE
        s = string_to_set($1)
        raise "Set #{$1} is unkown" unless s
        return internal_to_external(s.x_data($2), 
                                    s.y_data($2),
                                    s.z_data($2))
      else
        s = string_to_set(set)
        raise "Set #{set} is unkown" unless s
        return internal_to_external(s.x_data, s.y_data, s.z_data)
      end
    end

    # This is called by the architecture to get the data. It splits
    # the set name into filename@cols, reads the file if necessary and
    # calls get_data
    def query_xy_data(set)
      data = get_data(set)
      return Function.new(data.x.dup, data.y.dup)
    end

    # Transforms a string into a set. If the set starts with a #, it means
    # we're directly interested in the number, not in the name. Can be a great
    # deal useful, and much shorter...
    def string_to_set(str)
      if str =~ /^\#(\d+)$/
        return @data_sets_ids[$1.to_i]
      else
        return @data_sets[str]
      end
    end

    def expand_sets(set)
      if s = string_to_set(set)
        subs = s.subsets # we sort so that everything ids
        # looking fine.
        if subs.empty?
          return [set]
        else
          return subs.collect {|s| "#{set}@#{s}"}
        end
      end
      return [set]
    end

    # I think it is better not to take the cycles into account: they will
    # hinder the display for no good reason.
    def sets_available
      return @data_sets.keys.sort
    end

  end

  class MDBCyclicVoltammogram < MDBBackend

    describe 'mdbcv', 'PAR PowerSuite MDB files, Cyclic Voltammetry', <<EOD
Reads the MS Access files produced by the PAR electrochemistry suite,
to extract cyclic voltammetry data.
EOD
    def initialize
      super
      @data_set_table_name = "CVDataPoints_280"
      @data_set_x_col = "E"
      @data_set_y_col = "I"
      @data_set_z_col = "T"

      @type = :cv
      @neg = true
    end

    # We import parameters from the parent
    inherit_parameters :read_mdb_file, :base_line

    param_accessor :type, 'type', "Type", 
    { :type => :list, :list => {
        :cv => "I = f(E)",
        :et => "E = f(t)",
        :it => "I = f(t)"} }, 
    "The kind of plot wanted"

    param_accessor :neg, 'polarity', "Current polarity", 
    { :type => :boolean } , "True inverses current polarity (on by default)"

    # Returns the set wanted. (x = E, y = I, z = T);
    def internal_to_external(x,y,z)
      if @neg
        y = y.neg
      end
      case @type
      when :cv
        return Function.new(x,y)
      when :et
        return Function.new(z.sub(z.min).mul!(86400),x)
      when :it
        return Function.new(z.sub(z.min).mul!(86400),y)
      end
    end

  end

  class MDBPStep < MDBCyclicVoltammogram

    describe 'mdbps', 'PAR PowerSuite MDB files, Potential Steps', <<EOD
Reads the MS Access files produced by the PAR electrochemistry suite,
to extract potential step data.
EOD

    inherit_parameters :read_mdb_file, :base_line, :type, :neg

    def initialize
      super
      @data_set_table_name = "CADataPoints_28"
      @type = :it
    end

  end


  class MDBEIS < MDBBackend

    describe 'mdbeis', 'PAR PowerSuite MDB files, Impedance Spectroscopy', <<EOD
Reads the MS Access files produced by the PAR electrochemistry suite,
to extract impedance spectroscopy data.
EOD
    def initialize
      super
      @data_set_table_name = "ImpDataPoints_20"
      @data_set_x_col = "Freq"
      @data_set_y_col = "Zre"
      @data_set_z_col = "Zim"

      @type = :wzm
    end

    # We import parameters from the parent
    inherit_parameters :read_mdb_file, :base_line

    param_accessor :type, 'type', "Type", {:type => :list,
      :list => {
        :wzm => "|Z| = f(f)",
        :wzi => "Zi = f(f)", 
        :wzr => "Zr = f(f)",
        :wphi => "Phi = f(f)",
        :zrzi => "Zi = f(Zr)",
        :yryi => "- Yi = f(Yr)",
      }}, "The kind of plot wanted"

    # Returns the set wanted. (x = Freq, y = Zre, z = Zim);
    def internal_to_external(x,y,z)
      case @type
      when :wzm
        return Function.new(x,y.pow(2).add!(z.pow(2)).sqrt)
      when :wzi
        return Function.new(x,z)
      when :wzr
        return Function.new(x,y)
      when :zrzi
        return Function.new(y,z)
      when :wphi
        return Function.new(x,z.div(y).atan!)
      when :yryi
        # We need to compute real and imaginary part of admittance
        den = y**2 + z**2
        return Function.new(y.div(den), 
                            z.div(den))
      end
    end

  end



end


