/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* main.c */

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "config.h"
#include "directory_menu.h"
#include "dirtree.h"
#include "edit_menu.h"
#include "file_menu.h"
#include "help_menu.h"
#include "image_cache.h"
#include "intl.h"
#include "main.h"
#include "option_menu.h"
#include "pop_menu_dirtree.h"
#include "thumbnail.h"
#include "thumbnail_menu.h"
#include "thumbnail_table.h"
#include "utils.h"
#include "version.h"
#include "viewer.h"

#include "icon/closed_folder.xpm"
#include "icon/copy_icon.xpm"
#include "icon/cut_icon.xpm"
#include "icon/danpei.xpm"
#include "icon/dotfile_icon.xpm"
#include "icon/edit_icon.xpm"
#include "icon/home_icon.xpm"
#include "icon/next_icon.xpm"
#include "icon/opened_folder.xpm"
#include "icon/paste_icon.xpm"
#include "icon/prev_icon.xpm"
#include "icon/print_icon.xpm"
#include "icon/refresh_dir_icon.xpm"
#include "icon/refresh_icon.xpm"
#include "icon/remove_icon.xpm"
#include "icon/rotate_left_icon.xpm"
#include "icon/rotate_right_icon.xpm"
#include "icon/view_icon.xpm"
#include "icon/viewer_icon.xpm"

/* External variables. */
/* For drag and drop. */
const GtkTargetEntry dnd_te[] = { {"text/uri-list", GTK_TARGET_SAME_APP, 0} };
gint  dnd_te_num = 1;

/* Static variables. */
/* For menu. */
static GtkItemFactoryEntry menu_items[] = {
    { N_("/_File")                                      , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/File/E_xternal viewer")                      , NULL        ,
                        file_menu_cb_view               , 0, "<Item>"      },
    { N_("/File/_Edit ")                                , NULL        ,
                        file_menu_cb_edit               , 0, "<Item>"      },
    { N_("/File/---")                                  , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/Start _simple viewer")                  , NULL        ,
                        viewer_cb_all                   , 0, "<Item>"      },
    { N_("/File/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/_Prev ")                                , "<control>B",
                        file_menu_cb_prev               , 0, "<Item>"      },
    { N_("/File/_Next ")                                , "<control>N",
                        file_menu_cb_next               , 0, "<Item>"      },
    { N_("/File/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/_Refresh ")                             , NULL        ,
                        file_menu_cb_refresh            , 0, "<Item>"      },
    { N_("/File/Reload selected thumbnails")            , NULL        ,
                        file_menu_cb_reload_selected    , 0, "<Item>"      },
    { N_("/File/Re_load ")                              , NULL        ,
                        file_menu_cb_reload             , 0, "<Item>"      },
    { N_("/_File/---")                                  , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/Go to _Home")                           , NULL        ,
                        file_menu_cb_goto_home          , 0, "<Item>"      },
    { N_("/File/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/_Print ")                               , NULL        ,
                        file_menu_cb_print              , 0, "<Item>"      },
    { N_("/File/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/Remove ")                               , NULL        ,
                        file_menu_cb_remove             , 0, "<Item>"      },
    { N_("/File/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/File/_Quit ")                                , "<control>Q",
                        file_menu_cb_quit               , 0, "<Item>"      },
    { N_("/_Edit")                                      , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/Edit/C_ut ")                                  , NULL        ,
                        edit_menu_cb_cut                , 0, "<Item>"      },
    { N_("/Edit/_Copy ")                                , NULL        ,
                        edit_menu_cb_copy               , 0, "<Item>"      },
    { N_("/Edit/_Paste ")                               , NULL        ,
                        edit_menu_cb_paste              , 0, "<Item>"      },
    { N_("/Edit/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Edit/Rotate _left ")                         , NULL        ,
                        edit_menu_cb_rotate_left        , 0, "<Item>"      },
    { N_("/Edit/Rotate _right ")                        , NULL        ,
                        edit_menu_cb_rotate_right       , 0, "<Item>"      },
    { N_("/Edit/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Edit/Select _All ")                          , NULL        ,
                        edit_menu_cb_select_all         , 0, "<Item>"      },
    { N_("/Edit/Unselect All ")                         , NULL        ,
                        edit_menu_cb_unselect_all       , 0, "<Item>"      },
    { N_("/Edit/---")                                   , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Edit/Show the clip board ")                 , NULL        ,
                        edit_menu_cb_show_clipboard     , 0, "<Item>"      },
    { N_("/_Directory")                                 , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/Directory/_New ")                            , NULL        ,
                        directory_menu_cb_new           , 0, "<Item>"      },
    { N_("/Directory/---")                              , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Directory/Remove ")                          , NULL        ,
                        directory_menu_cb_remove        , 0, "<Item>"      },
    { N_("/Directory/---")                              , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Directory/_Refresh the tree")                , NULL        ,
                        directory_menu_cb_refresh       , 0, "<Item>"      },
    { N_("/Directory/---")                              , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Directory/_Dot file on-off")                 , NULL        ,
                        directory_menu_cb_dotfile       , 0, "<Item>"      },
    { N_("/_Thumbnail")                                 , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/Thumbnail/_Size")                            , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/Thumbnail/---")                              , NULL        ,
                        0                               , 0, "<Separator>" },
    { N_("/Thumbnail/_Format")                          , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/Thumbnail/So_rt")                            , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/_Option")                                    , NULL        ,
                        NULL                            , 0, "<Branch>"    },
    { N_("/Option/_Configuration")                      , NULL        ,
                        option_menu_cb_configuration    , 0, "<Item>"      },
    { N_("/_Help")                                      , NULL        ,
                        NULL                            , 0, "<LastBranch>"},
    { N_("/Help/_About...")                             , NULL        ,
                        help_menu_cb_about              , 0, "<Item>"      }
  };

static GtkItemFactoryEntry menu_items1[] = {
    { N_("/Cut     ")      , NULL        , 
                             edit_menu_cb_cut             , 0, "<Item>"     },
    { N_("/Copy    ")      , NULL        , 
                             edit_menu_cb_copy            , 0, "<Item>"     },
    { N_("/---")           , NULL        , 
                             0                            , 0, "<Separator>"},
    { N_("/Simple viewer ") 
                           , NULL        , 
                             file_menu_cb_simple_viewer   , 0, "<Item>"     },
    { N_("/External viewer ") 
                           , NULL        , 
                             file_menu_cb_view            , 0, "<Item>"     },
    { N_("/Edit    ")      , NULL        , 
                             file_menu_cb_edit            , 0, "<Item>"     },
    { N_("/---")           , NULL        , 
                             0                            , 0, "<Separator>"},
    { N_("/Rotate left  ") , NULL        , 
                             edit_menu_cb_rotate_left     , 0, "<Item>"     },
    { N_("/Rotate right ") , NULL        , 
                             edit_menu_cb_rotate_right    , 0, "<Item>"     },
    { N_("/---")           , NULL        , 
                             0                            , 0, "<Separator>"},
    { N_("/Print   ")      , NULL        , 
                             file_menu_cb_print           , 0, "<Item>"     },
    { N_("/---")           , NULL        , 
                             0                            , 0, "<Separator>"},
    { N_("/Remove  ")      , NULL        , 
                             file_menu_cb_remove          , 0, "<Item>"     }
  };

static GtkItemFactoryEntry menu_items2[] = {
    { N_("/Paste   ")   , NULL        ,
                          pop_menu_dirtree_cb_paste    , 0, "<Item>"     },
    { N_("/---")        , NULL        ,
                          0                            , 0, "<Separator>"},
    { N_("/New     ")   , NULL        ,
                          pop_menu_dirtree_cb_new      , 0, "<Item>"     },
    { N_("/---")        , NULL        ,
                          0                            , 0, "<Separator>"},
    { N_("/Remove  ")   , NULL        ,
                          pop_menu_dirtree_cb_remove   , 0, "<Item>"     }
  };

static gint nmenu_items  = sizeof(menu_items ) / sizeof(menu_items [0]);

static gint nmenu_items1 = sizeof(menu_items1) / sizeof(menu_items1[0]);

static gint nmenu_items2 = sizeof(menu_items2) / sizeof(menu_items2[0]);

/* Static functions declaration. */
static void     toplevel_display_usage             (void);

static void     toplevel_get_window_size           (gchar*         ,
                                                    gint*          ,
                                                    gint*            );

static void     toplevel_gui_config                (TopLevel*      , 
                                                    gint           ,
                                                    gint           ,
                                                    gchar*           );

static gchar    *menu_translate                    (const gchar *  , 
                                                    gpointer         );

static void     toplevel_radio_menu_create         (TopLevel*        );

static void     toplevel_init                      (TopLevel*        );

static gint     toplevel_cb_delete                 (GtkWidget*     ,
                                                    GdkEvent*      ,
                                                    gpointer         );

static gboolean toplevel_cb_button_pressed         (GtkWidget*     ,
                                                    GdkEventButton*,
                                                    gpointer         );

static void     toplevel_cb_toolbar_entry_activate (GtkEditable*   ,
                                                    gpointer         );

static void     toplevel_remove_danpei_cache_file  (TopLevel*        );

/* Function definitions. */
/*
 * @main
 *
 *
 *
 */
int main(int argc, char **argv) {
  TopLevel  toplevel;
  gchar     *init_dir;
  gint      pane_width, pane_height;
  GdkPixmap *icon_pixmap;
  GdkBitmap *icon_mask;

  /* Initialize the local variables. */
  init_dir    = NULL;
  icon_pixmap = NULL;
  icon_mask   = NULL;

  /* for gettext */
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);

  gtk_set_locale();
  gtk_init(&argc, &argv);
  gdk_rgb_init();
  gtk_rc_parse("./gtkrc");

  /* Initialize toplevel's variables. */
  toplevel_init(&toplevel);

  /* Clean up cache files. 
   * -- When danpei has normal end, cache files will be cleaned up.
   *    But abnormal end (ex. Segmentation fault), cache files will be rest.
   *    So, clean up them at starting danpei.
   */
  if (toplevel.app_option.image_cache.remain_on != TRUE) {
    toplevel_clean_cache_files(&toplevel);
    toplevel_remove_danpei_cache_file(&toplevel);
  }

  /* Get arguments. */
  switch (argc) {
    case 1:
      pane_width  = DEFAULT_PANE_WIDTH;
      pane_height = DEFAULT_PANE_HEIGHT;
      init_dir    = getcwd(NULL, 256);
      break;

    case 2:
      pane_width  = DEFAULT_PANE_WIDTH;
      pane_height = DEFAULT_PANE_HEIGHT;
      if (chdir(argv[1]) == 0) {
        init_dir = getcwd(NULL, 256);
      }
      else {
        fprintf(stderr, "danpei: chdir(%s) failed.\n", argv[1]);
        fprintf(stderr, "        main.c: error -- 01.\n");
        fprintf(stderr, "        <errno.h> - errno = %d.\n", errno);
        gtk_exit(-1);
      }
      break;

    case 3:
      if (strcmp(argv[1], "-size") != 0) {
        toplevel_display_usage();
      }
      /* Get window size parameter. */
      toplevel_get_window_size(argv[2], &pane_width, &pane_height);

      /* Get current directory. */
      init_dir = getcwd(NULL, 256);
      break;

    case 4:
      if (strcmp(argv[1], "-size") != 0) {
        toplevel_display_usage();
      }
      /* Get window size parameter. */
      toplevel_get_window_size(argv[2], &pane_width, &pane_height);

      /* Get the specified directory. */
      if (chdir(argv[3]) == 0) {
        init_dir = getcwd(NULL, 256);
      }
      else {
        fprintf(stderr, "danpei: chdir(%s) failed.\n", argv[3]);
        fprintf(stderr, "        main.c: error -- 02.\n");
        fprintf(stderr, "        <errno.h> - errno = %d.\n", errno);
        gtk_exit(-1);
      }
      break;

    default:
      toplevel_display_usage();
      break;
  }

  /* GUI configuration. */
  toplevel_gui_config(&toplevel, pane_width, pane_height, init_dir);

  /* Set the application icon. */
  /* -- TopLevel's window id is required, so do after the function
   *    gtk_widget_show(toplevel).
   */
  icon_pixmap = gdk_pixmap_create_from_xpm_d((toplevel.window)->window, 
                                             &icon_mask, NULL, 
                                             (gchar**)app_icon_xpm);
  gdk_window_set_icon((toplevel.window)->window, 
                      NULL, icon_pixmap, icon_mask);

  gtk_widget_set_usize(toplevel.window, 1, 1);

  gdk_pixmap_unref(icon_pixmap);
  gdk_bitmap_unref(icon_mask);
 
  gtk_main();

  free(init_dir);

  return 0;
}

/*
 * @toplevel_set_menu_sensitive
 *
 *  Set the menu sensitive or unsensitive.
 *
 */
void toplevel_set_menu_sensitive(TopLevel  *tp     ,
                                 AppStatus app_stat  ) {
  if (app_stat > STATUS_NO_THUMBNAIL - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Start simple viewer"), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Select All "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Unselect All "), FALSE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->viewer_button, FALSE);
    app_stat -= STATUS_NO_THUMBNAIL;
  }
  if (app_stat > STATUS_THUMBNAIL - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Start simple viewer"), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Select All "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Unselect All "), TRUE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->viewer_button, TRUE);
    app_stat -= STATUS_THUMBNAIL;
  }
  if (app_stat > STATUS_NO_CLIPBOARD - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Paste "), FALSE);
    /* Pop up menu(dirtree). */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_dirtree.item_factory, 
                                    "/Paste   "), FALSE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->paste_button, FALSE);
    app_stat -= STATUS_NO_CLIPBOARD;
  }
  if (app_stat > STATUS_CLIPBOARD - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Paste "), TRUE);
    /* Pop up menu(dirtree). */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_dirtree.item_factory, 
                                    "/Paste   "), TRUE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->paste_button, TRUE);
    app_stat -= STATUS_CLIPBOARD;
  }
  if (app_stat > STATUS_NO_NEXT - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Next "), FALSE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->next_button, FALSE);
    app_stat -= STATUS_NO_NEXT;
  }
  if (app_stat > STATUS_NEXT - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Next "), TRUE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->next_button, TRUE);
    app_stat -= STATUS_NEXT;
  }
  if (app_stat > STATUS_NO_PREV - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Prev "), FALSE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->prev_button, FALSE);
    app_stat -= STATUS_NO_PREV;
  }
  if (app_stat > STATUS_PREV - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Prev "), TRUE);
    /* Toolbar. */
    gtk_widget_set_sensitive(tp->prev_button, TRUE);
    app_stat -= STATUS_PREV;
  }
  if (app_stat > STATUS_NO_SELECTED - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/External viewer") , FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Edit "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Print "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Remove "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Cut "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Copy "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Rotate left "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Rotate right "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Unselect All "), FALSE);

    /* Popup menu on Thumbnail. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Simple viewer "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/External viewer "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Edit    "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Rotate left  "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Rotate right "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Print   "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Remove  "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Cut     "), FALSE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Copy    "), FALSE);

    /* Toolbar. */
    gtk_widget_set_sensitive(tp->view_button        , FALSE);
    gtk_widget_set_sensitive(tp->edit_button        , FALSE);
    gtk_widget_set_sensitive(tp->rotate_left_button , FALSE);
    gtk_widget_set_sensitive(tp->rotate_right_button, FALSE);
    gtk_widget_set_sensitive(tp->print_button       , FALSE);
    gtk_widget_set_sensitive(tp->remove_button      , FALSE);
    gtk_widget_set_sensitive(tp->cut_button         , FALSE);
    gtk_widget_set_sensitive(tp->copy_button        , FALSE);
    app_stat -= STATUS_NO_SELECTED;
  }
  if (app_stat > STATUS_SELECTED - 1) {
    /* Menubar. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/External viewer") , TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Edit "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Print "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/File/Remove "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Cut "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Copy "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Rotate left "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Rotate right "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                    "/Edit/Unselect All "), TRUE);

    /* Popup menu on Thumbnail. */
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Simple viewer "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/External viewer "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Edit    "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Rotate left  "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Rotate right "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Print   "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Remove  "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Cut     "), TRUE);
    gtk_widget_set_sensitive(
        gtk_item_factory_get_widget(tp->popup_thumbnail.item_factory, 
                                    "/Copy    "), TRUE);

    /* Toolbar. */
    gtk_widget_set_sensitive(tp->view_button        , TRUE);
    gtk_widget_set_sensitive(tp->edit_button        , TRUE);
    gtk_widget_set_sensitive(tp->rotate_left_button , TRUE);
    gtk_widget_set_sensitive(tp->rotate_right_button, TRUE);
    gtk_widget_set_sensitive(tp->print_button       , TRUE);
    gtk_widget_set_sensitive(tp->remove_button      , TRUE);
    gtk_widget_set_sensitive(tp->cut_button         , TRUE);
    gtk_widget_set_sensitive(tp->copy_button        , TRUE);
    app_stat -= STATUS_SELECTED;
  }

  return;
}

/*
 * @toplevel_clean_cache_files
 *
 *  Clean up cache files.
 *
 */
void toplevel_clean_cache_files(TopLevel *tp) {
  DIR           *dirp;
  struct dirent *ent;
  gchar         *filename;
  gchar         head[10];

  /* Check if the cache directory exists. */
  if (tp->cache.cache_dir == NULL) { return; }
  if ((dirp = opendir(tp->cache.cache_dir)) == NULL) { return; }

  while((ent = readdir(dirp)) != NULL) {
    if (strlen(ent->d_name) > strlen("danpei")) {
      strncpy(head, ent->d_name, strlen("danpei"));
      head[6] = '\0';
      if (strcmp(head, "danpei") == 0) {
        filename = (gchar*)malloc(sizeof(gchar) *
                                  (strlen(tp->cache.cache_dir) +
                                   strlen("/")                 +
                                   strlen(ent->d_name)         + 1) + 1);
        if (filename == NULL) {
          /* Out of memory. */
          fprintf(stderr, "danpei: Out of memory.\n");
          fprintf(stderr, "        main.c: error -- 05.\n");
          gtk_exit(-1);
        }
        else {
          sprintf(filename, "%s/%s", tp->cache.cache_dir, ent->d_name);
          remove(filename);
        }
        free(filename);
      }
    }
  }
  closedir(dirp);

  return;
}

/* Static functions definitions. */
/*
 * @toplevel_display_usage
 *
 *  Display the usage to stderr and exit.
 *
 */
static void toplevel_display_usage() {
  fprintf(stderr, "usage: danpei [-size widthxheight] [directory]\n");
  gtk_exit(-1);
}

/*
 * @toplevel_get_window_size
 *
 *  Get the width and height of the toplevel-window from the argument.
 *
 */
static void toplevel_get_window_size(gchar *size  ,
                                     gint  *width ,
                                     gint  *height  ) {
  gchar *wk_size1, *wk_size2; 

  /* Initialize the local variables. */
  wk_size1 = wk_size2 = NULL;

  if ((wk_size2 = strchr(size, 'x')) == NULL) {
    *width  = DEFAULT_PANE_WIDTH;
    *height = DEFAULT_PANE_HEIGHT;
    return;
  }
  else {
    wk_size1  = strdup(size);
    if (wk_size1 == NULL) {
      /* Out of memory. */
      fprintf(stderr, "danpei: Out of memory.\n");
      fprintf(stderr, "        main.c: error -- 03.\n");
      gtk_exit(-1);
    }
    else {
      *wk_size2 = '\0';
      if ((*width = atoi(wk_size1)) == 0) {
        *width  = DEFAULT_PANE_WIDTH;
      }

      wk_size2++;
      strcpy(wk_size1, wk_size2); 
      if ((*height = atoi(wk_size1)) == 0) {
        *height  = DEFAULT_PANE_HEIGHT;
      }
    }
    free(wk_size1);
  }

  return;
}

/*
 * @toplevel_gui_config
 *
 *  Construct the GUI.
 *
 */
static void toplevel_gui_config(TopLevel *tp        , 
                                gint     pane_width ,
                                gint     pane_height,
                                gchar    *directory   ) {
  GdkPixmap *pixmap_data;
  GdkBitmap *pixmap_mask;
  gchar     title[32];
  gint      i;            /* counter */
  GtkWidget *tmp_widget;

  /* Create the main window. */
  tp->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  sprintf(title,"Danpei version %s", VERSION);
  gtk_window_set_title(GTK_WINDOW(tp->window), title);
  gtk_signal_connect(GTK_OBJECT(tp->window), 
                     "delete_event",
                     GTK_SIGNAL_FUNC(toplevel_cb_delete), tp);
  gtk_signal_connect(GTK_OBJECT(tp->window), 
                     "destroy",
                     GTK_SIGNAL_FUNC(toplevel_cb_destroy), tp);

  gtk_container_border_width(GTK_CONTAINER(tp->window), 0);

  tp->ev_box = gtk_event_box_new();
/************************************************************************
  gtk_widget_set_events(GTK_WIDGET(tp->ev_box), GDK_BUTTON_PRESS_MASK);
************************************************************************/
  gtk_signal_connect(GTK_OBJECT(tp->ev_box), 
                     "button_press_event",
                     GTK_SIGNAL_FUNC(toplevel_cb_button_pressed), tp);
  gtk_container_add(GTK_CONTAINER(tp->window), tp->ev_box);
  gtk_widget_show(tp->ev_box);

  tp->vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(tp->ev_box), tp->vbox);
  gtk_widget_show(tp->vbox);

  /* Create the menu bar. */
  tp->main_menu.accel_group  = gtk_accel_group_new();
  tp->main_menu.item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR,
                                  "<Main>", tp->main_menu.accel_group);
  gtk_item_factory_set_translate_func(tp->main_menu.item_factory, 
                                      menu_translate, NULL, NULL);

  for (i = 0; i < nmenu_items; i++) {
    gtk_item_factory_create_item(tp->main_menu.item_factory, &menu_items[i],
                                 tp, 2);
  }
  gtk_accel_group_attach(tp->main_menu.accel_group, GTK_OBJECT(tp->window)); 

  toplevel_radio_menu_create(tp);

  gtk_box_pack_start(GTK_BOX(tp->vbox),
      gtk_item_factory_get_widget(tp->main_menu.item_factory, "<Main>"),
      FALSE, FALSE, 0);
  gtk_widget_show(gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                  "<Main>"));

  /* Create the pop up menu for thumbnail table. */
  tp->popup_thumbnail.accel_group  = gtk_accel_group_new();
  tp->popup_thumbnail.item_factory = 
              gtk_item_factory_new(GTK_TYPE_MENU, 
                                   "<Thumbnail>", 
                                   tp->popup_thumbnail.accel_group);
  gtk_item_factory_set_translate_func(tp->popup_thumbnail.item_factory, 
                                      menu_translate, NULL, NULL);
  for (i = 0; i < nmenu_items1; i++) {
    gtk_item_factory_create_item(tp->popup_thumbnail.item_factory, 
                                 &menu_items1[i], tp, 2);
  }
  gtk_accel_group_attach(tp->popup_thumbnail.accel_group, 
                         GTK_OBJECT(gtk_item_factory_get_widget(
                                    tp->popup_thumbnail.item_factory , 
                                                         "<Thumbnail>" )));

  /* Create the pop up menu for dirtree table. */
  tp->popup_dirtree.accel_group  = gtk_accel_group_new();
  tp->popup_dirtree.item_factory = 
              gtk_item_factory_new(GTK_TYPE_MENU, 
                                   "<DirTree>", 
                                   tp->popup_dirtree.accel_group);
  gtk_item_factory_set_translate_func(tp->popup_dirtree.item_factory, 
                                      menu_translate, NULL, NULL);
  for (i = 0; i < nmenu_items2; i++) {
    gtk_item_factory_create_item(tp->popup_dirtree.item_factory, 
                                 &menu_items2[i], tp, 2);
  }
  gtk_accel_group_attach(tp->popup_dirtree.accel_group, 
                         GTK_OBJECT(gtk_item_factory_get_widget(
                                    tp->popup_dirtree.item_factory , 
                                                         "<DirTree>" )));
  gtk_signal_connect(GTK_OBJECT(gtk_item_factory_get_widget(
                                    tp->popup_dirtree.item_factory, 
                                    "<DirTree>"                    )),
                     "deactivate",
                     GTK_SIGNAL_FUNC(pop_menu_dirtree_cb_deactivate), tp);

  /* Create the toolbar. */
  tp->handlebox = gtk_handle_box_new();
  gtk_box_pack_start(GTK_BOX(tp->vbox), tp->handlebox, FALSE, FALSE, 0);
  gtk_widget_show(tp->handlebox);

  tp->toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
                                GTK_TOOLBAR_BOTH);
  gtk_container_border_width   (GTK_CONTAINER(tp->toolbar), 2);
  gtk_toolbar_set_space_size   (GTK_TOOLBAR(tp->toolbar), 16);
  gtk_toolbar_set_button_relief(GTK_TOOLBAR(tp->toolbar), GTK_RELIEF_NONE);
  gtk_toolbar_set_space_style  (GTK_TOOLBAR(tp->toolbar), 
                                GTK_TOOLBAR_SPACE_LINE);
  gtk_container_add(GTK_CONTAINER(tp->handlebox), tp->toolbar);
  gtk_widget_show(tp->toolbar);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)prev_icon_xpm);
  tp->prev_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->prev_button = 
           gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                   NULL, _("Previous page"), 
                                   NULL, tp->prev_icon,
                                   GTK_SIGNAL_FUNC(file_menu_cb_prev),tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)next_icon_xpm);
  tp->next_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->next_button = 
           gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                   NULL, _("Next page"), 
                                   NULL, tp->next_icon,
                                   GTK_SIGNAL_FUNC(file_menu_cb_next), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  tp->tb_hbox  = gtk_hbox_new(FALSE, 0);
  tp->tb_entry = gtk_entry_new_with_max_length(5);
  gtk_widget_set_usize(tp->tb_entry, 50, 20);
  gtk_signal_connect(GTK_OBJECT(tp->tb_entry), "activate",
                     GTK_SIGNAL_FUNC(toplevel_cb_toolbar_entry_activate),
                     tp);
  gtk_box_pack_start(GTK_BOX(tp->tb_hbox), tp->tb_entry, FALSE, FALSE, 2);
  gtk_widget_show(tp->tb_entry);

  tp->tb_label = gtk_label_new("/      ");
  gtk_widget_set_usize(tp->tb_label, 60, 20);
  gtk_box_pack_start(GTK_BOX(tp->tb_hbox), tp->tb_label, FALSE, FALSE, 0);
  gtk_widget_show(tp->tb_label);
  gtk_toolbar_append_widget(GTK_TOOLBAR(tp->toolbar), 
                            tp->tb_hbox, NULL, NULL);
  gtk_widget_show(tp->tb_hbox);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)dotfile_icon_xpm);
  tp->dotfile_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->dotfile_button = 
      gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                              NULL, _("Dot file ON/OFF"), 
                              NULL, tp->dotfile_icon,
                              GTK_SIGNAL_FUNC(directory_menu_cb_dotfile), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)refresh_dir_icon_xpm);
  tp->refresh_dir_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->refresh_dir_button = 
              gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                  NULL, _("Refresh the directory tree"), 
                                  NULL, tp->refresh_dir_icon,
                                  GTK_SIGNAL_FUNC(directory_menu_cb_refresh),
                                  tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)home_icon_xpm);
  tp->home_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->home_button = gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                  NULL, _("Go to Home directory"), 
                                  NULL, tp->home_icon,
                                  GTK_SIGNAL_FUNC(file_menu_cb_goto_home),
                                  tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)refresh_icon_xpm);
  tp->refresh_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->refresh_button = 
              gtk_toolbar_append_item(
                      GTK_TOOLBAR(tp->toolbar), NULL, 
                      _("Refresh thumbnail"), 
                      NULL, tp->refresh_icon,
                      GTK_SIGNAL_FUNC(file_menu_cb_refresh),tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)viewer_icon_xpm);
  tp->viewer_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->viewer_button = 
           gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                   NULL, _("Start simple viewer"), 
                                   NULL, tp->viewer_icon,
                                   GTK_SIGNAL_FUNC(viewer_cb_all), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)view_icon_xpm);
  tp->view_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->view_button = 
           gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                   NULL, _("Start external viewer"), 
                                   NULL, tp->view_icon,
                                   GTK_SIGNAL_FUNC(file_menu_cb_view), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)edit_icon_xpm);
  tp->edit_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->edit_button = 
           gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                   NULL, _("Edit"), 
                                   NULL, tp->edit_icon,
                                   GTK_SIGNAL_FUNC(file_menu_cb_edit), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)rotate_left_icon_xpm);
  tp->rotate_left_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->rotate_left_button = 
      gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                              NULL, _("Rotate left"), 
                              NULL, tp->rotate_left_icon,
                              GTK_SIGNAL_FUNC(edit_menu_cb_rotate_left), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)rotate_right_icon_xpm);
  tp->rotate_right_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->rotate_right_button = 
      gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                              NULL, _("Rotate right"), 
                              NULL, tp->rotate_right_icon,
                              GTK_SIGNAL_FUNC(edit_menu_cb_rotate_right), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)cut_icon_xpm);
  tp->cut_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->cut_button = 
          gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                  NULL, _("Cut"), 
                                  NULL, tp->cut_icon,
                                  GTK_SIGNAL_FUNC(edit_menu_cb_cut), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)copy_icon_xpm);
  tp->copy_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->copy_button = 
           gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                   NULL, _("Copy"), 
                                   NULL, tp->copy_icon,
                                   GTK_SIGNAL_FUNC(edit_menu_cb_copy), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)paste_icon_xpm);
  tp->paste_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->paste_button = 
            gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                    NULL, _("Paste"), 
                                    NULL, tp->paste_icon,
                                    GTK_SIGNAL_FUNC(edit_menu_cb_paste), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)print_icon_xpm);
  tp->print_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->print_button = 
            gtk_toolbar_append_item(GTK_TOOLBAR(tp->toolbar),
                                    NULL, _("Print"), 
                                    NULL, tp->print_icon,
                                    GTK_SIGNAL_FUNC(file_menu_cb_print), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  gtk_toolbar_append_space(GTK_TOOLBAR(tp->toolbar));

  pixmap_data = gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT() ,
                                             &pixmap_mask, NULL,
                                             (gchar**)remove_icon_xpm);
  tp->remove_icon   = gtk_pixmap_new(pixmap_data, pixmap_mask);
  tp->remove_button = 
             gtk_toolbar_append_item(
                                GTK_TOOLBAR(tp->toolbar),
                                NULL, _("Remove"), 
                                NULL, tp->remove_icon,
                                GTK_SIGNAL_FUNC(file_menu_cb_remove), tp);
  gdk_pixmap_unref(pixmap_data);
  gdk_bitmap_unref(pixmap_mask);

  /* Create the Horizontal paned area for the directory tree and the 
   * thumbnail table.
   */
  tp->hpaned = gtk_hpaned_new();
  gtk_widget_set_usize(tp->hpaned, pane_width, pane_height);
  gtk_box_pack_start(GTK_BOX(tp->vbox), tp->hpaned, TRUE, TRUE, 0);
  gtk_widget_show(tp->hpaned);

  /* Create status bar. */
  tp->hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tp->vbox), tp->hbox, FALSE, FALSE, 0);
  gtk_widget_show(tp->hbox);

  tp->statusbar1.bar = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(tp->hbox), 
                     tp->statusbar1.bar, FALSE, FALSE, 0);
  gtk_widget_set_usize(tp->statusbar1.bar, pane_width * .7, 20);
  gtk_widget_show(tp->statusbar1.bar);

  tp->statusbar2.bar = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(tp->hbox), 
                     tp->statusbar2.bar, TRUE, TRUE, 0);
  gtk_widget_show(tp->statusbar2.bar);

  /* Create the Directory tree. */
  dirtree_create(tp, "/");
  gtk_signal_connect(GTK_OBJECT(tp->dir_tree.ctree), 
                     "button_press_event",
                     GTK_SIGNAL_FUNC(toplevel_cb_button_pressed), tp);
  dirtree_directory_set_selected(tp, directory, TRUE);
  gtk_paned_add1(GTK_PANED(tp->hpaned), tp->dir_tree.scrolled_window);

  /* Set the grid position. */
  gtk_paned_set_position(GTK_PANED(tp->hpaned), pane_width * .3);

  /* Set drag and drop */
  gtk_drag_dest_set(tp->dir_tree.ctree  ,
                    GTK_DEST_DEFAULT_ALL,
                    dnd_te, dnd_te_num  ,
                    GDK_ACTION_ASK  | GDK_ACTION_COPY | 
                    GDK_ACTION_MOVE | GDK_ACTION_LINK    );

  /* Set the menu sensitivity. */
  toplevel_set_menu_sensitive(tp, STATUS_NO_CLIPBOARD);

  /* Create the thumbnail table. */
  tp->thumbnail_table.scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(
                  GTK_SCROLLED_WINDOW(tp->thumbnail_table.scrolled_window),
                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_paned_add2(GTK_PANED(tp->hpaned), tp->thumbnail_table.scrolled_window);
  gtk_widget_show(tp->thumbnail_table.scrolled_window);

  tp->thumbnail_table.ev_box = gtk_event_box_new();
  gtk_signal_connect(GTK_OBJECT(tp->thumbnail_table.ev_box),
                     "button_press_event",
                     GTK_SIGNAL_FUNC(toplevel_cb_button_pressed  ), tp);
  gtk_scrolled_window_add_with_viewport(
               GTK_SCROLLED_WINDOW(tp->thumbnail_table.scrolled_window),
               tp->thumbnail_table.ev_box);
  gtk_widget_show(tp->thumbnail_table.ev_box);

  /* Show the toplevel window. */
  gtk_widget_show(tp->window);

  /* TRUE: Display the thumbnails from a head. */
  thumbnail_table_create(tp, directory, TRUE);

  gtk_widget_set_usize(tp->window, 1, 1);

  /*
   * Delete this line because configure_event will be emitted only when
   * the window is moved.
   * -- It seems that configure_event will not be emitted when the window 
   *    is resized.
   */
  /*
    gtk_signal_connect(GTK_OBJECT(tp->window),
                       "configure_event",
                       GTK_SIGNAL_FUNC(thumbnail_table_cb_configure), tp);
  */
  return;
}

/*
 * @menu_translate
 *
 *  Translate the menu by using gettext().
 *
 */
static gchar *menu_translate(const gchar *path,
                             gpointer    data   ) {
  gchar *val;

  val = gettext(path);

  return val;
}

/*
 * @toplevel_radio_menu_create
 *
 *  Create radio menu items.
 *  -- GtkItemFactory can't create radio menu items?
 *
 */
static void toplevel_radio_menu_create(TopLevel *tp) {
  GSList    *group1, *group2, *group3;
  GtkWidget *menu_small , *menu_middle, *menu_large;
  GtkWidget *menu_jpg   , *menu_bmp   , *menu_png  , *menu_xpm;
  GtkWidget *menu_tif   , *menu_xbm   , *menu_gif  , *menu_pcx;
  GtkWidget *menu_file  , *menu_ctime , *menu_atime, *menu_mtime;
  GtkWidget *menu_ascend, *menu_descend;
  GtkWidget *menu_separator;

  /* Initialize the local variables. */
  group1 = group2 = group3 = NULL;
  menu_small = menu_middle = menu_large = NULL;
  menu_jpg   = menu_bmp    = menu_png   = menu_xpm   = NULL;
  menu_tif   = menu_xbm    = menu_gif   = menu_pcx   = NULL;
  menu_file  = menu_ctime  = menu_atime = menu_mtime = NULL;

  /* Create the radio menu items. */
  /* for Thumbnail/Size menu.     */
  menu_small = gtk_radio_menu_item_new_with_label(group1, _("Small"));
  group1 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_small));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Size"           )), 
      menu_small);
  gtk_signal_connect(GTK_OBJECT(menu_small), "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_size_small_activate),
                     tp);
  gtk_widget_show(menu_small);

  menu_middle = gtk_radio_menu_item_new_with_label(group1, _("Middle"));
  group1 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_middle));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Size"           )), 
      menu_middle);
  gtk_signal_connect(GTK_OBJECT(menu_middle), "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_size_middle_activate),
                     tp);
  gtk_widget_show(menu_middle);
  
  menu_large = gtk_radio_menu_item_new_with_label(group1, _("Large"));
  group1 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_large));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Size"           )), 
      menu_large);
  gtk_signal_connect(GTK_OBJECT(menu_large), "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_size_large_activate),
                     tp);
  gtk_widget_show(menu_large);
  
  switch(tp->app_option.icon_size) {
    case ICON_SMALL:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_small), TRUE);
      break;
    case ICON_MIDDLE:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_middle), TRUE);
      break;
    case ICON_LARGE:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_large), TRUE);
      break;
    default:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_middle), TRUE);
      break;
  }

  /* for Thumbnail/Format menu.     */
  menu_jpg = gtk_check_menu_item_new_with_label("JPEG");
  gtk_widget_set_name(menu_jpg, "CHECK_MENU_ITEM_JPEG");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_jpg);
  gtk_signal_connect(GTK_OBJECT(menu_jpg), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.jpg_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_jpg), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_jpg), FALSE);
  }
  gtk_widget_show(menu_jpg);

  menu_bmp = gtk_check_menu_item_new_with_label("BMP");
  gtk_widget_set_name(menu_bmp, "CHECK_MENU_ITEM_BMP");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_bmp);
  gtk_signal_connect(GTK_OBJECT(menu_bmp), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.bmp_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_bmp), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_bmp), FALSE);
  }
  gtk_widget_show(menu_bmp);

  menu_png = gtk_check_menu_item_new_with_label("PNG");
  gtk_widget_set_name(menu_png, "CHECK_MENU_ITEM_PNG");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_png);
  gtk_signal_connect(GTK_OBJECT(menu_png), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.png_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_png), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_png), FALSE);
  }
  gtk_widget_show(menu_png);

  menu_xpm = gtk_check_menu_item_new_with_label("XPM");
  gtk_widget_set_name(menu_xpm, "CHECK_MENU_ITEM_XPM");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_xpm);
  gtk_signal_connect(GTK_OBJECT(menu_xpm), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.xpm_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_xpm), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_xpm), FALSE);
  }
  gtk_widget_show(menu_xpm);

  menu_tif = gtk_check_menu_item_new_with_label("TIFF");
  gtk_widget_set_name(menu_tif, "CHECK_MENU_ITEM_TIFF");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_tif);
  gtk_signal_connect(GTK_OBJECT(menu_tif), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.tif_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_tif), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_tif), FALSE);
  }
  gtk_widget_show(menu_tif);

  menu_xbm = gtk_check_menu_item_new_with_label("XBM");
  gtk_widget_set_name(menu_xbm, "CHECK_MENU_ITEM_XBM");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_xbm);
  gtk_signal_connect(GTK_OBJECT(menu_xbm), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.xbm_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_xbm), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_xbm), FALSE);
  }
  gtk_widget_show(menu_xbm);

  menu_gif = gtk_check_menu_item_new_with_label("GIF");
  gtk_widget_set_name(menu_gif, "CHECK_MENU_ITEM_GIF");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_gif);
  gtk_signal_connect(GTK_OBJECT(menu_gif), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.gif_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_gif), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_gif), FALSE);
  }
  gtk_widget_show(menu_gif);

  menu_pcx = gtk_check_menu_item_new_with_label("PCX");
  gtk_widget_set_name(menu_pcx, "CHECK_MENU_ITEM_PCX");
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Format"         )), menu_pcx);
  gtk_signal_connect(GTK_OBJECT(menu_pcx), "toggled",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_format_button_toggled),
                     tp);
  if (tp->current_format.pcx_on == TRUE) {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_pcx), TRUE);
  }
  else {
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_pcx), FALSE);
  }
  gtk_widget_show(menu_pcx);

  /* for Thumbnail/Sort menu.     */
  menu_file = gtk_radio_menu_item_new_with_label(group2, _("File name"));
  group2 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_file));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort"           )), 
      menu_file);
  gtk_signal_connect(GTK_OBJECT(menu_file), "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_sort_file_activate),
                     tp);
  gtk_widget_show(menu_file);

  menu_ctime = gtk_radio_menu_item_new_with_label(group2, _("Created time"));
  group2 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_ctime));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort"           )),
      menu_ctime);
  gtk_signal_connect(GTK_OBJECT(menu_ctime),
                     "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_sort_ctime_activate),
                     tp);
  gtk_widget_show(menu_ctime);

  menu_atime = gtk_radio_menu_item_new_with_label(group2, _("Access time"));
  group2 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_atime));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort"           )),
      menu_atime);
  gtk_signal_connect(GTK_OBJECT(menu_atime),
                     "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_sort_atime_activate),
                     tp);
  gtk_widget_show(menu_atime);

  menu_mtime = gtk_radio_menu_item_new_with_label(group2, _("Modified time"));
  group2 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_mtime));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort"           )),
      menu_mtime);
  gtk_signal_connect(GTK_OBJECT(menu_mtime),
                     "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_sort_mtime_activate),
                     tp);
  gtk_widget_show(menu_mtime);

  menu_separator = gtk_menu_item_new();
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort")), menu_separator);
  gtk_widget_show(menu_separator);

  menu_ascend = gtk_radio_menu_item_new_with_label(group3, _("Ascend"));
  group3 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_ascend));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort"           )),
      menu_ascend);
  gtk_signal_connect(GTK_OBJECT(menu_ascend),
                     "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_sort_ascend_activate),
                     tp);
  gtk_widget_show(menu_ascend);

  menu_descend = gtk_radio_menu_item_new_with_label(group3, _("Descend"));
  group3 = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_descend));
  gtk_menu_append(GTK_MENU(
      gtk_item_factory_get_widget(tp->main_menu.item_factory, 
                                  "/Thumbnail/Sort"           )),
      menu_descend);
  gtk_signal_connect(GTK_OBJECT(menu_descend),
                     "activate",
                     GTK_SIGNAL_FUNC(thumbnail_menu_cb_sort_descend_activate),
                     tp);
  gtk_widget_show(menu_descend);

  switch(tp->app_option.sort) {
    case SORT_FILE_NAME_A:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_file)  , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_ascend), TRUE);
      break;
    case SORT_CTIME_A:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_ctime) , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_ascend), TRUE);
      break;
    case SORT_ATIME_A:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_atime) , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_ascend), TRUE);
      break;
    case SORT_MTIME_A:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_mtime) , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_ascend), TRUE);
      break;
    case SORT_FILE_NAME_D:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_file)   , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_descend), TRUE);
      break;
    case SORT_CTIME_D:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_ctime)  , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_descend), TRUE);
      break;
    case SORT_ATIME_D:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_atime)  , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_descend), TRUE);
      break;
    case SORT_MTIME_D:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_mtime)  , TRUE);
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_descend), TRUE);
      break;
  }

  return;
}

/*
 * @toplevel_init
 *
 *  Initialize the TopLevel structure object.
 *
 */
static void toplevel_init(TopLevel *tp) {
  /* Initialize the member 'thumbnail_table'. */
  thumbnail_table_init_structure(&(tp->thumbnail_table));

  /* Initialize the member 'statusbar1'. */
  tp->statusbar1.bar        = NULL;
  tp->statusbar1.context_id = 0;
  tp->statusbar1.message_id = 0;

  /* Initialize the member 'statusbar2'. */
  tp->statusbar2.bar        = NULL;
  tp->statusbar2.context_id = 0;
  tp->statusbar2.message_id = 0;

  /* Initialize the member 'main_menu'.       */
  tp->main_menu.item_factory = NULL;
  tp->main_menu.accel_group  = NULL;

  /* Initialize the member 'app_option'.      */
  option_menu_init_app_option_structure(&(tp->app_option));
  option_menu_read_config_file(&(tp->app_option));

  /* Initialize the member 'cache'.           */
  image_cache_manage_cache_structure_init(&(tp->cache));
  tp->cache.cache_dir = (gchar*)malloc(sizeof(gchar) *
                                       (strlen(getenv("HOME")) +
                                        strlen("/.danpei")     + 1) + 1);
  if (tp->cache.cache_dir == NULL) {
    /* Out of memory. */
    fprintf(stderr, "danpei: Out of memory.\n");
    fprintf(stderr, "        main.c: error -- 04.\n");
    gtk_exit(-1);
  }
  else {
    sprintf(tp->cache.cache_dir, "%s/.danpei", getenv("HOME"));
  }
  if (tp->app_option.image_cache.remain_on == TRUE) {
    image_cache_read_info_from_disk(tp);
  }

  /* Initialize the member 'clipboard'.       */
  tp->clipboard.type = EDIT_NONE;
  tp->clipboard.top  = NULL;
  tp->clipboard.num  = 0;

  /* Initialize DanpeiViewer members. */
  (tp->viewer).idle_id         = 0;
  (tp->viewer).original_pixmap = NULL;
  (tp->viewer).scaled_pixmap   = NULL;
  (tp->viewer).thumb           = NULL;
  (tp->viewer).current_thumb   = NULL;
  (tp->viewer).area_x          = -1;
  (tp->viewer).area_y          = -1;
  (tp->viewer).alive           = FALSE;

  /* Initialize the options which user can set while danpei working. */
  tp->current_icon_size  = tp->app_option.icon_size;
  tp->current_format     = tp->app_option.format;
  tp->current_sort_order = tp->app_option.sort;
  tp->dot_file_on        = tp->app_option.dot_file;

  /* Create foleder pximaps. */
  tp->opened_folder_pixmap = 
      gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT(), 
                                   &(tp->opened_folder_mask), 
                                   NULL,
                                   (gchar**)opened_folder_xpm);
  tp->closed_folder_pixmap = 
      gdk_pixmap_create_from_xpm_d(GDK_ROOT_PARENT(), 
                                   &(tp->closed_folder_mask), 
                                   NULL,
                                   (gchar**)closed_folder_xpm);

  /* Initialize the temporary variables.      */
  tp->save_filename    = NULL;
  tp->from_2button     = FALSE;
  tp->op_thumb         = NULL;
  tp->dnd_selected_row = -1;
  tp->app_status       = STATUS_NONE;
  tp->all_yes          = FALSE;
  tp->focused_entry    = NULL;
  tp->columns          = 0;

  return;
}

/* Callback funtion definitions. */
/*
 * @toplevel_cb_destory
 *
 *  This function is public because it will be called when 'Quit' menu 
 *  selected.
 *
 */
void toplevel_cb_destroy(GtkWidget *widget,
                         gpointer  data     ) {
  TopLevel *tp;
  gchar    *cache_file;

  tp = (TopLevel*)data;

  if (tp->app_status == STATUS_APP_QUIT) { return; }

  tp->app_status = STATUS_APP_QUIT;
  if (tp->app_option.image_cache.remain_on == TRUE) {
    if (image_cache_write_info_to_disk(tp) == FALSE) {
      /* If writing cache information failed, remove all cache files. */
      toplevel_clean_cache_files(tp);
      toplevel_remove_danpei_cache_file(tp);
    }
  }
  else {
    toplevel_clean_cache_files(tp);
    toplevel_remove_danpei_cache_file(tp);
  }
  thumbnail_table_destroy_structure(&(tp->thumbnail_table));
  if ((tp->viewer).alive == TRUE) {
    gtk_widget_destroy((tp->viewer).window);
  }

  option_menu_destroy_app_option_structure(&(tp->app_option));
  image_cache_manage_cache_structure_destroy(&(tp->cache), TRUE);
  edit_menu_destroy_clipboard(&(tp->clipboard));

  /* Free the folder pixmaps and bitmaps. */
  gdk_pixmap_unref(tp->opened_folder_pixmap);
  gdk_pixmap_unref(tp->closed_folder_pixmap);
  gdk_bitmap_unref(tp->opened_folder_mask); 
  gdk_bitmap_unref(tp->closed_folder_mask); 

  gtk_main_quit();

  return;
}

/*
 * @toplevel_cb_delete
 *
 *
 *
 */
static gint toplevel_cb_delete(GtkWidget *widget, 
                               GdkEvent  *event , 
                               gpointer  data     ) {
  return FALSE;
}

/*
 * @toplevel_cb_button_press
 *
 *
 *
 */
static gboolean toplevel_cb_button_pressed(GtkWidget      *widget, 
                                           GdkEventButton *event , 
                                           gpointer       data     ) {
  TopLevel *tp;

  /* Initialize the local variables. */
  tp = (TopLevel*)data;

  if (tp->focused_entry != NULL) {
    gtk_window_set_focus(GTK_WINDOW(tp->window), tp->dir_tree.ctree);
  }

  return FALSE;
}

/*
 * @toplevel_cb_toolbar_entry_activate
 *
 *  Move the start_pos to the value of the toolbar-entry, and
 *  display thumbnails.
 *
 */
static void toplevel_cb_toolbar_entry_activate (GtkEditable *widget,
                                                gpointer    data     ) {
  TopLevel *tp;
  gint     start_pos, end_pos;
  gint     prev_start_pos, prev_end_pos;
  gchar    *num_pt;

  /* Initialize the local variables. */
  tp = (TopLevel*)data;

  prev_start_pos = tp->thumbnail_table.start_pos;
  prev_end_pos   = tp->thumbnail_table.end_pos;

  start_pos = atoi(gtk_entry_get_text(GTK_ENTRY(widget)));

  /* If the specified value is invalid, then return. */
  if ((start_pos == (prev_start_pos + 1))        ||
      (start_pos < 1)                            ||
      (start_pos > tp->thumbnail_table.file_num)    ) {
    utils_gint_to_gchar(tp->thumbnail_table.start_pos + 1, 5, 
                        TRUE, &num_pt);
    gtk_entry_set_text(GTK_ENTRY(tp->tb_entry), num_pt);
    gtk_editable_set_position(GTK_EDITABLE(tp->tb_entry), 0);
    free(num_pt);
    return;
  }

  /* Display the thumbnails after setting the start position and 
   * end position.
   */
  start_pos--;
  if (start_pos + tp->app_option.max_num > 
                          tp->thumbnail_table.file_num - 1) {
    end_pos = tp->thumbnail_table.file_num - 1;
  }
  else {
    end_pos = start_pos + tp->app_option.max_num - 1;
  }

  tp->thumbnail_table.start_pos = start_pos;
  tp->thumbnail_table.end_pos   = end_pos;
  thumbnail_table_display_current_page(tp, prev_start_pos, prev_end_pos);

  return;
}

/*
 * @toplevel_remove_danpei_cache_file
 *
 *
 *
 */
static void toplevel_remove_danpei_cache_file (TopLevel *tp) {
  gchar *cache_file;

  /* Initialize the local variables. */
  cache_file = NULL;

  cache_file = (gchar*)malloc(sizeof(gchar) * 
                              (strlen(tp->cache.cache_dir) +
                               strlen("/.danpei.cache")    + 1));
  if (cache_file == NULL) {
    /* Out of memory. */
    fprintf(stderr, "danpei: Out of memory.\n");
    fprintf(stderr, "        image_cache.c: error -- 07.\n");
    gtk_exit(-1);
  }
  sprintf(cache_file, "%s/.danpei.cache", tp->cache.cache_dir);

  remove(cache_file);

  free(cache_file);

  return;
}
