/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   Copyright (C) 2004 by Scott Wheeler                                   *
 *   wheeler@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kiconloader.h>
#include <kdebug.h>

#include <qpixmap.h>
#include <qpainter.h>
#include <qregexp.h>

#include "viewmode.h"
#include "datatable.h"
#include "datareport.h"

ViewMode::ViewMode( DataTableBox *b ) :
        QObject( b ),
        m_dataTableBox( b ),
        m_visible( false ),
        m_needsRefresh( false )
{
    m_dataTableBox->viewport() ->installEventFilter( this );
}

ViewMode::~ViewMode()
{}

void ViewMode::paintCell( DataTableBox::Item *item,
                          QPainter *painter,
                          const QColorGroup &colorGroup,
                          int column, int width, int )
{

    const QPixmap *pm = item->pixmap( column );

    if ( !pm )
        return;

    if ( width < pm ->width() )
        return ;

    if ( m_needsRefresh )
        updateHeights();

    QFontMetrics fm = painter->fontMetrics();

    int y = item->listView() ->itemMargin() + border;

    if ( item->isSelected() )
    {
        painter->eraseRect( 0, 0, width, item->height() );

        QPen oldPen = painter->pen();
        QPen newPen = oldPen;

        newPen.setWidth( 5 );
        newPen.setJoinStyle( RoundJoin );
        newPen.setColor( QColorGroup::Highlight );

        painter->setPen( newPen );
        painter->drawRect( border, border, width - border * 2, item->height() - border * 2 + 1 );
        painter->setPen( oldPen );

        painter->fillRect( border, border, width - border * 2, item->height() - border * 2 + 1,
                           colorGroup.brush( QColorGroup::Highlight ) );
        painter->setPen( colorGroup.highlightedText() );
    }
    else
        painter->eraseRect( 0, 0, width, item->height() );

    if ( !pm->isNull() )
    {
        int x = ( width - pm->width() ) / 2;
        x = QMAX( x, item->listView() ->itemMargin() );
        painter->drawPixmap( x, y, *pm );
    }
    y += pm->height() + fm.height() - fm.descent();
    for ( QStringList::Iterator it = m_lines[ item ].begin(); it != m_lines[ item ].end(); ++it )
    {
        int x = ( width - fm.width( *it ) ) / 2;
        x = QMAX( x, item->listView() ->itemMargin() );
        painter->drawText( x, y, *it );
        y += fm.height() - fm.descent();
    }

    if ( item == item->listView() ->dropItem() )
        paintDropIndicator( painter, width, item->height() );
}

bool ViewMode::eventFilter( QObject *watched, QEvent *e )
{
    if ( m_visible && watched == m_dataTableBox->viewport() && e->type() == QEvent::Resize )
    {
        QResizeEvent * re = static_cast<QResizeEvent *>( e );
        if ( re->size().width() != re->oldSize().width() )
            m_needsRefresh = true;
    }

    if ( e->type() == QEvent::Hide )
        m_needsRefresh = true;

    return QObject::eventFilter( watched, e );
}

void ViewMode::setShown( bool shown )
{
    dataTableBox() ->setTreeStepSize( 0 );
    m_visible = shown;
    if ( shown )
    {
        updateIcons( 32 );
        m_needsRefresh = true;
    }
}

void ViewMode::updateIcons( int size )
{
    for ( QListViewItemIterator it( m_dataTableBox ); it.current(); ++it )
    {
        DataTableBox::Item *i = static_cast<DataTableBox::Item *>( *it );
        i->setPixmap( 0, SmallIcon( i->iconName(), size ) );
    }
}

void ViewMode::setupItem( DataTableBox::Item *item ) const
{
    const DataTableBox * box = item->listView();
    const int width = box->width() - box->verticalScrollBar() ->width() - border * 2;
    const int baseHeight = 2 * box->itemMargin() + 32 + border * 2;
    const QFontMetrics fm = box->fontMetrics();
    item->setHeight( baseHeight + ( fm.height() - fm.descent() ) * lines( item, fm, width ).count() );
}

void ViewMode::updateHeights()
{
    const int width = m_dataTableBox->width() - m_dataTableBox->verticalScrollBar() ->width() - border * 2;

    const int baseHeight = 2 * m_dataTableBox->itemMargin() + 32 + border * 2;
    const QFontMetrics fm = m_dataTableBox->fontMetrics();

    for ( QListViewItemIterator it( m_dataTableBox ); it.current(); ++it )
    {
        DataTableBox::Item *i = static_cast<DataTableBox::Item *>( it.current() );
        m_lines[ i ] = lines( i, fm, width );
        const int height = baseHeight + ( fm.height() - fm.descent() ) * m_lines[ i ].count();
        i->setHeight( height );
    }

    m_needsRefresh = false;
}

void ViewMode::paintDropIndicator( QPainter *painter, int width, int height )       // static
{
    static const int border = 1;
    static const int lineWidth = 2;

    QPen oldPen = painter->pen();
    QPen newPen = oldPen;

    newPen.setWidth( lineWidth );
    newPen.setStyle( DotLine );

    painter->setPen( newPen );
    painter->drawRect( border, border, width - border * 2, height - border * 2 );
    painter->setPen( oldPen );
}

QStringList ViewMode::lines( const DataTableBox::Item *item,
                             const QFontMetrics &fm,
                             int width )
{
    // Here 32 is a bit arbitrary, but that's the width of the icons in this
    // mode and seems to a reasonable lower bound.

    if ( width < 32 )
        return QStringList();


    QString line;
    if ( DataTable * dataTable = dynamic_cast<DataTable*>( item->itemWidget() ) )
    {
        QStringList lst;
        lst.append( item->text() );
        lst.append( " (" );
        lst.append( QString::number( dataTable->dataTableView()->numRows() ) );
        lst.append( ")" );
        line = lst.join( "" );
    }
    else if ( DataReport * dataReport = dynamic_cast<DataReport*>( item->itemWidget() ) )
    {
        QStringList lst;
        lst.append( item->text() );
        lst.append( " (" );
        lst.append( QString::number( dataReport->numRows() ) );
        lst.append( ")" );
        line = lst.join( "" );
    }
    QStringList l;

    while ( !line.isEmpty() )
    {
        int textLength = line.length();
        while ( textLength > 0 &&
                fm.width( line.mid( 0, textLength ).stripWhiteSpace() ) +
                item->listView() ->itemMargin() * 2 > width )
        {
            int i = line.findRev( QRegExp( "\\W" ), textLength - 1 );
            if ( i > 0 )
                textLength = i;
            else
                textLength--;
        }

        l.append( line.mid( 0, textLength ).stripWhiteSpace() );
        line = line.mid( textLength );
    }
    return l;
}

CompactViewMode::CompactViewMode( DataTableBox *b ) :
        ViewMode( b )
{
}

CompactViewMode::~CompactViewMode()
{}

void CompactViewMode::paintCell( DataTableBox::Item *item,
                                 QPainter *painter,
                                 const QColorGroup &colorGroup,
                                 int column, int width, int )
{
    const QPixmap *pm = item->pixmap( column );

    if ( !pm )
        return;

    if ( width < pm ->width() )
        return ;

    if ( m_needsRefresh )
        updateHeights();

    QFontMetrics fm = painter->fontMetrics();

    int x = item->listView() ->itemMargin() + border;
    int y = item->listView() ->itemMargin() + border;

    if ( item->isSelected() )
    {
        painter->eraseRect( 0, 0, width, item->height() );

        QPen oldPen = painter->pen();
        QPen newPen = oldPen;

        newPen.setWidth( 5 );
        newPen.setJoinStyle( RoundJoin );
        newPen.setColor( QColorGroup::Highlight );

        painter->setPen( newPen );
        painter->drawRect( border, border, width - border * 2, item->height() - border * 2 + 1 );
        painter->setPen( oldPen );

        painter->fillRect( border, border, width - border * 2, item->height() - border * 2 + 1,
                           colorGroup.brush( QColorGroup::Highlight ) );
        painter->setPen( colorGroup.highlightedText() );
    }
    else
        painter->eraseRect( 0, 0, width, item->height() );

    if ( !pm->isNull() )
    {
        painter->drawPixmap( x, y, *pm );
        x += pm->width() + border;
    }
    for ( QStringList::Iterator it = m_lines[ item ].begin(); it != m_lines[ item ].end(); ++it )
    {
        y += fm.height() - fm.descent();
        painter->drawText( x, y, *it );
    }

    if ( item == item->listView() ->dropItem() )
        paintDropIndicator( painter, width, item->height() );
}

void CompactViewMode::setupItem( DataTableBox::Item *item ) const
{
    item->KListViewItem::setup();
}

void CompactViewMode::setShown( bool shown )
{
    dataTableBox() ->setTreeStepSize( 20 );
    setVisible( shown );

    if ( shown )
    {
        updateIcons( 16 );
        updateHeights();
    }
}

void CompactViewMode::updateHeights()
{
    const int width = m_dataTableBox->width() - m_dataTableBox->verticalScrollBar() ->width() - border * 2;

    const int baseHeight = 2 * m_dataTableBox->itemMargin() + 16 + border * 2;
    const QFontMetrics fm = m_dataTableBox->fontMetrics();

    for ( QListViewItemIterator it( m_dataTableBox ); it.current(); ++it )
    {
        DataTableBox::Item *i = static_cast<DataTableBox::Item *>( it.current() );
        m_lines[ i ] = lines( i, fm, width );
        const int height = baseHeight + ( fm.height() - fm.descent() ) * ( m_lines[ i ].count() - 1 );
        i->setHeight( height );
    }

    m_needsRefresh = false;
}

TreeViewMode::TreeViewMode( DataTableBox *b ) : CompactViewMode( b )
{}

TreeViewMode::~TreeViewMode()
{}

void TreeViewMode::setShown( bool show )
{
    CompactViewMode::setShown( show );
    dataTableBox() ->setRootIsDecorated( show );
}

#include "viewmode.moc"
