/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "relationeditordialogimpl.h"

#include "datafield.h"
#include "datatable.h"
#include "project.h"
#include "databaseconnection.h"

#include <qlistbox.h>
#include <qwidget.h>
#include <qcheckbox.h>
#include <qlineedit.h>

#include <qlabel.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qpushbutton.h>
#include <qmultilineedit.h>
#include <qlistview.h>
#include <qfeatures.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <limits.h>

#include <qsqlindex.h>
#include <qsqlcursor.h>
#include <qsqldatabase.h>

#include <kiconloader.h>
#include <kinputdialog.h>

RelationEditorDialog::RelationEditorDialog( DataField *dataField,
        DataTable *parentTable,
        bool configure,
        QWidget *parent,
        const char *name )
        : RelationEditorDialogBase( parent, name ),
        m_dataField( dataField ),
        m_parentTable( parentTable ),
        m_configure( configure )
{
    setMinimumSize( 800, 600 );
    setCaption( i18n("Relation Editor Dialog") );

    wizard1->setPixmap( UserIcon( "wizard_table_4" ) );
    buttonFieldUp->setPixmap( UserIcon( "up" ) );
    buttonFieldDown->setPixmap( UserIcon( "down" ) );

    setup();
}

RelationEditorDialog::~RelationEditorDialog()
{}

void RelationEditorDialog::currentFieldChanged( QListBoxItem *i )
{
    DataField * field = findField( i );
    if ( !i || !field )
    {
        checkBoxHidden->setEnabled( false );
        checkBoxForeign->setEnabled( false );
        checkBoxReport->setEnabled( false );
        comboTargetTable->setEnabled( false );
        comboTargetKey->setEnabled( false );
        comboTargetField->setEnabled( false );
        comboTargetConstraint->setEnabled( false );
        return ;
    }

    buttonDelete->setEnabled( !field->primary() );

    checkBoxHidden->setEnabled( !field->primary() );
    checkBoxHidden->setChecked( field->hidden() );
    checkBoxForeign->setEnabled( !field->primary() );
    checkBoxForeign->setChecked( field->foreign() );

    checkBoxReport->setEnabled( !field->primary() );
    checkBoxReport->setChecked( field->reporter() );

    comboTargetTable->setEnabled( false );
    comboTargetKey->setEnabled( false );
    comboTargetField->setEnabled( false );
    comboTargetConstraint->setEnabled( false );

    comboTargetTable->clear();
    comboTargetKey->clear();
    comboTargetField->clear();
    comboTargetConstraint->clear();

    foreignToggled( field->foreign() );
}

void RelationEditorDialog::hiddenToggled( bool toggled )
{
    DataField * field = findField( listBoxField->selectedItem() );
    if ( !field )
        return ;

    field->setHidden( toggled, false );

    if ( field->primary() || field->foreign() )
    {
        return ;
    }

    listBoxField->blockSignals( true );

    if ( field->hidden() )
        listBoxField->changeItem( UserIcon( "hidden" ), listBoxField->currentText(), listBoxField->currentItem() );
    else
        listBoxField->changeItem( UserIcon( "visible" ), listBoxField->currentText(), listBoxField->currentItem() );

    listBoxField->blockSignals( false );
}

void RelationEditorDialog::foreignToggled( bool toggled )
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || field->primary() )
        return ;

    if ( !toggled )
    {
        comboTargetTable->setEnabled( false );
        comboTargetKey->setEnabled( false );
        comboTargetField->setEnabled( false );
        comboTargetConstraint->setEnabled( false );
        comboTargetTable->clear();
        comboTargetKey->clear();
        comboTargetField->clear();
        comboTargetConstraint->clear();
        if ( field->relation() )
        {
            DataRelation *relation = field->relation();
            delete relation;
            relation = 0L;
        }
    }

    field->setForeign( toggled );
    listBoxField->blockSignals( true );

    if ( field->foreign() )
    {
        if ( !field->relation() )
            m_dataField->relation()->addDataRelation( field->name(),
                                                      QString::null,
                                                      QString::null,
                                                      QString::null,
                                                      QString::null,
                                                      QVariant::Type( 0 ), 0, 0
                                                      );

        DatabaseConnection * conn =
            m_parentTable->project() ->databaseConnection( m_parentTable->connection() );

        comboTargetTable->setEnabled( true );
        comboTargetTable->insertItem( i18n("<select target table>") );
        comboTargetTable->insertStringList( conn->tables() );

        if ( !field->relation() ->table().isEmpty() )
        {
            comboTargetTable->setCurrentText( field->relation() ->table() );
            targetTableActivated(); //Goto the next step
        }
        listBoxField->changeItem( UserIcon( "foreignkey" ), listBoxField->currentText(), listBoxField->currentItem() );
    }
    else if ( field->hidden() )
        listBoxField->changeItem( UserIcon( "hidden" ), listBoxField->currentText(), listBoxField->currentItem() );
    else
        listBoxField->changeItem( UserIcon( "visible" ), listBoxField->currentText(), listBoxField->currentItem() );

    listBoxField->blockSignals( false );
}

void RelationEditorDialog::reportToggled( bool toggled )
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field )
        return ;

    if ( toggled )
    {
        DataFieldList fields = m_dataField->relation() ->fieldList();
        for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
        {
            ( *it )->setReporter( false );
        }
    }

    field->setReporter( toggled );
}

void RelationEditorDialog::targetTableActivated()
{
    DatabaseConnection * conn =
        m_parentTable->project() ->databaseConnection( m_parentTable->connection() );
    comboTargetKey->clear();
    comboTargetField->clear();
    comboTargetKey->setEnabled( true );
    comboTargetField->setEnabled( true );
    comboTargetConstraint->setEnabled( true );
    comboTargetKey->insertItem( i18n("<select target key>") );
    comboTargetField->insertItem( i18n("<select target field>") );
    comboTargetConstraint->insertItem( i18n("<select target constraint>") );
    QStringList fields = conn->fields( comboTargetTable->currentText() );
    QStringList constraints = conn->fields( m_parentTable->tableName() );
    comboTargetKey->insertStringList( fields );
    comboTargetField->insertStringList( fields );
    comboTargetConstraint->insertStringList( constraints );

    DataField *field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || field->primary() || comboTargetTable->currentText() == i18n("<select target table>")  )
    {
        comboTargetKey->setEnabled( false );
        comboTargetField->setEnabled( false );
        comboTargetConstraint->setEnabled( false );
        comboTargetKey->clear();
        comboTargetField->clear();
        comboTargetConstraint->clear();
        return ;
    }

    field->relation() ->setTable( comboTargetTable->currentText() );
    field->relation() ->getPrimaryField()->setTable( comboTargetTable->currentText() );

    if ( !field->relation() ->key().isEmpty() &&
         fields.contains( field->relation() ->key() ) )
    {
        comboTargetKey->setCurrentText( field->relation() ->key() );
        targetKeyActivated(); //Goto the next step
    }
    if ( !field->relation() ->field().isEmpty() &&
         fields.contains( field->relation() ->field() ) )
    {
        comboTargetField->setCurrentText( field->relation() ->field() );
        targetFieldActivated(); //Goto the next step
    }
    if ( !field->relation() ->constraint().isEmpty() &&
         constraints.contains( field->relation() ->constraint() ) )
    {
        comboTargetConstraint->setCurrentText( field->relation() ->constraint() );
        targetConstraintActivated(); //Goto the next step
    }
}

void RelationEditorDialog::targetKeyActivated()
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || field->primary() || comboTargetKey->currentText() == i18n("<select target key>") )
        return ;

    field->relation() ->setKey( comboTargetKey->currentText() );
}

void RelationEditorDialog::targetFieldActivated()
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || field->primary() || comboTargetField->currentText() == i18n("<select target field>") )
        return ;

    field->relation() ->setField( comboTargetField->currentText() );

    DatabaseConnection * conn =
        m_parentTable->project() ->databaseConnection( m_parentTable->connection() );
    QVariant::Type fieldType = conn->getFieldType( field->relation() ->table(), field->relation() ->field() );
    field->relation() ->setType( fieldType );
}

void RelationEditorDialog::targetConstraintActivated()
{
    DataField * field = findField( listBoxField->item( listBoxField->currentItem() ) );
    if ( !field || field->primary() || comboTargetConstraint->currentText() == i18n("<select target constraint>") )
        return ;

    field->relation() ->setConstraint( comboTargetConstraint->currentText() );
}

void RelationEditorDialog::fieldUp()
{
    if ( listBoxField->currentItem() <= 0 ||
            listBoxField->count() < 2 )
        return ;
    int index = listBoxField->currentItem() - 1;

    //Do not allow an item to ascend the primary field
    if ( index == 0 )
        return ;

    QListBoxItem *i = listBoxField->item( listBoxField->currentItem() );
    DataField *field = findField( i );

    DataFieldList fields = m_dataField->relation() ->fieldList();
    for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
    {
        if ( ( *it ) ->number() == index )
            ( *it ) ->setNumber( index + 1 );
    }
    field->setNumber( index );

    listBoxField->takeItem( i );
    listBoxField->insertItem( i, index );
    listBoxField->setCurrentItem( i );
}

void RelationEditorDialog::fieldDown()
{
    if ( listBoxField->currentItem() == -1 ||
            listBoxField->currentItem() == ( int ) listBoxField->count() - 1 ||
            listBoxField->count() < 2 )
        return ;
    int index = listBoxField->currentItem() + 1;

    //Do not allow the primary field to descend
    if ( index == 1 )
        return ;

    QListBoxItem *i = listBoxField->item( listBoxField->currentItem() );
    DataField *field = findField( i );

    DataFieldList fields = m_dataField->relation() ->fieldList();
    for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
    {
        if ( ( *it ) ->number() == index )
            ( *it ) ->setNumber( index - 1 );
    }
    field->setNumber( index );

    listBoxField->takeItem( i );
    listBoxField->insertItem( i, index );
    listBoxField->setCurrentItem( i );
}

void RelationEditorDialog::newField()
{
    DatabaseConnection * conn =
        m_parentTable->project() ->databaseConnection( m_parentTable->connection() );

    QStringList lst = conn->fields( m_dataField->relation() ->table() );
    QString fieldName = KInputDialog::getItem(
                            i18n( "Choose Field" ), i18n( "Field:" ), lst, 0, false, 0, this );

    if ( fieldName.isEmpty() )
        return ;

    QVariant::Type fieldType = conn->getFieldType( m_dataField->relation() ->table(), fieldName );
    m_dataField->relation() ->addDataField(
        fieldName,                       //name
        fieldType,                       //type
        -1,                              //number
        -1,                              //editorwidth
        false,                           //foreign
        false,                           //primary
        false,                           //reporter
        false                            //hidden
    );
    listBoxField->insertItem( new QListBoxPixmap( UserIcon( "visible" ), fieldName ) );
    conn->close();

    if ( listBoxField->count() )
        buttonDelete->setEnabled( true );
}

void RelationEditorDialog::deleteField()
{
    int index = listBoxField->currentItem() + 1;

    QListBoxItem * i = listBoxField->item( listBoxField->currentItem() );
    DataField *field = findField( i );

    DataFieldList fields = m_dataField->relation() ->fieldList();
    for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
    {
        if ( ( *it ) ->number() == index )
            ( *it ) ->setNumber( index - 1 );
    }

    listBoxField->takeItem( i );
    m_dataField->relation() ->removeDataField( field );
    delete i;
    i = 0;

    if ( !listBoxField->count() )
        buttonDelete->setEnabled( false );
}

DataField *RelationEditorDialog::findField( QListBoxItem * i )
{
    if ( !i )
        return 0;

    DataFieldList fields = m_dataField->relation() ->fieldList();
    for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
    {
        if ( ( *it ) ->number() == listBoxField->index( i ) )
            return * it;
    }

    return 0;
}

void RelationEditorDialog::setup()
{
    listBoxField->clear();
    editTable->setText( m_dataField->relation() ->table() );
    editKey->setText( m_dataField->relation() ->key() );

    connect( comboTargetTable, SIGNAL( activated( int ) ), this, SLOT( targetTableActivated() ) );
    connect( comboTargetKey, SIGNAL( activated( int ) ), this, SLOT( targetKeyActivated() ) );
    connect( comboTargetField, SIGNAL( activated( int ) ), this, SLOT( targetFieldActivated() ) );
    connect( comboTargetConstraint, SIGNAL( activated( int ) ), this, SLOT( targetConstraintActivated() ) );

    DataFieldList fields = m_dataField->relation() ->fieldList();
    for ( DataFieldList::Iterator it = fields.begin(); it != fields.end(); ++it )
    {
        if ( ( *it ) ->primary() )
            listBoxField->insertItem( new QListBoxPixmap( UserIcon( "primarykey" ), ( *it ) ->name() ), ( *it ) ->number() );
        else if ( ( *it ) ->foreign() )
            listBoxField->insertItem( new QListBoxPixmap( UserIcon( "foreignkey" ), ( *it ) ->name() ), ( *it ) ->number() );
        else if ( ( *it ) ->hidden() )
            listBoxField->insertItem( new QListBoxPixmap( UserIcon( "hidden" ), ( *it ) ->name() ), ( *it ) ->number() );
        else
            listBoxField->insertItem( new QListBoxPixmap( UserIcon( "visible" ), ( *it ) ->name() ), ( *it ) ->number() );
    }

    if ( listBoxField->count() )
        buttonDelete->setEnabled( true );
}

void RelationEditorDialog::accept()
{
    RelationEditorDialogBase::accept();
}

#include "relationeditordialogimpl.moc"
