/* dc_gui2 - a GTK+2 GUI for DCTC
 * Copyright (C) 2002 Eric Prevoteau
 *
 * hublist_converter.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hublist_converter.c,v 1.4 2003/12/26 14:35:04 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include "hublist_converter.h"
#include "misc.h"

static int comp_hub_entry(const void *a, const void *b)
{
	return strcmp(((HUB_ENTRY*)a)->hubaddr, ((HUB_ENTRY*)b)->hubaddr);
}

/*********************************************************/
/* tell which entry contains better information of a hub */
/*********************************************************/
/* output: <0 => h1 is better */
/*         =0 => undefined    */
/*         >0 => h2 is better */
/******************************/
static int which_is_better(const HUB_ENTRY *h1,const HUB_ENTRY *h2)
{
	/* criteria: */
	/* = minshare */
	/* = maxusers */
	/* = minslots */
	/* = shares */
	/* = maxhub */
	/* = country */
	/* = status */
	/* ignored: rating, reliability */

#define COMPARE_VAL(v1,v2,undefined_val)			if((v1)!=(v2)){if((v2)==(undefined_val)){return -1;}else{return 1;}}
	COMPARE_VAL(h1->minshare,h2->minshare,0)
	COMPARE_VAL(h1->maxusers,h2->maxusers,0)
	COMPARE_VAL(h1->minslots,h2->minslots,0)
	COMPARE_VAL(h1->shares,h2->shares,0)
	COMPARE_VAL(h1->maxhub,h2->maxhub,0)

	if(strcmp(h1->country,h2->country))		/* country differs ? */
	{
		if(strlen(h2->country)==0)				/* h2 has no country => prefers 1st */
			return -1;
		else
			return 1;
	}

	if(strcmp(h1->status,h2->status))		/* status differs ? */
	{
		if(strlen(h2->status)==0)				/* h2 has no status => prefers 1st */
			return -1;
		else
			return 1;
	}
	
	return 0;	/* no entry is better than the other */
}

/**************************************************************/
/* convert the output of 'hublist' into a GArray of HUB_ENTRY */
/**************************************************************/
/* hub addresses are uniq in the array */
/*************************************************/
/* The returned array must be free by the caller */
/*************************************************/
GArray *hublist_output_to_hub_entry_array(const char *hublist_output)
{
	GArray *ga;
	gchar **row=NULL;
	int i;

	ga=g_array_new(FALSE,FALSE,sizeof(HUB_ENTRY));
	
	row=g_strsplit(hublist_output,"\r\n",0); /* expand data (warning, this means a "\r" followed by a "\n", not ("\r" or "\n")) */
	if(row)
	{
		/* insert only valid rows in the HUB_ENTRY array */
		for(i=0;row[i]!=NULL;i++)
		{
			gchar **splitted;
			
			splitted=g_strsplit(row[i],"|",0);		/* expand row */
			if(splitted)
			{
				int ln;

				ln=size_of_null_array((void**)splitted);

				if(ln>=4)
				{
					HUB_ENTRY he;
					
					/* it is the mandatory part */
					strncpy_max(he.hubname,splitted[0],sizeof(he.hubname));
					strncpy_max(he.hubaddr,splitted[1],sizeof(he.hubaddr));
					strncpy_max(he.hubdesc,splitted[2],sizeof(he.hubdesc));
					he.nb_users=atoi(splitted[3]);

					/* optional part */
					strncpy_max(he.country,((ln>=5)?splitted[4]:""),sizeof(he.country));

					he.shares=0;
					if((ln>=6)&&(strlen(splitted[5])))
						sscanf(splitted[5],"%llu",&(he.shares));

					strncpy_max(he.status,((ln>=7)?splitted[6]:""),sizeof(he.status));

					he.minshare=0;
					if((ln>=8)&&(strlen(splitted[7])))
						sscanf(splitted[7],"%llu",&(he.minshare));

					he.minslots=0;
					if((ln>=9)&&(strlen(splitted[8])))
						sscanf(splitted[8],"%u",&(he.minslots));

					he.maxhub=0;
					if((ln>=10)&&(strlen(splitted[9])))
						sscanf(splitted[9],"%u",&(he.maxhub));

					he.maxusers=0;
					if((ln>=11)&&(strlen(splitted[10])))
						sscanf(splitted[10],"%u",&(he.maxusers));

					he.reliability=-1;
					if((ln>=12)&&(strlen(splitted[11])))
						sscanf(splitted[11],"%f",&(he.reliability));

					if(ln>=13)
					{
						he.rating=strlen(splitted[12]);
					}
					else
						he.rating=-1;
					
					g_array_append_val(ga,he);
				}
			}
			g_strfreev(splitted);
		}
		g_strfreev(row);

		/* sanitize the HUB_ENTRY array */
		if(ga->len!=0)
		{
			qsort(ga->data,ga->len,sizeof(HUB_ENTRY), comp_hub_entry);

			for(i=ga->len-2;i>=0;i--)
			{
				HUB_ENTRY *h1,*h2;

				h1=&(g_array_index(ga,HUB_ENTRY,i));
				h2=&(g_array_index(ga,HUB_ENTRY,i+1));

				if(!strcmp(h1->hubaddr,
								h2->hubaddr))
				{
					if(which_is_better(h1,h2)<0)
						g_array_remove_index_fast(ga,i+1);	/* the 1st is better, remove the 2nd */
					else
						g_array_remove_index_fast(ga,i);	/* the 2nd is better, remove the 1st */
				}
			}
		}
	}

	return ga;
}

