#ifndef ADD_BUTTONS_H
#define ADD_BUTTONS_H

/*
 * Menu buttons to select tags
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtkmm/menu.h>
#include "TagMenu.h"
//#include "DebtagsDocument.h"
#include <gtkmm/eventbox.h>

template<typename Doc>
class AddButton : public Gtk::Button
{
public:
	typedef SigC::Signal1<void, typename Doc::Tag> type_signal_selected;

protected:
	type_signal_selected _signal_selected;

	typedef typename Doc::Facet Facet;
	typedef typename Doc::Tag Tag;
	typedef typename Doc::Package Package;

protected:
	virtual bool invoke(GdkEvent* e) = 0;

	Doc& doc;

	void do_selected(Tag tag)
	{
		_signal_selected.emit(tag);
	}

public:
	AddButton(Doc& doc, const std::string& label) : Gtk::Button(label), doc(doc)
	{
		add_events(Gdk::BUTTON_PRESS_MASK);
		signal_event().connect(sigc::mem_fun(*this, &AddButton<Doc>::invoke));
	}

	type_signal_selected signal_selected() throw () { return _signal_selected; }
};

template<typename Doc, typename Sel>
class AddAllButton : public AddButton<Doc>
{
protected:
	typedef typename Doc::Facet Facet;
	typedef typename Doc::Tag Tag;
	typedef typename Doc::Package Package;

	virtual bool invoke(GdkEvent* e)
	{
		if (e->type == GDK_BUTTON_PRESS)
		{
			TagMenu<Doc>* addMenu = manage(new TagMenu<Doc>());
			addMenu->populateUnselected(this->doc, this->sel.currentTags());
			addMenu->signal_selected().connect(sigc::mem_fun(*this, &AddAllButton<Doc, Sel>::do_selected));
			addMenu->popup(e->button.button, e->button.time);
			return true;
		}
		return false;
	}

	const Sel& sel;

public:
	AddAllButton(Doc& doc, const Sel& sel, const std::string& label)
		: AddButton<Doc>(doc, label), sel(sel) {}
};

template<typename Doc, typename Sel>
AddAllButton<Doc, Sel>* newAddAllButton(Doc& doc, const Sel& sel, const std::string& label)
{
	return new AddAllButton<Doc, Sel>(doc, sel, label);
}

template<typename Doc, typename Sel>
class AddCompanionButton : public AddButton<Doc>
{
protected:
	typedef typename Doc::Facet Facet;
	typedef typename Doc::Tag Tag;
	typedef typename Doc::Package Package;

	virtual bool invoke(GdkEvent* e)
	{
		if (e->type == GDK_BUTTON_PRESS)
		{
			TagMenu<Doc>* addMenu = manage(new TagMenu<Doc>());
			if (this->doc.getComputeIntensive())
			{
				addMenu->populateAvailableSmart(this->doc, this->sel.currentTags(), this->doc.subCollection);
			} else {
				addMenu->populateAvailable(this->doc, this->sel.currentTags(), this->doc.subCollection);
			}
			addMenu->signal_selected().connect(sigc::mem_fun(*this, &AddCompanionButton<Doc, Sel>::do_selected));
			addMenu->popup(e->button.button, e->button.time);
			return true;
		}
		return false;
	}

	const Sel& sel;

public:
	AddCompanionButton(Doc& doc, const Sel& sel, const std::string& label)
		: AddButton<Doc>(doc, label), sel(sel) {}
};

template<typename Doc, typename Sel>
AddCompanionButton<Doc, Sel>* newAddCompanionButton(Doc& doc, const Sel& sel, const std::string& label)
{
	return new AddCompanionButton<Doc, Sel>(doc, sel, label);
}

template<typename Doc, typename Sel>
class AddDiscriminantButton : public AddButton<Doc>
{
protected:
	typedef typename Doc::Facet Facet;
	typedef typename Doc::Tag Tag;
	typedef typename Doc::Package Package;

	virtual bool invoke(GdkEvent* e)
	{
		if (e->type == GDK_BUTTON_PRESS)
		{
			std::vector<Tag> tags = this->doc.subCollection.tagsInDiscriminanceOrder();
			std::set<Tag> current = this->sel.currentTags();
			Gtk::Menu* addMenu = manage(new Gtk::Menu());

			int count = 0;
			for (typename std::vector<Tag>::const_reverse_iterator i = tags.rbegin();
					i != tags.rend(); ++i)
			{
				// Only show tags that are not in the package yet
				if (current.find(*i) != current.end())
					continue;
				if (count >= 21)
					break;

				addMenu->items().push_back(Gtk::Menu_Helpers::MenuElem(
							"[" + i->fullname() + "] " + i->shortDescription(),
							sigc::bind<Tag>(
								sigc::mem_fun(*this, &AddDiscriminantButton<Doc, Sel>::do_selected), *i)));
				++count;
			}

			addMenu->popup(e->button.button, e->button.time);
			return true;
		}
		return false;
	}

	const Sel& sel;

public:
	AddDiscriminantButton(Doc& doc, const Sel& sel, const std::string& label)
		: AddButton<Doc>(doc, label), sel(sel) {}
};

template<typename Doc, typename Sel>
AddDiscriminantButton<Doc, Sel>* newAddDiscriminantButton(Doc& doc, const Sel& sel, const std::string& label)
{
	return new AddDiscriminantButton<Doc, Sel>(doc, sel, label);
}

template<typename Doc, typename Sel>
class AddRelevantButton : public AddButton<Doc>
{
protected:
	typedef typename Doc::Facet Facet;
	typedef typename Doc::Tag Tag;
	typedef typename Doc::Package Package;

	virtual bool invoke(GdkEvent* e)
	{
		if (e->type == GDK_BUTTON_PRESS)
		{
			std::vector<Tag> tags = this->doc.subCollection.tagsInRelevanceOrder(this->doc.debtags());
			std::set<Tag> current = sel.currentTags();
			Gtk::Menu* addMenu = manage(new Gtk::Menu());

			int count = 0;
			for (typename std::vector<Tag>::const_reverse_iterator i = tags.rbegin();
					i != tags.rend(); ++i)
			{
				// Only show tags that are not in the package yet
				if (current.find(*i) != current.end())
					continue;
				if (count >= 21)
					break;

				addMenu->items().push_back(Gtk::Menu_Helpers::MenuElem(
							"[" + i->fullname() + "] " + i->shortDescription(),
							sigc::bind<Tag>(
								sigc::mem_fun(*this, &AddRelevantButton<Doc, Sel>::do_selected), *i)));
				++count;
			}

			addMenu->popup(e->button.button, e->button.time);
			return true;
		}
		return false;
	}

	const Sel& sel;

public:
	AddRelevantButton(Doc& doc, const Sel& sel, const std::string& label)
		: AddButton<Doc>(doc, label), sel(sel) {}
};

template<typename Doc, typename Sel>
AddRelevantButton<Doc, Sel>* newAddRelevantButton(Doc& doc, const Sel& sel, const std::string& label)
{
	return new AddRelevantButton<Doc, Sel>(doc, sel, label);
}

// vim:set ts=4 sw=4:
#endif
