/*
 * Tagged Collection document for a Document-View editing architecture
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 3B
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "DebtagsDocument.h"
#include "Environment.h"

#include <wibble/operators.h>
#include <wibble/regexp.h>
#include <cctype>

using namespace std;
using namespace ept;
using namespace ept::debtags;

template<typename OUT>
class FilterBesidesTags : public wibble::mixin::OutputIterator< FilterBesidesTags<OUT> >
{
	const Filter& filter;
	OUT out;

public:
	FilterBesidesTags<OUT>(const Filter& filter, const OUT& out) : filter(filter), out(out) {}

	template<typename ITEMS, typename TAGS>
	FilterBesidesTags<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set<string> filtered;
		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
		{
			if (filter.matchBesidesTags(*i, data.second))
				filtered.insert(*i);
		}
		if (!filtered.empty())
		{
			*out = make_pair(filtered, data.second);
			++out;
		}
		return *this;
	}
};

template<typename OUT>
FilterBesidesTags<OUT> filterBesidesTags(const Filter& filter, const OUT& out)
{
	return FilterBesidesTags<OUT>(filter, out);
}

template<typename OUT>
class TagAdder : public wibble::mixin::OutputIterator< TagAdder<OUT> >
{
	DebtagsDocument& doc;
	OUT out;

public:
	TagAdder(DebtagsDocument& doc, const OUT& out) : doc(doc), out(out) {}

    TagAdder& operator=(const std::string& pkg);
};

template<typename OUT>
TagAdder<OUT> tagAdder(DebtagsDocument& doc, const OUT& out)
{
	return TagAdder<OUT>(doc, out);
}

template<typename OUT>
TagAdder<OUT>& TagAdder<OUT>::operator=(const std::string& pkg)
{
	using namespace std;
	using namespace ept::debtags;

	set<Tag> tags = doc.debtags().getTagsOfItem(pkg);
	if (tags.empty())
	{
		int idx = tolower(pkg[0]) - 'a';
		if (doc.nyt.valid()) tags.insert(doc.nyt);
		if (idx >= 0 && idx < 26 && doc.nytx[idx].valid())
			tags.insert(doc.nytx[idx]);
	}
	*out = make_pair(wibble::singleton(pkg), tags);
	++out;
	return *this;
}

DebtagsDocument::DebtagsDocument() :
	_debtags(true), _changed(false),
	_lastPatch(tagcoll::Patch<Package, Tag>(Package())),
	_undoTail(0), /*specialsGen(*this),*/ computeIntensive(false),
	_filter(*this)/*, specials(100)*/, hasSpecials(NO)
{
	nyt = vocabulary().tagByName("special::not-yet-tagged");
	for (int i = 0; i < 26; ++i)
	{
		nytx[i] = vocabulary().tagByName(string("special::not-yet-tagged::") + (char)('a' + i));
		//cerr << (char)('a' + i) << ": " << nytx[i].valid() << endl;
	}
#if 0
	regenerateSpecials();
#endif
}

DebtagsDocument::~DebtagsDocument()
{
#if 0
	MutexLock lock(specialsMutex);
	if (hasSpecials == GENERATING)
	{
		specialsGen.interrupt();
		specialsMutex.unlock();
		specialsGen.join();
		specialsMutex.lock();
	}
#endif
}

//#define AUTO(y, x...) typeof(x) y = x

string toLower(const std::string& s)
{
	string res;
	for (string::const_iterator i = s.begin(); i != s.end(); ++i)
		res += tolower(*i);
	return res;
}

Filter::Filter(DebtagsDocument& doc)
	: doc(doc), apt(doc.apt()), debtags(doc.debtags()), wantedState(0), hasExpression(false) {}

void Filter::setExpression(const tagcoll::Expression& expr)
{
	hasExpression = true;
	tagExpression = expr;
}

void Filter::addName(const std::string& str)
{
	wibble::Tokenizer tok(str, "[^[:blank:]]+", REG_EXTENDED);
	for (wibble::Tokenizer::const_iterator i = tok.begin();
			i != tok.end(); ++i)
		namePatterns.push_back(toLower(*i));
}

void Filter::addDescription(const std::string& str)
{
	wibble::Tokenizer tok(str, "[^[:blank:]]+", REG_EXTENDED);
	for (wibble::Tokenizer::const_iterator i = tok.begin();
			i != tok.end(); ++i)
		patterns.push_back(toLower(*i));
}

void Filter::addMaintainer(const std::string& str)
{
	wibble::Tokenizer tok(str, "[^[:blank:]]+", REG_EXTENDED);
	for (wibble::Tokenizer::const_iterator i = tok.begin();
			i != tok.end(); ++i)
		maintPatterns.push_back(toLower(*i));
}

void Filter::addTags(const std::set<ept::debtags::Tag>& tags)
{
	using namespace wibble::operators;
	wantedTags |= tags;
}

bool Filter::matchesName(const std::string& name) const
{
	using namespace std;
	string n = toLower(name);
	for (vector<string>::const_iterator i = namePatterns.begin();
			i != namePatterns.end(); ++i)
		if (n.find(*i) == string::npos)
			return false;
	return true;
}

bool Filter::matchesPattern() const
{
	using namespace std;
	string desc = toLower(rec.package()) + " " + toLower(rec.description());
	for (vector<string>::const_iterator i = patterns.begin();
			i != patterns.end(); ++i)
		if (desc.find(*i) == string::npos)
			return false;
	return true;
}

bool Filter::matchesMaint() const
{
	using namespace std;
	string maint = toLower(rec.maintainer());
	for (vector<string>::const_iterator i = maintPatterns.begin();
			i != maintPatterns.end(); ++i)
		if (maint.find(*i) == string::npos)
			return false;
	return true;
}

bool Filter::matchesState() const
{
	if ((wantedState & Installed) && !pstate.isInstalled())
		return false;
	if ((wantedState & NotInstalled) && pstate.isInstalled())
		return false;
	return true;
}

template<typename OUT>
void Filter::output(OUT out) const
{
	if (!wantedTags.empty())
	{
		// We have wantedTags, so we feed the generation with the Debtags
		// subset
		debtags.outputHavingTags(wantedTags, filterBesidesTags(*this, out));
	} else if (!namePatterns.empty() || !patterns.empty() || !maintPatterns.empty() || hasExpression) {
		TagAdder<OUT> adder(doc, out);
		// No wantedTags, so we feed the generation with the Apt cache
		for (ept::apt::Apt::record_iterator i = apt.recordBegin();
				i != apt.recordEnd(); ++i)
		{
			rec.scan(*i);
			if (matchBesidesTags())
			{
				*adder = rec.package();
				++adder;
			}
		}
	} else {
		// No filters at all, we can quickly iterate all package names
		std::copy(apt.begin(), apt.end(), tagAdder(doc, out));
	}
}


void DebtagsDocument::commitFilter()
{
	subCollection.clear();
	_filter.output(inserter(subCollection));
}

/*
void DebtagsDocument::setFilter(const Predicate& p)
{
	_filter = p;

	subCollection.clear();
	if (_filter == ept::predicate::True<Package>())
		std::copy(_agg.index().begin(), _agg.index().end(), TagAdder(*this));
	else
	{
		AUTO(r, filteredRange(_agg.index().range(), _filter));
		std::copy(r.begin(), r.end(), TagAdder(*this));
	}
}
*/

#if 0
bool DebtagsDocument::SpecialsGen::advance(int steps) throw ()
{
	fprintf(stderr, "SpecialsGen thread: %d step(s)\n", steps);

	return !_interrupt;
}

template<class ITEM>
class TFLimit : public Tagcoll::Filter<ITEM, Tag>
{
protected:
	int count;

	void consumeItemUntagged(const ITEM& pkg) { if (count > 0) { this->consumer->consume(pkg); count--; } }
	void consumeItem(const ITEM& pkg, const Tagcoll::OpSet<Tag>& tags)
	{
		if (count > 0) { this->consumer->consume(pkg, tags); count--; }
	}

public:
	TFLimit(int count) : count(count) {}
	TFLimit(int count, Tagcoll::Consumer<ITEM, Tag>& cons)
		: Tagcoll::Filter<ITEM, Tag>(cons), count(count) {}
};

void* DebtagsDocument::SpecialsGen::main() throw ()
{
	time_t start = time(NULL);
	fprintf(stderr, "SpecialsGen thread start\n");
	{
		MutexLock lock(doc.specialsMutex);
		doc.hasSpecials = GENERATING;
	}

	doc.specials.clear();
	fprintf(stderr, "SpecialsGen thread: specials cleared\n");
	if (_interrupt) return 0;
	if (0)
	{
		// Limit the number of packages, to have decent times during debugging
		TFLimit<package> limFilter(100, &doc.specials);
		coll.output(limFilter);
	} else {
		coll.output(doc.specials);
	}
	fprintf(stderr, "SpecialsGen thread: specials fed data\n");
	if (_interrupt) return 0;
	doc.specials.compute(this);
	fprintf(stderr, "SpecialsGen thread: specials computed\n");

	{
		MutexLock lock(doc.specialsMutex);
		if (doc.specials.isComputed())
			doc.hasSpecials = YES;
		else
		{
			doc.specials.clear();
			doc.hasSpecials = NO;
		}
	}
	int elapsed = time(NULL) - start;
	fprintf(stderr, "SpecialsGen thread end (%d seconds)\n", elapsed);
	return 0;
}

void DebtagsDocument::regenerateSpecials() throw ()
{
	return;
	{
		MutexLock lock(specialsMutex);
		if (hasSpecials == GENERATING)
		{
			specialsGen.interrupt();
			specialsMutex.unlock();
			specialsGen.join();
			specialsMutex.lock();
		}
		hasSpecials = GENERATING;
	}
	specialsGen.interrupt(false);
	specialsGen.coll = Tagcoll::ItemGrouper<aptFront::cache::entity::Package, Tag>();
	tagdb().output(specialsGen.coll);
	specialsGen.start();
}

void DebtagsDocument::wantSpecials() throw ()
{
	if (hasSpecials == NO)
	{
		specialsMutex.unlock();
		regenerateSpecials();
		specialsMutex.lock();
	}

	if (hasSpecials == GENERATING)
	{
		specialsMutex.unlock();
		specialsGen.join();
		specialsMutex.lock();
	}
}
#endif

void DebtagsDocument::do_changed()
{
	fprintf(stderr, "changed\n");
	if (computeIntensive)
	{
#if 0
		regenerateSpecials();
#endif
	}

	_signal_changed.emit();
}

void DebtagsDocument::do_reselected()
{
	fprintf(stderr, "reselected\n");

	_signal_reselected.emit();
}

bool DebtagsDocument::canUndo() const
{
	return _undoTail > 0;
}

bool DebtagsDocument::canRedo() const
{
	return _undoTail < _undoBuffer.size();
}

bool DebtagsDocument::canReapply(const Package& pkg) const
{
	return _lastPatch.item != Package() &&
		   _lastPatch.item != pkg;
}

void DebtagsDocument::undo()
{
	if (!canUndo())
		return;

	debug("Pre undo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	_undoTail--;
	debtags().applyChange(_undoBuffer[_undoTail]);
	_undoBuffer[_undoTail] = _undoBuffer[_undoTail].getReverse();

	debug("Post undo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	do_changed();
}

void DebtagsDocument::redo()
{
	if (!canRedo())
		return;

	debug("Pre redo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	debtags().applyChange(_undoBuffer[_undoTail]);
	_undoBuffer[_undoTail] = _undoBuffer[_undoTail].getReverse();
	_undoTail++;


	debug("Post redo: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

	do_changed();
}

void DebtagsDocument::reapply(const Package& pkg)
{
	if (!canReapply(pkg))
		return;

	// Build the new patch
	tagcoll::Patch<Package, Tag> newPatch(
			pkg,
			_lastPatch.added,
			_lastPatch.removed);

	applyChange(newPatch);
}

void DebtagsDocument::setCurrent(const Package& pkg)
{
	if (_current != pkg)
	{
		_current = pkg;
		do_reselected();
	}
}

void DebtagsDocument::applyChange(const tagcoll::Patch<Package, Tag>& change)
{
	tagcoll::PatchList<Package, Tag> patchList;
	patchList.addPatch(change);
	applyChange(patchList);
}

void DebtagsDocument::applyChange(const tagcoll::PatchList<Package, Tag>& change)
{
	debtags().applyChange(change);
	
	if (1 /*reverse != change*/)
	{
		debug("Pre change: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

		// Push the reverse change in the undo buffer
		if (_undoTail == _undoBuffer.size())
		{
			_undoBuffer.push_back(change.getReverse());
			_undoTail++;
		}
		else
		{
			_undoBuffer[_undoTail++] = change.getReverse();
			_undoBuffer.resize(_undoTail);
		}

		// Save 'last patch' if it's only about one package
		if (change.size() == 1)
			_lastPatch = change.begin()->second;
		else
			_lastPatch = tagcoll::Patch<Package, Tag>(Package());

		debug("Post change: tail: %d, size: %d\n", _undoTail, _undoBuffer.size());

		_changed = true;
			
		do_changed();
	} else
		debug("Change had no effect\n");
}

/*
template<class T>
void DebtagsDocument<T>::load()
	throw (FileException, ParserException)
{
	//_collection.clear();
	DebtagsEnvironment::get().outputPatched(_collection);

	// Output the merged collection to consumer
	/ *
	TagcollBuilder builder(_handleMaker);
	merger.output(builder);
	_collection = builder.collection();
	* /
	_undoBuffer.clear();
	_undoTail = 0;
	_fileName = "";

	do_filename_changed();
	do_changed();
}
*/

void DebtagsDocument::save() throw (wibble::exception::File)
{
	debtags().savePatch();
	_changed = false;
	//DebtagsEnvironment::get().savePatch(_collection);
	/*
	FILE* out = fopen(file.c_str(), "wt");
	if (!out)
		throw FileException(errno, "opening file " + file);
	
	// Output the collection, grouping the items
	TagcollSerializer serializer(out);
	ItemGrouper<string> grouper;
	output(grouper);
	grouper.output(serializer);

	fclose(out);

	if (file != _fileName)
	{
		_fileName = file;
		do_filename_changed();
	}
	*/
}

void DebtagsDocument::send() throw (wibble::exception::File)
{
	debtags().sendPatch();
}

#include <tagcoll/coll/fast.tcc>
#include <ept/debtags/debtags.tcc>

// vim:set ts=4 sw=4:
