#ifndef DEBTAGS_DOCUMENT_H
#define DEBTAGS_DOCUMENT_H

/*
 * Tagged Collection document for a Document-View editing architecture
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sigc++/sigc++.h>

#include <ept/apt/apt.h>
#include <ept/apt/packagerecord.h>
#include <ept/debtags/tag.h>
#include <ept/debtags/debtags.h>
#include <ept/debtags/expression.h>

#include <tagcoll/patch.h>
#include <tagcoll/coll/fast.h>

#include <wibble/sys/mutex.h>

#include <vector>

class DebtagsDocument;

class Filter
{
protected:
	DebtagsDocument& doc;
	ept::apt::Apt& apt;
	ept::debtags::Debtags& debtags;

	std::vector<std::string> namePatterns;
	std::vector<std::string> patterns;
	std::vector<std::string> maintPatterns;
	std::set<ept::debtags::Tag> wantedTags;
	unsigned int wantedState;
	tagcoll::Expression tagExpression;
	bool hasExpression;

	// Package record parser, shared here to avoid recreating it all the time
	mutable ept::apt::PackageRecord rec;
	mutable ept::apt::PackageState pstate;

public:
	Filter(DebtagsDocument& doc);

	enum Query {
		Installed = 1 << 0,
		NotInstalled = 1 << 1
	};

	void clear()
	{
		namePatterns.clear();
		patterns.clear();
		maintPatterns.clear();
		wantedTags.clear();
		hasExpression = false;
		wantedState = 0;
	}

	void unsetExpression() { hasExpression = false; }
	void setExpression(const tagcoll::Expression& expr);

	void unsetName() { namePatterns.clear(); }
	void addName(const std::string& str);

	void unsetDescription() { patterns.clear(); }
	void addDescription(const std::string& str);

	void unsetMaintainer() { maintPatterns.clear(); }
	void addMaintainer(const std::string& str);

	void usetTags() { wantedTags.clear(); }
	void addTags(const std::set<ept::debtags::Tag>& tags);

	void resetState() { wantedState = 0; }
	void addState(unsigned int s) { wantedState |= s; }

	bool matchesName(const std::string& name) const;
	bool matchesPattern() const;
	bool matchesMaint() const;
	bool matchesState() const;

	template<typename TAGS>
	bool matchBesidesTags(const std::string& pkg, const TAGS& tags) const
	{
		if (!namePatterns.empty() && !matchesName(pkg))
			return false;
		if (hasExpression && !tagExpression(tags))
			return false;
		if (!patterns.empty() || !maintPatterns.empty())
		{
			rec.scan(apt.rawRecord(pkg));
			if (!patterns.empty() && !matchesPattern())
				return false;
			if (!maintPatterns.empty() && !matchesMaint())
				return false;
		}
		if (wantedState)
		{
			pstate = apt.state(pkg);
			if (!matchesState())
				return false;
		}
		return true;
	}

	bool matchBesidesTags() const
	{
		// Assume that the PackageRecord has already been scanned
		if (!namePatterns.empty() && !matchesName(rec.package()))
			return false;
		if (hasExpression && !tagExpression(debtags.getTagsOfItem(rec.package())))
			return false;
		if (!patterns.empty() && !matchesPattern())
			return false;
		if (!maintPatterns.empty() && !matchesMaint())
			return false;
		if (wantedState)
		{
			pstate = apt.state(rec.package());
			if (!matchesState())
				return false;
		}
		return true;
	}

	template<typename OUT>
	void output(OUT out) const;
};

class DebtagsDocument
{
public:
	typedef SigC::Signal0<void> type_signal_changed;
	typedef SigC::Signal0<void> type_signal_reselected;
	typedef SigC::Signal0<void> type_signal_filename_changed;

	typedef ept::apt::Apt Apt;
	typedef ept::debtags::Debtags Debtags;
	typedef ept::debtags::Vocabulary Vocabulary;

	typedef std::string Package;
	typedef ept::debtags::Facet Facet;
	typedef ept::debtags::Tag Tag;

	//typedef ept::predicate::Predicate<Package> Predicate;

protected:
#if 0
	class SpecialsGen : public Thread, public Debtags::Status
	{
	protected:
		DebtagsDocument& doc;
		Tagcoll::ItemGrouper<package, Tag> coll;
		bool _interrupt;

		virtual const char* threadTag() { return "SpecialsGen"; }

		virtual void* main() throw ();

	public:
		SpecialsGen(DebtagsDocument& doc) : doc(doc), _interrupt(false) {}
		virtual ~SpecialsGen() throw () {}

		void interrupt(bool val = true) throw () { _interrupt = val; }
		bool advance(int steps = 1) throw ();
	
		friend class DebtagsDocument;
	};
#endif

	Apt _apt;
	Debtags _debtags;
	
	bool _changed;

	Package _current;

	tagcoll::Patch<Package, Tag> _lastPatch;
	std::vector<tagcoll::PatchList<Package, Tag> > _undoBuffer;
	unsigned int _undoTail;

	type_signal_changed _signal_changed;
	type_signal_reselected _signal_reselected;
	type_signal_filename_changed _signal_filename_changed;

	//OpSet<T> handlesToItems(const OpSet<int>& handles) const throw ();
	
#if 0
	SpecialsGen specialsGen;
#endif

	bool computeIntensive;

	Filter _filter;
	
public:
	DebtagsDocument();
	virtual ~DebtagsDocument();

	Apt& apt() { return _apt; }
	const Apt& apt() const { return _apt; }

	Debtags& debtags() { return _debtags; }
	const Debtags& debtags() const { return _debtags; }

	Vocabulary& vocabulary() { return _debtags.vocabulary(); }
	const Vocabulary& vocabulary() const { return _debtags.vocabulary(); }

	const tagcoll::PatchList<Package, Tag> getPatch() const
	{
		return debtags().changes();
	}

	bool getComputeIntensive() const throw () { return computeIntensive; }
	void setComputeIntensive(bool val) throw ()
	{
		if (!computeIntensive && val)
		{
			computeIntensive = val;
#if 0
			//regenerateSpecials();
#endif
		} else
			computeIntensive = val;
	}
	
	//HandleMaker<T>& handles() throw () { return _handleMaker; }
	//Debtags::Packages& collection() throw () { return _collection; }
	//const Debtags::Packages& collection() const throw () { return _collection; }

	bool changed() const { return _changed; }

	Package& current() { return _current; }
	const Package& current() const { return _current; }

	std::set<Tag> currentTags() const
	{
		return debtags().getTagsOfItem(_current);
	}

	void setCurrent(const Package& pkg);

	Filter& filter() { return _filter; }
	const Filter& filter() const { return _filter; }
	void commitFilter();

	// Apply a change, appending the reverse change in the undo buffer
	void applyChange(const tagcoll::Patch<Package, Tag>& change);
	void applyChange(const tagcoll::PatchList<Package, Tag>& change);

	bool canUndo() const;
	bool canRedo() const;
	bool canReapply(const Package& pkg) const;

	void undo();
	void redo();
	void reapplyToCurrent() { reapply(current()); }
	void reapply() { reapply(current()); }
	void reapply(const Package& pkg);

	//void load() throw (FileException, ParserException);
	void save() throw (wibble::exception::File);
	void send() throw (wibble::exception::File);

	//void output(TagcollConsumer<T>& consumer) throw ();
	//void output(TagcollConsumer<int>& consumer) throw ();

	// Results of the package filter
	//Debtags::PackageSet currentPackages;
	tagcoll::coll::Fast<Package, Tag> subCollection;

	//Debtags::Specials<Package> specials;
	enum { YES, NO, GENERATING } hasSpecials;
	wibble::sys::Mutex specialsMutex;

	// Cached not-yet-tagged tags
	Tag nyt;
	Tag nytx[26];

	/**
	 * Fork a subthread to regenerate the specials.
	 *
	 * If a previous subthread is stull running, it is interrupted and restarted.
	 */
#if 0
	void regenerateSpecials() throw ();

	/**
	 * Request the specials to be fully generated
	 *
	 * Must be called when specialsMutex is locked.  On exit, the specials are
	 * guaranteed to have been generated.
	 */
	void wantSpecials() throw ();
#endif

	type_signal_changed signal_changed() throw () { return _signal_changed; }
	type_signal_reselected signal_reselected() throw () { return _signal_reselected; }
	type_signal_filename_changed signal_filename_changed() throw () { return _signal_filename_changed; }

	virtual void do_changed();
	virtual void do_reselected();

	friend class SpecialsGen;
};


// vim:set ts=4 sw=4:
#endif
