/*
 * A single view on a TagcollDocument
 *
 * Copyright (C) 2003--2007  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "FilterPanel.h"

#include "Environment.h"

#include <sstream>
#include <iostream>

#include <wibble/regexp.h>

#include <tagcoll/stream/filters.h>
#include <tagcoll/stream/expression.h>

#include "DebtagsDocument.h"

#include <gtkmm/table.h>
#include <gtkmm/label.h>
#include <gtkmm/menu.h>
#include <gtkmm/treemodel.h>
#include <gtkmm/dialog.h>
#include <gtkmm/stock.h>

using namespace std;
using namespace ept::apt;

//#define MAX_VISIBLE_PKG 200


template<class DOC>
FilterPanel<DOC>::FilterPanel(DOC& doc)
	: Gtk::Frame("Filter"), doc(doc), submitButton("Submit"), specialQuery("No TODO query"), tagSelector(doc), specialID(0)
{
	Gtk::Table* table = manage(new Gtk::Table(7, 2));
	add(*table);


	// Add the TreeView, inside a ScrolledWindow
	scrolledItemList.add(itemList);
	// Only show the scrollbars when they are necessary:
	scrolledItemList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	Gtk::Menu* smenu = Gtk::manage(new Gtk::Menu());
	smenu->items().push_back(Gtk::Menu_Helpers::MenuElem("_None"));
	smenu->items().push_back(Gtk::Menu_Helpers::MenuElem("_Installed"));
	smenu->items().push_back(Gtk::Menu_Helpers::MenuElem("N_ot installed"));
	statusFilter.set_menu(*smenu);

	if (char* defmail = getenv("DEBEMAIL"))
	{
		//filter.setMaintainer(defmail);
		maintFilter.set_text(defmail);
		statusFilter.set_history(0);
	} else {
		//filter.setInstalled(Debtags::BasicPackageMatcher<string>::YES);
		statusFilter.set_history(1);
	}

	Gtk::Frame* tagFrame = manage(new Gtk::Frame("Tags:"));
	tagFrame->add(tagSelector);

	submitButton.signal_clicked().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::updateList));
	submitButton.set_sensitive(false);

	table->attach(*manage(new Gtk::Label("Name:", Gtk::ALIGN_RIGHT, Gtk::ALIGN_CENTER)), 0, 1, 0, 1, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(nameFilter, 1, 2, 0, 1, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(*manage(new Gtk::Label("Text:", Gtk::ALIGN_RIGHT, Gtk::ALIGN_CENTER)), 0, 1, 1, 2, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(ftextFilter, 1, 2, 1, 2, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(*manage(new Gtk::Label("Maintainer:", Gtk::ALIGN_RIGHT, Gtk::ALIGN_CENTER)), 0, 1, 2, 3, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(maintFilter, 1, 2, 2, 3, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(*manage(new Gtk::Label("Status:", Gtk::ALIGN_RIGHT, Gtk::ALIGN_CENTER)), 0, 1, 3, 4, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(statusFilter, 1, 2, 3, 4, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(specialQuery, 1, 2, 4, 5, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(*tagFrame, 0, 2, 5, 6, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(submitButton, 0, 2, 6, 7, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
	table->attach(scrolledItemList, 0, 2, 7, 8);
	table->attach(foundStats, 0, 2, 8, 9, Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);

	nameFilter.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::filterChanged));
	nameFilter.signal_activate().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::updateList));
	maintFilter.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::filterChanged));
	maintFilter.signal_activate().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::updateList));
	ftextFilter.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::filterChanged));
	ftextFilter.signal_activate().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::updateList));
	statusFilter.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::filterChanged));
	//specialQuery.signal_clicked().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::on_specialQuery_clicked));
	tagSelector.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::updateList));
	/*
	statusFilter.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::on_filter_changed));
	//specialQuery.signal_clicked().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::on_specialQuery_clicked));
	tagSelector.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::on_filter_changed));
	*/

	// Create the Tree model
	itemListModel = Gtk::ListStore::create(itemListModelColumns);
	sortedModel = Gtk::TreeModelSort::create(itemListModel);

	noModel = itemList.get_model();
	itemList.set_model(sortedModel);

	// Add the view columns
	Gtk::TreeViewColumn* column;
	int count;

	count = itemList.append_column("Name", itemListModelColumns.name);
	column = itemList.get_column(count - 1);
	column->set_sizing(Gtk::TREE_VIEW_COLUMN_FIXED);
	column->set_fixed_width(85);
	column->set_resizable(true);
	column->set_sort_column(itemListModelColumns.name);

	count = itemList.append_column("Description", itemListModelColumns.desc);
	column = itemList.get_column(count - 1);
	column->set_sizing(Gtk::TREE_VIEW_COLUMN_FIXED);
	//column->set_fixed_width(40);
	column->set_resizable(true);
	column->set_sort_column(itemListModelColumns.desc);

	itemList.set_fixed_height_mode(true);

	Glib::RefPtr<Gtk::TreeSelection> itemListSelection = itemList.get_selection();
	itemListSelection->set_mode(Gtk::SELECTION_SINGLE);

	itemListSelection->signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::on_selection_changed));
	
	doc.signal_reselected().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::reselect));
	doc.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::reselect));
	//doc.signal_changed().connect(sigc::mem_fun(*this, &FilterPanel<DOC>::updateList));

	//itemList.add_events(Gdk::BUTTON_PRESS_MASK);
	//itemList.signal_event().connect(SigC::slot(*this, &ItemList::on_event));

	//itemList.signal_focus_in_event().connect(SigC::slot(*this, &FilterPanel::do_list_focus_in_event));
	updateList();
	reselect();
}

/*
template<class DOC>
void FilterPanel<DOC>::on_filter_changed()
{
	filter.setName(nameFilter.get_text());
	filter.setMaintainer(maintFilter.get_text());
	filter.setString(ftextFilter.get_text());

	int cur = statusFilter.get_history();
	filter.setInstalled(
			cur == 0 ? Debtags::BasicPackageMatcher<Package>::ANY :
			cur == 1 ? Debtags::BasicPackageMatcher<Package>::YES :
			           Debtags::BasicPackageMatcher<Package>::NO);

	filter.setDebtags(tagSelector.selected());

	updateList();
}
*/

template<class DOC>
class FacetsDialog : public Gtk::Dialog
{
protected:
	typedef typename DOC::Package Package;
	typedef typename DOC::Facet Facet;
	typedef typename DOC::Tag Tag;

	// Tree model columns
	class FacetListModelColumns : public Gtk::TreeModel::ColumnRecord
	{
		public:
			FacetListModelColumns() { add(facet); add(label); }

			Gtk::TreeModelColumn<Facet> facet;
			Gtk::TreeModelColumn<Glib::ustring> label;
	};

	FacetListModelColumns masterListModelColumns;
	FacetListModelColumns slaveListModelColumns;

	Glib::RefPtr<Gtk::ListStore> masterListModel;
	Glib::RefPtr<Gtk::ListStore> slaveListModel;

	Gtk::TreeView masterList;
	Gtk::TreeView slaveList;

	Gtk::ScrolledWindow scrolledMasterList;
	Gtk::ScrolledWindow scrolledSlaveList;


	Gtk::HBox hbox;

	DOC& doc;

	class Stats : public std::map<Facet, int>
	{
	protected:
		DOC& doc;
		const Facet& pivot;
		std::set<Facet> facets;

		void update(const std::set<Tag>& tags, int count)
		{
			std::set<Facet> has;
			for (typename std::set<Tag>::const_iterator i = tags.begin();
					i != tags.end(); i++)
				has |= i->facet();
			if (pivot <= has)
			{
				has = facets - has;
				for (typename std::set<Facet>::const_iterator i = has.begin(); i != has.end(); i++)
					(*this)[*i] |= count;
			}
		}

	public:
		Stats(DOC& doc, const Facet& pivot) : doc(doc), pivot(pivot)
		{
			facets = doc.vocabulary().facets();
		}
		virtual ~Stats() {}

		template<typename ITEMS, typename TAGS>
		void insert(const ITEMS& items, const TAGS& tags)
		{
			update(tags, items.size());
		}
	};

	void on_master_changed()
	{
		Facet master = getMasterFacet();

		cerr << "masterchanged " << master.name() << endl;

		slaveListModel->clear();

		Gtk::TreeModel::Row row;
		Stats stats(doc, master);
		doc.debtags().outputPatched(tagcoll::coll::inserter(stats));
		std::map<int, Facet> rstats;
		for (typename Stats::const_iterator i = stats.begin();
				i != stats.end(); i++)
		{
			rstats.insert(make_pair<int, Facet>(i->second, i->first));
		}
		for (typename std::map<int, Facet>::const_iterator i = rstats.begin();
				i != rstats.end(); i++)
		{
			row = *(slaveListModel->append());
			row[slaveListModelColumns.facet] = i->second;
			stringstream str;
			str << i->second.name() << " (" << i->first << ")";
			row[slaveListModelColumns.label] = str.str();
		}
	}

	void on_slave_changed()
	{
		Facet master = getMasterFacet();
		Facet slave = getSlaveFacet();

		cerr << "-> " << master.name() << "::* && ! " << slave.name() << "::*" << endl;
	}

public:
	FacetsDialog(DOC& doc) : doc(doc)
	{
		set_default_size(500, 400);

		masterListModel = Gtk::ListStore::create(masterListModelColumns);
		masterList.set_model(masterListModel);
		masterList.append_column("Has", masterListModelColumns.label);
		masterList.get_column(0)->set_resizable(false);
		Glib::RefPtr<Gtk::TreeSelection> masterListSelection = masterList.get_selection();
		masterListSelection->set_mode(Gtk::SELECTION_SINGLE);
		masterListSelection->signal_changed().connect(sigc::mem_fun(*this, &FacetsDialog<DOC>::on_master_changed));
		slaveListModel = Gtk::ListStore::create(slaveListModelColumns);
		slaveList.set_model(slaveListModel);
		slaveList.append_column("And not", slaveListModelColumns.label);
		slaveList.get_column(0)->set_resizable(false);
		Glib::RefPtr<Gtk::TreeSelection> slaveListSelection = slaveList.get_selection();
		slaveListSelection->set_mode(Gtk::SELECTION_SINGLE);
		slaveListSelection->signal_changed().connect(sigc::mem_fun(*this, &FacetsDialog<DOC>::on_slave_changed));

		scrolledMasterList.add(masterList);
		scrolledMasterList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		scrolledSlaveList.add(slaveList);
		scrolledSlaveList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

		hbox.pack_start(scrolledMasterList, Gtk::PACK_EXPAND_WIDGET);
		hbox.pack_start(scrolledSlaveList, Gtk::PACK_EXPAND_WIDGET);

		masterList.show();
		slaveList.show();
		scrolledMasterList.show();
		scrolledSlaveList.show();
		hbox.show();

		get_vbox()->pack_start(hbox);

		add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
		add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);

		Gtk::TreeModel::Row row;
		std::set<Facet> facets = doc.vocabulary().facets();
		for (typename std::set<Facet>::const_iterator i = facets.begin();
				i != facets.end(); i++)
		{
			row = *(masterListModel->append());
			row[masterListModelColumns.facet] = *i;
			row[masterListModelColumns.label] = i->name();
		}
	}

	Facet getMasterFacet()
	{
		Glib::RefPtr<Gtk::TreeSelection> sel = masterList.get_selection();
		Gtk::TreeModel::iterator iter = sel->get_selected();
		Facet masterFacet;
		if (iter)
		{
			Gtk::TreeModel::Row row = *iter;
			masterFacet = row[masterListModelColumns.facet];
		}
		return masterFacet;
	}

	Facet getSlaveFacet()
	{
		Glib::RefPtr<Gtk::TreeSelection> sel = slaveList.get_selection();
		Gtk::TreeModel::iterator iter = sel->get_selected();
		Facet slaveFacet;
		if (iter)
		{
			Gtk::TreeModel::Row row = *iter;
			slaveFacet = row[slaveListModelColumns.facet];
		}
		return slaveFacet;
	}

	std::string getExpression()
	{
		Facet master = getMasterFacet();
		Facet slave = getSlaveFacet();

		return master.name() + "::* && ! " + slave.name() + "::*";
	}
};

template<class DOC>
class TODODialog : public Gtk::Dialog
{
protected:
	typedef typename DOC::Package Package;
	typedef typename DOC::Facet Facet;
	typedef typename DOC::Tag Tag;

	DOC& doc;
	Gtk::RadioButtonGroup group;
	std::vector<Gtk::RadioButton*> buttons;
	std::vector<std::string> labels;
	int selected;
	std::string selectedLabel;
	Facet specialFacet;
	std::set<Package> specialPackages;

	void on_selected_changed(int val)
	{
		selected = val;
		fprintf(stderr, "%d/%zd\n", selected, labels.size());
		selectedLabel = labels[selected];
		if (val < 5)
		{
			specialFacet = Facet();
			specialPackages.clear();
		}
		else
		{
#if 0
			MutexLock lock(doc.specialsMutex);
			doc.wantSpecials();
	
			int count = 0;
			for (typename std::map<Facet, OpSet<Package> >::const_iterator i = doc.specials.specials.begin();
					i != doc.specials.specials.end(); i++, count++)
				if (count == val - 5)
				{
					specialFacet = i->first;
					specialPackages = i->second;
				}
#endif
		}
	}

	void addButton(const std::string& label, int id, bool active = false)
	{
		Gtk::RadioButton* rb;
		rb = manage(new Gtk::RadioButton(group, label));
		if (active) rb->set_active(true);
		buttons.push_back(rb);
		labels.push_back(label);
		get_vbox()->pack_start(*rb, Gtk::PACK_EXPAND_WIDGET);
		rb->signal_clicked().connect(
				sigc::bind<int>(
					sigc::mem_fun(*this, &TODODialog<DOC>::on_selected_changed), id));
		rb->show();

		fprintf(stderr, "AddButton %zd %d\n", labels.size(), id);
	}

public:
	TODODialog(DOC& doc) : doc(doc), selected(0)
	{
		int count = 0;
		
		addButton("No TODO query", count++, true);
		addButton("Empty tagset", count++);
		addButton("Uitoolkit and not Interface", count++);
		addButton("Uitoolkit and not Implemented-in", count++);
		addButton("Missing Role", count++);

#if 0
		{
			MutexLock lock(doc.specialsMutex);
			if (doc.hasSpecials == DebtagsDocument::YES)
			{
				doc.wantSpecials();
			
				for (typename std::map<Facet, OpSet<Package> >::const_iterator i = doc.specials.specials.begin();
						i != doc.specials.specials.end(); i++, count++)
					addButton("Specials: [" + i->first.name() + "] " + i->first.sdesc() + " (" + stringf::fmt(i->second.size()) + ")", count);
			} else if (doc.hasSpecials == DebtagsDocument::GENERATING) {
				Gtk::Label* label = manage(new Gtk::Label("Special tag lists are being generated"));
				label->show();
				get_vbox()->pack_start(*label);
			} else {
				fprintf(stderr, "HASSPECIALS: %d\n", (int)doc.hasSpecials);
			}
		}
#endif

		add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
		add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	}

	std::string getSelectedLabel() { return selectedLabel; }
	int getSelected() { return selected; }
	Facet getSpecialFacet() { return specialFacet; }
	std::set<Package> getSpecialPackages() { return specialPackages; }
};


template<class DOC>
void FilterPanel<DOC>::do_facets_dialog()
{
	FacetsDialog<DOC> dialog(doc);
	Gtk::ResponseType result = (Gtk::ResponseType)dialog.run();
	if (result == Gtk::RESPONSE_OK)
	{
		specialExpression = dialog.getExpression();
		specialQuery.set_label(specialExpression);
		specialID = -1;
	}
	updateList();
}

template<class DOC>
void FilterPanel<DOC>::do_todo_dialog()
{
	TODODialog<DOC> dialog(doc);
	Gtk::ResponseType result = (Gtk::ResponseType)dialog.run();
	if (result == Gtk::RESPONSE_OK)
	{
		specialID = dialog.getSelected();
		specialFacet = dialog.getSpecialFacet();
		specialPackages = dialog.getSpecialPackages();
		if (!specialFacet)
			specialQuery.set_label(dialog.getSelectedLabel());
		else
		{
			stringstream str;
			str << "Special: " << specialFacet.name() << " (" << specialPackages.size() << ")";
			specialQuery.set_label(str.str());
		}
	}
	updateList();
}

template<class DOC>
void FilterPanel<DOC>::on_selection_changed()
{
	Glib::RefPtr<Gtk::TreeSelection> sel = itemList.get_selection();
	Gtk::TreeModel::iterator iter = sel->get_selected();
	if (iter)
	{
		Gtk::TreeModel::Row row = *iter;
		//Debtags::Package p = doc.packageDB().getPackage(row[itemListModelColumns.pkg]);
		//fprintf(stderr, "Selected %.*s\n", PFSTR(p.name()));
		Package up = row[itemListModelColumns.pkg];
		signal_selected().emit(up);
	}
}

template<class DOC> template<typename OUT>
void FilterPanel<DOC>::outputSpecialColl(OUT& consumer)
{
	if (specialID == -1)
	{
		doc.debtags().output(tagcoll::stream::filterItemsByExpression(specialExpression, consumer));
	} else if (specialID == 0) {
		doc.debtags().output(consumer);
	} else if (specialID == 1) {
		doc.debtags().output(tagcoll::stream::untaggedRemover(consumer, true));
	} else if (specialID == 2) {
		doc.debtags().output(tagcoll::stream::filterItemsByExpression("uitoolkit::* && !interface::*", consumer));
	} else if (specialID == 3) {
		doc.debtags().output(tagcoll::stream::filterItemsByExpression("uitoolkit::* && !implemented-in::*", consumer));
	} else if (specialID == 4) {
		doc.debtags().output(tagcoll::stream::filterItemsByExpression("!role::*", consumer));
	} else {
		if (!specialFacet)
		{
			warning("I don't know how to handle special ID %d", specialID);
		} else {
			for (typename std::set<Package>::const_iterator i = specialPackages.begin();
					i != specialPackages.end(); i++)
			{
				std::set<Tag> tags = doc.debtags().tagdb().getTags(*i);
				if (tags.empty())
					consumer.consume(*i);
				else
					consumer.consume(*i, tags);
			}
		}
	}
}

template<class DOC>
void FilterPanel<DOC>::filterChanged()
{
	submitButton.set_sensitive(true);
}

template<class DOC>
void FilterPanel<DOC>::reselect()
{
	Glib::RefPtr<Gtk::TreeSelection> sel = itemList.get_selection();
	
	sel->unselect_all();

	if (doc.subCollection.hasItem(doc.current()))
	{
		for (Gtk::ListStore::const_iterator i = sortedModel->children().begin();
				i != sortedModel->children().end(); i++)
		{
			Package thisPkg = (*i)[itemListModelColumns.pkg];
			if (thisPkg == doc.current())
			{
				sel->select(sortedModel->get_path(i));
				itemList.scroll_to_row(sortedModel->get_path(i));
				break;
			}
		}
	}
}

#define AUTO(y, x...) typeof(x) y = x

template<class DOC>
void FilterPanel<DOC>::updateList()
{
#ifdef MAX_VISIBLE_PKG
	static const unsigned int max_pkg = MAX_VISIBLE_PKG;
#endif
	printf("UPDATELIST start\n");

	string name_f = nameFilter.get_text();
	string desc_f = ftextFilter.get_text();
	string maint_f = maintFilter.get_text();
	std::set<Tag> tags_f = tagSelector.selected();

	doc.filter().clear();
	if (!name_f.empty())
		doc.filter().addName(name_f);
	if (!desc_f.empty())
		doc.filter().addDescription(desc_f);
	if (!maint_f.empty())
		doc.filter().addMaintainer(maint_f);
	if (!tags_f.empty())
		doc.filter().addTags(tags_f);
	switch (statusFilter.get_history())
	{
		case 1: doc.filter().addState(Filter::Installed); break;
		case 2: doc.filter().addState(Filter::NotInstalled); break;
	}

	switch (specialID)
	{
		case -1:
			//p = p and predicate::tagexprMatch<Package>(specialExpression);
			break;
			/*
	} else if (specialID == 0) {
		doc.tagdb().output(consumer);
	} else if (specialID == 1) {
		UntaggedFilter<Package> filter(&consumer);
		doc.tagdb().output(filter);
		*/
		case 2:
			//p = p and predicate::tagexprMatch<Package>("uitoolkit::* && !interface::*");
			break;
		case 3:
			//p = p and predicate::tagexprMatch<Package>("uitoolkit::* && !implemented-in::*");
			break;
		case 4:
			//p = p and predicate::tagexprMatch<Package>("!role::*");
			break;
			/*
	} else {
		if (!specialFacet)
		{
			warning("I don't know how to handle special ID %d", specialID);
		} else {
			for (typename OpSet<Package>::const_iterator i = specialPackages.begin();
					i != specialPackages.end(); i++)
			{
				TagSet tags = doc.tagdb().getTags(*i);
				if (tags.empty())
					consumer.consume(*i);
				else
					consumer.consume(*i, tags);
			}
		}
		*/
	}

	printf("UPDATELIST built filter\n");

	doc.commitFilter();

	printf("UPDATELIST created subcoll\n");

/*
	if (specialID != 0)
	{
		filter.setConsumer(&doc.subCollection);
		//PMFilter f(filter, &doc.subCollection);
		outputSpecialColl(filter);
	}
	else
	{
#if 0
		//doc.serializer().packageDB().output(doc.subCollection, filter);
		if (filter.hasOnlyDebtags())
		{
			//doc.tagdb().outputHavingTags(filter.debtagsTagset(), doc.subCollection);
			//OpSet<Package> items(doc.tagdb().getItems(filter.debtagsTagset()));
			TagSet tags(filter.debtagsTagset());
			if (!tags.empty())
			{
				TagSet::const_iterator i = tags.begin();
				OpSet<Package> items = doc.tagdb().getItems(*i);
				for (; i != tags.end(); i++)
					items ^= doc.tagdb().getItems(*i);
				
				fprintf(stderr, "A %d to start with\n", items.size());
				for (typename OpSet<Package>::const_iterator i = items.begin();
						i != items.end(); i++)
					doc.subCollection.consume(*i, doc.tagdb().getTags(*i));
					//consumer.consume(*i, getTags(*i));
			}
		}
		else if (filter.hasDebtags())
		{
			filter.setConsumer(&doc.subCollection);
			//doc.tagdb().outputHavingTags(filter.debtagsTagset(), filter);

			OpSet<Package> items(doc.tagdb().getItems(filter.debtagsTagset()));
			fprintf(stderr, "B %d to start with\n", items.size());
			for (typename OpSet<Package>::const_iterator i = items.begin();
					i != items.end(); i++)
				filter.consume(*i, doc.tagdb().getTags(*i));
		}
		else
		{
#endif
			filter.setConsumer(&doc.subCollection);
			doc.tagdb().output(filter);
//		}

	}
	*/

	//PackageBuffer::iterator lastIter = pkgs.size() > max_pkg ? pkgs.begin() + max_pkg : pkgs.end();
	
	stringstream state;
#ifdef MAX_VISIBLE_PKG
	if (doc.subCollection.itemCount() > max_pkg)
	{
		state << ">" << max_pkg << " (" << doc.subCollection.itemCount() << ") packages found";
		//printf("The limit of %d visualized packages has been reached: please narrow your search.\n", max_pkg);
	} else {
		state << doc.subCollection.itemCount() << " packages found";
	}
#else
	state << doc.subCollection.itemCount() << " packages found";
#endif
	foundStats.set_text(state.str());

	// To make things faster when inserting a large number of items, I create a
	// new ListStore and fill it up while it's not connected to anything.
	// Then, when the data is there, I finally attach it to a new TreeModelSort
	// and to the TreeView.

	int sortColumnID;
	Gtk::SortType sortOrder;
	sortedModel->get_sort_column_id(sortColumnID, sortOrder);

	itemList.set_model(noModel);

	itemListModel = Gtk::ListStore::create(itemListModelColumns);
	doc.subCollection.output(tagcoll::coll::inserter(*this));

	printf("UPDATELIST filled up model\n");

	sortedModel = Gtk::TreeModelSort::create(itemListModel);
	sortedModel->set_sort_column(sortColumnID, sortOrder);
	itemList.set_model(sortedModel);

	printf("UPDATELIST end\n");

	submitButton.set_sensitive(false);
}

template<class DOC> template<typename ITEMS, typename TAGS>
void FilterPanel<DOC>::insert(const ITEMS& pkgs, const TAGS& tags)
{
#ifdef MAX_VISIBLE_PKG
	static const unsigned int max_pkg = 500;
#endif

	for (typename ITEMS::const_iterator pkg = pkgs.begin(); pkg != pkgs.end(); ++pkg)
	{
#ifdef MAX_VISIBLE_PKG
		if (itemListModel->children().size() > max_pkg)
			return;
#endif
		if (*pkg != Package())
		{
			PackageRecord rec(doc.apt().rawRecord(*pkg));

			Gtk::TreeModel::Row row;
			row = *(itemListModel->append());
			row[itemListModelColumns.pkg] = *pkg;
			row[itemListModelColumns.name] = *pkg;
			row[itemListModelColumns.desc] = rec.shortDescription(string("(description not available)"));
		}
		//if (sel.find(*i) != sel.end())
		//itemList.get_selection()->select(row);
	}
}

/*
bool FilterPanel::do_list_focus_in_event(GdkEventFocus*)
{
	warning("LFIE\n");
	return false;
}
*/

template class FilterPanel<DebtagsDocument>;

#include <ept/debtags/debtags.tcc>

// vim:set ts=4 sw=4:
