#ifndef PATCH_REVIEW_H
#define PATCH_REVIEW_H

/*
 * Patch review panel
 *
 * Copyright (C) 2003,2004,2005,2006  Enrico Zini <enrico@debian.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtkmm/textview.h>
#include <gtkmm/box.h>
#include <gtkmm/treeview.h>
#include <gtkmm/liststore.h>
#include <gtkmm/scrolledwindow.h>


/**
 * Patch review panel
 *
 * Contains:
 * \verbatim
 * +------------------------------------+
 * |                                    |
 * | Raw patch list view                |
 * |                                    |
 * |-------------+----------------------+
 * | Details of  | Details of package   |
 * | tags added  |                      |
 * | (allows     |                      |
 * |  modifying) |                      |
 * +-------------+----------------------+
 * \endverbatim
 */
template<class DOC>
class PatchReview : public Gtk::VBox
{
protected:
	typedef typename DOC::Package Package;
	typedef typename DOC::Facet Facet;
	typedef typename DOC::Tag Tag;

	DOC& doc;

	// Tree model columns
	class PatchListModelColumns : public Gtk::TreeModel::ColumnRecord
	{
		public:
			PatchListModelColumns() { add(pkg); add(name); add(patch); }

			Gtk::TreeModelColumn<Package> pkg;
			Gtk::TreeModelColumn<Glib::ustring> name;
			Gtk::TreeModelColumn<Glib::ustring> patch;
	};
	class TagListModelColumns : public Gtk::TreeModel::ColumnRecord
	{
		public:
			TagListModelColumns() { add(tag); add(name); add(sdesc); }

			Gtk::TreeModelColumn<Tag> tag;
			Gtk::TreeModelColumn<Glib::ustring> name;
			Gtk::TreeModelColumn<Glib::ustring> sdesc;
	};

	PatchListModelColumns patchListModelColumns;
	Glib::RefPtr<Gtk::ListStore> patchListModel;
	Gtk::TreeView patchList;
	Gtk::ScrolledWindow scrolledPatchList;

	TagListModelColumns tagListModelColumns;
	Glib::RefPtr<Gtk::ListStore> tagListModel;
	Gtk::TreeView tagList;
	Gtk::ScrolledWindow scrolledTagList;
	Gtk::TextView descr;
	Glib::RefPtr<Gtk::TextBuffer> descrBuf;

	void on_patchList_selection_changed();
	void updatePackages();
	void updateCurrent();

#if 0

	DOC& doc;

	class Stats : public TagcollConsumer<typename DOC::package, Tag>, public std::map<Facet, int>
	{
	protected:
		DOC& doc;
		const Facet& pivot;
		FacetSet facets;

		void update(const Tagcoll::OpSet<Tag>& tags, int count)
		{
			FacetSet has;
			for (Tagcoll::OpSet<Tag>::const_iterator i = tags.begin();
					i != tags.end(); i++)
				has += i->facet();
			if (has.contains(pivot))
			{
				has = facets - has;
				for (FacetSet::const_iterator i = has.begin(); i != has.end(); i++)
					(*this)[*i] += count;
			}
		}

	public:
		Stats(DOC& doc, const Facet& pivot) : doc(doc), pivot(pivot)
		{
			facets = doc.vocabulary().getFacets();
		}
		virtual ~Stats() {}
		void consume(const typename DOC::package& pkg) {}
		void consume(const typename DOC::package& item, const Tagcoll::OpSet<Tag>& tags)
		{
			update(tags, 1);
		}
		void consume(const Tagcoll::OpSet<typename DOC::package>& items, const Tagcoll::OpSet<Tag>& tags)
		{
			update(tags, items.size());
		}
	};

	void on_master_changed()
	{
		Facet master = getMasterFacet();

		fprintf(stderr, "masterchanged %.*s\n", PFSTR(master.name()));

		slaveListModel->clear();

		Gtk::TreeModel::Row row;
		Stats stats(doc, master);
		doc.tagdb().outputPatched(stats);
		std::map<int, Facet> rstats;
		for (typename Stats::const_iterator i = stats.begin();
				i != stats.end(); i++)
		{
			rstats.insert(make_pair<int, Facet>(i->second, i->first));
		}
		for (typename std::map<int, Facet>::const_iterator i = rstats.begin();
				i != rstats.end(); i++)
		{
			row = *(slaveListModel->append());
			row[slaveListModelColumns.facet] = i->second;
			row[slaveListModelColumns.label] = i->second.name() + " (" + stringf::fmt(i->first) + ")";
		}
	}

	void on_slave_changed()
	{
		Facet master = getMasterFacet();
		Facet slave = getSlaveFacet();

		fprintf(stderr, "-> %.*s::* && ! %.*s::*\n", PFSTR(master.name()), PFSTR(slave.name()));
	}

public:
	FacetsDialog(DOC& doc) : doc(doc)
	{
		set_default_size(500, 400);

		masterListModel = Gtk::ListStore::create(masterListModelColumns);
		masterList.set_model(masterListModel);
		masterList.append_column("Has", masterListModelColumns.label);
		masterList.get_column(0)->set_resizable(false);
		Glib::RefPtr<Gtk::TreeSelection> masterListSelection = masterList.get_selection();
		masterListSelection->set_mode(Gtk::SELECTION_SINGLE);
		masterListSelection->signal_changed().connect(sigc::mem_fun(*this, &FacetsDialog<DOC>::on_master_changed));
		slaveListModel = Gtk::ListStore::create(slaveListModelColumns);
		slaveList.set_model(slaveListModel);
		slaveList.append_column("And not", slaveListModelColumns.label);
		slaveList.get_column(0)->set_resizable(false);
		Glib::RefPtr<Gtk::TreeSelection> slaveListSelection = slaveList.get_selection();
		slaveListSelection->set_mode(Gtk::SELECTION_SINGLE);
		slaveListSelection->signal_changed().connect(sigc::mem_fun(*this, &FacetsDialog<DOC>::on_slave_changed));

		scrolledMasterList.add(masterList);
		scrolledMasterList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		scrolledSlaveList.add(slaveList);
		scrolledSlaveList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

		hbox.pack_start(scrolledMasterList, Gtk::PACK_EXPAND_WIDGET);
		hbox.pack_start(scrolledSlaveList, Gtk::PACK_EXPAND_WIDGET);

		masterList.show();
		slaveList.show();
		scrolledMasterList.show();
		scrolledSlaveList.show();
		hbox.show();

		get_vbox()->pack_start(hbox);

		add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
		add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);

		Gtk::TreeModel::Row row;
		FacetSet facets = doc.vocabulary().getFacets();
		for (FacetSet::const_iterator i = facets.begin();
				i != facets.end(); i++)
		{
			row = *(masterListModel->append());
			row[masterListModelColumns.facet] = *i;
			row[masterListModelColumns.label] = i->name();
		}
	}

	Facet getMasterFacet()
	{
		Glib::RefPtr<Gtk::TreeSelection> sel = masterList.get_selection();
		Gtk::TreeModel::iterator iter = sel->get_selected();
		Facet masterFacet;
		if (iter)
		{
			Gtk::TreeModel::Row row = *iter;
			masterFacet = row[masterListModelColumns.facet];
		}
		return masterFacet;
	}

	Facet getSlaveFacet()
	{
		Glib::RefPtr<Gtk::TreeSelection> sel = slaveList.get_selection();
		Gtk::TreeModel::iterator iter = sel->get_selected();
		Facet slaveFacet;
		if (iter)
		{
			Gtk::TreeModel::Row row = *iter;
			slaveFacet = row[slaveListModelColumns.facet];
		}
		return slaveFacet;
	}

	std::string getExpression()
	{
		Facet master = getMasterFacet();
		Facet slave = getSlaveFacet();

		return master.name() + "::* && ! " + slave.name() + "::*";
	}
#endif
	
#if 0
	FilterPanel<DOC> fpanel;
	Gtk::HPaned mainHPaned;
	TagEditor<DOC> tagEditor;

	//TagPanel leftTagPanel;
	//TagPanel rightTagPanel;
	Gtk::VBox mainVBox;
	Gtk::HBox entryHBox;
	Gtk::Notebook dataArea;
	RelatedPanel<DOC> rpanel;
	Glib::RefPtr<Gtk::TextBuffer> descrBuf;

	//typename DOC::package current;
	
	//Gtk::Table panelTable;
	//bool changed;
	//int currentPanel;

	Gtk::MenuBar menuBar;
	Gtk::Menu fileMenu;
	Gtk::Menu editMenu;
	Gtk::Menu optionsMenu;
	Gtk::Menu helpMenu;

	Gtk::Widget* undoMenuItem;
	Gtk::Widget* redoMenuItem;
	Gtk::Widget* reapplyMenuItem;
	Gtk::CheckMenuItem* computeIntensiveMenuItem;
	/*
	Gtk::Widget* saveMenuItem;
	Gtk::Widget* saveasMenuItem;
	Gtk::Widget* mergeMenuItem;
	Gtk::Widget* intersectMenuItem;
	Gtk::Widget* copyToOtherMenuItem;
	Gtk::Widget* moveToOtherMenuItem;
	Gtk::Widget* deleteUnselectedMenuItem;
	*/

	void checkUndo();
	void checkReapply();
	//void setChanged(bool val);

	// Invoked when the contents of the document change
	void on_changed();

	// Invoked when the current element changes
	void on_reselected();
	
	//void on_package_selected(typename DOC::package pkg) throw ();
#endif

public:
	PatchReview(DOC& doc);
	virtual ~PatchReview() {}

#if 0
	//void on_open_debtags();
	void on_save();
	void on_send();
	//void on_save_as();
	void on_quit();
	void on_compute_intensive_data();
	//void on_merge();
	//void on_intersect();
	//void on_copy_to_other();
	//void on_move_to_other();
	//void on_delete_unselected();

	//void on_doc_change();
	void on_filename_change();
	void on_selection_change();
	//void on_tag_change() throw ();

	//void on_request_tagcoll_change(TagCollection<int>::Change change);
#endif
};

// vim:set ts=4 sw=4:
#endif
