/*
 * A single view on a TagcollDocument
 *
 * Copyright (C) 2003--2007  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "RelatedPanel.h"

#include "DebtagsDocument.h"

#include <map>
#include <algorithm>
#include <sstream>

#include <wibble/operators.h>

#include <gtkmm/table.h>
#include <gtkmm/label.h>
#include <gtkmm/menu.h>
#include <gtkmm/treemodel.h>

using namespace std;
using namespace wibble::operators;
using namespace ept::apt;

//template class RelatedPanel<DebtagsDocument>;

template<class DOC>
RelatedPanel<DOC>::RelatedPanel(DOC& doc)
	: doc(doc), pivot()
{
	// Add the TreeView, inside a ScrolledWindow
	scrolledItemList.add(itemList);
	// Only show the scrollbars when they are necessary:
	scrolledItemList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	pack_start(foundStats, Gtk::PACK_SHRINK);
	pack_start(scrolledItemList, Gtk::PACK_EXPAND_WIDGET);

	// Create the Tree model
	itemListModel = Gtk::ListStore::create(itemListModelColumns);
	itemList.set_model(itemListModel);

	// Add the view columns
	itemList.append_column("Name", itemListModelColumns.name);
	itemList.append_column("Desc", itemListModelColumns.desc);

	itemList.get_column(0)->set_resizable(true);

	Glib::RefPtr<Gtk::TreeSelection> itemListSelection = itemList.get_selection();
	itemListSelection->set_mode(Gtk::SELECTION_SINGLE);

	itemListSelection->signal_changed().connect(sigc::mem_fun(*this, &RelatedPanel<DOC>::on_selection_changed));
	
	updateList();
}

template<class DOC>
void RelatedPanel<DOC>::on_selection_changed() throw ()
{
	Glib::RefPtr<Gtk::TreeSelection> sel = itemList.get_selection();
	Gtk::TreeModel::iterator iter = sel->get_selected();
	if (iter)
	{
		Gtk::TreeModel::Row row = *iter;
		Package pkg = row[itemListModelColumns.pkg];
		signal_selected().emit(pkg);
	}
}

template<class DOC>
void RelatedPanel<DOC>::updateList() throw ()
{
	static const unsigned int max_pkg = 200;

	printf("rpUPDATELIST start\n");
	if (pivot == Package())
	{
		printf("rpUPDATELIST end\n");
		return;
	}

	unsigned int dist = 0;
	//pkgs.clear();
	std::set<Tag> ref = doc.debtags().getTagsOfItem(pivot);
	std::set<Package> related;
	for ( ; related.size() == 0 && dist <= ref.size(); dist++)
	{
		fprintf(stderr, "Trying distance %d...\n", dist);
		//Debtags::Environment::get().outputRelated(pkgs, pivot, dist);
		related |= doc.debtags().getRelatedItems(doc.debtags().getTagsOfItem(pivot), dist);
		//related.output(pkgs);
	}

	//PackageBuffer::const_iterator lastIter = pkgs.size() > max_pkg ? pkgs.begin() + max_pkg : pkgs.end();
	
	stringstream state;
	if (related.size() > max_pkg)
	{
		state << ">" << max_pkg << " (" << related.size() << ") packages at distance " << dist;
		//printf("The limit of %d visualized packages has been reached: please narrow your search.\n", max_pkg);
	} else {
		state << related.size() << " packages at distance " << dist;
	}
	foundStats.set_text(state.str());

	//pkgs.sort();

	printf("rpUPDATELIST populate\n");
	Gtk::TreeModel::Row row;
	itemListModel->clear();
	unsigned int count = 0;
	for (typename std::set<Package>::const_iterator i = related.begin(); count < max_pkg && i != related.end(); i++, count++)
	{
		//printf("rpUPDATELIST: %.*s\n", PFSTR((*i)->name()));
		if (*i != Package())
		{
			PackageRecord rec(doc.apt().rawRecord(*i));

			row = *(itemListModel->append());
			row[itemListModelColumns.pkg] = *i;
			row[itemListModelColumns.name] = *i;
			row[itemListModelColumns.desc] = rec.shortDescription(string("(description not available)"));
		}
	}
	printf("rpUPDATELIST end\n");
}


template class RelatedPanel<DebtagsDocument>;

#include <ept/debtags/debtags.tcc>

// vim:set ts=4 sw=4:
