/*
 * Tag list and selector widget
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "TagSelector.h"
#include "TagMenu.h"
#include "AddButtons.h"

#include "Environment.h"

#include <wibble/operators.h>

#include <gtkmm/eventbox.h>

using namespace std;
using namespace wibble::operators;

bool TagSelector::TagList::fillMenu(const Tag& tag, const std::set<Tag>& selected)
{
	if (tag == Tag())
		return false;

	if (selected.size() < 2)
	{
		this->itemPopup.items().push_back(
				Gtk::Menu_Helpers::MenuElem("Unselect tag " + tag.fullname(),
				sigc::bind< Tag >(sigc::mem_fun(ts, &TagSelector::on_del_from_selection), tag)));
	}

	return true;
}

void TagSelector::do_changed()
{
	_signal_changed.emit();
}

void TagSelector::on_add_to_selection(DebtagsDocument::Tag tag)
{
	//printf("Add to sel: %.*s\n", PFSTR(tag));
	unsigned int count_pre = _selected.size();
	_selected |= tag;
	if (count_pre != _selected.size())
	{
		updateView();
		do_changed();
	}
}

void TagSelector::on_del_from_selection(DebtagsDocument::Tag tag)
{
	//printf("Del from sel: %.*s\n", PFSTR(tag));
	unsigned int count_pre = _selected.size();
	_selected -= tag;
	if (count_pre != _selected.size())
	{
		updateView();
		do_changed();
	}
}

void TagSelector::on_document_changed()
{
	time_t start = time(NULL);
	std::set<DebtagsDocument::Tag> allTags = doc.vocabulary().tags();
	_selected = _selected & allTags;
	updateView();
	time_t end = time(NULL);
	if (end != start)
		fprintf(stderr, "TagSelector::on_document_changed: %ld seconds\n", (end-start));
}

TagSelector::TagSelector(DebtagsDocument& doc)
	: doc(doc), itemList(doc, *this)
{
	// Add the TreeView, inside a ScrolledWindow
	scrolledItemList.add(itemList);
	// Only show the scrollbars when they are necessary:
	scrolledItemList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	pack_start(scrolledItemList, true, true);

	Gtk::HBox* hbox = manage(new Gtk::HBox());
	pack_start(*hbox, false, false);

	AddButton<DebtagsDocument>* add = newAddCompanionButton(doc, *this, "Add");
	manage(add);
	hbox->pack_start(*add, true, true);
	add->signal_selected().connect(sigc::mem_fun(*this, &TagSelector::on_add_to_selection));

	AddButton<DebtagsDocument>* addD = newAddDiscriminantButton(doc, *this, "Add disc");
	manage(addD);
	hbox->pack_start(*addD, true, true);
	addD->signal_selected().connect(sigc::mem_fun(*this, &TagSelector::on_add_to_selection));

	AddButton<DebtagsDocument>* addR = newAddRelevantButton(doc, *this, "Add rel");
	manage(addR);
	hbox->pack_start(*addR, true, true);
	addR->signal_selected().connect(sigc::mem_fun(*this, &TagSelector::on_add_to_selection));

	updateView();

	doc.signal_changed().connect(sigc::mem_fun(*this, &TagSelector::on_document_changed));
}

void TagSelector::on_selectedTagList_drag_data_get(
		const Glib::RefPtr<Gdk::DragContext>&, Gtk::SelectionData& selection_data, guint, guint, DebtagsDocument::Tag tag)
{
	string stag = tag.fullname();
	selection_data.set(selection_data.get_target(), 8, (const guchar*)stag.data(), stag.size());
}

void TagSelector::updateView()
{
	itemList.display(_selected);
}

#include "TagMenu.tcc"

// vim:set ts=4 sw=4:
