#ifndef TAGSETLIST_H
#define TAGSETLIST_H

/*
 * Tag list and selector widget
 *
 * Copyright (C) 2003--2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtkmm/treeview.h>
#include <gtkmm/liststore.h>
#include <gtkmm/menu.h>
#include <gtkmm/eventbox.h>
#include <set>

template<class DOC>
class TagsetList : public Gtk::TreeView
{
protected:
	typedef typename DOC::Facet Facet;
	typedef typename DOC::Tag Tag;
	typedef typename DOC::Package Package;

protected:
	Gtk::Menu itemPopup;

	Glib::RefPtr<Gtk::ListStore> itemListModel;

	// Tree model columns
	class ItemListModelColumns : public Gtk::TreeModel::ColumnRecord
	{
	public:
		ItemListModelColumns() { add(tag); add(desc); }

		Gtk::TreeModelColumn<Tag> tag;
		Gtk::TreeModelColumn<Glib::ustring> desc;
	};

	ItemListModelColumns itemListModelColumns;

	// Format the tag description from a tag
	virtual std::string format(const Tag& tag) { return tag.fullname(); }
	//			"[" + i->fullname() + "] " +
	//			i->facet().shortDescription() + " :: " + i->shortDescription();

	/**
	 * Fill up the popup menu after a right click.
	 *
	 * @param tag
	 *   The tag where the right click happened
	 * @param selected
	 *   The tags in the list selection
	 * @return
	 *   True if the menu is to be shown, else false
	 */
	virtual bool fillMenu(const Tag& tag, const std::set<Tag>& selected) { return false; }

	bool on_event_from_list(GdkEvent* e)
	{
		if (e->type == GDK_BUTTON_PRESS && e->button.button == 3)
		{
			Gtk::TreeModel::Path path;
			Gtk::TreeViewColumn* column;
			int cell_x, cell_y;
			if (get_path_at_pos(
					(int)e->button.x, (int)e->button.y,
					path, column,
					cell_x, cell_y))
			{
				// Clicked on an item
				//debug("Cell %d, %d\n", cell_x, cell_y);

				Gtk::TreeModel::Row row = *itemListModel->get_iter(path);
				Tag tag = row[itemListModelColumns.tag];
				
				//debug("Tag: %s\n", tag.fullname().c_str());

				Glib::RefPtr<Gtk::TreeSelection> sel = get_selection();

				//int rows = sel->count_selected_rows();
				//debug("%d selected rows\n");

				// Build the list of selected items
				std::set<Tag> selected;
				Gtk::TreeSelection::ListHandle_Path lhp = sel->get_selected_rows();
				for (Gtk::TreeSelection::ListHandle_Path::const_iterator i = lhp.begin();
						i != lhp.end(); ++i)
				{
					Gtk::TreeModel::Row row = *itemListModel->get_iter(*i);
					selected.insert(row[itemListModelColumns.tag]);
				}
				
				this->itemPopup.items().clear();
				if (fillMenu(tag, selected))
				{
					this->itemPopup.popup(e->button.button, e->button.time);
					return true;
				}
			} else {
				// Clicked outside
				//warning("itemList.get_path_at_pos failed\n");
				if (fillMenu(Tag(), std::set<Tag>()))
				{
					this->itemPopup.popup(e->button.button, e->button.time);
					return true;
				}
			}
		}
		return false;
	}

public:
	TagsetList()
	{
		// Create the Tree model
		itemListModel = Gtk::ListStore::create(itemListModelColumns);
		set_model(itemListModel);
		add_events(Gdk::BUTTON_PRESS_MASK);
		signal_event().connect(sigc::mem_fun(*this, &TagsetList<DOC>::on_event_from_list));

		// Add the view columns
		append_column("Tag", itemListModelColumns.desc);
	}
	virtual ~TagsetList() {}

	void display(const std::set<Tag>& tags)
	{
		itemListModel->clear();
		for (typename std::set<Tag>::const_iterator i = tags.begin(); i != tags.end(); ++i)
		{
			if (*i == Tag()) continue;
			Gtk::TreeModel::Row row;
			row = *(itemListModel->append());
			row[itemListModelColumns.tag] = *i;
			row[itemListModelColumns.desc] = format(*i);
		}
	}
};

// vim:set ts=4 sw=4:
#endif
