# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: fakeopen.py 115 2007-06-18 05:49:26Z camrdale-guest $

"""Fake file handling objects (not used)."""

from string import join

class FakeHandle:
    """Fake file handles stored in memory.
    
    @type name: C{string}
    @ivar name: the file name
    @type fakeopen: L{FakeOpen}
    @ivar fakeopen: the collection of all fake files
    @type pos: C{int}
    @ivar pos: the current position within the file
    
    """
    
    def __init__(self, name, fakeopen):
        """Initialize the instance.
        
        @type name: C{string}
        @param name: the file name
        @type fakeopen: L{FakeOpen}
        @param fakeopen: the collection of all fake files
        
        """
        
        self.name = name
        self.fakeopen = fakeopen
        self.pos = 0
    
    def flush(self):
        """Does nothing."""
        pass
    
    def close(self):
        """Does nothing."""
        pass
    
    def seek(self, pos):
        """Seek to a position within the file.
        
        @type pos: C{int}
        @param pos: the position to seek to
        
        """
        self.pos = pos
    
    def read(self, amount = None):
        """Read data from the file.
        
        @type amount: C{int}
        @param amount: the amount of data to read
        @rtype: C{string}
        @return: the read data
        
        """
        
        old = self.pos
        f = self.fakeopen.files[self.name]
        if self.pos >= len(f):
            return ''
        if amount is None:
            self.pos = len(f)
            return join(f[old:], '')
        else:
            self.pos = min(len(f), old + amount)
            return join(f[old:self.pos], '')
    
    def write(self, s):
        """Write data to the file.
        
        @type s: C{string}
        @param s: the data to write
        
        """
        
        f = self.fakeopen.files[self.name]
        while len(f) < self.pos:
            f.append(chr(0))
        self.fakeopen.files[self.name][self.pos : self.pos + len(s)] = list(s)
        self.pos += len(s)

class FakeOpen:
    """Fake file objects.
    
    @type files: C{dictionary} of {C{string}: C{list} of C{char}}
    @ivar files: the stored file data, keys are file names, values are lists 
        of characters that are the data
    
    """
    
    def __init__(self, initial = {}):
        """Initialize the instance.
        
        @type initial: C{dictionary}
        @param initial: the initial files to create
        
        """
        
        self.files = {}
        for key, value in initial.items():
            self.files[key] = list(value)
    
    def open(self, filename, mode):
        """Open a new or existing file.
        
        Treats everything as read/write, doesn't support append.
        
        @type filename: C{string}
        @param filename: the file name to open
        @type mode: C{string}
        @param mode: the file mode to open (not used)
        
        """
        self.files.setdefault(filename, [])
        return FakeHandle(filename, self)

    def exists(self, file):
        """Check if the file exists.
        
        @type file: C{string}
        @param file: the file name to check
        @rtype: C{boolean}
        @return: whether the file exists
        
        """
        
        return self.files.has_key(file)

    def getsize(self, file):
        """Get the current size of the file.
        
        @type file: C{string}
        @param file: the file name to check
        @rtype: C{int}
        @return: the file size
        
        """
        
        return len(self.files[file])

def test_normal():
    """Test the fake file handling objects."""
    f = FakeOpen({'f1': 'abcde'})
    assert f.exists('f1')
    assert not f.exists('f2')
    assert f.getsize('f1') == 5
    h = f.open('f1', 'rw')
    assert h.read(3) == 'abc'
    assert h.read(1) == 'd'
    assert h.read() == 'e'
    assert h.read(2) == ''
    h.write('fpq')
    h.seek(4)
    assert h.read(2) == 'ef'
    h.write('ghij')
    h.seek(0)
    assert h.read() == 'abcdefghij'
    h.seek(2)
    h.write('p')
    h.write('q')
    assert h.read(1) == 'e'
    h.seek(1)
    assert h.read(5) == 'bpqef'

    h2 = f.open('f2', 'rw')
    assert h2.read() == ''
    h2.write('mnop')
    h2.seek(1)
    assert h2.read() == 'nop'
    
    assert f.exists('f1')
    assert f.exists('f2')
    assert f.getsize('f1') == 10
    assert f.getsize('f2') == 4
