# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import cgi, gtk, gui.window, media, modules, os, tools, urllib

from gui     import fileChooser, help, extTreeview, extListview, selectPath
from tools   import consts, prefs
from media   import playlist
from gettext import gettext as _
from os.path import isdir, isfile
from gobject import idle_add, TYPE_STRING, TYPE_INT

MOD_INFO = ('File Explorer', _('File Explorer'), _('Browse your file system'), [], True, True)
MOD_L10N = MOD_INFO[modules.MODINFO_L10N]

# Default preferences
PREFS_DEFAULT_MEDIA_FOLDERS     = {_('Home'): consts.dirBaseUsr}    # List of media folders that are used as roots for the file explorer
PREFS_DEFAULT_SHOW_HIDDEN_FILES = False                             # True if hidden files should be shown


# The format of a row in the treeview
(
    ROW_PIXBUF,    # Item icon
    ROW_NAME,      # Item name
    ROW_TYPE,      # The type of the item (e.g., directory, file)
    ROW_FULLPATH   # The full path to the item
) = range(4)


# The possible types for a node of the tree
(
    TYPE_DIR,   # A directory
    TYPE_PLIST, # A playlist
    TYPE_FILE,  # A media file
    TYPE_NONE   # A fake item, used to display a '+' in front of a directory when needed
) = range(4)


class FileExplorer(modules.Module):
    """ This explorer lets the user browse the disk from a given root directory (e.g., ~/, /) """

    def __init__(self):
        """ Constructor """
        modules.Module.__init__(self, (consts.MSG_EVT_APP_STARTED, consts.MSG_EVT_EXPLORER_CHANGED, consts.MSG_EVT_APP_QUIT))


    def onModLoaded(self):
        """ The module has been loaded """
        columns = (('',   [(gtk.CellRendererPixbuf(), gtk.gdk.Pixbuf), (gtk.CellRendererText(), TYPE_STRING)], True),
                   (None, [(None, TYPE_INT)],                                                                  False),
                   (None, [(None, TYPE_STRING)],                                                               False))

        self.tree            = extTreeview.ExtTreeView(columns)
        self.popup           = None
        self.cfgWin          = None
        self.folders         = prefs.get(__name__, 'media-folders',     PREFS_DEFAULT_MEDIA_FOLDERS)
        self.scrolled        = gtk.ScrolledWindow()
        self.currFolder      = None
        self.showHiddenFiles = prefs.get(__name__, 'show-hidden-files', PREFS_DEFAULT_SHOW_HIDDEN_FILES)
        # Explorer
        self.tree.setIsDraggableFunc(self.isDraggable)
        self.tree.setDNDSources([consts.DND_TARGETS[consts.DND_DAP_URI]])
        self.scrolled.add(self.tree)
        self.scrolled.set_shadow_type(gtk.SHADOW_IN)
        self.scrolled.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        self.scrolled.show()
        # GTK handlers
        self.tree.connect('drag-data-get',              self.onDragDataGet)
        self.tree.connect('key-press-event',            self.onKeyPressed)
        self.tree.connect('exttreeview-row-expanded',   self.onRowExpanded)
        self.tree.connect('exttreeview-row-collapsed',  self.onRowCollapsed)
        self.tree.connect('exttreeview-button-pressed', self.onMouseButton)

        for name in self.folders:
            modules.postMsg(consts.MSG_CMD_EXPLORER_ADD, {'modName': MOD_L10N, 'expName': name, 'icon': None, 'widget': self.scrolled})


    def onModUnloaded(self):
        """ The module is going to be unloaded """
        prefs.set(__name__, 'media-folders',     self.folders)
        prefs.set(__name__, 'show-hidden-files', self.showHiddenFiles)


    def sortKey(self, row):
        """ Key function used to compare two rows of the tree """
        return row[ROW_NAME].lower()


    def isDraggable(self):
        """ Determine whether the selected rows can be dragged """
        for row in self.tree.iterSelectedRows():
            if row[ROW_TYPE] in (TYPE_FILE, TYPE_PLIST) or (row[ROW_TYPE] == TYPE_DIR and row[ROW_PIXBUF] == consts.icoMediaDir):
                return True
        return False


    def setShowHiddenFiles(self, showHiddenFiles):
        """ Show/hide hidden files """
        if showHiddenFiles != self.showHiddenFiles:
            # Update the configuration window if needed
            if self.cfgWin is not None and self.cfgWin.isVisible():
                self.cfgWin.getWidget('chk-hidden').set_active(showHiddenFiles)

            self.showHiddenFiles = showHiddenFiles
            self.refresh(self.tree)


    def getFilesFromSelection(self, tree):
        """ Return a list of playable files from the selected rows """
        return ['file://' + row[ROW_FULLPATH] for row in tree.getSelectedRows() if row[ROW_TYPE] != TYPE_NONE]


    def playSelection(self, tree, replace):
        """ Replace/extend the tracklist """
        if replace: modules.postMsg(consts.MSG_CMD_TRACKLIST_SET, {'files': self.getFilesFromSelection(tree), 'playNow': True})
        else:       modules.postMsg(consts.MSG_CMD_TRACKLIST_ADD, {'files': self.getFilesFromSelection(tree)})


    def renameFolder(self, oldName, newName):
        """ Rename a folder """
        self.folders[newName] = self.folders[oldName]
        del self.folders[oldName]

        modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': oldName})
        modules.postMsg(consts.MSG_CMD_EXPLORER_ADD,    {'modName': MOD_L10N, 'expName': newName, 'icon': None, 'widget': self.scrolled})


    # --== Tree management ==--


    def getDirContent(self, directory):
        """ Return a tuple of sorted rows (directories, playlists, mediaFiles) for the given directory """
        playlists   = []
        mediaFiles  = []
        directories = []

        for (file, path) in tools.listDir(directory, self.showHiddenFiles):
            if isdir(path):
                directories.append((consts.icoDir, unicode(file, errors='replace'), TYPE_DIR, path))
            elif isfile(path):
                if media.isSupported(file):
                    mediaFiles.append((consts.icoMediaFile, unicode(file, errors='replace'), TYPE_FILE, path))
                elif playlist.isSupported(file):
                    playlists.append((consts.icoMediaFile, unicode(file, errors='replace'), TYPE_PLIST, path))

        playlists.sort(key=self.sortKey)
        mediaFiles.sort(key=self.sortKey)
        directories.sort(key=self.sortKey)

        return (directories, playlists, mediaFiles)


    def exploreDir(self, tree, parent, directory):
        """ List the content of the given directory and append it to the tree as a child of parent """
        directories, playlists, mediaFiles = self.getDirContent(directory)

        tree.appendRows(directories, parent)
        tree.appendRows(playlists,   parent)
        tree.appendRows(mediaFiles,  parent)

        if directories != []:
            idle_add(self.updateChildren(tree, parent).next)


    def updateChildren(self, tree, parent):
        """ This generator updates (e.g., icon, fake child for directories) all children of the given parent """
        for child in tree.iterChildren(parent):
            # Only directories need to be updated and since they are always on top, we can stop if we find something else
            if tree.getItem(child, ROW_TYPE) != TYPE_DIR:
                yield False

            # Make sure it's readable
            directory = tree.getItem(child, ROW_FULLPATH)
            if not os.access(directory, os.R_OK | os.X_OK):
                continue

            hasContent      = False
            hasMediaContent = False
            for (file, path) in tools.listDir(directory, self.showHiddenFiles):
                if isdir(path):
                    hasContent = True
                elif isfile(path) and (media.isSupported(file) or playlist.isSupported(file)):
                    hasContent      = True
                    hasMediaContent = True
                    break

            # Append/remove children if needed
            if hasContent and tree.getNbChildren(child) == 0:      tree.appendRow((consts.icoDir, '', TYPE_NONE, ''), child)
            elif not hasContent and tree.getNbChildren(child) > 0: tree.removeAllChildren(child)

            # Change the icon based on whether the folder contains some media content
            if hasMediaContent: tree.setItem(child, ROW_PIXBUF, consts.icoMediaDir)
            else:               tree.setItem(child, ROW_PIXBUF, consts.icoDir)

            yield True

        yield False


    def refresh(self, tree, treePath=None):
        """ Refresh the tree, starting from treePath """
        if treePath is None: directory = self.folders[self.currFolder]
        else:                directory = tree.getItem(treePath, ROW_FULLPATH)

        directories, playlists, mediaFiles = self.getDirContent(directory)

        disk       = directories + playlists + mediaFiles
        diskIndex  = 0
        childIndex = 0
        while diskIndex < len(disk):
            file    = disk[diskIndex]
            rowPath = tree.getChild(treePath, childIndex)

            # We've reached the end of the tree, append the file and switch to the next one
            if rowPath is None:
                tree.appendRow(file, treePath)
                continue

            cmpResult = cmp(self.sortKey(tree.getRow(rowPath)), self.sortKey(file))
            if cmpResult < 0:
                tree.removeRow(rowPath)
            else:
                if cmpResult > 0:
                    tree.insertRowBefore(file, treePath, rowPath)
                diskIndex  += 1
                childIndex += 1

        # If there are tree rows left, all the corresponding files are no longer there
        while childIndex < tree.getNbChildren(treePath):
            tree.removeRow(tree.getChild(treePath, childIndex))

        # Update nodes' appearance
        if len(directories) != 0:
            idle_add(self.updateChildren(tree, treePath).next)

        # Recursively refresh expanded rows
        for child in tree.iterChildren(treePath):
            if tree.row_expanded(child):
                idle_add(self.refresh, tree, child)


    # --== GTK handlers ==--


    def onMouseButton(self, tree, event, path):
        """ A mouse button has been pressed """
        if event.button == 3:
            self.onShowPopupMenu(tree, event.button, event.time, path)
        elif event.button == 1 and event.type == gtk.gdk._2BUTTON_PRESS and path is not None:
            if   tree.getItem(path, ROW_PIXBUF) != consts.icoDir: self.playSelection(tree, True)
            elif tree.row_expanded(path):                         tree.collapse_row(path)
            else:                                                 tree.expand_row(path, False)


    def onShowPopupMenu(self, tree, button, time, path):
        """ Show a popup menu """
        if self.popup is None:
            self.popup = tools.loadGladeFile('FileExplorerMenu.glade')
            self.popup.get_widget('item-refresh').connect('activate', lambda widget: self.refresh(tree))
            self.popup.get_widget('item-collapse').connect('activate', lambda widget: tree.collapse_all())
            self.popup.get_widget('item-add').connect('activate', lambda widget: self.playSelection(tree, False))
            self.popup.get_widget('item-play').connect('activate', lambda widget: self.playSelection(tree, True))
            self.popup.get_widget('item-hidden').connect('toggled', lambda item: self.setShowHiddenFiles(item.get_active()))
            self.popup.get_widget('menu-popup').show_all()

        playable = path is not None and tree.getItem(path, ROW_PIXBUF) != consts.icoDir

        self.popup.get_widget('item-add').set_sensitive(playable)
        self.popup.get_widget('item-play').set_sensitive(playable)
        self.popup.get_widget('item-hidden').set_active(self.showHiddenFiles)
        self.popup.get_widget('menu-popup').popup(None, None, None, button, time)


    def onKeyPressed(self, tree, event):
        """ A key has been pressed """
        if gtk.gdk.keyval_name(event.keyval) == 'F5':
            self.refresh(self.tree)


    def onRowExpanded(self, tree, path):
        """ Replace the fake child by the real children """
        fakeChild = tree.getChild(path, 0)
        self.exploreDir(tree, path, tree.getItem(path, ROW_FULLPATH))
        tree.removeRow(fakeChild)


    def onRowCollapsed(self, tree, path):
        """ Replace all children by a fake child """
        tree.removeAllChildren(path)
        tree.appendRow((consts.icoDir, '', TYPE_NONE, ''), path)


    def onDragDataGet(self, tree, context, selection, info, time):
        """ Provide information about the data being dragged """
        selection.set('text/uri-list', 8, ' '.join([urllib.pathname2url(file) for file in self.getFilesFromSelection(tree)]))


   # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if msg == consts.MSG_EVT_EXPLORER_CHANGED and params['modName'] == MOD_L10N and self.currFolder != params['expName']:
            self.currFolder = params['expName']
            self.tree.clear()
            self.exploreDir(self.tree, None, self.folders[self.currFolder])
            if len(self.tree) != 0:
                self.tree.scroll_to_cell(0)
        elif msg == consts.MSG_EVT_APP_STARTED:
            self.onModLoaded()
        elif msg == consts.MSG_EVT_APP_QUIT:
            self.onModUnloaded()


    # --== Configuration ==--


    def configure(self, parent):
        """ Show the configuration dialog """
        if self.cfgWin is None:
            self.cfgWin = gui.window.Window('FileExplorer.glade', 'vbox1', __name__, MOD_L10N, 370, 400)
            # Create the list of folders
            txtRdr  = gtk.CellRendererText()
            pixRdr  = gtk.CellRendererPixbuf()
            columns = ((None, [(txtRdr, TYPE_STRING)],                           0, False),
                       ('',   [(pixRdr, gtk.gdk.Pixbuf), (txtRdr, TYPE_STRING)], 2, False))

            self.cfgList = extListview.ExtListView(columns, sortable=False, useMarkup=True)
            self.cfgList.set_headers_visible(False)
            self.cfgWin.getWidget('scrolledwindow1').add(self.cfgList)
            # Connect handlers
            self.cfgList.connect('key-press-event', self.onCfgKeyPressed)
            self.cfgList.get_selection().connect('changed', self.onCfgSelectionChanged)
            self.cfgWin.getWidget('btn-add').connect('clicked', self.onAddFolder)
            self.cfgWin.getWidget('btn-remove').connect('clicked', lambda btn: self.onRemoveSelectedFolder(self.cfgList))
            self.cfgWin.getWidget('btn-ok').connect('clicked', self.onBtnOk)
            self.cfgWin.getWidget('btn-rename').connect('clicked', self.onRenameFolder)
            self.cfgWin.getWidget('btn-cancel').connect('clicked', lambda btn: self.cfgWin.hide())
            self.cfgWin.getWidget('btn-help').connect('clicked', self.onHelp)

        if not self.cfgWin.isVisible():
            self.populateFolderList()
            self.cfgWin.getWidget('chk-hidden').set_active(self.showHiddenFiles)
            self.cfgWin.getWidget('btn-ok').grab_focus()

        self.cfgWin.show()


    def populateFolderList(self):
        """ Populate the list of known folders """
        self.cfgList.replaceContent([(name, consts.icoBtnDir, '<b>%s</b>\n<small>%s</small>' % (cgi.escape(name), cgi.escape(path)))
                                     for name, path in sorted(self.folders.iteritems())])


    def onAddFolder(self, btn):
        """ Let the user add a new folder to the list """
        result = selectPath.SelectPath(MOD_L10N, self.cfgWin, self.folders.keys()).run()

        if result is not None:
            name, path = result
            self.folders[name] = path
            self.populateFolderList()
            modules.postMsg(consts.MSG_CMD_EXPLORER_ADD, {'modName': MOD_L10N, 'expName': name, 'icon': None, 'widget': self.scrolled})


    def onRemoveSelectedFolder(self, list):
        """ Remove the selected media folder """
        if list.getSelectedRowsCount() == 1:
            remark   = _('You will be able to add this root folder again later on if you wish so.')
            question = _('Remove the selected entry?')
        else:
            remark   = _('You will be able to add these root folders again later on if you wish so.')
            question = _('Remove all selected entries?')

        if gui.questionMsgBox(self.cfgWin, question, '%s %s' % (_('Your media files will not be deleted.'), remark)) == gtk.RESPONSE_YES:
            for row in self.cfgList.getSelectedRows():
                name = row[0]
                modules.postMsg(consts.MSG_CMD_EXPLORER_REMOVE, {'modName': MOD_L10N, 'expName': name})
                del self.folders[name]
                if self.currFolder == name:
                    self.currFolder = None
            self.cfgList.removeSelectedRows()


    def onRenameFolder(self, btn):
        """ Let the user rename a folder """
        name         = self.cfgList.getSelectedRows()[0][0]
        forbidden    = [rootName for rootName in self.folders if rootName != name]
        pathSelector = selectPath.SelectPath(MOD_L10N, self.cfgWin, forbidden)

        pathSelector.setPathSelectionEnabled(False)
        result = pathSelector.run(name, self.folders[name])

        if result is not None and result[0] != name:
            self.renameFolder(name, result[0])
            self.populateFolderList()


    def onCfgKeyPressed(self, list, event):
        """ Remove the selection if possible """
        if gtk.gdk.keyval_name(event.keyval) == 'Delete':
            self.onRemoveSelectedFolder(list)


    def onCfgSelectionChanged(self, selection):
        """ The selection has changed """
        self.cfgWin.getWidget('btn-remove').set_sensitive(selection.count_selected_rows() != 0)
        self.cfgWin.getWidget('btn-rename').set_sensitive(selection.count_selected_rows() == 1)


    def onBtnOk(self, btn):
        """ The user has clicked on the OK button """
        self.cfgWin.hide()
        self.setShowHiddenFiles(self.cfgWin.getWidget('chk-hidden').get_active())


    def onHelp(self, btn):
        """ Display a small help message box """
        helpDlg = help.HelpDlg(MOD_L10N)
        helpDlg.addSection(_('Description'),
                           _('This module allows you to browse the files on your drives.'))
        helpDlg.addSection(_('Usage'),
                           _('At least one root folder must be added to browse your files. This folder then becomes the root of the '
                             'file explorer tree in the main window.'))
        helpDlg.show(self.cfgWin)
