/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/macros/invariance_annotation.hpp
 *
 * $Id: invariance_annotation.hpp,v 1.12 2005/06/23 09:54:22 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] @ref DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER
 * and @ref DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT, @ref
 * DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD, and @ref
 * DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT.
 *
 * @test diagnostics/macros/invariance_annotation.t.cpp
 */
#ifndef DIAGNOSTICS__MACROS__INVARIANCE_ANNOTATION_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__MACROS__INVARIANCE_ANNOTATION_HPP__INCLUDE_GUARD

// ::diagnostics::logging_facility::log
#include <diagnostics/frame/logging_facility.hpp>

// DIAGNOSTICS_BASE_CONCAT
#include <diagnostics/util/preprocessor.hpp>

////////////////////////////////////////////////////////////////////////////////

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

/**
 * @brief Helper for Class-Invariance Checking. On construction and
 * description, the invariance is checked. The check is performed by
 * calling an argumentless void method. This helper is used when the
 * invariance is allowed to throw.
 */
template<typename CLASS>
class Class_Invariance_Guard_Throw
{
public:
	explicit inline Class_Invariance_Guard_Throw(CLASS const * const object);
	inline ~Class_Invariance_Guard_Throw();
	inline void no_final_check() { m_final_check=false; }
private:
	Class_Invariance_Guard_Throw(Class_Invariance_Guard_Throw const & other);
	Class_Invariance_Guard_Throw & operator=(Class_Invariance_Guard_Throw const & other);
    
private:
    CLASS const & m_object;
    bool m_final_check;
};

template<typename CLASS>
Class_Invariance_Guard_Throw<CLASS>::Class_Invariance_Guard_Throw(CLASS const * const object) 
    : m_object(*object),
      m_final_check(true)
{
    (m_object.m_class_invariance)();
}

template<typename CLASS>
Class_Invariance_Guard_Throw<CLASS>::~Class_Invariance_Guard_Throw()
{
    if(m_final_check) (m_object.m_class_invariance)();
}


/**
 * @brief Helper for Class-Invariance Checking. On construction and
 * description, the invariance is checked. The check is performed by
 * calling an argumentless void method. 
 */
template<typename CLASS>
class Class_Invariance_Guard
{
public:
	explicit inline Class_Invariance_Guard(CLASS const * const object);
	inline ~Class_Invariance_Guard();
private:
	Class_Invariance_Guard(Class_Invariance_Guard const & other);
	Class_Invariance_Guard & operator=(Class_Invariance_Guard const & other);
    
private:
    CLASS const & m_object;
};

template<typename CLASS>
Class_Invariance_Guard<CLASS>::Class_Invariance_Guard(CLASS const * const object) 
    : m_object(*object)
{
    (m_object.m_class_invariance)();
}

template<typename CLASS>
Class_Invariance_Guard<CLASS>::~Class_Invariance_Guard()
{
    (m_object.m_class_invariance)();
}


INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;



/**
 * @brief an invariance block. The type Self must be defined in this
 * context and has to refer to the current class. Self must have a
 * public method void Self::m_class_invariance() const. This method
 * must implement the invariance check. A new block is
 * opened. Exceptions thrown by the invariance method are handled as
 * well as possible.
 *
 * @attention If the block is left by an exception AND the invariance
 * throws an exception, then the invariance's exception (if of type @a
 * BASE) is logged with (@a LEVEL,@ref TYPE_TRACE,@a NAME,@a WHAT),
 * and is discared afterwards. Execptions of type @ref
 * diagnostics::unittest::Test_Exception can be handled explicitly --
 * all other exceptions are unkown and coresponding message is
 * generated. The exception of the method is rethrown.  This might
 * lead to counter-intuitive behaviors: Depending on whether an
 * excpetion is thrown or not by the method, invariance violations
 * lead to different behaviors. However, since invariance checking is
 * used for mainly for debuging purposes, and any exception thrown by
 * an invariance check is accompanied by a corresponding log-message,
 * which indicates failure anyhow, this is acceptable.  BUT IT IS NOT
 * ACCEPTABLE TO EXPOSE SUCH A BEHAVIOR TO A CLIENT OF A MODULE.
 *
 * @attention THIS FEATURE MIGHT BE DELETED IN THE FUTURE
 */
#define DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER \
          do { ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Class_Invariance_Guard_Throw<Self> \
	    internal__invariance_guard(this); try { 


#define DIAGNOSTICS_INTERNAL_INVARIANCE_HANDLE_TE1(LEVEL,NR_WHAT) \
	      catch(::diagnostics::unittest::Test_Exception & e) { \
		  DIAGNOSTICS_BASE_LOG(LEVEL, \
			    ::diagnostics::TYPE_TRACE, \
                            NR_WHAT, \
			    ::std::string("dropped invariance exception EXCEPTION=\"") \
			    + e.name() \
			    + "\" WHAT=\"" \
			    + e.what() \
			    + "\""); \
	      } 
#define DIAGNOSTICS_INTERNAL_INVARIANCE_HANDLE_TE0(LEVEL,NR_WHAT) 

#define DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(LEVEL,NR_WHAT,HANDLE_TE_EXLPLICITLY,BASE,STR_WHAT) \
          } \
          catch(...) { \
              internal__invariance_guard.no_final_check(); \
	      try { \
		  this->m_class_invariance(); \
	      } \
              DIAGNOSTICS_BASE_CONCAT(DIAGNOSTICS_INTERNAL_INVARIANCE_HANDLE_TE,HANDLE_TE_EXLPLICITLY)((LEVEL),(NR_WHAT)) \
	      catch(BASE & e) { \
		  DIAGNOSTICS_BASE_LOG((LEVEL), \
				       ::diagnostics::TYPE_TRACE, \
                                       (NR_WHAT), \
				       ::std::string("dropped invariance exception ") \
                                       + STR_WHAT(e)); \
	      } \
	      catch(...){ \
		  DIAGNOSTICS_BASE_LOG((LEVEL), \
				       ::diagnostics::TYPE_TRACE, \
                                       (NR_WHAT), \
				       "dropped invariance exception"); \
	      } \
	      throw; \
	  } \
	  } while(false)





/**
 * @brief an invariance guard. The type Self must be defined in this
 * context and has to refer to the current class. Self must have a
 * public method void Self::m_class_invariance() const. This method
 * must implement the invariance check. A new block is opened.
 *
 * @attention if the invariance method throws, the behavior is undefined. 
 */
#define DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD \
            ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Class_Invariance_Guard<Self> \
	    internal__invariance_guard(this)

/**
 * @brief an invariance block. Sets up a @ref
 * DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD within a new block.
 *
 * @attention if the invariance method throws, the behavior is undefined. 
 */
#define DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER \
          do { DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD; {

/**
 * @brief end of an invariance block.
 */
#define DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT \
          } } while(false)

/**
 * @brief Assert the current class invariance, i.e., calls the invariance.
 */ 
#define DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT (this->m_class_invariance())


#endif

// vim:ts=4:sw=4
