/*****************************************************************

  ddj.c

  Copyright (c) 1999-2003 by Mike Oliphant - oliphant@gtk.org

    http://www.nostatic.org/grip

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

*****************************************************************/

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/times.h>
#include <sys/param.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <math.h>
#include "mysql.h"
#include "ddj.h"
#include "xpm.h"
#include "parsecfg.h"
#include "x10.h"
#include "cmdsocket.h"
#include "mixer.h"
#include "mp3db.h"
#include "idlist.h"
#include "song.h"
#include "artist.h"
#include "disc.h"
#include "rrand.h"
#include "listgui.h"
#include "dialog.h"
#include "uihelper.h"
#include "playlist.h"
#include "burn.h"

#ifdef HAVE_GHTTP_H
#include <ghttp.h>
#endif

#ifdef HAVE_LIRC_LIRC_CLIENT_H
#include "lirc.h"
#endif

static void ShutDown(GtkWidget *widget, GdkEvent *event, gpointer data);
void Debug(char *fmt,...);
static void IncVolume(void);
static void DecVolume(void);
static void ToggleMute(void);
static void DescribeSong(void);
static int CommandFromName(char *name,char **arg);
static void LircCB(gpointer data,gint source,GdkInputCondition condition);
static void CommandSocketCB(gpointer data,gint source,
			    GdkInputCondition condition);
static void SerialCB(gpointer data,gint source,GdkInputCondition condition);
static void DoCommand(int command,char *arg);
static void StartClock(void);
static void StopClock(void);
static void ResetClock(void);
static int GetClockSecs(void);
static void CheckWindowPosition(void);
static void DoSuspend(void);
static gint TimeOut(gpointer data);
static void DoSaveConfig(void);
static void KillMP3(void);
static void RedirectIO(gboolean redirect_output);
static char *FindRoot(char *str);
static void MakeArgs(char *str,char **args,int maxargs);
static pid_t LaunchProg(char *prog);
static void SavePlaylist(GtkWidget *widget,gpointer data);
static void DoSavePlaylist(GtkWidget *widget,gpointer data);
static void LoadPlaylist(void);
static void DoLoadPlaylist(GtkWidget *widget,gpointer data);
static void PopPlaylist(void);
static void MinMax(void);
static void ToggleControlButtons(void);
static void ParseMP3Cmd(char *instr,char *outstr,char *filename,int disc_gain);
static void ParseFileFmt(char *instr,char *outstr);
static void PlayCB(void);
static void Play(void);
static void StopPlay(void);
static gboolean NextSong(void);
static gboolean PrevSong(void);
static gboolean LookUpData(int id,Song *song,Artist *artist,
			   Artist *disc_artist,Disc *disc);
static void SelectRow(GtkWidget *widget,gint row,gint column,
		      GdkEventButton *event,gpointer data);
static void UpdateScrollDisplay(gboolean init);
static char *BaseName(char *string);
static void ResetCurrentPlaylist(void);
static void LoadAlbumImage(void);
static void SelectSong(IDList *slist,int song);
static void DeleteUser(void);
static void CreateUser(GtkWidget *widget,gpointer data);
static void DoInitSQL(void);
static void FillUserCList(void);
static void LogInAsRoot(void);
static void SelectUser(GtkWidget *widget,gint row,gint column,
		       GdkEventButton *event,gpointer data);
static void MakeConfigPage(void);
static void Homepage(void);
static void MakeAboutPage(void);
static GtkWidget *MakeNewPage(char *name);
static void MakePlayPage(void);
static void FillArtistCList(void);
static void ArtistChanged(void);
static void FillDiscCList(void);
static void QueryGenres(void);
static gboolean SingleFileList(char *filename);
static gboolean M3UFileList(char *filename);
static void GeneratePlaylist(void);
static void AddToPlayList(IDList *idlist,int discid);
static void SetSongOrder(GtkWidget *widget,gpointer data);
static void SetQueryListType(GtkWidget *widget,gpointer data);
static void FillArtistCombo(void);
static void FillDiscCombo(void);
static void FillGenreCombo(void);
static void UpdateSongTitle(void);
static void UpdateSongArtist(void);
static void Beat(void);
static void BeatReset(void);
static gboolean GetAlbumImageHTTP(char *url,char *filename);
static void OverwriteCB(GtkWidget *widget,gpointer data);
static void NoOverwriteCB(GtkWidget *widget,gpointer data);
static gboolean GetAlbumImageFromFile(char *filename,gboolean overwrite);
static void AlbumSelectionReceived(GtkWidget *widget,
				   GtkSelectionData *selection_data, 
				   gpointer data);
static void AlbumImageClicked(GtkWidget *widget,GdkEvent *event,gpointer data);
static void MakeEditPage(void);
static void AutoInstallDB(void);
static void UpdateDB(void);
static void DBScan(void);
static void MakeQueryPage(void);
static void MakeStyles(void);
static GdkGC *MakePen(GtkWidget *widget,int red,int green,int blue);
static void HideMouse(GtkWidget *widget);
static GdkPixbuf *LoadImage(char *filename);
static GtkWidget *LoadImageAsWidget(char *filename);
static void OutputStatus(void);
static void DrawScreen(void);
static IDList *SelectionToIDList(ListWindow *lwin);
static void ListWinCut(GtkWidget *widget,gpointer data);
static void ListWinCopy(GtkWidget *widget,gpointer data);
static void ListWinPaste(GtkWidget *widget,gpointer data);
static void ListWinPasteAtEnd(GtkWidget *widget,gpointer data);
static void DoPaste(ListWindow *lwin,int pos);
static void SongWinToPlaylist(GtkWidget *widget,gpointer data);
static void RandomizeListWin(GtkWidget *widget,gpointer data);
static void SongWinTimeCalc(GtkWidget *widget,gpointer data);
static void SongWinBurn(GtkWidget *widget,gpointer data);
static ListWindow *NewSongWindow(IDList *idlist);
static void SongWindowFromArtistWindow(GtkWidget *widget,gpointer data);
static void DiscWindowFromArtistWindow(GtkWidget *widget,gpointer data);
static void SongWindowFromDiscWindow(GtkWidget *widget,gpointer data);
static void ArtistWindowFromDiscWindow(GtkWidget *widget,gpointer data);
static ListWindow *NewArtistWindow(IDList *idlist);
static void DiscWinButton(GtkWidget *widget,GdkEvent *event,gpointer data);
static void DiscWinTimeCalc(GtkWidget *widget,gpointer data);
static ListWindow *NewDiscWindow(IDList *idlist);

static void Usage(void);
static void ParseArgs(int numargs,char *args[]);

int ser_fd;
int lirc_fd;
int cmdsock_fd;
extern char *x10_button_names[];

gboolean volume_muted=FALSE;
gint save_vol;

GdkColor gdkblack;
GdkColor gdkwhite;
GdkColor *color_LCD;
GdkColor *color_dark_grey;

GtkStyle *style_wb;
GtkStyle *style_LCD;
GtkStyle *style_dark_grey;

int display_mode=1;

MYSQL *mysql=NULL;
MYSQL *root_mysql=NULL;
char sql_host[61]="localhost";
char sql_user[17]="ddj";
char sql_pswd[17]="ddj";
char sql_dbname[33]="ddj_mp3";

gboolean do_x10=FALSE;
char x10_serdevice[256]="/dev/ttyS0";
char status_file[256]="/var/tmp/ddj.dat";
char cover_dir[256]="";
int songlist_limit=250;

int db_version=0;

GtkWidget *main_draw;
GtkWidget *empty_image;

GdkFont *text_font;
GdkGC *text_gc;

GdkPixbuf *album_pixbuf=NULL;

GtkWidget *album_image=NULL;
GtkWidget *no_image=NULL;

GtkWidget *window;
GtkWidget *winbox;
GtkWidget *control_box;
GtkWidget *control_button_box;
GtkWidget *play_clist=NULL;
gchar *play_titles[]={"Title","Artist","Disc","Track","Length"};
GtkWidget *artist_clist;
gchar *artist_titles[]={"Artist"};
GtkWidget *disc_clist;
gchar *disc_titles[]={"Disc","Artist"};
GtkWidget *genre_clist;
gchar *genre_titles[]={"Genre"};
GtkWidget *user_clist;
gchar *user_titles[]={"User","Host"};
GtkWidget *main_notebook;
GtkWidget *play_page;
GtkWidget *query_page;
GtkWidget *edit_page;
GtkWidget *config_page;
GtkWidget *about_page;
GtkWidget *current_song_label;
GtkWidget *play_time_label;
gboolean have_db=FALSE;
gboolean minimized=FALSE;
gboolean control_buttons_visible=TRUE;
gboolean playing=FALSE;
gboolean paused=FALSE;
gboolean auto_start_play=FALSE;
gboolean remote_mode=FALSE;
gboolean console_mode=FALSE;
gboolean tv_mode=FALSE;
char mp3exename[256]="/usr/local/bin/madplay %f";
char mp3fileformat[256]="~/mp3/%a/%d/%n.mp3";
char scroll_display_fmt[256]="  %n -- %a -- %d  ";
pid_t mp3_pid;
gboolean do_debug=FALSE;
int num_songs=0;
int song_order=SONG_ORDER_RANDOM;
int query_list_type=LIST_TYPE_PLAYLIST;
GtkWidget *artist_combo;
GtkWidget *disc_combo;
GtkWidget *genre_combo;
GtkWidget *song_title_entry;
GtkWidget *beat_spin;
GtkWidget *year_spin_button;
int num_beats=0;
clock_t start_time=0,init_time,elapsed_time=0;
gboolean clock_ticking=FALSE;
int idle_count=0;
GtkWidget *low_bpm_spin, *high_bpm_spin;
int display_win_size=20;
GtkWidget *root_pswd_entry;
gboolean have_root_pswd=FALSE;
GtkWidget *tmp_user_name_entry;
GtkWidget *tmp_user_host_entry;
GtkWidget *tmp_user_pswd_entry;

IDStack *song_stack=NULL;
IDList juke_list;

GList *clist_list=NULL;

IDList *clipboard_list=NULL;
int clipboard_type;

Song song_data;
Artist artist_data,disc_artist_data;
Disc disc_data;

char cmdline_file[256]="";

gchar *id3_genres[] = {
  "Blues",
  "Classic Rock",
  "Country",
  "Dance",
  "Disco",
  "Funk",
  "Grunge",
  "Hip-Hop",
  "Jazz",
  "Metal",
  "New Age",
  "Oldies",
  "Other",
  "Pop",
  "R&B",
  "Rap",
  "Reggae",
  "Rock",
  "Techno",
  "Industrial",
  "Alternative",
  "Ska",
  "Death Metal",
  "Pranks",
  "Soundtrack",
  "Euro-Techno",
  "Ambient",
  "Trip-Hop",
  "Vocal",
  "Jazz+Funk",
  "Fusion",
  "Trance",
  "Classical",
  "Instrumental",
  "Acid",
  "House",
  "Game",
  "Sound Clip",
  "Gospel",
  "Noise",
  "AlternRock",
  "Bass",
  "Soul",
  "Punk",
  "Space",
  "Meditative",
  "Instrumental Pop",
  "Instrumental Rock",
  "Ethnic",
  "Gothic",
  "Darkwave",
  "Techno-Industrial",
  "Electronic",
  "Pop-Folk",
  "Eurodance",
  "Dream",
  "Southern Rock",
  "Comedy",
  "Cult",
  "Gangsta",
  "Top 40",
  "Christian Rap",
  "Pop/Funk",
  "Jungle",
  "Native American",
  "Cabaret",
  "New Wave",
  "Psychadelic",
  "Rave",
  "Showtunes",
  "Trailer",
  "Lo-Fi",
  "Tribal",
  "Acid Punk",
  "Acid Jazz",
  "Polka",
  "Retro",
  "Musical",
  "Rock & Roll",
  "Hard Rock",
  "Folk",
  "Folk/Rock",
  "National Folk",
  "Swing",
  "Bebob",
  "Latin",
  "Revival",
  "Celtic",
  "Bluegrass",
  "Avantgarde",
  "Gothic Rock",
  "Progressive Rock",
  "Psychedelic Rock",
  "Symphonic Rock",
  "Slow Rock",
  "Big Band",
  "Chorus",
  "Easy Listening",
  "Acoustic",
  "Humour",
  "Speech",
  "Chanson",
  "Opera",
  "Chamber Music",
  "Sonata",
  "Symphony",
  "Booty Bass",
  "Primus",
  "Porn Groove",
  "Satire",
  "Slow Jam",
  "Club",
  "Tango",
  "Samba",
  "Folklore",
  "Ballad",
  "Power Ballad",
  "Rhythmic Soul",
  "Freestyle",
  "Duet",
  "Punk Rock",
  "Drum Solo",
  "Acapella",
  "Euro-house",
  NULL
};

CmdMap cmd_map[]={
  {"play",DDJ_PLAY},
  {"stop",DDJ_STOP},
  {"next_song",DDJ_NEXT_SONG},
  {"prev_song",DDJ_PREV_SONG},
  {"vol_up",DDJ_VOL_UP},
  {"vol_dn",DDJ_VOL_DN},
  {"mute",DDJ_MUTE},
  {"describe",DDJ_DESCRIBE},
  {"display",DDJ_DISPLAY},
  {"exit",DDJ_EXIT},
  {"all_songs",DDJ_ALL_SONGS},
  {"artist_songs",DDJ_ARTIST_SONGS},
  {"disc_songs",DDJ_DISC_SONGS},
  {"genre_songs",DDJ_GENRE_SONGS},
  {"title_songs",DDJ_TITLE_SONGS},
  {"year_songs",DDJ_YEAR_SONGS},
  {"2year_songs",DDJ_2YEAR_SONGS},
  {"decade_songs",DDJ_DECADE_SONGS},
  {"recent_songs",DDJ_RECENT_SONGS},
  {"recall",DDJ_RECALL},
  {"randomize",DDJ_RANDOMIZE},
  {"juke_add_song",DDJ_JUKE_ADD_SONG},
  {"load_playlist",DDJ_LOAD_PLAYLIST},
  {"",0}
};

CFGEntry cfg_entries[]={
  {"sql_host",CFG_ENTRY_STRING,61,sql_host},
  {"sql_dbname",CFG_ENTRY_STRING,33,sql_dbname},
  {"sql_user",CFG_ENTRY_STRING,17,sql_user},
  {"sql_pswd",CFG_ENTRY_STRING,17,sql_pswd},
  {"mp3fileformat",CFG_ENTRY_STRING,256,mp3fileformat},
  {"mp3exename",CFG_ENTRY_STRING,256,mp3exename},
  {"x10_serdevice",CFG_ENTRY_STRING,256,x10_serdevice},
  {"do_x10",CFG_ENTRY_BOOL,0,&do_x10},
  {"status_file",CFG_ENTRY_STRING,256,status_file},
  {"cover_dir",CFG_ENTRY_STRING,256,cover_dir},
  {"songlist_limit",CFG_ENTRY_INT,0,&songlist_limit},
  {"",CFG_ENTRY_LAST,0,NULL}
};

static char * empty_xpm[] = {
  "1 1 1 1",
  "     c None",
  " "
};

static void ShutDown(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  KillMP3();

  gtk_main_quit();

  if(have_db) DisconnectSQL(mysql);

  mysql_close(root_mysql);

  if(do_x10) {
    x10_close(ser_fd);
  }

#ifdef HAVE_LIRC_LIRC_CLIENT_H
  close_lirc();
#endif
}

void Debug(char *fmt,...)
{
  va_list args;

  if(do_debug) {
    va_start(args,fmt);

    vprintf(fmt,args);
  }

  va_end(args);
}

/* See if the window is going off of the screen, and if so, move it */

static void CheckWindowPosition(void)
{
  int x,y,width,height,newx,newy;
  int x_pos,y_pos;
  int scr_width,scr_height;
  gboolean doit=FALSE;

  gdk_window_get_origin(window->window,&x,&y);
  gdk_window_get_size(window->window,&width,&height);
  scr_width=gdk_screen_width();
  scr_height=gdk_screen_height();

  newx=x; newy=y;

  if((x+width)>scr_width) {
    newx=scr_width-width;
    doit=TRUE;
  }

  if((y+height)>scr_height) {
    newy=scr_height-height;
    doit=TRUE;
  }

  if(doit) {
    gdk_window_get_root_origin(window->window,&x_pos,&y_pos);

    gdk_window_move(window->window,newx-(x-x_pos),newy-(y-y_pos));
  }
}

static void IncVolume(void)
{
  gint fd;
  gint vol;

  if(volume_muted) return;

  if((fd=OpenMixer(MIXER_DEVICE))>=0) {
    vol=ReadMixer(fd);

    SetMixer(fd,vol+2);

    CloseMixer(fd);
  }
}

static void DecVolume(void)
{
  gint fd;
  gint vol;

  if(volume_muted) return;

  if((fd=OpenMixer(MIXER_DEVICE))>=0) {
    vol=ReadMixer(fd);

    SetMixer(fd,vol-2);

    CloseMixer(fd);
  }
}

static void ToggleMute(void)
{
  gint fd;

  if((fd=OpenMixer(MIXER_DEVICE))>=0) {
    if(volume_muted) {
      SetMixer(fd,save_vol);
    }
    else {
      save_vol=ReadMixer(fd);
      SetMixer(fd,0);
    }

    volume_muted=!volume_muted;

    CloseMixer(fd);
  }
}

static void DescribeSong(void)
{
  char buf[1024];

  if(!song_stack->current_idlist->num_ids) return;

  sprintf(buf,"perl %s/fest.pl \"%s\" \"%s\"",
	  AUXDIR,song_data.title,artist_data.name);

  if(playing) PlayCB();
  system(buf);
  if(paused) PlayCB();
}

/* Get a command number from a command name */
static int CommandFromName(char *name,char **arg)
{
  int cmd;
  char *c;

  *arg=NULL;

  for(c=name;*c&&*c!=' ';c++);

  if(*c==' ') {
    *c='\0';
    *arg=c+1;
  }

  for(cmd=0;*cmd_map[cmd].cmd_name;cmd++) {
    if(!strcasecmp(cmd_map[cmd].cmd_name,name))
      return cmd;
  }

  return -1;
}

#ifdef HAVE_LIRC_LIRC_CLIENT_H

/* Handle incoming lirc commands */
static void LircCB(gpointer data,gint source,GdkInputCondition condition)
{
  char cmd_name[256];
  int cmd;
  char *arg=NULL;

  while(lirc_get_event(lirc_fd,cmd_name)==0) {
    cmd=CommandFromName(cmd_name,&arg);

    if(cmd!=-1) DoCommand(cmd,arg);
  }
}
#endif /* HAVE_LIRC_LIRC_CLIENT_H */

/* Handle incoming command socket commands */
static void CommandSocketCB(gpointer data,gint source,
			    GdkInputCondition condition)
{
  char cmd_name[256];
  int cmd;
  char *arg=NULL;

  while(GetSocketCommand(cmdsock_fd,cmd_name,256)>=0) {
    cmd=CommandFromName(cmd_name,&arg);
    
    if(cmd!=-1) DoCommand(cmd,arg);
  }
}

/* Conditional to allow us to skip every other command */
static int x10_doit=1;

/* Handle x10 events coming in over the serial port */
static void SerialCB(gpointer data,gint source,GdkInputCondition condition)
{
  int button;
  int cmd=-1;

  while((button=x10_get_event(ser_fd))>=0) {
    if(x10_doit) {
      x10_doit=0;

      switch(button) {
      case X10_PLAY:
	cmd=DDJ_PLAY;
	break;
      case X10_STOP:
	cmd=DDJ_STOP;
	break;
      case X10_REW:
	cmd=DDJ_PREV_SONG;
	break;
      case X10_FF:
	cmd=DDJ_NEXT_SONG;
	break;
      case X10_PAUSE:
	cmd=DDJ_PLAY;
	break;
      case X10_VOL_UP:
	cmd=DDJ_VOL_UP;
	break;
      case X10_VOL_DN:
	cmd=DDJ_VOL_DN;
	break;
      case X10_MUTE:
	cmd=DDJ_MUTE;
	break;
      case X10_SKIP_FF:
	cmd=DDJ_NEXT_SONG;
	break;
      case X10_SKIP_REW:
	cmd=DDJ_PREV_SONG;
	break;
      case X10_DISPLAY:
	cmd=DDJ_DESCRIBE;
	break;
      case X10_DVD_UP:
	cmd=DDJ_PREV_SONG;
	break;
      case X10_DVD_DN:
	cmd=DDJ_NEXT_SONG;
	break;
      case X10_DVD_RIGHT:
	cmd=DDJ_NEXT_SONG;
	break;
      case X10_DVD_LEFT:
	cmd=DDJ_PREV_SONG;
	break;
      case X10_DVD_OK:
	cmd=DDJ_PLAY;
	break;
      case X10_A_B:
	cmd=DDJ_RANDOMIZE;
	break;
      case X10_TITLE:
	cmd=DDJ_DISPLAY;
	break;
      case X10_RETURN:
	cmd=DDJ_EXIT;
	break;
      case X10_RECALL:
	cmd=DDJ_RECALL;
	break;
      case X10_1:
	cmd=DDJ_ALL_SONGS;
	break;
      case X10_2:
	cmd=DDJ_ARTIST_SONGS;
	break;
      case X10_3:
	cmd=DDJ_DISC_SONGS;
	break;
      case X10_4:
	cmd=DDJ_GENRE_SONGS;
	break;
      case X10_5:
	cmd=DDJ_TITLE_SONGS;
	break;
      case X10_7:
	cmd=DDJ_YEAR_SONGS;
	break;
      case X10_8:
	cmd=DDJ_2YEAR_SONGS;
	break;
      case X10_9:
	cmd=DDJ_DECADE_SONGS;
	break;
      case X10_0:
	cmd=DDJ_RECENT_SONGS;
	break;
      }

      if(cmd!=-1) DoCommand(cmd,NULL);
    }
    else x10_doit=1;
  }
}

static void DoCommand(int command,char *arg)
{
  char buf[256];
  char query[1024];
  PlaylistInfo pinfo;
  IDList *tmp_idlist=NULL;
  IDList *dest_idlist=NULL;

  switch(command) {
  case DDJ_PLAY:
    PlayCB();
    break;
  case DDJ_STOP:
    StopPlay();
    break;
  case DDJ_NEXT_SONG:
    NextSong();
    break;
  case DDJ_PREV_SONG:
    PrevSong();
    break;
  case DDJ_VOL_UP:
    IncVolume();
    break;
  case DDJ_VOL_DN:
    DecVolume();
    break;
  case DDJ_MUTE:
    ToggleMute();
    break;
  case DDJ_DESCRIBE:
    DescribeSong();
    break;
  case DDJ_EXIT:
    ShutDown(NULL,NULL,NULL);
    break;
  case DDJ_DISPLAY:
    if(tv_mode) {
      display_mode=(display_mode+1)%2;
    
      DrawScreen();
    }
    break;
  case DDJ_ALL_SONGS:
    PushIDList(song_stack);
    
    GetAllSongs(mysql,song_stack->current_idlist);
    
    if(!(tv_mode || console_mode))
       TrimIDList(song_stack->current_idlist,songlist_limit);

    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);
    
    ResetCurrentPlaylist();
    
    break;
  case DDJ_ARTIST_SONGS:
    PushIDList(song_stack);
    
    sprintf(query,"SELECT id FROM song WHERE artistid=%d",
	    song_data.artistid);
    
    GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
    
    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);
    
    ResetCurrentPlaylist();
    
    break;
  case DDJ_DISC_SONGS:
    PushIDList(song_stack);
    
    sprintf(query,"SELECT id FROM song WHERE discid=%d ORDER BY track_num",
	    song_data.discid);
    
    GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
    
    ResetCurrentPlaylist();
    
    break;
  case DDJ_GENRE_SONGS:
    PushIDList(song_stack);
    
    sprintf(query,"SELECT id FROM song WHERE genre='%s'",
	    song_data.genre);
    
    GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
    
    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);
    
    ResetCurrentPlaylist();
    
    break;
  case DDJ_TITLE_SONGS:
    PushIDList(song_stack);
    
    EscapeQuote(song_data.title,buf,256);
    
    sprintf(query,"SELECT id FROM song WHERE title='%s'",
	    buf);
    
    GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
    
    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);
    
    ResetCurrentPlaylist();
    break;
  case DDJ_YEAR_SONGS:
    if(song_data.year>0) {
      PushIDList(song_stack);
      
      sprintf(query,"SELECT id FROM song WHERE year=%d",
	      song_data.year);
      
      GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
      
      ScrambleInts(song_stack->current_idlist->ids,
		   song_stack->current_idlist->num_ids);
      
      ResetCurrentPlaylist();
    }
    break;
  case DDJ_2YEAR_SONGS:
    if(song_data.year>0) {
      PushIDList(song_stack);
      
      sprintf(query,"SELECT id FROM song WHERE year>%d and year<%d",
	      song_data.year-2,song_data.year+2);
      
      GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
      
      ScrambleInts(song_stack->current_idlist->ids,
		   song_stack->current_idlist->num_ids);
      
      ResetCurrentPlaylist();
    }
    break;
  case DDJ_DECADE_SONGS:
    if(song_data.year>0) {
      PushIDList(song_stack);
      
      sprintf(query,"SELECT id FROM song WHERE year>%d and year<%d",
	      ((song_data.year/10)*10)-1,((song_data.year/10+1)*10));
      
      GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
      
      ScrambleInts(song_stack->current_idlist->ids,
		   song_stack->current_idlist->num_ids);
      
      ResetCurrentPlaylist();
    }
    break;
  case DDJ_RECENT_SONGS:
    PushIDList(song_stack);
    
    sprintf(query,"SELECT id FROM song ORDER BY date_added DESC "
	    "LIMIT 100");
    
    GetListFromQuery(mysql,song_stack->current_idlist,query,TRUE);
    
    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);
    
    ResetCurrentPlaylist();
    break;
  case DDJ_RECALL:
    PopPlaylist();
    break;
  case DDJ_RANDOMIZE:
    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);	
    ResetCurrentPlaylist();
    break;
  case DDJ_JUKE_ADD_SONG:
    InsertID(&juke_list,atoi(arg),juke_list.num_ids);
    break;
  case DDJ_LOAD_PLAYLIST:
    if(GetPlaylistInfoFromDB(mysql,atoi(arg),&pinfo)) {
      PushIDList(song_stack);

      if(pinfo.type==LIST_TYPE_SONG)
	dest_idlist=song_stack->current_idlist;
      else {
	tmp_idlist=g_new(IDList,1);

	dest_idlist=tmp_idlist;
      }

      LoadPlaylistFromDB(mysql,atoi(arg),dest_idlist);

      switch(pinfo.type) {
      case LIST_TYPE_SONG:
	break;
      case LIST_TYPE_ARTIST:
	ArtistListToSongList(mysql,tmp_idlist,song_stack->current_idlist);
	break;
      case LIST_TYPE_DISC:
	DiscListToSongList(mysql,tmp_idlist,song_stack->current_idlist);
	break;
      }

      if(tmp_idlist) {
	FreeIDList(tmp_idlist);
	g_free(tmp_idlist);
      }		     

      ResetCurrentPlaylist();
    }

    break;
  }  
}

static void StartClock(void)
{
  struct tms t;

  if(!clock_ticking) {
    init_time=times(&t);
    clock_ticking=TRUE;
  }
}

static void StopClock(void)
{
  clock_t end_time;
  struct tms t;

  if(clock_ticking) {
    end_time=times(&t);
  
    elapsed_time+=(end_time-init_time);
    clock_ticking=FALSE;
  }
}

static void ResetClock(void)
{
  StopClock();
  elapsed_time=0;
}

static int GetClockSecs(void)
{
  clock_t curr_time;
  struct tms t;

  if(!clock_ticking) return elapsed_time/CLK_TCK;
  else {
    curr_time=times(&t);

    return (elapsed_time+(curr_time-init_time))/CLK_TCK;
  }
}

/* Put the computer in a sleep state */
static void DoSuspend(void)
{
  idle_count=0;
  //system("apm --suspend");
}

static gint TimeOut(gpointer data)
{
  int secs;
  char buf[6];
  static int counter=0;

  if(playing) {
    idle_count=0;

    if(!paused||(counter++%4)) {
      secs=GetClockSecs();

      if(!(tv_mode || console_mode)) {
	g_snprintf(buf,6,"%02d:%02d",secs/60,secs%60);

	gtk_label_set_text(GTK_LABEL(play_time_label),buf);
      }
    }
    else {
      if(!(tv_mode || console_mode)) {
	gtk_label_set_text(GTK_LABEL(play_time_label),"     ");
      }
    }

    if(waitpid(mp3_pid,NULL,WNOHANG)) {
      waitpid(mp3_pid,NULL,0);

      playing=FALSE;
      if(NextSong()) Play();
    }
  }
  else {
   idle_count++;

    if(idle_count>(4*120)) {
      DoSuspend();
    }
  }

  if(!tv_mode && !console_mode)
    UpdateScrollDisplay(FALSE);

  return TRUE;
}

static void DoSaveConfig(void)
{
  char filename[256];

  g_snprintf(filename,256,"%s/.ddj",getenv("HOME"));

  if(!SaveConfig(filename,"DDJ",1,cfg_entries))
    printf("Error: Unable to save config file to %s\n",filename);
}

static void KillMP3(void)
{
  if(playing) {
    if(paused) {
      kill(-mp3_pid,SIGCONT);
      paused=FALSE;
    }

    kill(-mp3_pid,SIGINT);
    waitpid(mp3_pid,NULL,0);
    ResetClock();

    if(!(tv_mode || console_mode)) {
      gtk_label_set_text(GTK_LABEL(play_time_label),"00:00");
    }
  }
}

static void RedirectIO(gboolean redirect_output)
{
  int fd;

  fd=open("/dev/null",O_RDWR);
  dup2(fd,0);

  if(redirect_output) {
    dup2(fd,1);
    dup2(fd,2);
  }

  close(fd); 

  /* Close any other filehandles that might be around */
  for(fd=3;fd<NOFILE;fd++) {
    close(fd);
  }
}

static char *FindRoot(char *str)
{
  char *c;

  for(c=str+strlen(str);c>str;c--) {
    if(*c=='/') return c+1;
  }

  return c;
}

static void MakeArgs(char *str,char **args,int maxargs)
{
  int arg;
  gboolean inquotes=FALSE;

  for(arg=0,args[0]=str;*str&&arg<(maxargs-1);str++) {
    if(*str=='"') {
      if(inquotes) *str='\0';
      else args[arg]=str+1;

      inquotes=!inquotes;
    }
    else if(*str==' '&&!inquotes) {
      *str='\0';
      args[++arg]=str+1;
    }
  }

  args[++arg]=NULL;
}

static pid_t LaunchProg(char *prog)
{
  pid_t pid;
  char *args[50];
  char *exename,*argstr;

  exename=strtok(prog," ");

  if(!exename) {
    printf("Error: Unable to launch [%s]\n",prog);

    return -1;
  }

  args[0]=FindRoot(exename);

  argstr=strtok(NULL,"");

  if(argstr)
    MakeArgs(argstr,args+1,49);
  else args[1]=NULL;

  pid=fork();

  if(pid==-1) {
    printf("Error: Unable to launch [%s]\n",prog);

    return -1;
  }

  if(pid==0) {
    /* Set process group id so we can kill it better */
    setpgid(0,0);

    if(!console_mode)
      close(ConnectionNumber(GDK_DISPLAY()));

    RedirectIO(TRUE);
    execv(exename,args);
    _exit(0);
  }

  return pid;
}

static void SavePlaylist(GtkWidget *widget,gpointer data)
{
  PlaylistInfo *list_info;
  ListWindow *list_win;

  list_info=g_new(PlaylistInfo,1);

  list_win=(ListWindow *)data;

  if(list_win) {
    list_info->type=list_win->list_type;
    list_info->idlist=list_win->idlist;
  }
  else {
    list_info->type=LIST_TYPE_SONG;
    list_info->idlist=song_stack->current_idlist;
  }

  SelectPlaylist(mysql,TRUE,DoSavePlaylist,list_info);
}

static void DoSavePlaylist(GtkWidget *widget,gpointer data)
{
  PlaylistInfo *list_info;

  if(!data) return;

  list_info=(PlaylistInfo *)data;

  StorePlaylistToDB(mysql,list_info,list_info->idlist,TRUE);

  g_free(list_info);
}

static void LoadPlaylist(void)
{
  PlaylistInfo *list_info;

  list_info=g_new(PlaylistInfo,1);

  SelectPlaylist(mysql,FALSE,DoLoadPlaylist,list_info);
}

static void DoLoadPlaylist(GtkWidget *widget,gpointer data)
{
  PlaylistInfo *list_info;
  IDList *new_idlist;

  if(!data) return;

  list_info=(PlaylistInfo *)data;

  switch(list_info->type) {
  case LIST_TYPE_SONG:
    new_idlist=g_new(IDList,1);

    /*    PushIDList(song_stack);*/

    if(LoadPlaylistFromDB(mysql,list_info->id,new_idlist)) {
      NewSongWindow(new_idlist);
      
      /*      ResetCurrentPlaylist();*/
      break;
    }
    else {
      g_free(new_idlist);
      /*      PopIDList(song_stack);*/
    }
    break;

  case LIST_TYPE_ARTIST:
    new_idlist=g_new(IDList,1);

    if(LoadPlaylistFromDB(mysql,list_info->id,new_idlist)) {
      NewArtistWindow(new_idlist);
      break;
    }
    else {
      g_free(new_idlist);
    }
    break;

  case LIST_TYPE_DISC:
    new_idlist=g_new(IDList,1);

    if(LoadPlaylistFromDB(mysql,list_info->id,new_idlist)) {
      NewDiscWindow(new_idlist);
      break;
    }
    else {
      g_free(new_idlist);
    }
    break;
  }

  g_free(list_info);
}

/* Revert to the previsou playlist */
static void PopPlaylist(void)
{
  if(PopIDList(song_stack)) {
    if(tv_mode || console_mode) {
      SelectSong(song_stack->current_idlist,
		 song_stack->current_idlist->current_id);
    }
    else {
      IDListToSongCList(mysql,song_stack->current_idlist,&play_clist);
      
      gtk_clist_select_row(GTK_CLIST(play_clist),
			   song_stack->current_idlist->current_id,0);    
    }
  }
}

static void MinMax(void)
{
  if(minimized) {
    gtk_container_border_width(GTK_CONTAINER(winbox),3);
    gtk_widget_show(main_notebook);
  }
  else {
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,TRUE);
    gtk_container_border_width(GTK_CONTAINER(winbox),0);
    gtk_widget_hide(main_notebook);

    gtk_label_set(GTK_LABEL(current_song_label),"");

    UpdateGTK();
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);

    UpdateScrollDisplay(FALSE);
  }

  minimized=!minimized;
}

static void ToggleControlButtons(void)
{
  if(control_buttons_visible) {
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,TRUE);
    gtk_widget_hide(control_button_box);

    UpdateGTK();
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);
  }
  else {
    gtk_widget_show(control_button_box);
  }

  control_buttons_visible=!control_buttons_visible;
}

static void ParseMP3Cmd(char *instr,char *outstr,char *filename,int disc_gain)
{
  int left=1024;

  for(;*instr&&(left>0);instr++) {
    if(*instr=='%') {
      switch(*(++instr)) {
      case '%':
	g_snprintf(outstr,left,"%%");
	break;
      case 'f':
	g_snprintf(outstr,left,"\"%s\"",filename);
	break;
      case 'r':
	g_snprintf(outstr,left,"\"%+6.2f\"",((double)disc_gain)/100.0);
	break;
      }
      
      left-=strlen(outstr);
      outstr+=strlen(outstr);
    }
    else {
      if(left>0) {
	*outstr++=*instr;
	left--;
      }
    }
  }
  
  *outstr='\0';
}

static void ParseFileFmt(char *instr,char *outstr)
{
  int left=1024;
  char *tok;
  struct passwd *pwd;

  *outstr='\0';

  /* Expand ~ in dir -- modeled loosely after code from gtkfind by
                        Matthew Grossman */

  if(*instr=='~') {
    instr++;

    if((*instr=='\0') || (*instr=='/')) {   /* This user's dir */
      left-=g_snprintf(outstr,left,"%s",getenv("HOME"));
    }
    else {  /* Another user's dir */
      tok=strchr(instr,'/');

      if(tok) {   /* Ugly, but it works */
	*tok='\0';
	pwd=getpwnam(instr);
	instr+=strlen(instr);
	*tok='/';
      }
      else {
	pwd=getpwnam(instr);
	instr+=strlen(instr);
      }

      if(!pwd)
	printf("Error: unable to translate filename. No such user as %s\n",
	       tok);
      else {
	left-=g_snprintf(outstr,left,"%s",pwd->pw_dir);
      }
    }
    
    outstr+=strlen(outstr);
  }

  for(;*instr&&left>0;instr++) {
    if(*instr=='%') {
      switch(*(++instr)) {
      case '%':
	g_snprintf(outstr,left,"%%");
	break;
      case 't':
	g_snprintf(outstr,left,"%02d",song_data.track_num+1);
	break;
      case 'n':
	g_snprintf(outstr,left,"%s",song_data.title);
	break;
      case 'a':
	g_snprintf(outstr,left,"%s",artist_data.name);
	break;
      case 'A':
	if(disc_artist_data.id==artist_data.id)
	  g_snprintf(outstr,left,"%s",artist_data.name);
	else g_snprintf(outstr,left,"%s",disc_artist_data.name);
	break;
      case 'd':
	g_snprintf(outstr,left,"%s",disc_data.title);
	break;
      }
	  
      left-=strlen(outstr);
      outstr+=strlen(outstr);
    }
    else {
      if(left>0) {
	*outstr++=*instr;
	left--;
      }
    }
  }
  
  *outstr='\0';
}

static void PlayCB(void)
{
  if(!song_stack->current_idlist->num_ids) return;

  if(playing) {
    if(paused) {
      kill(-mp3_pid,SIGCONT);
      paused=FALSE;

      StartClock();
    }
    else {
      kill(-mp3_pid,SIGSTOP);
      paused=TRUE;

      StopClock();
    }
  }
  else Play();
}

static void Play(void)
{
  char buf[1024];
  char exe_string[1024];

  if(!song_stack->current_idlist->num_ids) return;

  if(remote_mode) {
    g_snprintf(buf,1024,"http://somewhere/cgi-bin/ddj/play.cgi?song_id=%d&play_type=mp3",song_data.id);
    ParseMP3Cmd(mp3exename,exe_string,buf,disc_data.gain_adjustment);

    Debug("URL is %s\n",buf);
  }
  else {
    ParseMP3Cmd(mp3exename,exe_string,song_data.filename,
		disc_data.gain_adjustment);
  }

  Debug("MP3 exe is [%s]\n",exe_string);

  mp3_pid=LaunchProg(exe_string);

  if(mp3_pid!=-1) {
    playing=TRUE;
    
    g_snprintf(buf,1024,"UPDATE song SET num_plays=num_plays+1 WHERE id=%d",
	     song_data.id);
    
    mysql_query(mysql,buf);
    
    g_snprintf(buf,1024,"UPDATE song SET last_play=NOW() WHERE id=%d",
	     song_data.id);
    
    mysql_query(mysql,buf);
    
    g_snprintf(buf,1024,"UPDATE artist SET num_plays=num_plays+1 WHERE id=%d",
	     song_data.artistid);
    
    mysql_query(mysql,buf);
    
    g_snprintf(buf,1024,"UPDATE artist SET last_play=NOW() WHERE id=%d",
	     song_data.artistid);
    
    mysql_query(mysql,buf);
    
    g_snprintf(buf,1024,"UPDATE disc SET num_plays=num_plays+1 WHERE id=%d",
	     song_data.discid);
    
    mysql_query(mysql,buf);
    
    g_snprintf(buf,1024,"UPDATE disc SET last_play=NOW() WHERE id=%d",
	     song_data.discid);
    
    mysql_query(mysql,buf);
  }

  ResetClock();
  StartClock();
}

static void StopPlay(void)
{
  if(playing) {
    KillMP3();
    playing=FALSE;
    paused=FALSE;
  }
}

static gboolean NextSong(void)
{
  IDList *slist;

  if(juke_list.num_ids) {
    SelectSong(&juke_list,0);
    RemoveIDs(&juke_list,0,1);

    return TRUE;
  }
  else {
    slist=song_stack->current_idlist;
    
    if(slist->current_id<(slist->num_ids-1)) {
      if(tv_mode || console_mode) {
	SelectSong(slist,slist->current_id+1);
      }
      else {
	gtk_clist_select_row(GTK_CLIST(play_clist),
			     slist->current_id+1,0);
      }
      
      return TRUE;
    }
    else {
      StopPlay();
      
      return FALSE;
    }
  }
}

static gboolean PrevSong(void)
{
  IDList *slist;

  slist=song_stack->current_idlist;

  if(!playing||(GetClockSecs()<2)) {
      if(slist->current_id)
	if(tv_mode || console_mode) {
	  SelectSong(slist,slist->current_id-1);
	}
	else {
	  gtk_clist_select_row(GTK_CLIST(play_clist),
			       slist->current_id-1,0);
	}
      else {
	StopPlay();
	return FALSE;
      }
  }
    
  if(playing) {
    KillMP3();
    Play();
  }

  return TRUE;
}

static gboolean LookUpData(int id,Song *song,Artist *artist,
			   Artist *disc_artist,Disc *disc)
{
  if(!have_db) return FALSE;

  LookUpSong(mysql,id,song);

  LookUpDisc(mysql,song->discid,disc);

  LookUpArtist(mysql,song->artistid,artist);

  LookUpArtist(mysql,disc->artistid,artist);

  return TRUE;
}

static void SelectRow(GtkWidget *widget,gint row,gint column,
		      GdkEventButton *event,gpointer data)
{
  SelectSong(song_stack->current_idlist,row);

  if(gtk_clist_row_is_visible(GTK_CLIST(play_clist),row)!=GTK_VISIBILITY_FULL)
    gtk_clist_moveto(GTK_CLIST(play_clist),row,0,0,0);
}

static void UpdateScrollDisplay(gboolean init)
{
  static char display_window[1024];
  static char display_buf[1024];
  static int display_buf_len=0;
  static int display_pos=0;
  static int curr_display_size=0;
  int left;
  int points_per_char;

  if(!have_db || !song_stack->current_idlist->num_ids) return;

  points_per_char=gdk_string_width(current_song_label->style->font,"W");
  display_win_size=current_song_label->allocation.width/points_per_char;

  if(display_win_size<15) display_win_size=15;

  if(init||(curr_display_size!=display_win_size)) {
    ParseFileFmt(scroll_display_fmt,display_buf);

    curr_display_size=display_win_size;
    display_pos=0;
    display_buf_len=strlen(display_buf);
  }
  else {
    if(display_buf_len<display_win_size)
      return;
    
    display_pos++;
    
    if(display_pos==display_buf_len) display_pos=0;
  }

  left=display_buf_len-display_pos;

  if((left>=display_win_size)||(display_buf_len<=display_win_size))
    g_snprintf(display_window,display_win_size,"%s",display_buf+display_pos);
  else {
    sprintf(display_window,"%s",display_buf+display_pos);
    g_snprintf(display_window+left,display_win_size-left,"%s",display_buf);
  }

  gtk_label_set(GTK_LABEL(current_song_label),display_window);
}

static char *BaseName(char *string)
{
  char *c;

  for(c=string+strlen(string);*c!='/'&&c!=string;c--);

  if(*c=='/') c++;

  return c;
}

/* Reset conditions to reflect a new active playlist */
static void ResetCurrentPlaylist(void)
{
  if(tv_mode || console_mode) {
    SelectSong(song_stack->current_idlist,0);
  }
  else {
    IDListToSongCList(mysql,song_stack->current_idlist,&play_clist);

    gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    
  }
}

static void LoadAlbumImage(void)
{
  char data_filename[1024];
  GtkWidget *image;

  if(tv_mode) {
    if(album_pixbuf)
      gdk_pixbuf_unref(album_pixbuf);
    
    sprintf(data_filename,"%s/%d.jpg",cover_dir,disc_data.id);
    
    album_pixbuf=LoadImage(data_filename);

    DrawScreen();
  }
  else if(!console_mode) {
    sprintf(data_filename,"%s/%d.jpg",cover_dir,disc_data.id);
    
    image=LoadImageAsWidget(data_filename);
    
    if(image) {
      CopyPixmap(GTK_PIXMAP(image),GTK_PIXMAP(album_image));
    }
    else {
      CopyPixmap(GTK_PIXMAP(no_image),GTK_PIXMAP(album_image));
    }
  }
}

static void SelectSong(IDList *slist,int song)
{
  int id;
  char *filename;

  if(!song_stack->current_idlist->num_ids) return;

  slist->current_id=song;

  LookUpData(slist->ids[song],&song_data,&artist_data,
	     &disc_artist_data,&disc_data);

  LoadAlbumImage();

  if(!(tv_mode || console_mode)) {
    id=(int)gtk_clist_get_row_data(GTK_CLIST(play_clist),song);
    
    if(id==-1) {
      gtk_clist_get_text(GTK_CLIST(play_clist),song,0,
			 (gchar **)&filename);
      strncpy(song_data.filename,filename,256);
      strncpy(song_data.title,BaseName(filename),80);
    }
    
    UpdateScrollDisplay(TRUE);
    
    gtk_entry_set_text(GTK_ENTRY(song_title_entry),song_data.title);
    
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(artist_combo)->entry),
		       artist_data.name);
    
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(disc_combo)->entry),
		       disc_data.title);
    
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(genre_combo)->entry),
		       song_data.genre);
    
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(beat_spin),song_data.bpm);
    
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(year_spin_button),
			      song_data.year);
    
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,TRUE);
    UpdateGTK();
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);
    
    num_beats=0;
  }

  if(playing) {
    KillMP3();
    Play();
  }

  OutputStatus();
}

static void MakePlayPage(void)
{
  GtkWidget *listobj;

  play_page=MakeNewPage("Playlist");

  listobj=IDListToSongCList(mysql,NULL,&play_clist);
  gtk_widget_set_usize(play_clist,585,320);

  gtk_signal_connect(GTK_OBJECT(play_clist),"select_row",
		     GTK_SIGNAL_FUNC(SelectRow),
		     NULL);

  gtk_container_add(GTK_CONTAINER(play_page),listobj);
  gtk_widget_show(listobj);
}

static void FillArtistCList(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;

  if(!have_db) return;

  gtk_clist_clear(GTK_CLIST(artist_clist));

  if(mysql_query(mysql,"SELECT artist FROM artist order by artist")) {
    printf("Query error\n");

    return;
  }

  if(!(res = mysql_store_result(mysql))) {
    printf("Query error\n");

    return;
  }

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(artist_clist),row);
  }

  mysql_free_result(res);
}

static void ArtistChanged(void)
{
  FillDiscCList();
}

static void FillDiscCList(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  GString *query;
  GList *alist;
  char *artist_name;
  gboolean first=TRUE;
  char tartist[256];
  int pos;

  if(!have_db) return;

  gtk_clist_clear(GTK_CLIST(disc_clist));

  query=g_string_new(NULL);

  g_string_append(query,"SELECT title,artist,disc.id FROM disc,artist WHERE "
		  "disc.artistid=artist.id");

  alist=GTK_CLIST(artist_clist)->selection;

  while(alist) {
    gtk_clist_get_text(GTK_CLIST(artist_clist),
		       GPOINTER_TO_INT(alist->data),0,
		       &artist_name);
    
    EscapeQuote(artist_name,tartist,256);

    if(first) {
      first=FALSE;
      g_string_sprintfa(query," AND (artist.artist='%s'",tartist);
    }
    else
      g_string_sprintfa(query," OR artist.artist='%s'",tartist);

    alist=alist->next;
  }

  if(!first) g_string_append(query,")");

  g_string_append(query," order by title");

  if(mysql_query(mysql,query->str)) {
    printf("Query error\n");

    g_string_free(query,TRUE);

    return;
  }

  g_string_free(query,TRUE);

  if(!(res = mysql_store_result(mysql))) {
    printf("Query error\n");

    return;
  }

  pos=0;

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(disc_clist),row);

    gtk_clist_set_row_data(GTK_CLIST(disc_clist),pos++,(gpointer)atoi(row[2]));
  }

  mysql_free_result(res);
}

static void QueryGenres(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;

  if(!have_db) return;

  gtk_clist_clear(GTK_CLIST(genre_clist));

  if(mysql_query(mysql,"SELECT distinct genre FROM song")) {
    printf("Query error\n");

    return;
  }

  if(!(res = mysql_store_result(mysql))) {
    printf("Query error\n");

    return;
  }

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(genre_clist),row);
  }

  mysql_free_result(res);
}

static gboolean SingleFileList(char *filename)
{
  char *cols[4];

  StopPlay();

  gtk_clist_freeze(GTK_CLIST(play_clist));
  gtk_clist_clear(GTK_CLIST(play_clist));

  cols[0]=filename;
  cols[1]=NULL;
  cols[2]=NULL;
  cols[3]=NULL;

  gtk_clist_append(GTK_CLIST(play_clist),cols);
  gtk_clist_set_row_data(GTK_CLIST(play_clist),0,(gpointer)-1);

  gtk_clist_thaw(GTK_CLIST(play_clist));
  gtk_widget_queue_resize(play_clist);

  gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    

  num_songs=1;

  return TRUE;
}

static gboolean M3UFileList(char *filename)
{
  char *cols[4];
  FILE *mfp;
  char buf[1024];

  num_songs=0;

  StopPlay();

  mfp=fopen(filename,"r");
  if(!mfp) return FALSE;

  gtk_clist_freeze(GTK_CLIST(play_clist));
  gtk_clist_clear(GTK_CLIST(play_clist));
 
  cols[1]=NULL;
  cols[2]=NULL;
  cols[3]=NULL;

  while(fgets(buf,1024,mfp)) {
    if(buf[strlen(buf)-1]=='\n') buf[strlen(buf)-1]='\0';

    cols[0]=buf;

    gtk_clist_append(GTK_CLIST(play_clist),cols);
    gtk_clist_set_row_data(GTK_CLIST(play_clist),num_songs++,(gpointer)-1);
  }    

  fclose(mfp);
    
  gtk_clist_thaw(GTK_CLIST(play_clist));
  gtk_widget_queue_resize(play_clist);

  gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    

  return TRUE;
}

static void GeneratePlaylist(void)
{
  GList *disc_list;
  int num_discs,disc;
  int *num_list,pos;
  IDList *new_idlist;

  if(!have_db) return;

  StopPlay();

  if(query_list_type==LIST_TYPE_PLAYLIST)
    PushIDList(song_stack);

  if(song_order==SONG_ORDER_DISC) {
    disc_list=GTK_CLIST(disc_clist)->selection;
  
    if(disc_list) {
      num_discs=g_list_length(disc_list);

      num_list=(int *)malloc(num_discs*sizeof(int));
      
      disc=0;

      while(disc_list) {
	num_list[disc++]=(int)gtk_clist_get_row_data
	  (GTK_CLIST(disc_clist),GPOINTER_TO_INT(disc_list->data));

	disc_list=disc_list->next;
      }
    }
    else {
      num_discs=g_list_length(GTK_CLIST(disc_clist)->row_list);

      num_list=(int *)malloc(num_discs*sizeof(int));

      for(disc=0;disc<num_discs;disc++)
	num_list[disc]=(int)gtk_clist_get_row_data(GTK_CLIST(disc_clist),
						   disc);
    }

    ScrambleInts(num_list,num_discs);

    for(pos=0;pos<num_discs;pos++)
      AddToPlayList(song_stack->current_idlist,num_list[pos]);

    new_idlist=song_stack->current_idlist;
  }
  else {
    if(query_list_type==LIST_TYPE_PLAYLIST)
      new_idlist=song_stack->current_idlist;
    else {
      new_idlist=g_new(IDList,1);
    }

    AddToPlayList(new_idlist,-1);
  }

  if(song_order==SONG_ORDER_RANDOM)
    ScrambleInts(new_idlist->ids,new_idlist->num_ids);

  switch(query_list_type) {
  case LIST_TYPE_PLAYLIST:
    TrimIDList(new_idlist,songlist_limit);
  
    IDListToSongCList(mysql,song_stack->current_idlist,&play_clist);

    gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    
    break;
  case LIST_TYPE_SONG:
    NewSongWindow(new_idlist);;
    break;
  case LIST_TYPE_ARTIST:
    NewArtistWindow(new_idlist);;
    break;
  case LIST_TYPE_DISC:
    NewDiscWindow(new_idlist);;
    break;
  }
}

static void AddToPlayList(IDList *idlist,int discid)
{
  GString *query;
  GList *artist_list,*disc_list,*genre_list;
  gboolean first=TRUE;
  char *artist_name,*disc_name,*genre_name;
  int low_bpm,high_bpm;
  char tartist[256],tdisc[256],tgenre[256];
  IDList new_idlist;

  if(discid!=-1) Debug("Adding disc %d\n",discid);
  
  query=g_string_new(NULL);

  switch(query_list_type) {
  case LIST_TYPE_SONG:
  case LIST_TYPE_PLAYLIST:
    g_string_append(query,"SELECT song.id ");
    break;
  case LIST_TYPE_ARTIST:
    g_string_append(query,"SELECT DISTINCT artist.id ");
    break;
  case LIST_TYPE_DISC:
    g_string_append(query,"SELECT DISTINCT disc.id ");
    break;
  }

  g_string_append(query,"FROM artist,disc,song WHERE "
		  "artist.id=song.artistid AND disc.id=song.discid");
  
  artist_list=GTK_CLIST(artist_clist)->selection;

  while(artist_list) {
    gtk_clist_get_text(GTK_CLIST(artist_clist),
		       GPOINTER_TO_INT(artist_list->data),0,
		       &artist_name);

    EscapeQuote(artist_name,tartist,256);

    if(first) {
      first=FALSE;
      g_string_sprintfa(query," AND (artist.artist='%s'",tartist);
    }
    else
      g_string_sprintfa(query," OR artist.artist='%s'",tartist);

    artist_list=artist_list->next;
  }

  if(!first) g_string_append(query,")");

  if(discid==-1) {
    first=TRUE;

    disc_list=GTK_CLIST(disc_clist)->selection;

    while(disc_list) {
      gtk_clist_get_text(GTK_CLIST(disc_clist),
			 GPOINTER_TO_INT(disc_list->data),0,
			 &disc_name);
      
      EscapeQuote(disc_name,tdisc,256);
      
      if(first) {
	first=FALSE;
	g_string_sprintfa(query," AND (disc.title='%s'",tdisc);
      }
      else
	g_string_sprintfa(query," OR disc.title='%s'",tdisc);
      
      disc_list=disc_list->next;
    }

    if(!first) g_string_append(query,")");
  }
  else
    g_string_sprintfa(query," AND (disc.id=%d)",discid);

  first=TRUE;

  genre_list=GTK_CLIST(genre_clist)->selection;

  while(genre_list) {
    gtk_clist_get_text(GTK_CLIST(genre_clist),
		       GPOINTER_TO_INT(genre_list->data),0,
		       &genre_name);

    EscapeQuote(genre_name,tgenre,256);

    if(first) {
      first=FALSE;
      g_string_sprintfa(query," AND (song.genre='%s'",tgenre);
    }
    else
      g_string_sprintfa(query," OR song.genre='%s'",tgenre);

    genre_list=genre_list->next;
  }

  if(!first) g_string_append(query,")");

  low_bpm=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(low_bpm_spin));
  high_bpm=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(high_bpm_spin));

  if((low_bpm>LOW_BPM)||(high_bpm<HIGH_BPM)) {
    g_string_sprintfa(query," AND song.bpm BETWEEN %d and %d",
		    low_bpm,high_bpm);
  }

  if(song_order==SONG_ORDER_DISC)
    g_string_append(query," order by discid,track_num");
  else if(song_order==SONG_ORDER_OLDEST) {
    switch(query_list_type) {
    case LIST_TYPE_SONG:
    case LIST_TYPE_PLAYLIST:
      g_string_append(query," order by song.last_play");
      break;
    case LIST_TYPE_ARTIST:
      g_string_append(query," order by artist.last_play");
      break;
    case LIST_TYPE_DISC:
      g_string_append(query," order by disc.last_play");
      break;
    }
  }
  else if(song_order==SONG_ORDER_SLOWEST)
    g_string_append(query," AND song.bpm != 'NULL' order by bpm");

  Debug("Query is [%s]\n",query->str);

  if(discid==-1)
    GetListFromQuery(mysql,idlist,query->str,TRUE);
  else {
    GetListFromQuery(mysql,&new_idlist,query->str,TRUE);

    InsertIDs(idlist,&new_idlist,idlist->num_ids);

    FreeIDList(&new_idlist);
  }

  g_string_free(query,TRUE);
}

static void SetSongOrder(GtkWidget *widget,gpointer data)
{
  song_order=(int)data;
}

static void SetQueryListType(GtkWidget *widget,gpointer data)
{
  query_list_type=(int)data;
}

static void FillArtistCombo(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  GtkWidget *item;

  if(!have_db) return;

  if(mysql_query(mysql,"SELECT artist FROM artist order by artist")) {
    printf("Query error\n");

    return;
  }

  if(!(res = mysql_store_result(mysql))) {
    printf("Query error\n");

    return;
  }

  while((row=mysql_fetch_row(res))) {
    item=gtk_list_item_new_with_label(row[0]);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(artist_combo)->list),item);
    gtk_widget_show(item);
  }

  mysql_free_result(res);
}

static void FillDiscCombo(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  GtkWidget *item;

  if(!have_db) return;

  if(mysql_query(mysql,"SELECT title FROM disc order by title")) {
    printf("Query error\n");

    return;
  }

  if(!(res = mysql_store_result(mysql))) {
    printf("Query error\n");

    return;
  }

  while((row=mysql_fetch_row(res))) {
    item=gtk_list_item_new_with_label(row[0]);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(disc_combo)->list),item);
    gtk_widget_show(item);
  }

  mysql_free_result(res);
}

static void FillGenreCombo(void)
{
  GtkWidget *item;
  int genre;

  for(genre=0;id3_genres[genre];genre++) {
    item=gtk_list_item_new_with_label(id3_genres[genre]);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(genre_combo)->list),item);
    gtk_widget_show(item);
  }
}

static void UpdateSongTitle(void)
{
  char query[1024];
  char ttitle[256];

  strncpy(song_data.title,gtk_entry_get_text(GTK_ENTRY(song_title_entry)),80);
  EscapeQuote(song_data.title,ttitle,256);

  g_snprintf(query,1024,"UPDATE song set title='%s' where id=%d",ttitle,
	   song_data.id);

  SQLQuery(mysql,query);
}

static void UpdateSongArtist(void)
{
  char query[1024];
  char ttitle[256];

  strncpy(song_data.title,gtk_entry_get_text(GTK_ENTRY(song_title_entry)),80);
  EscapeQuote(song_data.title,ttitle,256);

  g_snprintf(query,1024,"UPDATE song set title='%s' where id=%d",ttitle,
	   song_data.id);

  SQLQuery(mysql,query);
}

static void UpdateSongDisc(void)
{
}

static void UpdateSongGenre(void)
{
  char query[1024];
  char tgenre[256];

  strncpy(song_data.genre,
	  gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(genre_combo)->entry)),20);

  EscapeQuote(song_data.genre,tgenre,256);

  g_snprintf(query,1024,"UPDATE song set genre='%s' where id=%d",tgenre,
	   song_data.id);

  SQLQuery(mysql,query);
}

static void UpdateSongBPM(void)
{
  char query[1024];

  song_data.bpm=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(beat_spin));
  if(song_data.bpm==0) song_data.bpm=-1;

  if(song_data.bpm==-1)
    g_snprintf(query,1024,"UPDATE song set bpm=NULL where id=%d",song_data.id);
  else
    g_snprintf(query,1024,"UPDATE song set bpm=%d where id=%d",song_data.bpm,
	     song_data.id);

  SQLQuery(mysql,query);
}

static void UpdateSongYear(void)
{
  char query[1024];

  song_data.year=
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(year_spin_button));

  if(song_data.year)
    g_snprintf(query,1024,"UPDATE song set year=%d where id=%d",song_data.year,
	     song_data.id);
  else
    g_snprintf(query,1024,"UPDATE song set bpm=NULL where id=%d",
	     song_data.id);

  SQLQuery(mysql,query);
}

static void Beat(void)
{
  clock_t end_time;
  struct tms t;
  gfloat bpm;

  num_beats++;

  if(num_beats==1)
    start_time=times(&t);
  else {
    end_time=times(&t);

    bpm=rint((gfloat)(num_beats-1)/
      (((gfloat)(end_time-start_time))/(gfloat)(CLK_TCK*60)));

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(beat_spin),bpm);
  }
}

static void BeatReset(void)
{
  num_beats=0;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(beat_spin),0.0);
}

#ifdef HAVE_GHTTP_H

/* Grab an album image over http */
static gboolean GetAlbumImageHTTP(char *url,char *filename)
{
  ghttp_request *request=NULL;
  char *result;
  int fd;

  request=ghttp_request_new();

  ghttp_set_uri(request,url);
  ghttp_set_header(request,http_hdr_Connection,"close");
  ghttp_prepare(request);
  ghttp_process(request);

  result=ghttp_get_body(request);
  if(!result) return FALSE;

  strcpy(filename,"/tmp/ddj-image.XXXXXX");
  fd=mkstemp(filename);

  if(fd==-1)
    return FALSE;

  write(fd,ghttp_get_body(request),ghttp_get_body_len(request));
  close(fd);

  return TRUE;
}

#endif /* ifdef HAVE_GHTTP_H */

/* Copy alubm image, forcing overwrite */
static void OverwriteCB(GtkWidget *widget,gpointer data)
{
  char *filename;

  filename=(char *)data;

  GetAlbumImageFromFile(filename,TRUE);

  free(filename);
}

/* Just here to free the filename data */
static void NoOverwriteCB(GtkWidget *widget,gpointer data)
{
  char *filename;
  
  filename=(char *)data;

  free(filename);
}

/* Make a file the album's new image */
static gboolean GetAlbumImageFromFile(char *filename,gboolean overwrite)
{
  int in_fd,out_fd;
  char buf[1024];
  int num_read;
  GdkPixbuf *tst_pixbuf;
  char *new_filename;

  tst_pixbuf=LoadImage(filename);

  if(!tst_pixbuf) return FALSE;

  gdk_pixbuf_unref(tst_pixbuf);

  g_snprintf(buf,1024,"%s/%d.jpg",cover_dir,disc_data.id);
  if(!overwrite && FileExists(buf)) {
    /* Silliness to work around filename being freed on us */
    new_filename=(char *)malloc(strlen(filename)+1);
    strcpy(new_filename,filename);

    BoolDialog("Overwrite existing album image?","Yes",OverwriteCB,
	       (gpointer)new_filename,"No",NoOverwriteCB,
	       (gpointer)new_filename);

    return TRUE;
  }

  in_fd=open(filename,O_RDONLY);
  if(in_fd==-1) return FALSE;

  out_fd=creat(buf,0666);
  if(out_fd==-1) {
    close(in_fd);

    return FALSE;
  }

  while((num_read=read(in_fd,buf,1024))) {
    write(out_fd,buf,num_read);
  }

  close(in_fd);
  close(out_fd);

  LoadAlbumImage();

  return TRUE;
}

/* Try to load an album image from the url/file in the selection */
static void AlbumSelectionReceived(GtkWidget *widget,
				   GtkSelectionData *selection_data, 
				   gpointer data)
{
  char filename[80];
  int len;
  char *source;

  source=selection_data->data;

  if(!source) return;

  len=strlen(source);

  if(len>4 && !strcasecmp(source+(len-4),".jpg")) {
    if(len>7 && !strncasecmp(source,"http://",7)) {
#ifdef HAVE_GHTTP_H
      if(!GetAlbumImageHTTP(selection_data->data,filename)) {
	DisplayMsg("Unable to retrieve image url.");

	return;
      }
      else {
	if(!GetAlbumImageFromFile(filename,FALSE)) {
	  DisplayMsg("Unable to read image file.");
	}
      }
#else
      DisplayMsg("libghttp must be compiled in\n"
		 "in order to get images via http");
#endif /* #ifdef HAVE_GHTTP_H */
    }
    else {
      if(!GetAlbumImageFromFile(source,FALSE)) {
	DisplayMsg("Unable to read image file.");
      }
    }
  }
  else DisplayMsg("Image source must be a .jpg file.");
}

/* Trigger a selection grab when middle clicked */
static void AlbumImageClicked(GtkWidget *widget,GdkEvent *event,gpointer data)
{
  GdkEventButton *event_button;

  if(event->type==GDK_BUTTON_PRESS) {
    event_button=(GdkEventButton *)event;
    
    if(event_button->button==2) {
      gtk_selection_convert(window,GDK_SELECTION_PRIMARY,
			    gdk_atom_intern("STRING",FALSE),
			    time(NULL));
    }
  }
}

static void MakeEditPage(void)
{
  GtkWidget *vbox,*vbox2,*vbox3,*hbox,*hbox2,*hbox3;
  GtkWidget *label;
  GtkWidget *button;
  GtkObject *adj;
  GtkWidget *ebox;
  GtkWidget *bpm_image;
  GtkWidget *frame;
  
  edit_page=MakeNewPage("Edit");

  vbox=gtk_vbox_new(FALSE,3);

  hbox3=gtk_hbox_new(FALSE,3);
  
  ebox=gtk_event_box_new();

  no_image=Loadxpm(window,noimage_xpm);
  album_image=NewBlankPixmap(window);
  gtk_signal_connect(GTK_OBJECT(window),"selection-received",
		     GTK_SIGNAL_FUNC(AlbumSelectionReceived),NULL);
  gtk_signal_connect(GTK_OBJECT(ebox),"button_press_event",
		     GTK_SIGNAL_FUNC(AlbumImageClicked),NULL);
  gtk_widget_set_usize(album_image,155,155);

  gtk_container_add(GTK_CONTAINER(ebox),album_image);
  gtk_widget_show(album_image);

  gtk_box_pack_start(GTK_BOX(hbox3),ebox,FALSE,FALSE,0);
  gtk_widget_show(ebox);
  
  gtk_selection_add_target(window,GDK_SELECTION_PRIMARY,
			   gdk_atom_intern("STRING",FALSE),1);

  vbox3=gtk_vbox_new(FALSE,3);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Title:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  song_title_entry=gtk_entry_new_with_max_length(80);
  gtk_signal_connect(GTK_OBJECT(song_title_entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongTitle),NULL);
  gtk_signal_connect(GTK_OBJECT(song_title_entry),"focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongTitle),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),song_title_entry,TRUE,TRUE,0);
  gtk_widget_show(song_title_entry);

  gtk_box_pack_start(GTK_BOX(vbox3),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Artist:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  artist_combo=gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(artist_combo)->entry),FALSE);

  FillArtistCombo();

  gtk_box_pack_start(GTK_BOX(hbox),artist_combo,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(artist_combo)->entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongArtist),NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(artist_combo)->entry),
		     "focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongArtist),NULL);
  gtk_widget_show(artist_combo);

  gtk_box_pack_start(GTK_BOX(vbox3),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Disc Name:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  disc_combo=gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(disc_combo)->entry),FALSE);

  FillDiscCombo();

  gtk_box_pack_start(GTK_BOX(hbox),disc_combo,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(disc_combo)->entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongDisc),NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(disc_combo)->entry),
		     "focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongDisc),NULL);
  gtk_widget_show(disc_combo);

  gtk_box_pack_start(GTK_BOX(vbox3),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Genre:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  genre_combo=gtk_combo_new();

  FillGenreCombo();

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(genre_combo)->entry),"activate",
		     GTK_SIGNAL_FUNC(UpdateSongGenre),NULL);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(genre_combo)->entry),
		     "focus_out_event",
		     GTK_SIGNAL_FUNC(UpdateSongGenre),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),genre_combo,TRUE,TRUE,0);
  gtk_widget_show(genre_combo);

  gtk_box_pack_start(GTK_BOX(vbox3),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Year:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(0,0,9999,1.0,5.0,0);

  year_spin_button=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_signal_connect(GTK_OBJECT(year_spin_button),"changed",
		     GTK_SIGNAL_FUNC(UpdateSongYear),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),year_spin_button,TRUE,TRUE,0);
  gtk_widget_show(year_spin_button);

  gtk_box_pack_start(GTK_BOX(vbox3),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(hbox3),vbox3,TRUE,TRUE,0);
  gtk_widget_show(vbox3);

  gtk_box_pack_start(GTK_BOX(vbox),hbox3,FALSE,FALSE,0);
  gtk_widget_show(hbox3);


  frame=gtk_frame_new(NULL);

  ebox=gtk_event_box_new();
  gtk_widget_set_style(ebox,style_wb);

  hbox=gtk_hbox_new(FALSE,3);

  bpm_image=Loadxpm(window,bpm_xpm);
  gtk_box_pack_start(GTK_BOX(hbox),bpm_image,TRUE,TRUE,0);
  gtk_widget_show(bpm_image);

  vbox2=gtk_vbox_new(FALSE,3);

  button=gtk_button_new_with_label("Beat");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BIN(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"pressed",
		     GTK_SIGNAL_FUNC(Beat),NULL);

  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Click here on the beat",NULL);
  gtk_box_pack_start(GTK_BOX(vbox2),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_button_new_with_label("Update");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BIN(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(UpdateSongBPM),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Update the database with current BPM",NULL);
  gtk_box_pack_start(GTK_BOX(vbox2),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(hbox),vbox2,TRUE,TRUE,0);
  gtk_widget_show(vbox2);

  vbox2=gtk_vbox_new(FALSE,3);

  button=gtk_button_new_with_label("Reset");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BIN(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(BeatReset),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Reset the BPM calclulation",NULL);
  gtk_box_pack_start(GTK_BOX(vbox2),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  hbox2=gtk_hbox_new(FALSE,3);
  
  label=gtk_label_new("BPM:");
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(hbox2),label,TRUE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(120,LOW_BPM,HIGH_BPM,1.0,5.0,0);

  beat_spin=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_widget_set_name(beat_spin,"darkgrey");
  gtk_box_pack_start(GTK_BOX(hbox2),beat_spin,TRUE,FALSE,0);
  gtk_widget_show(beat_spin);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox2,TRUE,FALSE,0);
  gtk_widget_show(hbox2);

  gtk_box_pack_start(GTK_BOX(hbox),vbox2,TRUE,TRUE,0);
  gtk_widget_show(vbox2);

  gtk_container_add(GTK_CONTAINER(ebox),hbox);
  gtk_widget_show(hbox);

  gtk_container_add(GTK_CONTAINER(frame),ebox);
  gtk_widget_show(frame);

  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,0);
  gtk_widget_show(ebox);

  gtk_container_add(GTK_CONTAINER(edit_page),vbox);
  gtk_widget_show(vbox);
}

/* Try to automatically set up the database */
static void AutoInstallDB(void)
{
  printf("log in as root\n");

  LogInAsRoot();

  printf("init sql\n");

  DoInitSQL();

  printf("make user\n");

  MakeSQLUser(root_mysql,"ddj","localhost","ddj",
	      gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  printf("scan\n");

  DBScan();
}

/* Update from a previous version of the database */
static void UpdateDB(void)
{
  char schemafile[1024];

  g_snprintf(schemafile,1024,
	     "%s/%d-%d.sql",AUXDIR,
	     db_version,CURRENT_DB_VERSION);

  ExecuteDBSchema(schemafile,root_mysql,
		  gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  DBScan();
}

/* Re-connect to the database, and rescan */
static void DBScan(void)
{
  if(have_db) DisconnectSQL(mysql);

  mysql=ConnectSQL();

  if(!mysql) {
    have_db=FALSE;

    if(console_mode || tv_mode) {
      printf("Unable to connect to SQL Server.\nPerhaps you need to "
	     "initialize the database?");

      exit(0);
    }
    else {
      BoolDialog("Unable to connect to SQL Server.\n"
		 "Attempt to configure it?\n","Yes",AutoInstallDB,
		 NULL,"No",NULL,NULL);
    }
  }
  else {
    have_db=TRUE;
  }

  if(have_db) {
    db_version=1;

    GetDBInfo(mysql,&db_version);

    Debug("database version is [%d]\n",db_version);

    if(db_version!=CURRENT_DB_VERSION) {
      if(console_mode || tv_mode) {
	printf("Your database format is out of date.\nUse the GUI version "
	       "of DigitalDJ to update it.\n");
	
	exit(0);
      }
      else {
	BoolDialog("Your database format is out of date.\nDo you wish to "
		   "update it?\n","Yes",UpdateDB,NULL,"No",NULL,NULL);

	have_db=FALSE;
      }
    }
  }

  if(have_db) {
    GetAllSongs(mysql,song_stack->current_idlist);
    ScrambleInts(song_stack->current_idlist->ids,
		 song_stack->current_idlist->num_ids);
    
    TrimIDList(song_stack->current_idlist,songlist_limit);

    if(console_mode || tv_mode) {
      SelectSong(song_stack->current_idlist,0);    
    }
    else {
      FillArtistCList();
      QueryGenres();
      FillDiscCList();

      IDListToSongCList(mysql,song_stack->current_idlist,&play_clist);
      gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    
    }
  }
}

static void MakeQueryPage(void)
{
  GtkWidget *vpaned,*hpaned;
  GtkWidget *vbox,*hbox;
  GtkWidget *button;
  GtkWidget *label;
  GtkObject *adj;
  GSList *group;
  GtkWidget *scroll;

  query_page=MakeNewPage("Query");

  vbox=gtk_vbox_new(FALSE,3);

  hpaned=gtk_hpaned_new();
  gtk_paned_set_position(GTK_PANED(hpaned),400);
  gtk_paned_set_gutter_size(GTK_PANED(hpaned),11);

  vpaned=gtk_vpaned_new();
  gtk_paned_set_position(GTK_PANED(vpaned),105);
  gtk_paned_set_gutter_size(GTK_PANED(vpaned),11);
  
  /* The Artist CList */

  artist_clist=gtk_clist_new_with_titles(1,artist_titles);

  gtk_signal_connect_after(GTK_OBJECT(artist_clist),"button_release_event",
			   GTK_SIGNAL_FUNC(ArtistChanged),NULL);
  gtk_signal_connect_after(GTK_OBJECT(artist_clist),"key_release_event",
			   GTK_SIGNAL_FUNC(ArtistChanged),NULL);

  gtk_clist_set_selection_mode(GTK_CLIST(artist_clist),GTK_SELECTION_EXTENDED);

  gtk_clist_column_titles_passive(GTK_CLIST(artist_clist));

  gtk_clist_set_column_width(GTK_CLIST(artist_clist),0,150);

  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),artist_clist);
  gtk_widget_show(artist_clist);

  gtk_paned_pack1(GTK_PANED(hpaned),scroll,TRUE,TRUE);
  gtk_widget_show(scroll);

  /* The Genre CList */

  genre_clist=gtk_clist_new_with_titles(1,genre_titles);

  gtk_clist_set_selection_mode(GTK_CLIST(genre_clist),GTK_SELECTION_EXTENDED);

  gtk_clist_column_titles_passive(GTK_CLIST(genre_clist));

  gtk_clist_set_column_width(GTK_CLIST(genre_clist),0,120);

  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),genre_clist);
  gtk_widget_show(genre_clist);

  gtk_paned_pack2(GTK_PANED(hpaned),scroll,FALSE,TRUE);
  gtk_widget_show(scroll);

  gtk_paned_pack1(GTK_PANED(vpaned),hpaned,TRUE,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),vpaned,TRUE,TRUE,0);
  gtk_widget_show(vpaned);
  gtk_widget_show(hpaned);

  /* The Disc CList */

  disc_clist=gtk_clist_new_with_titles(2,disc_titles);

  gtk_clist_set_selection_mode(GTK_CLIST(disc_clist),GTK_SELECTION_EXTENDED);

  gtk_clist_column_titles_passive(GTK_CLIST(disc_clist));

  gtk_clist_set_column_width(GTK_CLIST(disc_clist),0,250);
  gtk_clist_set_column_width(GTK_CLIST(disc_clist),1,250);

  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),disc_clist);
  gtk_widget_show(disc_clist);

  gtk_paned_pack2(GTK_PANED(vpaned),scroll,TRUE,TRUE);
  gtk_widget_show(scroll);

  /* BPM selection */

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("BPM: More than");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(LOW_BPM,LOW_BPM,HIGH_BPM,1.0,5.0,0);

  low_bpm_spin=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_box_pack_start(GTK_BOX(hbox),low_bpm_spin,TRUE,FALSE,0);
  gtk_widget_show(low_bpm_spin);

  label=gtk_label_new("Less than");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  adj=gtk_adjustment_new(HIGH_BPM,LOW_BPM,HIGH_BPM,1.0,5.0,0);

  high_bpm_spin=gtk_spin_button_new(GTK_ADJUSTMENT(adj),0.5,0);
  gtk_box_pack_start(GTK_BOX(hbox),high_bpm_spin,TRUE,FALSE,0);
  gtk_widget_show(high_bpm_spin);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  /* Play order */

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Play order:");
  gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,0);
  gtk_widget_show(label);

  button=gtk_radio_button_new_with_label(NULL,"Random");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_RANDOM);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),TRUE);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  group=gtk_radio_button_group(GTK_RADIO_BUTTON(button));

  button=gtk_radio_button_new_with_label(group,"Disc/Track");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_DISC);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"Age");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_OLDEST);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"BPM");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetSongOrder),
		     (gpointer)SONG_ORDER_SLOWEST);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);


  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Generate:");
  gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,0);
  gtk_widget_show(label);

  button=gtk_radio_button_new_with_label(NULL,"Song playlist");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetQueryListType),
		     (gpointer)LIST_TYPE_PLAYLIST);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),TRUE);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  group=gtk_radio_button_group(GTK_RADIO_BUTTON(button));

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"Song list");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetQueryListType),
		     (gpointer)LIST_TYPE_SONG);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"Artist list");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetQueryListType),
		     (gpointer)LIST_TYPE_ARTIST);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(button)),"Disc list");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(SetQueryListType),
		     (gpointer)LIST_TYPE_DISC);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);


  hbox=gtk_hbox_new(TRUE,3);

  button=gtk_button_new_with_label("Generate List");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(GeneratePlaylist),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  button=gtk_button_new_with_label("Re-Scan Database");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(DBScan),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  gtk_container_add(GTK_CONTAINER(query_page),vbox);
  gtk_widget_show(vbox);
}

static void DeleteUser(void)
{
  char *user,*host;

  if(!have_root_pswd) {
    DisplayMsg("You must log in as root first");

    return;
  }

  user=gtk_entry_get_text(GTK_ENTRY(tmp_user_name_entry));
  if(!*user) {
    DisplayMsg("Please enter a username");

    return;
  }

  host=gtk_entry_get_text(GTK_ENTRY(tmp_user_host_entry));
  if(!*host) {
    DisplayMsg("Please enter a host.\n(Use 'localhost' for local machine)");

    return;
  }

  DeleteSQLUser(root_mysql,user,host,
		gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  ReloadSQL(root_mysql,gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  FillUserCList();
}

static void CreateUser(GtkWidget *widget,gpointer data)
{
  char *user,*host,*pswd;

  if(!have_root_pswd) {
    DisplayMsg("You must log in as root first");

    return;
  }

  user=gtk_entry_get_text(GTK_ENTRY(tmp_user_name_entry));
  if(!*user) {
    DisplayMsg("Please enter a username");

    return;
  }

  host=gtk_entry_get_text(GTK_ENTRY(tmp_user_host_entry));
  if(!*host) {
    DisplayMsg("Please enter a host.\n(Use 'localhost' for local machine)");

    return;
  }

  pswd=gtk_entry_get_text(GTK_ENTRY(tmp_user_pswd_entry));
  if(!*pswd) {
    DisplayMsg("Please enter a password");

    return;
  }

  MakeSQLUser(root_mysql,user,host,pswd,
	      gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  ReloadSQL(root_mysql,gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

  FillUserCList();

  gtk_clist_select_row(GTK_CLIST(user_clist),
		       g_list_length(GTK_CLIST(user_clist)->row_list)-1,0);
}

static void DoInitSQL(void)
{
  char schemafile[1024];

  if(!have_root_pswd)
    DisplayMsg("You must log in as root first");
  else {
    g_snprintf(schemafile,1024,"%s/0-%d.sql",AUXDIR,
	       CURRENT_DB_VERSION);
    ExecuteDBSchema(schemafile,root_mysql,
		    gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));

    printf("reload\n");

    ReloadSQL(root_mysql,gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)));
  }
}

static void FillUserCList(void)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char query[256];

  gtk_clist_clear(GTK_CLIST(user_clist));

  if(!(mysql_real_connect(root_mysql,sql_host,"root",
			  gtk_entry_get_text(GTK_ENTRY(root_pswd_entry)),
                          "mysql",3306,NULL,0))) {
    printf("FillUserCList(): Could not connect to database: error %d: %s\n",
           mysql_errno(root_mysql),mysql_error(root_mysql));
    
    return;
  }

  g_snprintf(query,256,"SELECT User,Host FROM db WHERE Db='%s'",sql_dbname);

  if(mysql_query(root_mysql,query)) {
    printf("FillUserCList(): Query error: error %d: %s\n",
           mysql_errno(root_mysql),mysql_error(root_mysql));

    return;
  }

  if(!(res = mysql_store_result(root_mysql))) {
    printf("Query error %d: %s\n",mysql_errno(root_mysql),
	   mysql_error(root_mysql));

    return;
  }

  while((row=mysql_fetch_row(res))) {
    gtk_clist_append(GTK_CLIST(user_clist),row);
  }

  mysql_free_result(res);
}


static void LogInAsRoot(void)
{
  have_root_pswd=TRUE;

  FillUserCList();
}

static void SelectUser(GtkWidget *widget,gint row,gint column,
		       GdkEventButton *event,gpointer data)
{
  char *buf;

  if(gtk_clist_row_is_visible(GTK_CLIST(user_clist),row)!=GTK_VISIBILITY_FULL)
    gtk_clist_moveto(GTK_CLIST(user_clist),row,0,0,0);

  gtk_clist_get_text(GTK_CLIST(user_clist),row,0,&buf);
  gtk_entry_set_text(GTK_ENTRY(tmp_user_name_entry),buf);

  gtk_clist_get_text(GTK_CLIST(user_clist),row,1,&buf);
  gtk_entry_set_text(GTK_ENTRY(tmp_user_host_entry),buf);
}

static void MakeConfigPage(void)
{
  GtkWidget *cfgbox,*vbox,*vbox2,*hbox,*hbox2;
  GtkWidget *entry;
  GtkWidget *check;
  GtkWidget *button;
  GtkWidget *notebook;
  GtkWidget *page;
  GtkWidget *label;
  GtkWidget *scroll;
  GtkWidget *frame;
  
  config_page=MakeNewPage("Config");

  cfgbox=gtk_vbox_new(FALSE,3);

  notebook=gtk_notebook_new();

  page=gtk_frame_new(NULL);

  vbox=gtk_vbox_new(FALSE,3);

  entry=MakeStrEntry(NULL,mp3exename,"Player executable",256,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  check=MakeCheckButton(NULL,&do_x10,"Use X10 serial remote");
  gtk_box_pack_start(GTK_BOX(vbox),check,FALSE,FALSE,0);
  gtk_widget_show(check);

  entry=MakeStrEntry(NULL,x10_serdevice,"X10 serial device",256,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,status_file,"Status output file",256,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,cover_dir,"Album cover directory",256,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  entry=MakeNumEntry(NULL,&songlist_limit,"Song list query limit",5);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  gtk_container_add(GTK_CONTAINER(page),vbox);
  gtk_widget_show(vbox);

  label=gtk_label_new("Options");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),page,label);
  gtk_widget_show(page);

  page=gtk_frame_new(NULL);

  vbox=gtk_vbox_new(FALSE,3);

  entry=MakeStrEntry(NULL,sql_host,"SQL host",60,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,sql_dbname,"SQL database",32,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,sql_user,"SQL username",16,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  entry=MakeStrEntry(NULL,sql_pswd,"SQL password",16,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  gtk_widget_show(entry);

  gtk_container_add(GTK_CONTAINER(page),vbox);
  gtk_widget_show(vbox);

  label=gtk_label_new("Client");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),page,label);
  gtk_widget_show(page);

  page=gtk_frame_new(NULL);

  vbox=gtk_vbox_new(FALSE,3);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("MySQL Root Password (blank by default):");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  root_pswd_entry=gtk_entry_new_with_max_length(16);
  gtk_box_pack_start(GTK_BOX(hbox),root_pswd_entry,TRUE,TRUE,0);
  gtk_widget_show(root_pswd_entry);

  button=gtk_button_new_with_label("Log in");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(LogInAsRoot),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,0);
  gtk_widget_show(button);

  button=gtk_button_new_with_label("Initialize SQL database");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(DoInitSQL),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox2=gtk_hbox_new(TRUE,3);

  vbox2=gtk_vbox_new(FALSE,3);

  user_clist=gtk_clist_new_with_titles(2,user_titles);

  gtk_clist_set_selection_mode(GTK_CLIST(user_clist),GTK_SELECTION_SINGLE);

  gtk_clist_column_titles_passive(GTK_CLIST(user_clist));

  gtk_clist_set_column_width(GTK_CLIST(user_clist),0,125);
  gtk_clist_set_column_width(GTK_CLIST(user_clist),1,125);

  gtk_signal_connect(GTK_OBJECT(user_clist),"select_row",
		     GTK_SIGNAL_FUNC(SelectUser),
		     NULL);

  scroll=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll),user_clist);
  gtk_widget_show(user_clist);

  gtk_box_pack_start(GTK_BOX(vbox2),scroll,TRUE,TRUE,0);
  gtk_widget_show(scroll);

  gtk_box_pack_start(GTK_BOX(hbox2),vbox2,TRUE,TRUE,0);
  gtk_widget_show(vbox2);

  frame=gtk_frame_new(NULL);

  vbox2=gtk_vbox_new(FALSE,3);
  gtk_container_border_width(GTK_CONTAINER(vbox2),3);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Username:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  tmp_user_name_entry=gtk_entry_new_with_max_length(16);
  gtk_box_pack_start(GTK_BOX(hbox),tmp_user_name_entry,TRUE,TRUE,0);
  gtk_widget_show(tmp_user_name_entry);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Host:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  tmp_user_host_entry=gtk_entry_new_with_max_length(60);
  gtk_box_pack_start(GTK_BOX(hbox),tmp_user_host_entry,TRUE,TRUE,0);
  gtk_widget_show(tmp_user_host_entry);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  label=gtk_label_new("Password:");
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  tmp_user_pswd_entry=gtk_entry_new_with_max_length(16);
  gtk_box_pack_start(GTK_BOX(hbox),tmp_user_pswd_entry,TRUE,TRUE,0);
  gtk_widget_show(tmp_user_pswd_entry);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE,3);

  button=gtk_button_new_with_label("Add user account");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(CreateUser),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);
 
  button=gtk_button_new_with_label("Delete user account");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(DeleteUser),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);

  gtk_container_add(GTK_CONTAINER(frame),vbox2);
  gtk_widget_show(vbox2);

  gtk_box_pack_start(GTK_BOX(hbox2),frame,TRUE,TRUE,0);
  gtk_widget_show(frame);

  gtk_box_pack_start(GTK_BOX(vbox),hbox2,TRUE,TRUE,0);
  gtk_widget_show(hbox2);

  gtk_container_add(GTK_CONTAINER(page),vbox);
  gtk_widget_show(vbox);

  label=gtk_label_new("Server");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),page,label);
  gtk_widget_show(page);

  gtk_box_pack_start(GTK_BOX(cfgbox),notebook,TRUE,TRUE,0);
  gtk_widget_show(notebook);

  gtk_container_add(GTK_CONTAINER(config_page),cfgbox);
  gtk_widget_show(cfgbox);
}

static void Homepage(void)
{
  system("mozilla -remote openURL\\(http://www.nostatic.org/ddj\\)");
}

static void MakeAboutPage(void)
{
  GtkWidget *vbox,*vbox2,*hbox;
  GtkWidget *label;
  GtkWidget *logo;
  GtkWidget *ebox;
  GtkWidget *button;
  char versionbuf[20];

  about_page=MakeNewPage("About");

  ebox=gtk_event_box_new();
  gtk_widget_set_style(ebox,style_wb);

  vbox=gtk_vbox_new(TRUE,5);
  gtk_container_border_width(GTK_CONTAINER(vbox),3);

  logo=Loadxpm(window,ddj_xpm);

  gtk_box_pack_start(GTK_BOX(vbox),logo,FALSE,FALSE,0);
  gtk_widget_show(logo);

  vbox2=gtk_vbox_new(TRUE,0);

  label=gtk_label_new("DigitalDJ");
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  sprintf(versionbuf,"Version %s",VERSION);
  label=gtk_label_new(versionbuf);
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,0);
  gtk_widget_show(label);

  label=gtk_label_new("Copyright (c) 1999-2003, Mike Oliphant");
  gtk_widget_set_style(label,style_wb);
  gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,0);
  gtk_widget_show(label);


  hbox=gtk_hbox_new(TRUE,0);

  button=gtk_button_new_with_label("http://www.nostatic.org/ddj");
  gtk_widget_set_style(button,style_dark_grey);
  gtk_widget_set_style(GTK_BUTTON(button)->child,style_wb);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(Homepage),NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
  gtk_widget_show(hbox);
  

  gtk_container_add(GTK_CONTAINER(vbox),vbox2);
  gtk_widget_show(vbox2);

  gtk_container_add(GTK_CONTAINER(ebox),vbox);
  gtk_widget_show(vbox);

  gtk_container_add(GTK_CONTAINER(about_page),ebox);
  gtk_widget_show(ebox);
}

static GtkWidget *MakeNewPage(char *name)
{
  GtkWidget *page;
  GtkWidget *label;

  page=gtk_frame_new(NULL);
  gtk_widget_show(page);

  label=gtk_label_new(name);
  gtk_notebook_append_page(GTK_NOTEBOOK(main_notebook),page,label);

  return page;
}

static GtkWidget *MakeControls(void)
{
  GtkWidget *vbox,*vbox3,*hbox,*imagebox,*hbox2;
  GtkWidget *button;
  GtkWidget *ebox,*ebox2,*lcdbox;
  GtkWidget *image;

  ebox=gtk_event_box_new();
  gtk_widget_set_style(ebox,style_wb);

  vbox=gtk_vbox_new(FALSE,0);
  gtk_container_border_width(GTK_CONTAINER(vbox),0);

  vbox3=gtk_vbox_new(FALSE,2);
  gtk_container_border_width(GTK_CONTAINER(vbox3),2);

  lcdbox=gtk_event_box_new();
  gtk_signal_connect(GTK_OBJECT(lcdbox),"button_press_event",
		     GTK_SIGNAL_FUNC(ToggleControlButtons),NULL);
  gtk_widget_set_style(lcdbox,style_LCD);

  hbox2=gtk_hbox_new(FALSE,0);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(window,upleft_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(window,lowleft_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox2),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);
  
  hbox=gtk_hbox_new(FALSE,0);
  gtk_container_border_width(GTK_CONTAINER(hbox),0);

  current_song_label=gtk_label_new("");
  gtk_widget_set_style(current_song_label,style_LCD);
  gtk_box_pack_start(GTK_BOX(hbox),current_song_label,TRUE,TRUE,0);
  gtk_widget_show(current_song_label);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(window,upright_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(window,lowright_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);

  ebox2=gtk_event_box_new();
  gtk_widget_set_usize(ebox2,2,0);
  gtk_widget_set_style(ebox2,style_wb);
  gtk_box_pack_start(GTK_BOX(hbox),ebox2,FALSE,FALSE,0);
  gtk_widget_show(ebox2);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(window,upleft_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(window,lowleft_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);

  play_time_label=gtk_label_new("00:00");
  gtk_widget_set_style(play_time_label,style_LCD);
  gtk_box_pack_start(GTK_BOX(hbox),play_time_label,FALSE,FALSE,0);
  gtk_widget_show(play_time_label);

  gtk_container_add(GTK_CONTAINER(hbox2),hbox);
  gtk_widget_show(hbox);

  imagebox=gtk_vbox_new(FALSE,0);

  image=Loadxpm(window,upright_xpm);
  gtk_box_pack_start(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  image=Loadxpm(window,lowright_xpm);
  gtk_box_pack_end(GTK_BOX(imagebox),image,FALSE,FALSE,0);
  gtk_widget_show(image);

  gtk_box_pack_start(GTK_BOX(hbox2),imagebox,FALSE,FALSE,0);
  gtk_widget_show(imagebox);
  
  gtk_container_add(GTK_CONTAINER(lcdbox),hbox2);
  gtk_widget_show(hbox2);

  gtk_box_pack_start(GTK_BOX(vbox3),lcdbox,FALSE,FALSE,0);
  gtk_widget_show(lcdbox);

  gtk_box_pack_start(GTK_BOX(vbox),vbox3,FALSE,FALSE,0);
  gtk_widget_show(vbox3);

  control_button_box=gtk_vbox_new(TRUE,0);

  hbox=gtk_hbox_new(TRUE,0);

  image=Loadxpm(window,playpaus_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(PlayCB),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Play track / Pause play",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,stop_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(StopPlay),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Stop play",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,prevtrk_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(PrevSong),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Go to previous song",NULL);
  gtk_widget_show(button);

  image=Loadxpm(window,nexttrk_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(NextSong),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Go to next song",NULL);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(control_button_box),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(TRUE,0);

  image=Loadxpm(window,save_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SavePlaylist),NULL);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Save playlist",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,load_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(LoadPlaylist),0);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Load playlist",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,back_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(PopPlaylist),0);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Revert to previus playlist",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,minmax_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(MinMax),0);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Toggle playlist display",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,quit_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Exit DDJ",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
		     GTK_SIGNAL_FUNC(ShutDown),0);
  gtk_widget_show(button);
  
  gtk_box_pack_start(GTK_BOX(control_button_box),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);

  gtk_box_pack_start(GTK_BOX(vbox),control_button_box,TRUE,TRUE,0);

  if(control_buttons_visible)
    gtk_widget_show(control_button_box);


  gtk_container_add(GTK_CONTAINER(ebox),vbox);
  gtk_widget_show(vbox);

  return ebox;
}

static void MakeStyles(void)
{
  gdk_color_white(gdk_window_get_colormap(window->window),&gdkwhite);
  gdk_color_black(gdk_window_get_colormap(window->window),&gdkblack);

  color_LCD=MakeColor(33686,38273,29557);
  color_dark_grey=MakeColor(0x4444,0x4444,0x4444);
  
  style_wb=MakeStyle(&gdkwhite,&gdkblack,FALSE);
  style_LCD=MakeStyle(&gdkblack,color_LCD,FALSE);
  style_LCD->font=gdk_font_load
    ("-B&H-LucidaTypewriter-Medium-R-Normal-Sans-18-180-75-75-M-110-ISO8859-1");
  style_dark_grey=MakeStyle(&gdkwhite,color_dark_grey,TRUE);
}

static GdkGC *MakePen(GtkWidget *widget,int red,int green,int blue)
{
  GdkColor *c;
  GdkGC *gc;

  c=(GdkColor *)g_malloc(sizeof(GdkColor));
  c->red=red;
  c->green=green;
  c->blue=blue;

  gdk_color_alloc(gdk_colormap_get_system(),c);

  gc=gdk_gc_new(widget->window);

  gdk_gc_set_foreground(gc,c);

  gdk_gc_set_function(gc,GDK_COPY);

  return gc;
}

static void HideMouse(GtkWidget *widget)
{
  GdkPixmap *gdkpix;
  GdkBitmap *mask;
  GdkCursor *cursor;
  GdkColor col = {0,0,0,0};

  gtk_pixmap_get(GTK_PIXMAP(empty_image),&gdkpix,&mask);

  cursor=gdk_cursor_new_from_pixmap(mask,mask,&col,&col,0,0);

  gdk_window_set_cursor(widget->window,cursor);
}

static GdkPixbuf *LoadImage(char *filename)
{
  GdkPixbuf *pixbuf;
  GdkPixbuf *scaled_pixbuf;
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  pixbuf=gdk_pixbuf_new_from_file(filename);

  if(pixbuf) {
    scaled_pixbuf=gdk_pixbuf_scale_simple(pixbuf,350,350,GDK_INTERP_NEAREST);
    
    /* Keeps things for segfaulting for some strange reason */
    gdk_pixbuf_render_pixmap_and_mask(scaled_pixbuf,&pixmap,&mask,127);
    if(pixmap) gdk_pixmap_unref(pixmap);
    if(mask) gdk_bitmap_unref(mask);

    gdk_pixbuf_unref(pixbuf);
  }
  else return NULL;

  return scaled_pixbuf;
}

/* Load an image into a widget */
static GtkWidget *LoadImageAsWidget(char *filename)
{
  GdkPixbuf *pixbuf;
  GdkPixbuf *scaled_pixbuf;
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  GtkWidget *image;

  pixbuf=gdk_pixbuf_new_from_file(filename);

  if(pixbuf) {
    scaled_pixbuf=gdk_pixbuf_scale_simple(pixbuf,150,150,GDK_INTERP_NEAREST);
    
    gdk_pixbuf_render_pixmap_and_mask(scaled_pixbuf,&pixmap,&mask,127);

    image=gtk_pixmap_new(pixmap,mask);
    
    gdk_pixbuf_unref(scaled_pixbuf);
    gdk_pixbuf_unref(pixbuf);
  }
  else return NULL;

  return image;
}

static void OutputStatus(void)
{
  FILE *ofp;

  ofp=fopen(status_file,"w");

  if(ofp) {
    fprintf(ofp,"track_id: %d\n",song_data.id);
    fprintf(ofp,"track_num: %d\n",song_data.track_num+1);

    if(!(song_data.mins<0)) {
      fprintf(ofp,"track_len: %d:%02d\n",song_data.mins,song_data.secs);
    }

    fprintf(ofp,"title: %s\n",song_data.title);
    fprintf(ofp,"artist: %s\n",artist_data.name);
    fprintf(ofp,"album: %s\n",disc_data.title);
    fprintf(ofp,"album_id: %d\n",disc_data.id);
    fprintf(ofp,"year: %d\n",disc_data.year);
    fprintf(ofp,"genre: %s\n",song_data.genre);
    fprintf(ofp,"gain: %d\n",disc_data.gain_adjustment);
    fprintf(ofp,"juke: %d\n",(juke_list.num_ids>1)?juke_list.num_ids-1:0);

    fclose(ofp);
  }
  else {
    printf("Unable to open %s for writing\n",status_file);
  }
}

static void DrawScreen(void)
{
  int win_width,win_height;
  char buf[1024];

  if(console_mode) {
    printf("%s\n%s\n",song_data.title,artist_data.name);

    if(disc_data.year>0)
      printf("%s (%d)\n",disc_data.title,disc_data.year);
    else
      printf("%s\n",disc_data.title);
  }
  else {
    win_width=main_draw->allocation.width;
    win_height=main_draw->allocation.height;
    
    gdk_draw_rectangle(main_draw->window,main_draw->style->black_gc,1,
		       0,0,win_width,win_height);
    
    if(album_pixbuf)
      gdk_pixbuf_render_to_drawable(album_pixbuf,main_draw->window,
				    text_gc,0,0,10,10,350,350,
				    GDK_RGB_DITHER_NONE,0,0);
    
    if(display_mode==1) {
      if(*song_data.genre) {
	gdk_draw_text(main_draw->window,text_font,
		      text_gc,400,40,song_data.genre,
		      strlen(song_data.genre));
      }
      
      sprintf(buf,"Track: %d",song_data.track_num+1);
      
      gdk_draw_text(main_draw->window,text_font,
		    text_gc,400,70,buf,
		    strlen(buf));
      
      if(!(song_data.mins<0)) {
	sprintf(buf,"%d:%02d",song_data.mins,song_data.secs);
	
	gdk_draw_text(main_draw->window,text_font,
		      text_gc,400,100,buf,
		      strlen(buf));
      }

      if(juke_list.num_ids>1) {
	sprintf(buf,"Juke: %d",juke_list.num_ids-1);

	gdk_draw_text(main_draw->window,text_font,
		      text_gc,400,130,buf,
		      strlen(buf));	
      }
    }
    
    sprintf(buf,"\"%s\"",song_data.title);

    gdk_draw_text(main_draw->window,text_font,
		  text_gc,10,400,buf,strlen(buf));
    
    gdk_draw_text(main_draw->window,text_font,
		  text_gc,10,430,artist_data.name,
		  strlen(artist_data.name));
    
    if(disc_data.year>0)
      sprintf(buf,"%s (%d)",disc_data.title,disc_data.year);
    else
      strcpy(buf,disc_data.title);
    
    gdk_draw_text(main_draw->window,text_font,
		  text_gc,10,460,buf,strlen(buf));
  }

}

static IDList *SelectionToIDList(ListWindow *lwin)
{
  GList *select_list;
  int num_selected;
  IDList *select_idlist,*win_idlist;
  int pos;
  GtkWidget *clist;

  clist=lwin->clist;
  win_idlist=lwin->idlist;

  select_list=GTK_CLIST(clist)->selection;
  
  if(select_list) {
    num_selected=g_list_length(select_list);

    select_idlist=g_new(IDList,1);
    select_idlist->ids=g_new(int,num_selected);
    select_idlist->num_ids=num_selected;
    
    pos=0;
    
    while(select_list) {
      select_idlist->ids[pos++]=
	win_idlist->ids[GPOINTER_TO_INT(select_list->data)];

      select_list=select_list->next;
    }

    return select_idlist;
  }
  else return NULL;
}

static void DeleteSelection(ListWindow *lwin)
{
  GList *select_list;
  GtkWidget *clist;
  IDList *idlist;
  int pos;

  clist=lwin->clist;
  idlist=lwin->idlist;

  while((select_list=GTK_CLIST(clist)->selection)) {
    pos=GPOINTER_TO_INT(select_list->data);

    gtk_clist_remove(GTK_CLIST(clist),pos);

    RemoveIDs(idlist,pos,1);
  }
}

static void ListWinCut(GtkWidget *widget,gpointer data)
{
  ListWindow *win;

  win=(ListWindow *)data;

  printf("Cut\n");

  if(clipboard_list) FreeIDList(clipboard_list);

  clipboard_list=SelectionToIDList(win);
  clipboard_type=win->list_type;
  DeleteSelection(win);
}

static void ListWinCopy(GtkWidget *widget,gpointer data)
{
  ListWindow *win;

  win=(ListWindow *)data;

  printf("Copy\n");

  if(clipboard_list) FreeIDList(clipboard_list);

  clipboard_list=SelectionToIDList(win);
  clipboard_type=win->list_type;
}

static void ListWinPaste(GtkWidget *widget,gpointer data)
{
  ListWindow *win;

  win=(ListWindow *)data;

  printf("Paste\n");

  DoPaste(win,win->idlist->current_id);
}

static void ListWinPasteAtEnd(GtkWidget *widget,gpointer data)
{
  ListWindow *win;

  win=(ListWindow *)data;

  printf("Paste\n");

  DoPaste(win,win->idlist->num_ids);
}

static void DoPaste(ListWindow *lwin,int pos)
{
  IDList *converted_list=NULL;

  if(clipboard_list) {
    switch(lwin->list_type) {
    case LIST_TYPE_SONG:
      switch(clipboard_type) {
      case LIST_TYPE_SONG:
	converted_list=clipboard_list;
	break;

      case LIST_TYPE_ARTIST:
	converted_list=g_new(IDList,1);
	ArtistListToSongList(mysql,clipboard_list,converted_list);
	break;
	
      case LIST_TYPE_DISC:
	converted_list=g_new(IDList,1);
	DiscListToSongList(mysql,clipboard_list,converted_list);
	break;
      }

      if(converted_list) {
	InsertIDs(lwin->idlist,converted_list,pos);
	IDListToSongCList(mysql,lwin->idlist,&lwin->clist);
      }
      break;
      
    case LIST_TYPE_ARTIST:
      switch(clipboard_type) {
      case LIST_TYPE_SONG:
	break;

      case LIST_TYPE_ARTIST:
	converted_list=clipboard_list;
	break;
	
      case LIST_TYPE_DISC:
	converted_list=g_new(IDList,1);
	DiscListToArtistList(mysql,clipboard_list,converted_list);
	break;
      }

      if(converted_list) {
	InsertIDs(lwin->idlist,converted_list,pos);
	IDListToArtistCList(mysql,lwin->idlist,&lwin->clist);
      }
      break;
      
    case LIST_TYPE_DISC:
      switch(clipboard_type) {
      case LIST_TYPE_SONG:
	break;

      case LIST_TYPE_ARTIST:
	converted_list=g_new(IDList,1);
	printf("converting from artist to disc\n");
	ArtistListToDiscList(mysql,clipboard_list,converted_list);
	break;
	
      case LIST_TYPE_DISC:
	converted_list=clipboard_list;
	break;
      }

      if(converted_list) {
	InsertIDs(lwin->idlist,converted_list,pos);
	IDListToDiscCList(mysql,lwin->idlist,&lwin->clist);
      }
      break;
    }

    if(converted_list) {
      /* If we created a converted list, free it */
      if(converted_list!=clipboard_list) {
	FreeIDList(converted_list);
	g_free(converted_list);
      }
    }
  }
  else printf("Nothing to paste\n");
}

/* Send the contents of a song window to the playlist window */
static void SongWinToPlaylist(GtkWidget *widget,gpointer data)
{
  ListWindow *swin;

  swin=(ListWindow *)data;

  PushIDList(song_stack);

  CopyIDList(swin->idlist,song_stack->current_idlist);

  IDListToSongCList(mysql,song_stack->current_idlist,&play_clist);

  gtk_clist_select_row(GTK_CLIST(play_clist),0,0);    
}

/* Randomize the rows of a list window */
static void RandomizeListWin(GtkWidget *widget,gpointer data)
{
  ListWindow *win;
  IDList *idlist;

  win=(ListWindow *)data;

  idlist=win->idlist;

  ScrambleInts(idlist->ids,idlist->num_ids);

  switch(win->list_type) {
  case LIST_TYPE_SONG:
    IDListToSongCList(mysql,idlist,&win->clist);
    break;
    
  case LIST_TYPE_ARTIST:
    IDListToArtistCList(mysql,idlist,&win->clist);
    break;
    
  case LIST_TYPE_DISC:
    IDListToDiscCList(mysql,idlist,&win->clist);
    break;
  }
}

/* Calculate time/size info from a song window */
static void SongWinTimeCalc(GtkWidget *widget,gpointer data)
{
  ListWindow *swin;
  unsigned long long bytes;
  unsigned long long secs;
  int frames;
  int disc_secs;
  int disc_mins;
  int disc_frames;

  swin=(ListWindow *)data;

  SongListTimeCalc(mysql,swin->idlist,&bytes,&secs);
  
  frames=bytes/2048;

  disc_mins=frames/(60*75);
  disc_secs=(frames%(60*75))/75;
  disc_frames=frames%75;

  printf("Total is %lld mbytes %lld secs %d:%02d:%02d\n",bytes/(1024*1024),
  secs,disc_mins,disc_secs,disc_frames);
}


/* Burn the contents of a song window */
static void SongWinBurn(GtkWidget *widget,gpointer data)
{
  ListWindow *swin;

  swin=(ListWindow *)data;

  SymlinkSongs(mysql,swin->idlist,"/home/oliphant/tmp");
}


/* Create a new song list window */
static ListWindow *NewSongWindow(IDList *idlist)
{
  ListWindow *swin;
  GtkWidget *listobj;
  GtkWidget *vbox,*hbox;
  GtkWidget *button;
  GtkWidget *button_box;
  GtkWidget *image;

  swin=g_new(ListWindow,1);
  if(!swin) return NULL;

  swin->list_type=LIST_TYPE_SONG;

  swin->window=gtk_window_new(GTK_WINDOW_TOPLEVEL);

  vbox=gtk_vbox_new(FALSE,3);

  swin->clist=NULL;
  listobj=IDListToSongCList(mysql,idlist,&swin->clist);
  gtk_clist_set_selection_mode(GTK_CLIST(swin->clist),GTK_SELECTION_EXTENDED);

  gtk_signal_connect(GTK_OBJECT(swin->clist),"button_press_event",
		     GTK_SIGNAL_FUNC(DiscWinButton),
		     (gpointer)swin);

  gtk_box_pack_start(GTK_BOX(vbox),listobj,TRUE,TRUE,0);
  gtk_widget_show(listobj);

  button_box=gtk_vbox_new(TRUE,0);

  hbox=gtk_hbox_new(TRUE,0);

  image=Loadxpm(window,note_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SongWinToPlaylist),
		     (gpointer)swin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Send songs to the playlist window",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,randomize_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(RandomizeListWin),
		     (gpointer)swin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Randomize list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,watch_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SongWinTimeCalc),
		     (gpointer)swin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Calculate Time/Size",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,todisc_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SongWinBurn),
		     (gpointer)swin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Make ISO filesystem",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(button_box),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);


  gtk_box_pack_start(GTK_BOX(vbox),button_box,FALSE,FALSE,0);
  gtk_widget_show(button_box);

  gtk_container_add(GTK_CONTAINER(swin->window),vbox);
  gtk_widget_show(vbox);

  gtk_widget_set_usize(swin->window,585,320);

  gtk_widget_show(swin->window);

  swin->idlist=idlist;

  return swin;
}

/* Create a song window from an artist window */
static void SongWindowFromArtistWindow(GtkWidget *widget,gpointer data)
{
  ListWindow *awin;
  IDList *slist;

  awin=(ListWindow *)data;

  printf("creating list from %d artists",awin->idlist->num_ids);

  slist=g_new(IDList,1);

  ArtistListToSongList(mysql,awin->idlist,slist);

  printf("here with %d songs\n",slist->num_ids);

  if(slist->num_ids) NewSongWindow(slist);
  else g_free(slist);
}

/* Create a disc window from an artist window */
static void DiscWindowFromArtistWindow(GtkWidget *widget,gpointer data)
{
  ListWindow *awin;
  IDList *dlist;

  awin=(ListWindow *)data;

  printf("creating list from %d artists",awin->idlist->num_ids);

  dlist=g_new(IDList,1);

  ArtistListToDiscList(mysql,awin->idlist,dlist);

  printf("here with %d discs\n",dlist->num_ids);

  if(dlist->num_ids) NewDiscWindow(dlist);
  else g_free(dlist);
}

/* Create a song window from a disc window */
static void SongWindowFromDiscWindow(GtkWidget *widget,gpointer data)
{
  ListWindow *dwin;
  IDList *slist;

  dwin=(ListWindow *)data;

  slist=g_new(IDList,1);

  DiscListToSongList(mysql,dwin->idlist,slist);

  if(slist->num_ids) NewSongWindow(slist);
  else g_free(slist);
}

/* Create a disc window from an artist window */
static void ArtistWindowFromDiscWindow(GtkWidget *widget,gpointer data)
{
  ListWindow *dwin;
  IDList *alist;

  dwin=(ListWindow *)data;

  printf("creating list from %d discs\n",dwin->idlist->num_ids);

  alist=g_new(IDList,1);

  DiscListToArtistList(mysql,dwin->idlist,alist);

  printf("here with %d artists\n",alist->num_ids);

  if(alist->num_ids) NewArtistWindow(alist);
  else g_free(alist);
}

/* Create a new artist list window */
static ListWindow *NewArtistWindow(IDList *idlist)
{
  ListWindow *awin;
  GtkWidget *listobj;
  GtkWidget *vbox,*hbox;
  GtkWidget *button;
  GtkWidget *button_box;
  GtkWidget *image;

  awin=g_new(ListWindow,1);
  if(!awin) return NULL;

  awin->list_type=LIST_TYPE_ARTIST;

  awin->window=gtk_window_new(GTK_WINDOW_TOPLEVEL);

  vbox=gtk_vbox_new(FALSE,3);

  awin->clist=NULL;
  listobj=IDListToArtistCList(mysql,idlist,&awin->clist);
  gtk_clist_set_selection_mode(GTK_CLIST(awin->clist),GTK_SELECTION_EXTENDED);

  gtk_signal_connect(GTK_OBJECT(awin->clist),"button_press_event",
		     GTK_SIGNAL_FUNC(DiscWinButton),
		     (gpointer)awin);

  gtk_box_pack_start(GTK_BOX(vbox),listobj,TRUE,TRUE,0);
  gtk_widget_show(listobj);



  button_box=gtk_vbox_new(TRUE,0);

  hbox=gtk_hbox_new(TRUE,0);

  image=Loadxpm(window,note_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SongWindowFromArtistWindow),
		     (gpointer)awin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Create song list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,disc_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(DiscWindowFromArtistWindow),
		     (gpointer)awin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Create disc list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,save_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SavePlaylist),(gpointer)awin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Save artist list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(button_box),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);


  gtk_box_pack_start(GTK_BOX(vbox),button_box,FALSE,FALSE,0);
  gtk_widget_show(button_box);

  gtk_container_add(GTK_CONTAINER(awin->window),vbox);
  gtk_widget_show(vbox);

  gtk_widget_set_usize(awin->window,400,300);

  gtk_widget_show(awin->window);

  awin->idlist=idlist;

  return awin;
}

static void DiscWinButton(GtkWidget *widget,GdkEvent *event,gpointer data)
{
  ListWindow *dwin;
  GdkEventButton *event_button;
  GtkWidget *menu;
  GtkWidget *menu_item;

  dwin=(ListWindow *)data;
  
  if(event->type==GDK_BUTTON_PRESS) {
    event_button=(GdkEventButton *)event;

    if(event_button->button==3) {
      menu=gtk_menu_new();

      menu_item=gtk_menu_item_new_with_label("Cut");
      gtk_signal_connect(GTK_OBJECT(menu_item),"activate",
			 GTK_SIGNAL_FUNC(ListWinCut),(gpointer)dwin);
      gtk_menu_append(GTK_MENU(menu),menu_item);
      gtk_widget_show(menu_item);

      menu_item=gtk_menu_item_new_with_label("Copy");
      gtk_signal_connect(GTK_OBJECT(menu_item),"activate",
			 GTK_SIGNAL_FUNC(ListWinCopy),(gpointer)dwin);
      gtk_menu_append(GTK_MENU(menu),menu_item);
      gtk_widget_show(menu_item);

      menu_item=gtk_menu_item_new_with_label("Paste");
      gtk_signal_connect(GTK_OBJECT(menu_item),"activate",
			 GTK_SIGNAL_FUNC(ListWinPaste),(gpointer)dwin);
      gtk_menu_append(GTK_MENU(menu),menu_item);
      gtk_widget_show(menu_item);

      menu_item=gtk_menu_item_new_with_label("Paste at end");
      gtk_signal_connect(GTK_OBJECT(menu_item),"activate",
			 GTK_SIGNAL_FUNC(ListWinPasteAtEnd),(gpointer)dwin);
      gtk_menu_append(GTK_MENU(menu),menu_item);
      gtk_widget_show(menu_item);

      gtk_menu_popup(GTK_MENU(menu),NULL,NULL,NULL,NULL, 
		     event_button->button,event_button->time);
    }
  }
}

/* Calculate time/size info from a disc window */
static void DiscWinTimeCalc(GtkWidget *widget,gpointer data)
{
  ListWindow *dwin;
  IDList slist;
  unsigned long long bytes;
  unsigned long long secs;

  dwin=(ListWindow *)data;

  DiscListToSongList(mysql,dwin->idlist,&slist);

  SongListTimeCalc(mysql,&slist,&bytes,&secs);

  printf("Total is %lld bytes %lld secs\n",bytes,secs);

  FreeIDList(&slist);
}

/* Create a new disc list window */
static ListWindow *NewDiscWindow(IDList *idlist)
{
  ListWindow *dwin;
  GtkWidget *listobj;
  GtkWidget *vbox,*hbox;
  GtkWidget *button;
  GtkWidget *button_box;
  GtkWidget *image;

  dwin=g_new(ListWindow,1);
  if(!dwin) return NULL;

  dwin->list_type=LIST_TYPE_DISC;

  dwin->window=gtk_window_new(GTK_WINDOW_TOPLEVEL);

  vbox=gtk_vbox_new(FALSE,3);

  dwin->clist=NULL;
  listobj=IDListToDiscCList(mysql,idlist,&dwin->clist);
  gtk_clist_set_selection_mode(GTK_CLIST(dwin->clist),GTK_SELECTION_EXTENDED);

  gtk_signal_connect(GTK_OBJECT(dwin->clist),"button_press_event",
		     GTK_SIGNAL_FUNC(DiscWinButton),
		     (gpointer)dwin);

  gtk_box_pack_start(GTK_BOX(vbox),listobj,TRUE,TRUE,0);
  gtk_widget_show(listobj);


  button_box=gtk_vbox_new(TRUE,0);

  hbox=gtk_hbox_new(TRUE,0);

  image=Loadxpm(window,note_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SongWindowFromDiscWindow),
  		     (gpointer)dwin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Create song list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,artist_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(ArtistWindowFromDiscWindow),
		     (gpointer)dwin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Create artist list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,save_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(SavePlaylist),(gpointer)dwin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Save disc list",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  image=Loadxpm(window,watch_xpm);
  button=ImageButton(window,image);
  gtk_widget_set_style(button,style_dark_grey);
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
  		     GTK_SIGNAL_FUNC(DiscWinTimeCalc),
		     (gpointer)dwin);
  gtk_tooltips_set_tip(MakeToolTip(),button,
		       "Calculate Time/Size",NULL);
  gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
  gtk_widget_show(button);

  gtk_box_pack_start(GTK_BOX(button_box),hbox,TRUE,TRUE,0);
  gtk_widget_show(hbox);


  gtk_box_pack_start(GTK_BOX(vbox),button_box,FALSE,FALSE,0);
  gtk_widget_show(button_box);

  gtk_container_add(GTK_CONTAINER(dwin->window),vbox);
  gtk_widget_show(vbox);

  gtk_widget_set_usize(dwin->window,400,300);

  gtk_widget_show(dwin->window);

  dwin->idlist=idlist;

  return dwin;
}



static void Usage(void)
{
  printf("\nUsage: ddj [-s] [-v] [-p] [-t]\n\n");
  printf("  -s                 Launch in \"small\" (cd-only) mode\n");
  printf("  -v                 Verbose (debug) mode\n");
  printf("  -p                 Start playing immediately\n");
  printf("  -t                 TV mode\n");
  printf("  -c                 Console mode\n");
  printf("\n");
  printf("See the README file for more information\n\n");

  exit(-1);
}

static void ParseArgs(int numargs,char *args[])
{
  int num=1;
  int pos;
  gboolean skipout;

  while(num<numargs) {
    if(*args[num]!='-') {
      strncpy(cmdline_file,args[num],256);
    }
    else {
      for(pos=1,skipout=FALSE;args[num][pos]&&!skipout;pos++) {
	switch(args[num][pos]) {
	case 'h':
	case 'H':
	case '?':
	  Usage();
	  break;
	case 's':
	  minimized=TRUE;
	  break;
	case 'v':
	  do_debug=TRUE;
	  break;
	case 'p':
	  auto_start_play=TRUE;
	  break;
	case 'r':
	  remote_mode=TRUE;
	  break;
	case 't':
	  tv_mode=TRUE;
	  control_buttons_visible=FALSE;
	  break;
	case 'c':
	  console_mode=TRUE;
	  break;
	default:
	  printf("Unrecognized argument [-%c]\n",args[num][1]);
	  Usage();
	  break;
	}
      }
    }
    
    num++;
  }
}

int main(int argc,char *argv[])
{
  char buf[256];
  GIOChannel *serchannel,*lirc_channel,*cmdsock_channel;
  GMainLoop *main_loop;

  ParseArgs(argc,argv);

  if(console_mode && tv_mode) {
    printf("Error: Can't be in both console mode tv mode\n\n");

    exit(1);
  }

  srand(time(NULL));

  g_snprintf(buf,256,"%s/.ddj",getenv("HOME"));

  LoadConfig(buf,"DDJ",1,1,cfg_entries);

  gtk_set_locale();

  if(console_mode) {
    gtk_type_init();
  }
  else {
    gtk_init(&argc,&argv);
  }

  if((GTK_MINOR_VERSION!=gtk_minor_version)) {
    printf("Warning: This version of DigitalDJ was compiled using gtk+ "
	   "version %d.%d.%d, and you are running gtk+ version %d.%d.%d...\n",
	   GTK_MAJOR_VERSION,GTK_MINOR_VERSION,GTK_MICRO_VERSION,
	   gtk_major_version,gtk_minor_version,gtk_micro_version);
  }

  root_mysql=mysql_init(NULL);

  gtk_rc_parse("~/.gtkrc");

  song_stack=NewIDStack(5);
  juke_list.ids=NULL;
  juke_list.num_ids=0;
  juke_list.current_id=0;
    
  if(do_x10) {
    ser_fd=x10_init(x10_serdevice);
    
    if(console_mode) {
      serchannel=g_io_channel_unix_new(ser_fd);
      g_io_add_watch(serchannel,G_IO_IN,(GIOFunc)SerialCB,0);
    }
    else gdk_input_add(ser_fd,GDK_INPUT_READ,SerialCB,0);
  }
  
#ifdef HAVE_LIRC_LIRC_CLIENT_H
  lirc_fd=init_lirc();
  if(console_mode) {
    lirc_channel=g_io_channel_unix_new(lirc_fd);
    g_io_add_watch(lirc_channel,G_IO_IN,(GIOFunc)LircCB,0);
  }
  else gdk_input_add(lirc_fd,GDK_INPUT_READ,LircCB,0);
#endif

  cmdsock_fd=InitCommandSocket(9999);
  if(console_mode) {
    cmdsock_channel=g_io_channel_unix_new(cmdsock_fd);
    g_io_add_watch(cmdsock_channel,G_IO_IN,(GIOFunc)CommandSocketCB,0);
  }
  else gdk_input_add(cmdsock_fd,GDK_INPUT_READ,CommandSocketCB,0);
  
  if(tv_mode || console_mode) {
    if(tv_mode) {
      window=gtk_window_new(GTK_WINDOW_POPUP);
      
      gtk_widget_set_usize(window,640,480);
          
      gtk_widget_realize(window);
      
      text_font=gdk_font_load
	("-Adobe-Helvetica-Medium-R-Normal--24-240-75-75-P-130-ISO8859-1");
      text_gc=MakePen(window,0xdddd,0xdddd,0xdddd);
      
      empty_image=Loadxpm(window,empty_xpm);
      HideMouse(window);
      
      main_draw=gtk_drawing_area_new();
      gtk_signal_connect(GTK_OBJECT(main_draw),"expose_event",
			 GTK_SIGNAL_FUNC(DrawScreen),NULL);
      gtk_container_add(GTK_CONTAINER(window),main_draw);
      gtk_widget_show(main_draw);
      
      gtk_window_set_title(GTK_WINDOW(window),"DigtialDJ");
      
      gtk_signal_connect(GTK_OBJECT(window),"delete_event",
			 GTK_SIGNAL_FUNC(ShutDown),NULL);
      
      gtk_widget_show(window);
    }
    
    DBScan();

    gtk_timeout_add(250,TimeOut,0);
    
    if(have_db && auto_start_play) Play();

    if(console_mode) {
      main_loop=g_main_new(FALSE);
      
      g_main_run(main_loop);
    }
    else {
      gtk_main();
    }
  }
  else {
    window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
    
    gtk_widget_realize(window);
    
    gtk_window_set_policy(GTK_WINDOW(window),FALSE,TRUE,FALSE);
    
    gtk_window_set_title(GTK_WINDOW(window),"DigitalDJ");
    
    gtk_signal_connect(GTK_OBJECT(window),"delete_event",
		       GTK_SIGNAL_FUNC(ShutDown),NULL);
    
    gtk_signal_connect_after(GTK_OBJECT(window),"size_allocate",
			     GTK_SIGNAL_FUNC(CheckWindowPosition),NULL);
    
    winbox=gtk_vbox_new(FALSE,3);
    if(!minimized) gtk_container_border_width(GTK_CONTAINER(winbox),3);
    
    MakeStyles();
    
    empty_image=Loadxpm(window,empty_xpm);

    main_notebook=gtk_notebook_new();
    
    MakePlayPage();
    MakeQueryPage();
    MakeEditPage();
    MakeConfigPage();
    MakeAboutPage();

    gtk_box_pack_start(GTK_BOX(winbox),main_notebook,TRUE,TRUE,0);
    if(!minimized) gtk_widget_show(main_notebook);
    
    control_box=MakeControls();
    gtk_box_pack_start(GTK_BOX(winbox),control_box,FALSE,FALSE,0);
    gtk_widget_show(control_box);
    
    gtk_container_add(GTK_CONTAINER(window),winbox);
    gtk_widget_show(winbox);
    
    gtk_widget_show(window);

    if(*cmdline_file)
      M3UFileList(cmdline_file);
    else
      DBScan();

    UpdateScrollDisplay(TRUE);

    gtk_timeout_add(250,TimeOut,0);
    
    if(have_db && auto_start_play) Play();

    gtk_main();
    
    DoSaveConfig();
  }

  FreeIDStack(song_stack);

  return 0;
}
