#include "config.h"
#include "scanthread.h"
#include "list.h"
#include "dirview.h"
#include <stdio.h>
#include "mp3map.h"
#include <qfileinfo.h>
#include <qapplication.h>
#ifdef HAVE_LIBID3
#undef HAVE_CONFIG_H
#include <id3/tag.h>
#endif
#ifdef HAVE_LIBVORBIS
#include <vorbis/vorbisfile.h>
#include <vorbis/codec.h>
#endif
#include "cache.h"
#include "cfgfile.h"

#define SCAN_USLEEP 40000

ScanItem::ScanItem()
{
	s_owner=0;
	s_thread=0;
	s_item=0;
}

ScanItem::ScanItem(const ScanItem& source)
{
	s_owner=source.s_owner;
	s_path=source.s_path;
	s_thread=source.s_thread;
	s_item=source.s_item;
}

ScanItem::ScanItem(ScanThread *t, List *o, QString p, DirViewItem *i)
{
	s_item=i;
	s_owner=o;
	s_path=p;
	s_thread=t;
}

ScanItem::~ScanItem()
{
}

ScanItem& ScanItem::operator= (const ScanItem& source)
{
	s_owner=source.s_owner;
	s_path=source.s_path;
	s_thread=source.s_thread;
	s_item=source.s_item;
	return *this;
}

DirViewItem *ScanItem::item()
{
	return s_item;
}

List *ScanItem::owner()
{
	return s_owner;
}

QString ScanItem::path()
{
	return s_path;
}

ScanThread *ScanItem::thread()
{
	return s_thread;
}

ScanThread::ScanThread()
{
	s_stop=false;
	s_skip=false;
	s_pause=false;
	cache=new Cache();
}

ScanThread::~ScanThread()
{
	stop();
	delete cache;
}

void ScanThread::run()
{
	for(;;)
	{
		if(s_pause)
		{
			usleep(50000);
			continue;
		}
		listlock.lock();
		s_skip=0;
		if(s_stop)
		{
			listlock.unlock();
			return;
		}
		QValueList<ScanItem>::Iterator it=list.begin();
		unsigned lcount=list.count();
		if(it == list.end())
		{
			listlock.unlock();
			listwait.wait();
		}
		else
		{
			ScanItem i=(*it);
			CacheItem ci;
			ci=cache->find(i.path());
			if(ci.path != QString::null)
			{
				listlock.unlock();
				QString time_string;
				time_string.sprintf("%d:%02d", ci.length/60, ci.length%60);
				QString bpm_string;
				bpm_string.sprintf("%.1lf", (double)ci.bpm/10.0);
				list.remove(it);
				lcount=list.count();
				listlock.unlock();
				app->lock();
				if(!s_skip)
				{
					if(list.count() != lcount)
					{
						app->unlock();
						continue;
					}
					if(!s_skip)
					{
						i.item()->setText(2, time_string);
						if(ci.bpm != -1)
							i.item()->setText(1, bpm_string);
					}
				}
				app->unlock();
				usleep(SCAN_USLEEP);
			}
			else
			{
				QFileInfo info(i.path());
				QString ext=info.extension(false);
printf("Scan %s\n", (const char *)info.absFilePath());
				if(ext.lower() == "mp3" || ext.lower() == "mp2" || ext.lower() == "mus")
				{
					int time_secs=-1;
					int bpm=-1;
#ifdef HAVE_LIBID3
					ID3_Tag tag(info.absFilePath());
					ID3_Frame *tlen=tag.Find(ID3FID_SONGLEN);
					ID3_Field *fld=0;
					ID3_Frame *tbpm=tag.Find(ID3FID_BPM);
					ID3_Field *fbpm=0;
#endif
					int res=-1;
					Mp3Map *map=0;
#ifdef HAVE_LIBID3
					if(tlen)
					{
						if(tbpm)
						{
							fbpm=tbpm->GetField(ID3FN_TEXT);
							if(fbpm)
								bpm=atoi(fbpm->GetRawText());
						}
						fld=tlen->GetField(ID3FN_TEXT);
						if(fld)
						{
							time_secs=atoi(fld->GetRawText())/1000;
							if(time_secs == 0)
							{
								time_secs=-1;
							}
							else
							{
								cache->add(i.path(), bpm, time_secs);
							}
						}
					}
#endif
					if(time_secs == -1)
					{
						map=new Mp3Map();
						if(config->getValue("QuickCheck", "0").toInt())
							res=map->basic_info(i.path());
						else
							res=map->attach(i.path());
					}
					if(time_secs == -1 && !res)
					{
						time_secs=map->samples/map->samplerate;
						cache->add(i.path(), -1, time_secs);
						QString time_string;
						time_string.sprintf("%d:%02d", time_secs/60, time_secs%60);
						list.remove(it);
						lcount=list.count();
						listlock.unlock();
						app->lock();
						if(!s_skip)
							i.item()->setText(2, time_string);
						app->unlock();
						usleep(SCAN_USLEEP);
#ifdef HAVE_LIBID3
						if(info.isWritable())
						{
							ID3_Frame tl(ID3FID_SONGLEN);
							tag.SetUnsync(true);
							QString fld_string;
							fld_string.sprintf("%d", time_secs*1000);
							if(!tlen)
							{
								tlen=&tl;
								if(!fld)
									fld=tlen->GetField(ID3FN_TEXT);
								if(fld)
									fld->Set(fld_string);
								tag.AddFrame(tlen);
							}
							else
							{
								if(!fld)
									fld=tlen->GetField(ID3FN_TEXT);
								fld->Set(fld_string);
							}
							tag.Update();
						}
#endif
					}
					else
					{
						list.remove(it);
						lcount=list.count();
						listlock.unlock();
						if(time_secs != -1)
						{
							QString time_string;
							time_string.sprintf("%d:%02d", time_secs/60, time_secs%60);
							QString bpm_string;
							bpm_string.sprintf("%.1lf", (double)bpm/10);

							app->lock();
							if(list.count() != lcount)
							{
								app->unlock();
								continue;
							}
							if(!s_skip)
							{
								i.item()->setText(2, time_string);
								if(bpm != -1)
									i.item()->setText(1, bpm_string);
							}
							app->unlock();
							usleep(SCAN_USLEEP);
						}
					}
					if(map)
						delete map;
				}
#ifdef HAVE_LIBVORBIS
				else if(ext.lower() == "ogg")
				{
					OggVorbis_File ov;
					FILE *fp;

					fp=fopen(info.absFilePath(), "r");
					if(fp)
					{
						if(ov_open(fp, &ov, NULL, 0) == 0)
						{
							if(ov_streams(&ov) == 1)
							{
								vorbis_info *info=ov_info(&ov, -1);
								if(info && info->channels == 2)
								{
									int sample_rate=info->rate;
									unsigned long frames=ov_pcm_total(&ov, -1);

									if(sample_rate)
									{
										int duration=frames/sample_rate;
										QString time_string;
										time_string.sprintf("%d:%02d", duration/60, duration%60);
										app->lock();
										if(list.count() != lcount)
										{
											app->unlock();
											continue;
										}
										if(!s_skip)
										{
											i.item()->setText(2, time_string);
										}
										app->unlock();
										usleep(SCAN_USLEEP);
									}
								}
							}
							ov_clear(&ov);
						}
						else
							fclose(fp);
					}
					list.remove(it);
					listlock.unlock();
				}
#endif
				else if(ext.lower() == "mpg" || ext.lower() == "mpeg")
				{
					list.remove(it);
					listlock.unlock();
				}
				else
				{
					list.remove(it);
					listlock.unlock();
				}
			}
		}
	}
}

void ScanThread::addFile(List *o, QString p, DirViewItem *i)
{
	if(!config->getValue("DoCheck", "1").toInt())
		return;

	CacheItem ci=cache->find(p);
	if(ci.path != QString::null)
	{
		QString time_string;
		time_string.sprintf("%d:%02d", ci.length/60, ci.length%60);
		QString bpm_string;
		bpm_string.sprintf("%.1lf", (double)ci.bpm/10.0);
		i->setText(2, time_string);
		if(ci.bpm != -1)
			i->setText(1, bpm_string);
		return;
	}
	list.append(ScanItem(this, o, p, i));
	listwait.wakeOne();
}

void ScanThread::removeFile(ScanItem it)
{
	removeFile(it.owner(), it.path());
}

void ScanThread::removeFile(List *o, QString p)
{
	listlock.lock();
	QValueList<ScanItem>::Iterator it;
	for(it=list.begin();it!=list.end();)
	{
		if((*it).owner() == o && (*it).path() == p)
			it=list.remove(it);
		else
			++it;
	}
	listlock.unlock();
}

void ScanThread::removeList(List *o)
{
	listlock.lock();
	QValueList<ScanItem>::Iterator it;
	for(it=list.begin();it!=list.end();)
	{
		if((*it).owner() == o)
			it=list.remove(it);
		else
			++it;
	}
	listlock.unlock();
}

void ScanThread::stop()
{
	pause(false);
	listlock.lock();
	s_stop=true;
	listlock.unlock();
	listwait.wakeOne();
	while(!finished())
		usleep(10000);
}

void ScanThread::setSkip()
{
	s_skip=true;
}

void ScanThread::pause(bool yes)
{
	s_pause=yes;
}

