# -*- coding: utf-8 -*-

# Copyright (c) 2007 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Package implementing lexers for the various supported programming languages.
"""

from PyQt4.QtCore import QStringList
from PyQt4.QtGui import QApplication

import Preferences

def getSupportedLanguages():
    """
    Module function to get a dictionary of supported lexer languages.
    
    @return dictionary of supported lexer languages. The keys are the
        internal language names. The items are lists of two entries.
        The first is the display string for the language, the second
        is a dummy file name, which can be used to derive the lexer.
        (QString, string)
    """
    supportedLanguages = {
        "Bash"       : [QApplication.translate('Lexers', "Bash"),         'dummy.sh'],
        "Batch"      : [QApplication.translate('Lexers', "Batch"),        'dummy.bat'],
        "C++"        : [QApplication.translate('Lexers', "C/C++"),        'dummy.cpp'],
        "C#"         : [QApplication.translate('Lexers', "C#"),           'dummy.cs'],
        "CMake"      : [QApplication.translate('Lexers', "CMake"),        'dummy.cmake'],
        "CSS"        : [QApplication.translate('Lexers', "CSS"),          'dummy.css'],
        "D"          : [QApplication.translate('Lexers', "D"),            'dummy.d'],
        "Diff"       : [QApplication.translate('Lexers', "Diff"),         'dummy.diff'],
        "HTML"       : [QApplication.translate('Lexers', "HTML/PHP/XML"), 'dummy.html'],
        "IDL"        : [QApplication.translate('Lexers', "IDL"),          'dummy.idl'],
        "Java"       : [QApplication.translate('Lexers', "Java"),         'dummy.java'],
        "JavaScript" : [QApplication.translate('Lexers', "JavaScript"),   'dummy.js'],
        "Lua"        : [QApplication.translate('Lexers', "Lua"),          'dummy.lua'],
        "Makefile"   : [QApplication.translate('Lexers', "Makefile"),     'dummy.mak'],
        "Perl"       : [QApplication.translate('Lexers', "Perl"),         'dummy.pl'],
        "Povray"     : [QApplication.translate('Lexers', "Povray"),       'dummy.pov'],
        "Properties" : [QApplication.translate('Lexers', "Properties"),   'dummy.ini'],
        "Python"     : [QApplication.translate('Lexers', "Python"),       'dummy.py'], 
        "Ruby"       : [QApplication.translate('Lexers', "Ruby"),         'dummy.rb'],
        "SQL"        : [QApplication.translate('Lexers', "SQL"),          'dummy.sql'],
        "TeX"        : [QApplication.translate('Lexers', "TeX"),          'dummy.tex'],
        "VHDL"       : [QApplication.translate('Lexers', "VHDL"),         'dummy.vhd'], 
    }
    # adjust list to various QScintilla versions
    removeLanguages = []
    for removeLanguage in removeLanguages:
        del supportedLanguages[removeLanguage]
    
    return supportedLanguages

def getLexer(language, parent = None):
    """
    Module function to instantiate a lexer object for a given language.
    
    @param language language of the lexer (string)
    @param parent reference to the parent object (QObject)
    @return reference to the instanciated lexer object (QsciLexer)
    """
    try:
        if language == "Python":
            from LexerPython import LexerPython
            return LexerPython(parent)
        elif language == "C++":
            from LexerCPP import LexerCPP
            return LexerCPP(parent, 
                Preferences.getEditor("CppCaseInsensitiveKeywords"))
        elif language == "C#":
            from LexerCSharp import LexerCSharp
            return LexerCSharp(parent)
        elif language == "IDL":
            from LexerIDL import LexerIDL
            return LexerIDL(parent)
        elif language == "Java":
            from LexerJava import LexerJava
            return LexerJava(parent)
        elif language == "JavaScript":
            from LexerJavaScript import LexerJavaScript
            return LexerJavaScript(parent)
        elif language == "SQL":
            from LexerSQL import LexerSQL
            return LexerSQL(parent)
        elif language == "HTML":
            from LexerHTML import LexerHTML
            return LexerHTML(parent)
        elif language == "Perl":
            from LexerPerl import LexerPerl
            return LexerPerl(parent)
        elif language == "Bash":
            from LexerBash import LexerBash
            return LexerBash(parent)
        elif language == "Ruby":
            from LexerRuby import LexerRuby
            return LexerRuby(parent)
        elif language == "Lua":
            from LexerLua import LexerLua
            return LexerLua(parent)
        elif language == "CSS":
            from LexerCSS import LexerCSS
            return LexerCSS(parent)
        elif language == "TeX":
            from LexerTeX import LexerTeX
            return LexerTeX(parent)
        elif language == "Diff":
            from LexerDiff import LexerDiff
            return LexerDiff(parent)
        elif language == "Makefile":
            from LexerMakefile import LexerMakefile
            return LexerMakefile(parent)
        elif language == "Properties":
            from LexerProperties import LexerProperties
            return LexerProperties(parent)
        elif language == "Batch":
            from LexerBatch import LexerBatch
            return LexerBatch(parent)
        elif language == "D":
            from LexerD import LexerD
            return LexerD(parent)
        elif language == "Povray":
            from LexerPOV import LexerPOV
            return LexerPOV(parent)
        elif language == "CMake":
            from LexerCMake import LexerCMake
            return LexerCMake(parent)
        elif language == "VHDL":
            from LexerVHDL import LexerVHDL
            return LexerVHDL(parent)
        else:
            return None
    except ImportError:
        return None

def getOpenFileFiltersList(includeAll = False, asString = False):
    """
    Module function to get the file filter list for an open file operation.
    
    @param includeAll flag indicating the inclusion of the 
        All Files filter (boolean)
    @param asString flag indicating the list should be returned 
        as a string (QString)
    @return file filter list (QStringList or QString)
    """
    openFileFiltersList = QStringList() \
        << QApplication.translate('Lexers', 
            'Python Files (*.py)') \
        << QApplication.translate('Lexers', 
            'Python Files win32 (*.pyw)') \
        << QApplication.translate('Lexers', 
            'Pyrex Files (*.pyx)') \
        << QApplication.translate('Lexers', 
            'Quixote Template Files (*.ptl)') \
        << QApplication.translate('Lexers', 
            'Ruby Files (*.rb)') \
        << QApplication.translate('Lexers', 
            'IDL Files (*.idl)') \
        << QApplication.translate('Lexers', 
            'C Files (*.h *.c)') \
        << QApplication.translate('Lexers', 
            'C++ Files (*.h *.hpp *.hh *.cxx *.cpp *.cc)') \
        << QApplication.translate('Lexers', 
            'C# Files (*.cs)') \
        << QApplication.translate('Lexers', 
            'HTML Files (*.html *.htm *.asp *.shtml)') \
        << QApplication.translate('Lexers', 
            'CSS Files (*.css)') \
        << QApplication.translate('Lexers', 
            'QSS Files (*.qss)') \
        << QApplication.translate('Lexers', 
            'PHP Files (*.php *.php3 *.php4 *.php5 *.phtml)') \
        << QApplication.translate('Lexers', 
            'XML Files (*.xml *.xsl *.xslt *.dtd)') \
        << QApplication.translate('Lexers', 
            'Qt Resource Files (*.qrc)') \
        << QApplication.translate('Lexers', 
            'D Files (*.d *.di)') \
        << QApplication.translate('Lexers', 
            'Java Files (*.java)') \
        << QApplication.translate('Lexers', 
            'JavaScript Files (*.js)') \
        << QApplication.translate('Lexers', 
            'SQL Files (*.sql)') \
        << QApplication.translate('Lexers', 
            'Docbook Files (*.docbook)') \
        << QApplication.translate('Lexers', 
            'Perl Files (*.pl *.pm *.ph)') \
        << QApplication.translate('Lexers', 
            'Lua Files (*.lua)') \
        << QApplication.translate('Lexers', 
            'Tex Files (*.tex *.sty *.aux *.toc *.idx)') \
        << QApplication.translate('Lexers', 
            'Shell Files (*.sh)') \
        << QApplication.translate('Lexers', 
            'Batch Files (*.bat *.cmd)') \
        << QApplication.translate('Lexers', 
            'Diff Files (*.diff *.patch)') \
        << QApplication.translate('Lexers', 
            'Makefiles (*.mak)') \
        << QApplication.translate('Lexers', 
            'Properties Files (*.properties *.ini *.inf *.reg *.cfg *.cnf *.rc)') \
        << QApplication.translate('Lexers', 
            'Povray Files (*.pov)') \
        << QApplication.translate('Lexers', 
            'CMake Files (CMakeLists.txt *.cmake *.ctest)') \
        << QApplication.translate('Lexers', 
            'VHDL Files (*.vhd *.vhdl)')
    openFileFiltersList.sort()
    if includeAll:
        openFileFiltersList.append(QApplication.translate('Lexers', 'All Files (*)'))
    
    if asString:
        return openFileFiltersList.join(';;')
    else:
        return openFileFiltersList

def getSaveFileFiltersList(includeAll = False, asString = False):
    """
    Module function to get the file filter list for a save file operation.
    
    @param includeAll flag indicating the inclusion of the 
        All Files filter (boolean)
    @param asString flag indicating the list should be returned 
        as a string (QString)
    @return file filter list (QStringList or QString)
    """
    saveFileFiltersList = QStringList() \
        << QApplication.translate('ViewManager', 
            "Python Files (*.py)") \
        << QApplication.translate('ViewManager', 
            "Python Files win32 (*.pyw)") \
        << QApplication.translate('ViewManager', 
            "Pyrex Files (*.pyx)") \
        << QApplication.translate('ViewManager', 
            "Quixote Template Files (*.ptl)") \
        << QApplication.translate('ViewManager', 
            "Ruby Files (*.rb)") \
        << QApplication.translate('ViewManager', 
            "IDL Files (*.idl)") \
        << QApplication.translate('ViewManager', 
            "C Files (*.c)") \
        << QApplication.translate('ViewManager', 
            "C++ Files (*.cpp)") \
        << QApplication.translate('ViewManager', 
            "C++/C Header Files (*.h)") \
        << QApplication.translate('ViewManager', 
            "C# Files (*.cs)") \
        << QApplication.translate('ViewManager', 
            "HTML Files (*.html)") \
        << QApplication.translate('ViewManager', 
            "PHP Files (*.php)") \
        << QApplication.translate('ViewManager', 
            "ASP Files (*.asp)") \
        << QApplication.translate('ViewManager', 
            "CSS Files (*.css)") \
        << QApplication.translate('ViewManager', 
            "QSS Files (*.qss)") \
        << QApplication.translate('ViewManager', 
            "XML Files (*.xml)") \
        << QApplication.translate('ViewManager', 
            "XSL Files (*.xsl)") \
        << QApplication.translate('ViewManager', 
            "DTD Files (*.dtd)") \
        << QApplication.translate('ViewManager', 
            "Qt Resource Files (*.qrc)") \
        << QApplication.translate('ViewManager', 
            "D Files (*.d)") \
        << QApplication.translate('ViewManager', 
            "D Interface Files (*.di)") \
        << QApplication.translate('ViewManager', 
            "Java Files (*.java)") \
        << QApplication.translate('ViewManager', 
            "JavaScript Files (*.js)") \
        << QApplication.translate('ViewManager', 
            "SQL Files (*.sql)") \
        << QApplication.translate('ViewManager', 
            "Docbook Files (*.docbook)") \
        << QApplication.translate('ViewManager', 
            "Perl Files (*.pl)") \
        << QApplication.translate('ViewManager', 
            "Perl Module Files (*.pm)") \
        << QApplication.translate('ViewManager', 
            "Lua Files (*.lua)") \
        << QApplication.translate('ViewManager', 
            "Shell Files (*.sh)") \
        << QApplication.translate('ViewManager', 
            "Batch Files (*.bat)") \
        << QApplication.translate('ViewManager', 
            "TeX Files (*.tex)") \
        << QApplication.translate('ViewManager', 
            "TeX Template Files (*.sty)") \
        << QApplication.translate('ViewManager', 
            "Diff Files (*.diff)") \
        << QApplication.translate('ViewManager', 
            "Make Files (*.mak)") \
        << QApplication.translate('ViewManager', 
            "Properties Files (*.ini)") \
        << QApplication.translate('ViewManager', 
            "Configuration Files (*.cfg)") \
        << QApplication.translate('ViewManager', 
            'Povray Files (*.pov)') \
        << QApplication.translate('ViewManager', 
            'CMake Files (CMakeLists.txt)') \
        << QApplication.translate('ViewManager', 
            'CMake Macro Files (*.cmake)') \
        << QApplication.translate('ViewManager', 
            'VHDL Files (*.vhd)')
    saveFileFiltersList.sort()
    
    if includeAll:
        saveFileFiltersList.append(QApplication.translate('Lexers', 'All Files (*)'))
    
    if asString:
        return saveFileFiltersList.join(';;')
    else:
        return saveFileFiltersList

def getDefaultLexerAssociations():
    return {
        '*.sh'              : "Bash",
        '*.bash'            : "Bash",
        "*.bat"             : "Batch",
        "*.cmd"             : "Batch",
        '*.cpp'             : "C++",
        '*.cxx'             : "C++",
        '*.cc'              : "C++",
        '*.c'               : "C++",
        '*.hpp'             : "C++",
        '*.hh'              : "C++",
        '*.h'               : "C++",
        '*.cs'              : "C#",
        'CMakeLists.txt'    : "CMake", 
        '*.cmake'           : "CMake",
        '*.cmake.in'        : "CMake",
        '*.ctest'           : "CMake",
        '*.ctest.in'        : "CMake",
        '*.css'             : "CSS",
        '*.qss'             : "CSS",
        "*.d"               : "D",
        "*.di"              : "D",
        "*.diff"            : "Diff",
        "*.patch"           : "Diff",
        '*.html'            : "HTML",
        '*.htm'             : "HTML",
        '*.asp'             : "HTML",
        '*.shtml'           : "HTML",
        '*.php'             : "HTML",
        '*.php3'            : "HTML",
        '*.php4'            : "HTML",
        '*.php5'            : "HTML",
        '*.phtml'           : "HTML",
        '*.xml'             : "HTML",
        '*.xsl'             : "HTML",
        '*.svg'             : "HTML",
        '*.xsd'             : "HTML",
        '*.xslt'            : "HTML",
        '*.dtd'             : "HTML",
        '*.rdf'             : "HTML",
        '*.docbook'         : "HTML",
        '*.ui'              : "HTML",
        '*.ts'              : "HTML",
        '*.qrc'             : "HTML",
        '*.e3d'             : "HTML",
        '*.e3k'             : "HTML",
        '*.e3p'             : "HTML",
        '*.e3s'             : "HTML",
        '*.e3t'             : "HTML",
        '*.e4d'             : "HTML",
        '*.e4k'             : "HTML",
        '*.e4p'             : "HTML",
        '*.e4q'             : "HTML",
        '*.e4s'             : "HTML",
        '*.e4t'             : "HTML",
        '*.kid'             : "HTML",
        '*.idl'             : "IDL",
        '*.java'            : "Java",
        '*.js'              : "JavaScript",
        '*.lua'             : "Lua",
        "*makefile"         : "Makefile",
        "Makefile*"         : "Makefile",
        "*.mak"             : "Makefile",
        '*.pl'              : "Perl",
        '*.pm'              : "Perl",
        '*.ph'              : "Perl",
        '*.pov'             : "Povray",
        "*.properties"      : "Properties",
        "*.ini"             : "Properties",
        "*.inf"             : "Properties",
        "*.reg"             : "Properties",
        "*.cfg"             : "Properties",
        "*.cnf"             : "Properties",
        "*.rc"              : "Properties",
        '*.py'              : "Python",
        '*.pyw'             : "Python",
        '*.pyx'             : "Python",
        '*.ptl'             : "Python",
        '*.rb'              : "Ruby",
        '*.rbw'             : "Ruby",
        '*.sql'             : "SQL",
        "*.tex"             : "TeX",
        "*.sty"             : "TeX",
        "*.aux"             : "TeX",
        "*.toc"             : "TeX",
        "*.idx"             : "TeX",
        '*.vhd'             : "VHDL", 
        '*.vhdl'            : "VHDL", 
    }
