/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-params.c,v 1.9 2003/12/15 16:08:53 hoa Exp $
 */

#include "etpan-msg-params.h"

#include <stdlib.h>

#include "etpan-errors.h"

struct etpan_msg_param * etpan_msg_param_new(void)
{
  struct etpan_msg_param * param;

  param = malloc(sizeof(struct etpan_msg_param));
  if (param == NULL)
    return param;

  param->ref_count = 0;
  param->ref_mime = 0;
  param->score = 0;
  param->stop_score = 0;
  param->hidden = 0;

  return param;
}

void etpan_msg_param_free(struct etpan_msg_param * param)
{
  free(param);
}

int etpan_msg_param_ref_message(struct etpan_msg_params * params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(params->hash, &key, &data);

  param = data.data;

  param->ref_count ++;

  return param->ref_count;
}

int etpan_msg_param_message_get_ref_count(struct etpan_msg_params * params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(params->hash, &key, &data);

  param = data.data;

  return param->ref_count;
}

int etpan_msg_param_ref_mime(struct etpan_msg_params * params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(params->hash, &key, &data);

  param = data.data;

  param->ref_mime ++;

  return param->ref_mime;
}

int etpan_msg_param_unref_message(struct etpan_msg_params * params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;
  int r;

  key.data = &msg;
  key.len = sizeof(msg);
  r = chash_get(params->hash, &key, &data);
  if (r < 0)
    return -1;
  
  param = data.data;

  param->ref_count --;
  
  return param->ref_count;
}

int etpan_msg_param_unref_mime(struct etpan_msg_params * params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(params->hash, &key, &data);

  param = data.data;

  param->ref_mime --;
  
  return param->ref_mime;
}


/*
  create the messages states relatives to the messages of 
  the given list of messages
*/

int etpan_msg_params_add_list(struct etpan_msg_params * params,
    carray * msg_list)
{
  chashdatum key;
  chashdatum data;
  struct etpan_msg_param * param;
  unsigned int i;
  int r;

  for(i = 0 ; i < carray_count(msg_list) ; i ++) {
    mailmessage * msg;

    key.data = &msg;
    key.len = sizeof(msg);

    msg = carray_get(msg_list, i);
    
    r = chash_get(params->hash, &key, &data);
    if (r == 0)
      continue;

    param = etpan_msg_param_new();
    if (param == NULL)
      goto err;

#if 0
    r = carray_add(params->tab, msg, NULL);
    if (r < 0) {
      msg_param_free(param);
      goto err;
    }
#endif

    data.data = param;
    data.len = 0;

    r = chash_set(params->hash, &key, &data, NULL);
    if (r < 0) {
      etpan_msg_param_free(param);
      goto err;
    }
  }

  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}

void etpan_msg_params_remove(struct etpan_msg_params * params,
    mailmessage * msg)
{
  chashdatum key;
  chashdatum data;
  struct etpan_msg_param * param;
  int r;
  
  key.data = &msg;
  key.len = sizeof(msg);
  
  r = chash_delete(params->hash, &key, &data);
  
  if (r == 0) {
    param = data.data;
    
    etpan_msg_param_free(param);
  }
}


/* initializes structures */

struct etpan_msg_params * etpan_msg_params_new(void)
{
  struct etpan_msg_params * params;

  params = malloc(sizeof(struct etpan_msg_params));
  if (params == NULL)
    goto err;

  params->hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (params->hash == NULL)
    goto free;
#if 0
  params->tab = carray_new(128);
  if (params->tab == NULL)
    goto free_hash;
#endif

  return params;

#if 0
 free_hash:
  chash_free(params->hash);
#endif
 free:
  free(params);
 err:
  return NULL;
}

void etpan_msg_params_free(struct etpan_msg_params * params)
{
  etpan_msg_params_clear(params);
#if 0
  carray_free(params->tab);
#endif
  chash_free(params->hash);
  free(params);
}


/*
  destroys the messages states relatives to the messages of
  the given list of messages
*/

void etpan_msg_params_clear(struct etpan_msg_params * params)
{
#if 0
  uint32_t i;
  int r;
  chashdatum key;
  chashdatum data;
  chashiter * cur;

  for(i = 0 ; i < carray_count(params->tab) ; i ++) {
    struct etpan_msg_param * param;
    mailmessage * msg;

    msg = carray_get(params->tab, i);

    key.data = (char *) &msg;
    key.len = sizeof(msg);
    r = chash_delete(params->hash, &key, &data);
    if (r == 0) {
      param = (struct etpan_msg_param *) data.data;
      msg_param_free(param);
    }
  }
  carray_set_size(params->tab, 0);
#endif
  chashiter * cur;

  for(cur = chash_begin(params->hash) ; cur != NULL ;
      cur = chash_next(params->hash, cur)) {
    chashdatum data;
    struct etpan_msg_param * param;
    
    chash_value(cur, &data);
    param = data.data;
    etpan_msg_param_free(param);
  }

  chash_clear(params->hash);
}


/* score */

void etpan_msg_set_score(struct etpan_msg_params * msg_params,
    mailmessage * msg, int32_t value)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(msg_params->hash, &key, &data);

  param = data.data;

  if (!param->stop_score)
    param->score = value;
}

void etpan_msg_add_score(struct etpan_msg_params * msg_params,
    mailmessage * msg, int32_t value)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(msg_params->hash, &key, &data);

  param = data.data;

  if (!param->stop_score)
    param->score += value;
}

int32_t etpan_msg_get_score(struct etpan_msg_params * msg_params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;
  
  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(msg_params->hash, &key, &data);

  param = data.data;

  return param->score;
}

void etpan_msg_stop_score(struct etpan_msg_params * msg_params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(msg_params->hash, &key, &data);

  param = data.data;
  param->stop_score = 1;
}

/*
  visibility of the message
*/

void etpan_msg_hide(struct etpan_msg_params * msg_params,
    mailmessage * msg, int do_hide)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(msg_params->hash, &key, &data);

  param = data.data;
  param->hidden = do_hide;
}

int etpan_msg_is_hidden(struct etpan_msg_params * msg_params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg;
  key.len = sizeof(msg);
  chash_get(msg_params->hash, &key, &data);

  param = data.data;

  return param->hidden;
}


struct etpan_node_msg_param *
etpan_node_msg_param_new(struct etpan_msg_params * msg_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  
  param = malloc(sizeof(struct etpan_node_msg_param));
  if (param == NULL)
    return NULL;
  
  if ((msg_tree->node_msg != NULL) && (msg_params != NULL))
    param->score = etpan_msg_get_score(msg_params, msg_tree->node_msg);
  else
    param->score = 0;

  param->selected = 0;
  param->opened_thread = 0;
  param->visibility = 0;
  param->thread_visible = 0;

  return param;
}

void etpan_node_msg_param_free(struct etpan_node_msg_param * param)
{
  free(param);
}




struct etpan_node_msg_params * etpan_node_msg_params_new(void)
{
  struct etpan_node_msg_params * params;

  params = malloc(sizeof(struct etpan_node_msg_params));
  if (params == NULL)
    goto err;

  params->hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (params->hash == NULL)
    goto free;
#if 0
  params->tab = carray_new(128);
  if (params->tab == NULL)
    goto free_hash;
#endif

  return params;

#if 0
 free_hash:
  chash_free(params->hash);
#endif
 free:
  free(params);
 err:
  return NULL;
}

void etpan_node_msg_params_free(struct etpan_node_msg_params * params)
{
  etpan_node_msg_params_clear(params);
#if 0
  carray_free(params->tab);
#endif
  chash_free(params->hash);
  free(params);
}


int etpan_node_msg_params_add_recursive(struct etpan_node_msg_params * params,
    struct etpan_msg_params * msg_params,
    struct mailmessage_tree * msg_tree)
{
  chashdatum key;
  chashdatum data;
  int r;
  unsigned int cur;

#if 0
  r = carray_add(params->tab, msg_tree, NULL);
  if (r < 0) {
    node_msg_param_free(param);
    goto err;
  }
#endif

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);

  r = chash_get(params->hash, &key, &data);
  if (r < 0) {
    struct etpan_node_msg_param * param;
    
    param = etpan_node_msg_param_new(msg_params, msg_tree);
    if (param == NULL)
      goto err;
    
    data.data = param;
    data.len = 0;
    
    r = chash_set(params->hash, &key, &data, NULL);
    if (r < 0) {
      etpan_node_msg_param_free(param);
      goto err;
    }
  }
  
  for(cur = 0 ; cur < carray_count(msg_tree->node_children) ; cur ++) {
    r = etpan_node_msg_params_add_recursive(params, msg_params,
        carray_get(msg_tree->node_children, cur));
    if (r != NO_ERROR)
      goto err;
  }

  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}

void etpan_node_msg_params_clear(struct etpan_node_msg_params * params)
{
#if 0
  uint32_t i;
  int r;
  chashdatum key;
  chashdatum data;

  for(i = 0 ; i < carray_count(params->tab) ; i ++) {
    struct etpan_node_msg_param * param;
    struct mailmessage_tree * msg_tree;

    msg_tree = carray_get(params->tab, i);

    key.data = (char *) &msg_tree;
    key.len = sizeof(msg_tree);
    r = chash_delete(params->hash, &key, &data);
    if (r == 0) {
      param = (struct etpan_node_msg_param *) data.data;
      node_msg_param_free(param);
    }
  }
  carray_set_size(params->tab, 0);
#endif

  chashiter * cur;

  for(cur = chash_begin(params->hash) ; cur != NULL ;
      cur = chash_next(params->hash, cur)) {
    chashdatum data;
    struct etpan_node_msg_param * param;
    
    chash_value(cur, &data);
    param = data.data;
    etpan_node_msg_param_free(param);
  }
  chash_clear(params->hash);
}


/*
  visibility of the children of the message in the thread
*/

int etpan_node_msg_is_opened(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  return param->opened_thread;
}

void etpan_node_msg_set_visibility(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree, int visible)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  param->visibility = visible;
}

int etpan_node_msg_is_visible(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  return param->visibility;
}

static void update_visibility(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree, int visible)
{
  unsigned int cur;

  for(cur = 0 ; cur < carray_count(msg_tree->node_children) ; cur ++) {
    chashdatum key;
    chashdatum data;
    struct mailmessage_tree * child;
    struct etpan_node_msg_param * child_param;
      
    child = carray_get(msg_tree->node_children, cur);
      
    key.data = &child;
    key.len = sizeof(child);
    chash_get(tree_params->hash, &key, &data);
      
    child_param = data.data;
      
    if (child_param->thread_visible != visible) {
      if (child->node_msg != NULL) {
        child_param->visibility = visible;
      }
      child_param->thread_visible = visible;

      if (!visible)
        update_visibility(tree_params, child, 0);
      else
        update_visibility(tree_params, child, child_param->opened_thread);
    }
  }
}

void etpan_node_msg_set_opened(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree,
    int opened)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  param->opened_thread = opened;

  update_visibility(tree_params, msg_tree, param->opened_thread);
}


void etpan_node_msg_change_opened(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  param->opened_thread = !param->opened_thread;

  update_visibility(tree_params, msg_tree, param->opened_thread);
}

void etpan_node_msg_set_selected(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree,
    int sel)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  param->selected = sel;
}

void etpan_node_msg_change_selected(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  param->selected = !param->selected;
}

int etpan_node_msg_is_selected(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  return param->selected;
}




/*
  score of the message
*/

void etpan_node_msg_set_score(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree, int32_t value)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  param->score = value;
}

int32_t etpan_node_msg_get_score(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  chash_get(tree_params->hash, &key, &data);

  param = data.data;

  return param->score;
}


struct etpan_node_msg_param *
etpan_node_msg_get_param(struct etpan_node_msg_params * tree_params,
    struct mailmessage_tree * msg_tree)
{
  struct etpan_node_msg_param * param;
  chashdatum key;
  chashdatum data;
  int r;

  key.data = &msg_tree;
  key.len = sizeof(msg_tree);
  r = chash_get(tree_params->hash, &key, &data);
  if (r < 0)
    return NULL;

  param = data.data;
  
  return param;
}

struct etpan_msg_param *
etpan_msg_get_param(struct etpan_msg_params * msg_params,
    mailmessage * msg)
{
  struct etpan_msg_param * param;
  chashdatum key;
  chashdatum data;
  int r;
  
  key.data = &msg;
  key.len = sizeof(msg);
  r = chash_get(msg_params->hash, &key, &data);
  if (r < 0)
    return NULL;

  param = data.data;
  
  return param;
}
