/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_dm.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"


/*
 *  Function: dl_compare_target_lists
 *
 *  Called to compare 2 drivelink target lists.
 *
 *  Returns:  0  ... if they match
 *            EINVAL if they dont match
 */
int dl_compare_target_lists(dm_target_t *list1, dm_target_t *list2)
{
        int rc=0;
        dm_target_t *l1 = list1;
        dm_target_t *l2 = list2;

        LOG_ENTRY();

        REQUIRE(l1 != NULL);
        REQUIRE(l2 != NULL);

        while (l1 && l2 && rc==0) {
                if ( l1->type   != l2->type  ||
                     l1->start  != l2->start ||
                     l1->length != l2->length ) {
                        rc = EINVAL;
                }
                l1=l1->next;
                l2=l2->next;
                if ( (l1==NULL && l2!=NULL) ||
                     (l1!=NULL && l2==NULL) ) {
                        rc = EINVAL;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
 }


/*
 *  Function:  dl_build_target_list
 *
 *  Called to build a device mapper target list for the
 *  specified drive link storage object.
 */
int dl_build_target_list(storage_object_t *drivelink, dm_target_t **target_list)
{
        int rc, i;
        drivelink_private_data_t *pdata=NULL;
        dm_target_t *target=NULL, *targets=NULL;
        dm_device_t *dev;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink) == TRUE);
        REQUIRE(target_list != NULL);

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        REQUIRE(pdata->drive_link_count > 0);

        for (i=0,rc=0; (!rc) && (i<pdata->drive_link_count); i++) {

                if (pdata->drive_link[i].flags & DL_FLAG_MISSING){
                        target = EngFncs->dm_allocate_target( DM_TARGET_ERROR,
                                                                     pdata->drive_link[i].start_lsn,
                                                                     pdata->drive_link[i].sector_count, 0, 0);
                }
                else {
                        target = EngFncs->dm_allocate_target( DM_TARGET_LINEAR,
                                                                     pdata->drive_link[i].start_lsn,
                                                                     pdata->drive_link[i].sector_count, 0, 0);
                        if (target) {
                                dev = target->data.linear;
                                dev->major = pdata->drive_link[i].object->dev_major;
                                dev->minor = pdata->drive_link[i].object->dev_minor;
                                dev->start = 0;
                        }
                }

                if (target) {
                        EngFncs->dm_add_target(target, &targets);
                }
                else {
                        rc = ENOMEM;
                }

        }

        if (rc==0) {
                *target_list = targets;
        }
        else {
                if (targets) EngFncs->dm_deallocate_targets(targets);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_get_devmap_info
 *
 *  Called to test if the drivelink has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
int dl_get_devmap_info( storage_object_t *drivelink )
{
        int rc;
        dm_target_t *list1=NULL, *list2=NULL;

        LOG_ENTRY();

        REQUIRE(drivelink);

        rc = EngFncs->dm_update_status(drivelink);
        if (rc==0) {

                if ( drivelink->flags & SOFLAG_ACTIVE ) {

                        rc = EngFncs->dm_get_targets(drivelink, &list1);
                        if (rc==0) {
                                rc = dl_build_target_list(drivelink, &list2);
                        }

                        if (rc==0) {
                                rc = dl_compare_target_lists(list1,list2);
                        }

                        if (rc) {
                                LOG_DEBUG("this drivelink is being marked needs_activate\n");
                                drivelink->flags |= SOFLAG_NEEDS_ACTIVATE;
                        }
			else {
				drivelink->flags &= ~SOFLAG_NEEDS_ACTIVATE;
			}

                }

        }

        if (list1) EngFncs->dm_deallocate_targets(list1);
        if (list2) EngFncs->dm_deallocate_targets(list2);

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_can_activate
 */
int dl_can_activate( storage_object_t *drivelink )
{
        LOG_ENTRY();

        REQUIRE( dl_isa_drivelink(drivelink) == TRUE );

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function: dl_activate
 *
 *  Called in order to create a mapping of an evms drivelink
 *  in the kernel.
 */
int dl_activate( storage_object_t *drivelink )
{
        int rc;
        dm_target_t *target_list = NULL;

        LOG_ENTRY();

        REQUIRE( dl_isa_drivelink(drivelink) == TRUE );

        rc = dl_build_target_list(drivelink, &target_list);
        if (rc==0) {
                rc = EngFncs->dm_activate(drivelink, target_list);
        }

        if (rc==0) {
                drivelink->flags &= ~SOFLAG_NEEDS_ACTIVATE;
        }

        if (target_list) EngFncs->dm_deallocate_targets(target_list);

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_can_deactivate
 */
int dl_can_deactivate( storage_object_t *drivelink )
{
        LOG_ENTRY();

        REQUIRE( dl_isa_drivelink(drivelink) == TRUE );

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function: dl_deactivate
 *
 *  Called in order to delete an existing mapping of an evms
 *  drivelink storage object in the kernel.
 */
int dl_deactivate( storage_object_t *drivelink )
{
        int rc;

        LOG_ENTRY();

        REQUIRE( drivelink != NULL );
        REQUIRE( drivelink->plugin == dl_plugin_record );

        rc = EngFncs->dm_deactivate(drivelink);

	if (!rc) {
		drivelink->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
	}

        LOG_EXIT_INT(rc);
        return rc;
}

