/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: snap_rollback.c
 *
 * Functions for implementing snapshot rollback.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "snapshot.h"


/**
 * can_rollback
 *
 * Can the current state of the snapshot be rolled-back onto the origin?
 * To roll-back a snapshot, the following conditions must be met:
 * - The specified snapshot must be the only snapshot of its origin.
 * - The snapshot must not be dirty/waiting for a commit.
 * - The snapshot must be valid.
 * - The snapshot must be active.
 * - The snapshot must not be scheduled for deactivation.
 * - Both the snapshot and the origin must be unmounted.
 **/
int can_rollback(snapshot_volume_t * snap_volume, int during_commit)
{
	snapshot_volume_t * org_volume = snap_volume->origin;
	int rc = EINVAL;

	LOG_ENTRY();

	if (org_volume->count != 1) {
		LOG_DETAILS("Cannot roll-back snapshot %s. Origin %s has more "
			    "than one snapshot.\n", snap_volume->parent->name,
			    org_volume->parent->volume->name);
		goto out;
	}

	if (is_invalid(snap_volume)) {
		/* The snapshot may have been filled after it was marked for
		 * roll-back. If so, turn off the roll-back flag.
		 */
		rollback_complete(snap_volume);
		LOG_DETAILS("Cannot roll-back full/disabled snapshot %s.\n",
			    snap_volume->parent->name);
		goto out;
	}

	if (!during_commit && commit_is_pending(snap_volume)) {
		LOG_DETAILS("Cannot roll-back snapshot %s. Please save pending "
			    "changes first.\n", snap_volume->parent->name);
		goto out;
	}

	if (!is_active(snap_volume)) {
		LOG_DETAILS("Snapshot %s must be active to be eligible for "
			    "roll-back.\n", snap_volume->parent->name);
		goto out;
	}

	if (deactivate_is_pending(snap_volume)) {
		LOG_DETAILS("Snapshot %s is already pending deactivation.\n",
			    snap_volume->parent->name);
		goto out;
	}

	if (!EngFncs->is_offline(snap_volume->parent, NULL) ||
	    !EngFncs->is_offline(org_volume->parent, NULL)) {
		LOG_DETAILS("Snapshot %s and origin %s must both be unmounted "
			    "to be eligible for roll-back.\n",
			    snap_volume->parent->name,
			    org_volume->parent->volume->dev_node);
		if (during_commit) {
			MESSAGE(_("Please unmount the snapshot and origin "
				"volumes, and try to save changes again."));
		}
		goto out;
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * start_rollback_progress
 *
 * Create a progress task to inform the user of the progress of the rollback.
 **/
static progress_t * start_rollback_progress(snapshot_volume_t * snap_volume)
{
	progress_t * progress;

	LOG_ENTRY();

	progress = EngFncs->engine_alloc(sizeof(*progress));
	if (!progress) {
		goto out;
	}

	progress->title = EngFncs->engine_alloc(350);
	if (!progress->title) {
		EngFncs->engine_free(progress);
		goto out;
	}

	progress->id = 0;
	snprintf(progress->title, 350, "Snapshot: Rolling-back %s to %s",
		 snap_volume->parent->name,
		 snap_volume->origin->parent->volume->name);
	progress->description = NULL;
	progress->type = INDETERMINATE;
	progress->count = 0;
	progress->total_count = 1;
	progress->remaining_seconds = 0;
	progress->plugin_private_data = NULL;

	EngFncs->progress(progress);

out:
	LOG_EXIT_PTR(progress);
	return progress;
}

/**
 * update_rollback_progress
 **/
static void update_rollback_progress(snapshot_volume_t * snap_volume,
				     progress_t * progress,
				     dm_disk_exception_t * table_entry)
{
	LOG_ENTRY();

	snprintf(progress->title, 350, "Snapshot: Rolling-back %s to %s: chunk "
		 "%"PRIu64" to chunk %"PRIu64, snap_volume->parent->name,
		 snap_volume->origin->parent->volume->name,
		 table_entry->new_chunk, table_entry->old_chunk);

	EngFncs->progress(progress);

	LOG_EXIT_VOID();
}

/**
 * end_rollback_progress
 **/
static void end_rollback_progress(snapshot_volume_t * snap_volume,
				  progress_t * progress)
{
	LOG_ENTRY();

	if (progress) {
		progress->count = progress->total_count;
		EngFncs->progress(progress);
		EngFncs->engine_free(progress->title);
		EngFncs->engine_free(progress);
	}

	LOG_EXIT_VOID();
}

/**
 * read_exception_table
 * @snap_volume:	Snapshot to read the table from.
 * @table:		Buffer to read the table into.
 * @table_index:	Index of the table to read.
 *
 * Read one exception table from the snapshot. Each table is one chunk in
 * size, and contains EXCEPTION_TABLE_ENTRIES(chunk_size) dm_disk_exception
 * entries.
 **/
static int read_exception_table(snapshot_volume_t * snap_volume,
				dm_disk_exception_t * table,
				u_int32_t table_index)
{
	lsn_t table_lsn;
	u_int32_t stride;
	int rc;

	LOG_ENTRY();

	stride = EXCEPTION_TABLE_ENTRIES(snap_volume->metadata->chunk_size) + 1;
	table_lsn = (1 + table_index * stride) *
		    snap_volume->metadata->chunk_size;

	rc = READ(snap_volume->child, table_lsn,
		  snap_volume->metadata->chunk_size, table);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * rollback_chunk
 *
 * Copy one chunk from the snapshot to the origin, using the Engine's copy
 * service.
 **/
static int rollback_chunk(snapshot_volume_t * snap_volume,
			  dm_disk_exception_t * table_entry)
{
	copy_job_t copy;
	u_int32_t chunk_size = snap_volume->metadata->chunk_size;
	int rc;

	LOG_ENTRY();
	LOG_DETAILS("Copying chunk %"PRIu64" from snapshot %s to chunk %"PRIu64
		    " on origin %s.\n", table_entry->new_chunk,
		    snap_volume->parent->name, table_entry->old_chunk,
		    snap_volume->origin->parent->name);

	copy.src.obj = snap_volume->child;
	copy.src.start = table_entry->new_chunk * chunk_size;
	copy.src.len = min(chunk_size,
			   snap_volume->child->size - copy.src.start);
	copy.trg.obj = snap_volume->origin->child;
	copy.trg.start = table_entry->old_chunk * chunk_size;
	copy.trg.len = min(chunk_size,
			   snap_volume->origin->child->size - copy.trg.start);
	copy.title = NULL;
	copy.description = NULL;

	rc = EngFncs->offline_copy(&copy);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * rollback_reset
 *
 * When a rollback is complete, the snapshot volume needs to be reset and
 * reactivated, so it will be a clean snapshot of its origin.
 **/
static int rollback_reset(snapshot_volume_t * snap_volume)
{
	int rc;

	LOG_ENTRY();

	/* Deactivate the snapshot. */
	rc = my_plugin_record->functions.plugin->deactivate(snap_volume->parent);
	if (rc) {
		goto out;
	}

	/* Erase the DM snapshot header. */
	rc = erase_snapshot_header(snap_volume, FALSE);
	if (rc) {
		goto out;
	}

	/* Clear the rollback-related info from the metadata. */
	snap_volume->metadata->flags &= ~SNAPSHOT_ROLLBACK;
	snap_volume->metadata->current_table_index = 0;
	snap_volume->metadata->current_table_entry = 0;
	rc = write_snapshot_metadata(snap_volume, FALSE);
	if (rc) {
		goto out;
	}

	/* Activate the snapshot. */
	rc = my_plugin_record->functions.plugin->activate(snap_volume->parent);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * rollback
 *
 * Copy the data from the snapshot back to the origin.
 **/
int rollback(snapshot_volume_t * snap_volume)
{
	dm_disk_exception_t * table = NULL;
	progress_t * rollback_progress = NULL;
	u_int32_t table_index, entries_per_table, i;
	int rc = 0, done = FALSE;

	LOG_ENTRY();

	/* Verify that the rollback can proceed. */
	if (!rollback_is_pending(snap_volume)) {
		goto out;
	}

	rc = can_rollback(snap_volume, TRUE);
	if (rc) {
		goto out;
	}

	/* Calculate the number of entries mapped by each exception table. */
	entries_per_table = EXCEPTION_TABLE_ENTRIES(snap_volume->metadata->chunk_size);

	/* Allocate a buffer to hold the exception tables. */
	table = EngFncs->engine_alloc(snap_volume->metadata->chunk_size *
				      EVMS_VSECTOR_SIZE);
	if (!table) {
		LOG_CRITICAL("Error allocating memory for exception table for "
			     "%s\n", snap_volume->parent->name);
		rc = ENOMEM;
		goto out;
	}

	/* Start the UI progress indicator. */
	rollback_progress = start_rollback_progress(snap_volume);
	if (!rollback_progress) {
		LOG_CRITICAL("Error allocating memory for progress indicator "
			     "for %s\n", snap_volume->parent->name);
		rc = ENOMEM;
		goto out;
	}

	/* Read each exception table from disk. */
	for (table_index = snap_volume->metadata->current_table_index;
	     !done; table_index++) {
		rc = read_exception_table(snap_volume, table, table_index);
		if (rc) {
			goto out;
		}

		/* For each entry in this exception table. */
		for (i = snap_volume->metadata->current_table_entry;
		     i < entries_per_table; i++) {

			table[i].old_chunk = DISK_TO_CPU64(table[i].old_chunk);
			table[i].new_chunk = DISK_TO_CPU64(table[i].new_chunk);

			update_rollback_progress(snap_volume,
						 rollback_progress, &table[i]);

			/* Update the snapshot metadata to reflect the table
			 * and table-entry that is currently being rolled-back.
			 */
			snap_volume->metadata->current_table_index = table_index;
			snap_volume->metadata->current_table_entry = i;
			rc = write_snapshot_metadata(snap_volume, FALSE);
			if (rc) {
				goto out;
			}

			if (table[i].new_chunk == 0) {
				done = TRUE;
				break;
			}

			/* Copy the data chunk. */
			rc = rollback_chunk(snap_volume, &table[i]);
			if (rc) {
				goto out;
			}
		}
	}

	rc = rollback_reset(snap_volume);
	if (!rc) {
		rollback_complete(snap_volume);
	}

out:
	end_rollback_progress(snap_volume, rollback_progress);
	EngFncs->engine_free(table);
	LOG_EXIT_INT(rc);
	return rc;
}


