/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: unassign.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "unassign.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   plugin_handle_t get_parent_plugin_for_object (object_handle_t)
 *
 *   Description:
 *      This routine attempts to return the plugin handle corresponding
 *      to plugin managing the first immediate parent object for the
 *      object given. 
 * 
 *   Entry:
 *      object - handle of object for the query
 *
 *   Exit:
 *      The parent plugin handle is returned or 0 on any error or if
 *      the object has no parent objects.
 *
 */
plugin_handle_t get_parent_plugin_for_object(object_handle_t object)
{
	gint rc;
	plugin_handle_t plugin = 0;
	handle_object_info_t *info;

	rc = evms_get_info(object, &info);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	} else {
		handle_array_t *users = info->info.object.parent_objects;

		if (users->count > 0) {
			handle_object_info_t *parent_info;

			rc = evms_get_info(users->handle[0], &parent_info);

			if (rc == 0) {
				plugin = parent_info->info.object.plugin;

				evms_free(parent_info);
			} else {
				log_error
				    ("%s: evms_get_info() returned error %d for first parent object.\n",
				     __FUNCTION__, rc);
			}
		}

		evms_free(info);
	}

	return plugin;
}

/*
 *
 *   plugin_type_t get_parent_plugin_type (object_handle_t)
 *
 *   Description:
 *      This routine attempts to return the plugin type for the 
 *      plugin managing the first immediate parent object for the
 *      object given. 
 * 
 *   Entry:
 *      object - handle of object for the query
 *
 *   Exit:
 *     The parent plugin type is returned or a type of EVMS_NO_PLUGIN
 *     on error.
 *
 */
plugin_type_t get_parent_plugin_type(object_handle_t object)
{
	gint rc;
	plugin_type_t type;
	plugin_handle_t plugin_handle;
	handle_object_info_t *info;

	plugin_handle = get_parent_plugin_for_object(object);

	rc = evms_get_info(plugin_handle, &info);

	if (rc != SUCCESS) {
		type = EVMS_NO_PLUGIN;
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	} else {
		type = GetPluginType(info->info.plugin.id);
		evms_free(info);
	}

	return type;
}

/*
 *
 *   inline gchar *get_parent_plugin_type_string (object_handle_t, gboolean)
 *
 *   Description:
 *      This routine attempts to return the plugin type string for
 *      the plugin managing the first immediate parent object for the
 *      object given. 
 * 
 *   Entry:
 *      object    - handle of object for the query
 *      lowercase - TRUE if first letter of each word should be lowercase
 *
 *   Exit:
 *      A string corresponding to the parent plugin type is returned.
 *
 */
inline gchar *get_parent_plugin_type_string(object_handle_t object, gboolean lowercase)
{
	return make_plugin_type_readable_string(get_parent_plugin_type(object), lowercase);
}

/*
 *
 *   void add_object_and_parent_plugin_to_selection_list (GtkCList *, object_handle_t, gboolean)
 *   
 *   Description:
 *      This routine appends a row to a GtkCList with information
 *      on the supplied object and the name of the parent object's
 *      plugin.
 * 
 *   Entry:
 *      clist       - address of the selections GtkCList widget
 *      object      - the handle to the object
 *      is_selected - whether to mark this item as selected
 *
 *   Exit:
 *      A new row corresponding to the object and parent object plugin name
 *      is added to the clist.
 *
 */
void add_object_and_parent_plugin_to_selection_list(GtkCList * clist, object_handle_t object,
						    gboolean is_selected)
{
	gint row;

	row = add_thing_to_selection_list(clist, object, is_selected);

	if (row != -1) {
		gint rc;
		handle_object_info_t *info;

		rc = evms_get_info(object, &info);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
		} else {
			gchar *plugin_name;

			plugin_name = get_plugin_name(get_parent_plugin_for_object(object));
			gtk_clist_set_text(clist, row, SL_MINMAX_SIZE_COLUMN, plugin_name);
			g_free(plugin_name);

			evms_free(info);
		}
	}
}

/*
 *
 *   void on_unassign_parent_plugin_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the unassign API call using the handle
 *      of the thing associated with the last row selected. It then
 *      creates and displays the results popup which destroys both
 *      windows when dismissed.
 * 
 *   Entry:
 *      button      - address of the GtkButton widget
 *      user_data   - not used
 *
 *   Exit:
 *      evms_unassign() is invoked on handle and results window is displayed afterwards
 *
 */
void on_unassign_parent_plugin_button_clicked(GtkButton * button, gpointer user_data)
{
	GtkCList *clist;
	object_handle_t handle;

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));

	handle = GPOINTER_TO_UINT(get_single_select_current_row_data(clist));

	if (handle != 0) {
		gint rc;

		rc = evms_unassign(handle);

		display_selection_window_results(GTK_WIDGET(button), rc,
						 _
						 ("An error was encountered removing the plug-in from the storage object."),
						 _
						 ("The plug-in removed itself successfully from the storage object."));
	}
}

/*
 *
 *   inline void set_parent_plugin_clist_properties (GtkCList *)
 *
 *   Description:
 *      This routine initializes some of the properties of the 
 *      clist used to display the objects and the parent object
 *      plugins that can be removed.
 * 
 *   Entry:
 *      clist - the address of the GtkCList to modify
 *
 *   Exit:
 *      See description.
 *
 */
inline void set_parent_plugin_clist_properties(GtkCList * clist)
{
	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);
	gtk_clist_set_column_visibility(clist, SL_MINMAX_SIZE_COLUMN, TRUE);
	gtk_clist_set_column_auto_resize(clist, SL_MINMAX_SIZE_COLUMN, TRUE);
}

/*
 *
 *   void on_unassign_parent_plugin_clist_realize (GtkWidget *, plugin_type_t)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of storage objects that can have its parent objects removed
 *      by their corresponding plugin.
 * 
 *   Entry:
 *      widget - address of the selections GtkCList widget
 *      type   - specifies what type of plugins we want removed
 *
 *   Exit:
 *      Selection list populated with storage objects that can have
 *      it parent objects removed by the owning plugin
 *
 */
void on_unassign_parent_plugin_clist_realize(GtkWidget * widget, plugin_type_t type)
{
	gint rc;
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *things;

	rc = evms_get_object_list(0, DATA_TYPE, 0, 0, 0, &things);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_object_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		guint i;
		gboolean is_selected = (things->count == 1);

		set_parent_plugin_clist_properties(clist);

		set_selection_window_clist_column_titles(clist, NULL,
							 _("Storage Object"),
							 make_plugin_type_readable_string(type,
											  FALSE));

		for (i = 0; i < things->count; i++) {
			if (evms_can_unassign(things->handle[i]) == 0
			    && get_parent_plugin_type(things->handle[i]) == type)
				add_object_and_parent_plugin_to_selection_list(clist,
									       things->handle[i],
									       is_selected);
		}

		adjust_clist_column_width(clist, SL_MINMAX_SIZE_COLUMN, 30);

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(things);
	}
}

/*
 *
 *   void add_single_object_and_parent_plugin_to_list (GtkCList *, gpointer)
 *   
 *   Description:
 *      This routine populates the standard selection window GtkCList
 *      with the information from one object handle. It also selects
 *      this single entry in the list so that the "Next" button is
 *      made active to complete an operation on the row.
 * 
 *   Entry:
 *      widget    - address of the selection window GtkCList widget
 *      user_data - contains handle of object to place in selection window
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void add_single_object_and_parent_plugin_to_list(GtkCList * clist, gpointer user_data)
{
	object_type_t type;
	object_handle_t handle = GPOINTER_TO_UINT(user_data);

	set_parent_plugin_clist_properties(clist);

	if (evms_get_handle_object_type(handle, &type) == SUCCESS)
		set_selection_window_clist_column_titles(clist, NULL,
							 make_object_type_readable_string(type),
							 get_parent_plugin_type_string(handle,
										       FALSE));

	add_object_and_parent_plugin_to_selection_list(clist, handle, TRUE);

	adjust_clist_column_width(clist, SL_MINMAX_SIZE_COLUMN, 30);
}

/*
 *
 *   void on_unassign_parent_plugin_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to have 
 *      its parent objects removed by the owning plugin to
 *      allow the user to confirm the operation on the thing.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_unassign_parent_plugin_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Remove Plug-in from Storage Object"),
						  _("Remove"),
						  NULL,
						  add_single_object_and_parent_plugin_to_list,
						  on_unassign_parent_plugin_button_clicked,
						  NULL, NULL, NULL, NULL, user_data);

	gtk_widget_show(window);
}
