
/* glob.c file filter for grep.*/
/* 
   Copyright 2000-2006 Edscott Wilson Garcia

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

/*****************************************************************/

#include "globber.h"

/** tripas */
#define GREP "grep"

static void *object = NULL;
static int initial;
static int terminated = 0;
static char *token;
static int options = 0, type = 0;
static long size = 0;
static long month_t = 0;
static long unsigned day_t = 0;
static long unsigned hour_t = 0;
static long unsigned min_t = 0;

/* two low order bytes are defined in globber.h */
#define GLOBRUN_PID     	0x10000
#define GLOBRUN_COUNT   	0x20000
#define GLOBRUN_FILTERED   	0x40000
#define GLOBRUN_IGNORE_CASE	0x80000
#define GLOBRUN_REG_EXP  	0x100000
#define GLOBRUN_INVERT       	0x200000
#define GLOBRUN_WORDS_ONLY   	0x400000
#define GLOBRUN_LINES_ONLY   	0x800000
#define GLOBRUN_ZERO_BYTE    	0x1000000
#define GLOBRUN_NOBINARIES   	0x2000000

#define MAX_ARG 25
#define COPYRIGHT "Copyright (c) 2002-2006 Edscott Wilson Garcia.\nAll rights reserved. GPL distribution licence."

static int
display (char *input)
{
  if (terminated)
    return terminated;		/* die quietly and quickly */
  printf ("%s\n", input);
  if (time (NULL) - initial > 3)
  {
    fflush (NULL);
    initial = time (NULL);
  }
  return terminated;
}

static int
grep (char *file)
{
  static char *arguments[MAX_ARG];
  int status = 0;
  if (terminated)
    return terminated;		/* die quietly and quickly */

  arguments[status++] = "grep";
#ifdef HAVE_GNU_GREP
  arguments[status++] = "-d";
  arguments[status++] = "skip";
  arguments[status++] = "-H";
  if (options & GLOBRUN_NOBINARIES) arguments[status++] = "-I";
  if (options & GLOBRUN_ZERO_BYTE)   arguments[status++] = "-Z";
  if (!(options & GLOBRUN_COUNT) && (options & GLOBRUN_INVERT))
  {
    arguments[status++] = "-L";
  }
  if (!(options & GLOBRUN_COUNT) && !(options & GLOBRUN_INVERT))
  {
    arguments[status++] = "-l";
  }
#else
  arguments[status++] = "-l";
#endif
  if (options & GLOBRUN_IGNORE_CASE) arguments[status++] = "-i";
  if (options & GLOBRUN_WORDS_ONLY)  arguments[status++] = "-w";
  if (options & GLOBRUN_LINES_ONLY)  arguments[status++] = "-x";

  if ((options & GLOBRUN_COUNT) && (options & GLOBRUN_INVERT))
  {
    arguments[status++] = "-c";
    arguments[status++] = "-v";
  }
  if ((options & GLOBRUN_COUNT) && !(options & GLOBRUN_INVERT))
  {
    arguments[status++] = "-c";
  }

#ifdef HAVE_GNU_GREP
  if (options & GLOBRUN_REG_EXP) arguments[status++] = "-E";
  else  arguments[status++] = "-e";
#endif
  arguments[status++] = token;

  arguments[status++] = file;
  arguments[status++] = (char *) 0;
  if (options & GLOBBER_VERBOSE)
  {
    int i;
    for (i = 0; arguments[i] != NULL; i++)
      fprintf (stderr,"%s ", arguments[i]);
    fprintf (stderr,"\n");
  }

  if (fork () == 0)
  {
    execvp (GREP, arguments);
    fprintf (stderr, "%s: %s\n", strerror(ENOENT),GREP);
    exit (1);
  }
  wait (&status);

  /*fflush(NULL); */
  return terminated;
}


static char *message[] = {
  " ",
  N_("[-vVPrMACaiIyLcwxZ] [-fpotkhsmudgeE (option)] path"), "\n", "\n",
  N_("Find-grep command tool."),"\n",
  N_("Find time options should be used with either  -M, -C, or -A."),"\n",
  N_("Grep options will enable piping of find results to GNU grep."), "\n",
  "\n",
  N_("*** Find options:"), 
  "\n","\n","-v","\t\t",
  N_("= verbose"),
  "\n","\n","-V","\t\t",
  N_("= print version number information"),
  "\n","\n","-a","\t\t",
  N_("= stay on a single filesystem."),
  "\n","\n","-P","\t\t",
  N_("= print process id (capital P)"),
  "\n","\n","-D","\t\t",
  N_("= include filtered dotfiles"),
  "\n","\n","-f ",
  N_("filter \t= file filter (enclosed in quotes if regexp *,? or [] is used)"),
  "\n","\n","-r","\t\t",
  N_("= recursive"),
  "\n","\n","-s ",
  N_("+kbytes\t= size greater than kbytes KBYTES"),
  "\n","\n","-s ",
  N_("-kbytes\t= size less than kbytes KBYTES"),
  "\n","\n","-p ",
  N_("perm\t\t= perm is either suid | exe"),
  "\n","\n","-o ",
  N_("octal_mode\t= octal mode is the file mode in octal notation"),
  "\n","\n","-t ",
  N_("type\t\t="), 
  " any | reg | dir | sym | sock | blk | chr | fifo", 
  N_(" (any [default], regular, directory, symlink, socket, blk_dev, chr_dev, fifo)"), 
  "\n","\n","-k ",
  N_("min\t\t= file time in the previous (int) min minutes (use with either -M, -C or -A)"),
  "\n","\n","-h ",
  N_("hh\t\t= file time in the previous (int) hh hours (use with either -M, -C or -A)"),
  "\n","\n","-d ",
  N_("dd\t\t= file time in the previous (int) dd days (use with either -M, -C or -A)"),
  "\n","\n","-m ",
  N_("mm\t\t= file time in the previous (int) mm months (use with either -M, -C or -A)"),
  "\n","\n","-M","\t\t",
  N_("= use mtime for file (modification time: mknod, truncate, utime,write"),
  "\n","\n","-A","\t\t",
  N_("= use atime for file (access time: exec, mknod, pipe, utime, read)"), 
  "\n","\n","-C","\t\t",
  N_("= use ctime for file (change time: setting inode information i.e., owner, group, link count, mode, etc.)"), 
  "\n","\n","-u "
  N_("user-id\t= only files matching numeric user-id"), 
  "\n","\n","-g ",
  N_("group-id\t= only files matching numeric group-id"), 
  "\n","\n","-Z","\t\t",
  N_("= Output  a  zero  byte  (the  ASCII  NULL  character) instead of the character that  normally  follows  a file  name"),
  "\n","\n",
  N_("*** Grep options:"), 
  "\n","\n","-e ",
  N_("string\t= containing string (if *,? or [], use quotes)"), 
  "\n","\n","-E ",
  N_("regexp\t= containing regexp: (use quotes). "), 
  "\n","\n","-i","\t\t",
  N_("= ignore case (for search string -c)"),
  "\n","\n","-I","\t\t",
  N_("= do not search into binary files"), 
  "\n","\n","-y","\t\t",
  N_("= same as -i (obsolete)"),   
  "\n","\n","-L","\t\t",
  N_("= print the  name  of each input file from which *no* output would normally have been printed."),
  "\n","\n","-c","\t\t",
  N_("= only print a count of matching lines for each input file."),
  "\n","\n","-w","\t\t",
  N_("= Select  only  those  lines  containing matches that form whole words. Word-constituent  characters  are letters, digits, and the underscore."),   
  "\n","\n","-x","\t\t",
  N_("= Select only those matches that  exactly  match  the whole line."), 
  "\n","\n",
  "fgr-",VERSION,"\n",
  COPYRIGHT,"\n",
  NULL
};

void
finish (int sig)
{
  /*printf("\n****\nglob terminated by signal\n****\n"); */
  terminated = 1;
  fflush (NULL);
}

void
halt (int sig)
{
  fflush (NULL);
  globber_destroy (object);
  exit (1);
}


#define CHECK_ARG if (argc <= i) goto error;
int
main (int argc, char **argv)
{
  int i, timetype = 0;
  char *filter = NULL, globbered = 0;
  int (*operate) (char *) = display;
  initial = time (NULL);

#ifdef ENABLE_NLS
    setlocale (LC_MESSAGES, "");
    setlocale (LC_ALL, "");
    bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
    textdomain(PACKAGE);
#endif
  /* initializations  */
  signal (SIGHUP, halt);
  signal (SIGSEGV, finish);
  signal (SIGKILL, finish);
  signal (SIGTERM, finish);


  if (argc < 2)
  {
  error:
    fprintf (stdout, "%s:\n", _("usage"));
    fprintf (stdout, "%s ", argv[0]);
    i = 0;
    while (message[i])
      fprintf (stdout, "%s", _(message[i++]));
    exit (1);
  }
  object = globber_create ();
  for (i = 1; i < argc; i++)
  {
    if (argv[i][0] == '-')
    {
      /* options for the globber : **************** */
      if (strstr (argv[i], "M") != NULL)
      {
	timetype = 1;
	glob_set_options (object, GLOBBER_MTIME);
	continue;
      }
      if (strstr (argv[i], "A") != NULL)
      {
	timetype = 1;
	glob_set_options (object, GLOBBER_ATIME);
	continue;
      }
      if (strstr (argv[i], "C") != NULL)
      {
	timetype = 1;
	glob_set_options (object, GLOBBER_CTIME);
	continue;
      }
      if (strstr (argv[i], "a") != NULL)
      {
	glob_set_options (object, GLOBBER_XDEV);
	options |= GLOBBER_XDEV;
	continue;
      }
      if (strstr (argv[i], "v") != NULL)
      {
	glob_set_options (object, GLOBBER_VERBOSE);
	options |= GLOBBER_VERBOSE;
	continue;
      }
      if (strstr (argv[i], "r") != NULL)
      {
	glob_set_options (object, GLOBBER_RECURSIVE);
	options |= GLOBBER_RECURSIVE;
	continue;
      }
      if (strstr (argv[i], "R") != NULL)
      {
	glob_set_options (object, GLOBBER_RECURSIVE);
	glob_set_options (object, GLOBBER_RECURSIVE_NO_HIDDEN);
	options |= GLOBBER_RECURSIVE;
	options |= GLOBBER_RECURSIVE_NO_HIDDEN;
	continue;
      }
      if (strstr (argv[i], "u") != NULL)
      {
	i++;
	CHECK_ARG;
	glob_set_user (object, atol (argv[i]));
	continue;
      }
      if (strstr (argv[i], "g") != NULL)
      {
	i++;
	CHECK_ARG;
	glob_set_group (object, atol (argv[i]));
	continue;
      }


      if (strstr (argv[i], "t") != NULL)
      {
	i++;
	type &= 07777;
	CHECK_ARG;
	/*if (strcmp (argv[i], "any") == 0) type &= 07777; */
	if (strcmp (argv[i], "reg") == 0)
	  type |= S_IFREG;
	if (strcmp (argv[i], "dir") == 0)
	  type |= S_IFDIR;
	if (strcmp (argv[i], "sym") == 0)
	  type |= S_IFLNK;
	if (strcmp (argv[i], "sock") == 0)
	  type |= S_IFSOCK;
	if (strcmp (argv[i], "blk") == 0)
	  type |= S_IFBLK;
	if (strcmp (argv[i], "chr") == 0)
	  type |= S_IFCHR;
	if (strcmp (argv[i], "fifo") == 0)
	  type |= S_IFIFO;
	if (strcmp (argv[i], "any") != 0)
	{
	  glob_set_options (object, GLOBBER_TYPE);
	  glob_set_type (object, type);
	}
	continue;
      }
      if (strstr (argv[i], "p") != NULL)
      {
	i++;
	/*type &= S_IFMT; */
	CHECK_ARG;
	if (strcmp (argv[i], "suid") == 0)
	  type |= S_ISUID;
	if (strcmp (argv[i], "exe") == 0)
	  type |= S_IXUSR;
	glob_set_options (object, GLOBBER_PERM);
	glob_set_type (object, type);
	continue;
      }
      if (strstr (argv[i], "o") != NULL)
      {
	int valor;
	i++;
	type &= S_IFMT;
	CHECK_ARG;
	sscanf (argv[i], "%o", &valor);
	type |= (07777 & valor);
	glob_set_options (object, GLOBBER_PERM);
	glob_set_type (object, type);
	continue;
      }

      if (strstr (argv[i], "s") != NULL)
      {
	i++;
	CHECK_ARG;
	size = atol (argv[i]);
	if (size < 0)
	  glob_set_sizeL (object, -size * 1024);
	else
	  glob_set_sizeG (object, size * 1024);
	continue;
      }

      if (strstr (argv[i], "k") != NULL)
      {
	i++;
	CHECK_ARG;
	min_t = atol (argv[i]);
	glob_set_time (object, month_t, day_t, hour_t, min_t);
	continue;
      }
      if (strstr (argv[i], "h") != NULL)
      {
	i++;
	CHECK_ARG;
	hour_t = atol (argv[i]);
	glob_set_time (object, month_t, day_t, hour_t, min_t);
	continue;
      }
      if (strstr (argv[i], "D") != NULL){
	glob_set_options (object, GLOBBER_ADD_DOT_FILTER);
	continue;	
      }
      if (strstr (argv[i], "d") != NULL)
      {
	i++;
	CHECK_ARG;
	day_t = atol (argv[i]);
	glob_set_time (object, month_t, day_t, hour_t, min_t);
	continue;
      }
      if (strstr (argv[i], "m") != NULL)
      {
	CHECK_ARG;
	month_t = atol (argv[i]);
	glob_set_time (object, month_t, day_t, hour_t, min_t);
	continue;
      }


      if (strstr (argv[i], "f") != NULL)
      {
	options |= GLOBRUN_FILTERED;
	i++;
	CHECK_ARG;
	filter = argv[i];
	/*if (options & GLOBBER_VERBOSE) 
	 *  fprintf (stderr, "filtering %s\n", filter);*/
	continue;
      }
      /* options for grep : ****************** */
      if (strstr (argv[i], "I") != NULL)
      {
	options |= GLOBRUN_NOBINARIES;
	continue;
      }
      if ((strstr (argv[i], "i") != NULL) || (strstr (argv[i], "y") != NULL))
      {
	options |= GLOBRUN_IGNORE_CASE;
	continue;
      }
      if (strstr (argv[i], "L") != NULL)
      {
	options |= GLOBRUN_INVERT;
	continue;
      }
      if (strstr (argv[i], "c") != NULL)
      {
	options |= GLOBRUN_COUNT;
	continue;
      }
      if (strstr (argv[i], "w") != NULL)
      {
	options |= GLOBRUN_WORDS_ONLY;
	continue;
      }
      if (strstr (argv[i], "x") != NULL)
      {
	options |= GLOBRUN_LINES_ONLY;
	continue;
      }
      if (strstr (argv[i], "Z") != NULL)
      {
	options |= GLOBRUN_ZERO_BYTE;
	continue;
      }
      if (strstr (argv[i], "P") != NULL)
      {
	options |= GLOBRUN_PID;
	printf ("PID=%d\n", (int) getpid ());
	fflush (NULL);
	continue;
      }
      if (strstr (argv[i], "E") != NULL)
      {
	i++;
	CHECK_ARG;
	token = argv[i];
	operate = grep;
	options |= GLOBRUN_REG_EXP;
	continue;
      }
      if (strstr (argv[i], "e") != NULL)
      {
	i++;
	CHECK_ARG;
	token = argv[i];
	operate = grep;
	options |= GLOBRUN_REG_EXP;
	options ^= GLOBRUN_REG_EXP;	/* turn off extended regexp */
	continue;
      }

      if (strstr (argv[i], "V") != NULL)
      {
	printf ("This is %s %s (find<-->grep)\nForward any bug reports to %s\n%s\n", PACKAGE_NAME, VERSION, PACKAGE_BUGREPORT,COPYRIGHT);
	return 0;
      }
      fprintf (stdout, "%s: %s\n", strerror(EINVAL),argv[i]);
      exit (1);
    }
    if (((min_t) || (hour_t) || (day_t) || (month_t)) && !timetype)
      glob_set_options (object, GLOBBER_MTIME);
    terminated = globber (object, argv[i], operate, filter);
    globbered = 1;
  }				/* end of argument processing */


  if (!globbered)
  {
    fprintf (stderr, "%s: %s\n",strerror(ENOENT),strerror(EINVAL));
    goto error;
  }
/*	if (terminated) printf("glob run was terminated.\n");*/
  if (!terminated)
  {				/* die quietly and quickly */
    if (options & GLOBRUN_PID)
      printf ("GLOB DONE=%d\n", (int) getpid ());
  }
  fflush (NULL);
  globber_destroy (object);
  exit (0);
}
