/*
 *
 * Copyright (c) 2003 The Regents of the University of California.  All 
 * rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Neither the name of the University nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include "fusd.h"


#define FUSD_MAX 100

struct fusd_file_info *outstanding_requests[FUSD_MAX];
int num_outstanding = 0;
int num_open = 0;
int last = 1;

int do_open(struct fusd_file_info *file) {
  file->private_data = (void *) last++;
  printf("oracle: opening: number of files now open: %d\n", ++num_open);
  return 0;
}

int do_close(struct fusd_file_info *file) {
  int i, dead_file;

  /* if the file that is closing is one that has a read outstanding,
     take it out of the list of outstanding requests */
  for (dead_file = 0; dead_file < num_outstanding; dead_file++)
    if (outstanding_requests[dead_file]->private_data == file->private_data)
      break;

  if (dead_file < num_outstanding) {
    printf("removing closed file from the outstanding request queue\n");
    for (i = dead_file; i < (num_outstanding-1); i++)
      outstanding_requests[i] = outstanding_requests[i+1];
    num_outstanding--;
  }

  printf("oracle: closing: file now open: %d, outstanding requests: %d\n",
	 --num_open, num_outstanding);
  return 0;
}


ssize_t do_read(struct fusd_file_info *file, char *buffer, size_t length,
	     loff_t *offset)
{
  if (num_outstanding >= FUSD_MAX) {
    return -EBUSY;
  } else {
    outstanding_requests[num_outstanding++] = file;
    return -FUSD_NOREPLY;
  }
}

ssize_t do_write(struct fusd_file_info *file, const char *buffer,
                 size_t len, loff_t *offset)
{
  if (!num_outstanding)
    return -EAGAIN;
  else {
    int i;

    struct fusd_file_info *supplicant = outstanding_requests[0];
    for (i = 1; i < num_outstanding; i++)
      outstanding_requests[i-1] = outstanding_requests[i];
    num_outstanding--;

    if (supplicant->fusd_msg->datalen < len)
      len = supplicant->fusd_msg->datalen;
    strncpy(supplicant->fusd_msg->data, buffer, len);
    fusd_return(supplicant, len);
  }

  return len;
}



int main(int argc, char *argv[])
{
  struct fusd_file_operations f = { open: do_open, close: do_close,
				    read: do_read, write: do_write };

  if (fusd_register("oracle", "misc", "oracle", 0666, NULL, &f) < 0)
    perror("/dev/oracle");
  else {
    printf("oracle ready to answer queries\n");
    fusd_run();
  }
  return 0;
}
