/*
 *
 * Copyright (c) 2003 The Regents of the University of California.  All 
 * rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Neither the name of the University nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 

/* 
 * thruput.c
 * 
 * tests "throughput" of a specified device by reading as quickly as possible
 *
 */

#include <stdio.h>
#include <sys/time.h>
#ifdef __QNX__
# include <fcntl.h>
#else
# include <sys/fcntl.h>
#endif
#include <stdlib.h>
#include <unistd.h>

int main(int argc, char **argv)
{
  int count = 2 << 20;
  int savecount;
  int blocksize = 4096;
  char *dev;
  int fd;
  struct timeval start;
  struct timeval end;
  char *buffer;
  double timelag;
  int opt;
  int graph = 0;

  if (argc < 2) {
    fprintf(stderr, "usage: %s devicefile [-g] [-b blocksize] [-l bytes]\n", argv[0]);
    fprintf(stderr, "       -g for 'graphable' output\n");
    exit(1);
  }

  dev = argv[1];
  argv++;
  argc--;

  /* parse args
   */
  do {
    opt = getopt (argc, argv, "b:l:g");
    switch (opt)
      {
      case 'g':
	graph = 1;
	break;
      case 'b':
	blocksize = atoi(optarg);
	break;
      case 'l':
	count = atoi(optarg);
	break;
      }
  }
  while (opt > 0);

  if (!graph)
    printf("%s: Reading %d bytes from %s, blocksize %d.\n", 
	   argv[0], count, dev, blocksize);

  /* open */
  fd = open(dev, O_RDONLY);
  if (fd < 0) {
    fprintf(stderr, "%s: Unable to open device %s\n", argv[0], dev);
    perror(argv[0]);
    exit(1);
  }

  /* allocate */
  buffer = malloc(blocksize);
  if (buffer == NULL) {
    fprintf(stderr, "%s: Unable to allocate %d bytes\n", argv[0], blocksize);
    perror(argv[0]);
    exit(1);
  }

  /* timestamp */
  gettimeofday(&start, NULL);

  /* read */
  savecount = count;
  while (count > 0) {
    int block = count;
    int status;
    if (block > blocksize) block = blocksize;
    
    status = read(fd, buffer, block);

    if (status < 0) {
      fprintf(stderr, "%s: Read failed\n", argv[0]);
      perror(argv[0]);
      exit(1);
    }

    count -= status;
    if (status == 0) {
      fprintf(stderr, "%s: Zero-length read\n", argv[0]);
    }
  }

  /* timestamp */
  gettimeofday(&end, NULL);

  /* calc */
  timelag = 
    ((end.tv_usec - start.tv_usec) / 1000000.0) +
    end.tv_sec - start.tv_sec;
  
  /* print */
  if (graph) 
    printf("%d \t%d \t%f\n", savecount, blocksize, timelag);
  else
    printf("%s: %d bytes transferred in %f seconds => %f bytes/sec.\n",
	   argv[0], savecount, timelag, savecount/timelag);
  
  return 0;
}





