/*
** 1998-12-19 -	Let's pretend we're one of those big, bad, desktop environments, and provide
**		users with an "Information" window in which we can show cool stuff... This
**		will basically repeat what stat() has told us, plus a few extras.
** 1999-03-06 -	Changed for the new selection/dirrow handling.
** 1999-03-28 -	Now uses the progress-indicating version of the fut_dir_size() routine.
** 1999-04-05 -	Added use of the new command configuration system, to allow user to have
**		some control over how this command operates. Nice.
** 2000-07-02 -	Internationalized. Lost some hard-coded plural suffix stuff (was English-only). :(
** 2002-02-08 -	Now uses the dialog module. Why it wasn't, nobody knows.
*/

#include "gentoo.h"

#include <time.h>

#include "errors.h"
#include "file.h"
#include "dialog.h"
#include "dirpane.h"
#include "iconutil.h"
#include "sizeutil.h"
#include "strutil.h"
#include "fileutil.h"
#include "userinfo.h"
#include "progress.h"
#include "cmdseq_config.h"

#include "cmd_info.h"

#define	CMD_ID	"info"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	gboolean modified;
	gboolean use_file;			/* Show 'file' output? */
	gboolean recurse_dirs;			/* Recurse directories? */
	gchar	 df_access[DP_DATEFMT_SIZE];	/* Formatter for dates of last access, */
	gchar	 df_modify[DP_DATEFMT_SIZE];	/*  modification, */
	gchar	 df_change[DP_DATEFMT_SIZE];	/*  and change. */
	gchar	 tick[2];
} OptInfo;

static OptInfo	info_options;
static CmdCfg	*info_cmc = NULL;

/* Special little struct used to hold information during background directory sizing. */
typedef struct {
	GtkWidget	*size, *contents;			/* Label widgets, updated. */
	gpointer	handle;						/* Handle from fut_dir_size_start(). */
} DirSizeInfo;

/* ----------------------------------------------------------------------------------------- */

static void dir_size_update(const FUCount *count, gpointer user)
{
	DirSizeInfo	*dsi = user;
	gchar	buf[64], sbuf[64], stbuf[64], btbuf[64], *st, *bt;

	if(!count)
		return;

	/* Replace contents of previously built size-label. */
	sze_put_size64(sbuf, sizeof sbuf, count->num_bytes, SZE_AUTO);
			
	stbuf[sizeof stbuf - 1] = '\0';
	st = stu_tickify(stbuf + sizeof stbuf - 1, count->num_bytes,  info_options.tick[0]);
	btbuf[sizeof btbuf - 1] = '\0';
	bt = stu_tickify(btbuf + sizeof btbuf - 1, count->num_blocks, info_options.tick[0]);
	g_snprintf(buf, sizeof buf, _("%s (%s bytes,\n%s blocks)"), sbuf, st, bt);
	gtk_label_set_text(GTK_LABEL(dsi->size), buf);

	/* And tell a little something about the contents. */
	g_snprintf(buf, sizeof buf, _("%u dirs, %u files, %u symlinks"),
			count->num_dirs, count->num_files, count->num_links);
	gtk_label_set_text(GTK_LABEL(dsi->contents), buf);
}

static GtkWidget * label_pack(GtkWidget *table, const gchar *text, gint x, gint y, gboolean left)
{
	GtkWidget	*algn, *lab;

	algn = gtk_alignment_new(left ? 0.0f : 0.9f, 0.0f, 0.0f, 0.0f);
	lab  = gtk_label_new(text);
	if(left)
		gtk_label_set_justify(GTK_LABEL(lab), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(algn), lab);
	gtk_table_attach(GTK_TABLE(table), algn, x, x + 1, y, y + 1, GTK_FILL,0,4,0);
	return lab;
}

static void build_date(const gchar *label, gint y, const time_t *date, const gchar *fmt, GtkWidget *tab)
{
	gchar		buf[64];
	struct tm	*tm;

	label_pack(tab, label, 0, y, FALSE);
	if((tm = localtime(date)) != NULL)
	{
		if(strftime(buf, sizeof buf, fmt, tm) >= 0)
			label_pack(tab, buf, 1, y, TRUE);
	}
}

static GtkWidget * build_info(MainInfo *min, const DirPane *dp, const DirRow *row, DirSizeInfo **dsi)
{
	gchar		buf[128], sbuf[32], *ptr;
	guint		y;
	const gchar	*iname;
	GtkWidget	*tab, *szlab, *sep, *entry;

	tab = gtk_table_new(17, 2, FALSE);

	y = 0;
	if(DP_ROW_TYPE(row)->style != NULL)
	{
		if((iname = stl_style_property_get_icon(DP_ROW_TYPE(row)->style, SPN_ICON_UNSEL)) != NULL)
		{
			GtkWidget	*ico;
			GdkPixmap	*ipix;
			GdkBitmap	*imsk;

			ipix = ico_icon_get(min, iname, &imsk);
			ico = gtk_pixmap_new(ipix, imsk);
			gtk_table_attach(GTK_TABLE(tab), ico, 0, 1, y, y + 1, 0,0,5,5);
		}
	}
	label_pack(tab, DP_ROW_NAME(row), 1, y++, TRUE);

	/* For symbolic links, display name of the target, at least. */
	if(S_ISLNK(DP_ROW_LSTAT(row).st_mode))
	{
		label_pack(tab, _("Link To"), 0, y, FALSE);
		entry = gtk_entry_new();
		gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), DP_ROW_LINKNAME(row));
		gtk_table_attach(GTK_TABLE(tab), entry, 1, 2, y, y + 1,  GTK_EXPAND|GTK_FILL,GTK_EXPAND|GTK_FILL,5,0);
		y++;
	}
	sep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
	y++;

	label_pack(tab, _("Type"), 0, y, FALSE);
	label_pack(tab, DP_ROW_TYPE(row)->name, 1, y++, TRUE);

	label_pack(tab, _("Location"), 0, y, FALSE);
	label_pack(tab, dp->dir.path, 1, y++, TRUE);

	label_pack(tab, _("Size"), 0, y, FALSE);
	sze_put_size(sbuf, sizeof sbuf, DP_ROW_LSTAT(row).st_size, SZE_AUTO);
	if(DP_ROW_LSTAT(row).st_size < 1024)
		stu_strncpy(buf, sbuf, sizeof buf);
	else
	{
		gchar	stbuf[64], *st, btbuf[64], *bt;

		stbuf[sizeof stbuf - 1] = '\0';
		st = stu_tickify(stbuf + sizeof stbuf - 1, DP_ROW_LSTAT(row).st_size, info_options.tick[0]);
		btbuf[sizeof btbuf - 1] = '\0';
		bt = stu_tickify(btbuf + sizeof btbuf - 1, DP_ROW_LSTAT(row).st_blocks, info_options.tick[0]);
		g_snprintf(buf, sizeof buf, _("%s (%s bytes,\n%s blocks)"), sbuf, st, bt);
	}
	szlab = label_pack(tab, buf, 1, y++, TRUE);
	/* Building info for a directory, and recursing enabled? */
	if(S_ISDIR(DP_ROW_LSTAT(row).st_mode) && info_options.recurse_dirs)
	{
		/* Start asynchronous directory sizing. Nice. */
		*dsi = g_malloc(sizeof *dsi);
		(*dsi)->size     = szlab;
		label_pack(tab, _("Contains"), 0, y, FALSE);
		(*dsi)->contents = label_pack(tab, "(content info)", 1, y++, TRUE);
		(*dsi)->handle   = fut_dir_size_start(dp_full_name(dp, DP_ROW_INDEX(dp, row)), dir_size_update, *dsi);
	}

	/* Is the "Show 'file' Output?" option enabled? Requires "file -n -f -" to work. */
#if defined HAVE_GOOD_FILE
	if(info_options.use_file)
	{
		gchar	buf[PATH_MAX];

		sep = gtk_hseparator_new();
		gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
		y++;
		label_pack(tab, _("'File' Info"), 0, y, FALSE);
		g_snprintf(buf, sizeof buf, "%s%s%s", dp->dir.path, G_DIR_SEPARATOR_S, DP_ROW_NAME(row));
		label_pack(tab, fle_file(min, buf), 1, y++, TRUE);
	}
#endif
	sep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
	y++;

	label_pack(tab, _("Owner"), 0, y, FALSE);
	if((ptr = (gchar *) usr_lookup_uname(DP_ROW_LSTAT(row).st_uid)) != NULL)
		g_snprintf(buf, sizeof buf, "%s (%d)", ptr, (gint) DP_ROW_LSTAT(row).st_uid);
	else
		g_snprintf(buf, sizeof buf, "%d", (gint) DP_ROW_LSTAT(row).st_uid);
	label_pack(tab, buf, 1, y++, TRUE);

	label_pack(tab, _("Group"), 0, y, FALSE);
	if((ptr = (gchar *) usr_lookup_gname(DP_ROW_LSTAT(row).st_gid)) != NULL)
		g_snprintf(buf, sizeof buf, "%s (%d)", ptr, (gint) DP_ROW_LSTAT(row).st_gid);
	else
		g_snprintf(buf, sizeof buf, "%d", (gint) DP_ROW_LSTAT(row).st_gid);
	label_pack(tab, buf, 1, y++, TRUE);

	sep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
	y++;

	build_date(_("Accessed"), y++, &DP_ROW_LSTAT(row).st_atime, info_options.df_access, tab);
	build_date(_("Modified"), y++, &DP_ROW_LSTAT(row).st_ctime, info_options.df_modify, tab);
	build_date(_("Changed"),  y++, &DP_ROW_LSTAT(row).st_mtime, info_options.df_change, tab);

	if(errno)
	{
		gtk_widget_destroy(tab);
		return NULL;
	}
	return tab;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-10-16 -	Callback when Information window is closing. Stop asynch. dir sizing, if in use. */
static void info_close(gint button, gpointer user)
{
	DirSizeInfo	*dsi = user;

	if(dsi)
		fut_dir_size_stop(dsi->handle);
	g_free(dsi);
}

/* 1999-03-06 -	Create window showing all about <row>, which sits in <dp>. */
static gint information(MainInfo *min, const DirPane *dp, const DirRow *row)
{
	DirSizeInfo	*dsi = NULL;
	GtkWidget	*fr;

	if((fr = build_info(min, dp, row, &dsi)) != NULL)
	{
		gtk_widget_show_all(fr);
		dlg_dialog_async_new(fr, DP_ROW_NAME(row), _("_OK"), info_close, dsi);
	}
	return fr != NULL;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-12-19 -	Show information about files in <src>. Knows about double-click, and handles
**		it like everybody else.
*/
gint cmd_information(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GSList	*slist, *iter;

	if((slist = dp_get_selection(src)) == NULL)
		return 1;

	pgs_progress_begin(min, _("Getting Information..."), PFLG_BUSY_MODE);
	for(iter = slist; iter != NULL; iter = g_slist_next(iter))
	{
		if(!information(min, src, DP_SEL_ROW(iter)))
			break;
		dp_unselect(src, DP_SEL_INDEX(src, iter));
	}
	pgs_progress_end(min);

	if(errno)
		err_set(min, errno, CMD_ID, iter ? DP_SEL_NAME(iter) : NULL);
	else
		err_clear(min);
	err_show(min);

	dp_free_selection(slist);

	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-04-05 -	Initialize configuration stuff. */
void cfg_information(MainInfo *min)
{
	if(info_cmc == NULL)
	{
		/* Initialize default option values. */
		info_options.modified	  = FALSE;
		info_options.use_file	  = FALSE;
		info_options.recurse_dirs = TRUE;
		stu_strncpy(info_options.df_access, "%Y-%m-%d %H:%M.%S", sizeof info_options.df_access);
		stu_strncpy(info_options.df_modify, "%Y-%m-%d %H:%M.%S", sizeof info_options.df_modify);
		stu_strncpy(info_options.df_change, "%Y-%m-%d %H:%M.%S", sizeof info_options.df_change);
		stu_strncpy(info_options.tick, ",", sizeof info_options.tick);

		info_cmc = cmc_config_new("Information", &info_options);
		cmc_field_add_boolean(info_cmc, "modified", NULL, offsetof(OptInfo, modified));
#if defined HAVE_GOOD_FILE
		cmc_field_add_boolean(info_cmc, "use_file", _("Show 'file' Output?"), offsetof(OptInfo, use_file));
#endif
		cmc_field_add_boolean(info_cmc, "recurse_dirs", _("Recurse Directories?"), offsetof(OptInfo, recurse_dirs));
		cmc_field_add_string(info_cmc, "df_access", _("Access Date Format"), offsetof(OptInfo, df_access), sizeof info_options.df_access);
		cmc_field_add_string(info_cmc, "df_modify", _("Modify Date Format"), offsetof(OptInfo, df_modify), sizeof info_options.df_modify);
		cmc_field_add_string(info_cmc, "df_change", _("Change Date Format"), offsetof(OptInfo, df_change), sizeof info_options.df_change);
		cmc_field_add_string(info_cmc, "tick", _("Size Tick Mark"), offsetof(OptInfo, tick), sizeof info_options.tick);

		cmc_config_register(info_cmc);
	}
}
