/*
** 1998-05-25 -	This module contains various file and directory related operations, that
**		are needed here and there in the application.
** 1999-11-13 -	Rewrote the core fut_copy() function. Shorter, simpler, better. Added
**		a complementary fut_copy_partial() function that copies just parts of files.
*/

#include "gentoo.h"

#include <ctype.h>
#include <fcntl.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "errors.h"
#include "progress.h"
#include "strutil.h"
#include "userinfo.h"

#include "fileutil.h"

/* ----------------------------------------------------------------------------------------- */

#define	USER_NAME_SIZE	(64)		/* Just an assumption, really. */

#define	MIN_COPY_CHUNK	(1 << 17)

/* Path components are relative to this if they're not absolute. */
static gchar	base_path[PATH_MAX];
/* The length of the base. Saves some calls to strlen() in the middle of everything. :) */
static gint	base_len;

/* ----------------------------------------------------------------------------------------- */

/* 1998-05-21 -	Enter a directory, saving the old path for restoration later.
**		If <old> is NULL, no storing is done. Returns boolean success.
*/
gboolean fut_cd(const gchar *new_dir, gchar *old, gsize old_max)
{
	if(old != NULL)
		getcwd(old, old_max);

	return chdir(new_dir) == 0 ? TRUE : FALSE;
}

#if 0
/* 2003-10-10 -	Compute a directory stat chain from <path>. It's made
**		up of a list of stat(2) structures up to the root.
*/
GList * fut_stat_chain_new(const gchar *path)
{
	gchar	canon[PATH_MAX];
	GList	*chain = NULL;
	GString	*dir;

	if(!fut_path_canonicalize(path, canon, sizeof canon))
		return NULL;

	dir = g_string_new(canon);

	while(dir->len && strcmp(dir->str, G_DIR_SEPARATOR_S) != 0)
	{
		struct stat	stbuf;

		if(stat(dir->str, &stbuf) == 0)
		{
			struct stat	*st = g_malloc(sizeof *st);
			const char *sep;

			*st = stbuf;
			chain = g_list_append(chain, st);
			if((sep = strrchr(dir->str, G_DIR_SEPARATOR)) != NULL)
				g_string_truncate(dir, sep - dir->str);
			else
				break;
		}
		else	/* A stat() call failed, then path is bad and no chain exists. Fail. */
		{
			fut_stat_chain_free(chain);
			return NULL;
		}
	}
	return chain;
}

/* 2003-10-10 -	Returns true if the second chain is a prefix of the first, which can also
**		be more clearly put as "chain1 represents a subdir of chain2". Um, at least
**		that's a different wording.
*/
gboolean fut_stat_chain_prefix_equal(const GList *chain1, const GList *chain2)
{
	if(!chain1 || !chain2)
		return FALSE;

	for(; chain1 && chain2; chain1 = g_list_next(chain1), chain2 = g_list_next(chain2))
	{
		const struct stat	*s1 = chain1->data, *s2 = chain2->data;

		if(s1->st_dev != s2->st_dev || s1->st_ino != s2->st_ino)
			return FALSE;
	}
	return TRUE;
}

void fut_stat_chain_free(GList *chain)
{
	GList	*iter;

	for(iter = chain; iter != NULL; iter = g_list_next(iter))
		g_free(iter->data);
	g_list_free(chain);
}
#endif

/* 2003-10-10 -	Get next path part, up to (but not including) the next separator. */
static GString * get_part(GString *in, char *out)
{
	const char	*sep = strchr(in->str, G_DIR_SEPARATOR);

	if(sep)
	{
		gsize	len = sep - in->str;
		strncpy(out, in->str, len);
		out[len] = '\0';
		g_string_erase(in, 0, len + 1);
		return in;
	}
	return NULL;
}

/* 2003-10-10 -	Make a path canonical, i.e. make it absolute and remove any "..", "."
**		and repeated slashes it might contain. Pure string operation.
*/
gboolean fut_path_canonicalize(const gchar *path, gchar *outpath, gsize maxout)
{
	GString	*dir, *out;
	char	part[PATH_MAX];

	if(!path || *path == '\0' || !outpath || maxout < 2)
		return FALSE;

	if(*path == G_DIR_SEPARATOR)
		dir = g_string_new(path);
	else
	{
		char	buf[PATH_MAX];

		if(getcwd(buf, sizeof buf) == NULL)
			return FALSE;
		if(buf[0] != G_DIR_SEPARATOR)
			dir = g_string_new(G_DIR_SEPARATOR_S);
		else
			dir = g_string_new(buf);
		g_string_append_c(dir, G_DIR_SEPARATOR);
		g_string_append(dir, path);
	}
	g_string_append_c(dir, G_DIR_SEPARATOR);		/* Must end in separator for get_part(). */

	out = g_string_new("");
	while(get_part(dir, part))
	{
		if(*part)
		{
			if(strcmp(part, ".") == 0)
				continue;
			else if(strcmp(part, "..") == 0)
			{
				const char	*sep = strrchr(out->str, G_DIR_SEPARATOR);
				g_string_truncate(out, sep - out->str);
				continue;
			}
			g_string_append_c(out, G_DIR_SEPARATOR);
			g_string_append(out, part);
		}
	}
	g_string_free(dir, TRUE);
	if(out->len < maxout)
	{
		strcpy(outpath, out->str);
		g_string_free(out, TRUE);
		return TRUE;
	}
	g_string_free(out, TRUE);
	return FALSE;
}

/* 2003-10-10 -	Check if the directory <to> is a child directory of <from>. If it is, copying <from>
**		into <to> is a genuine Bad Idea.
*/
gboolean fut_is_parent_of(const char *from, const char *to)
{
	gchar	buf1[PATH_MAX], buf2[PATH_MAX];

	if(fut_path_canonicalize(from, buf1, sizeof buf1) &&
	   fut_path_canonicalize(to,   buf2, sizeof buf2))
	{
		gsize	len = strlen(buf1);

		if(strncmp(buf1, buf2, len) == 0)
			return buf2[len] == G_DIR_SEPARATOR;	/* If last char is /, buf1 is really a prefix. */
	}
	return FALSE;
}

/* 1998-09-23 -	Return 1 if the named file exists, 0 otherwise. Since this is Unix
**		(I think), this matter is somewhat complex. But I choose to ignore
**		that for now, and pretend it's nice and simple. :)
*/
gboolean fut_exists(const gchar *name)
{
	gint		old_errno = errno;
	gboolean	ret;

	ret = (access(name, F_OK) == 0);
	errno = old_errno;

	return ret;
}

/* 1998-12-18 -	Sets <size> to the size of file <name>, in bytes. Returns 1 on success,
**		0 on failure (in case <size> is never touched).
*/
gboolean fut_size(const gchar *name, gsize *size)
{
	struct stat	s;
	gint		old_errno = errno, ret = 0;

	if(name != NULL && size != NULL && (ret = (stat(name, &s)) == 0))
		*size = (gsize) s.st_size;
	errno = old_errno;
	return ret ? TRUE : FALSE;
}

/* 1999-02-03 -	Works just like fut_size above, but takes an open file descriptor
**		instead of a filename. Sometimes handy.
*/
gboolean fut_fsize(gint fd, gsize *size)
{
	struct stat	s;
	gint		old_errno = errno, ret = 0;

	if(fd >= 0 && size != NULL && (ret = (fstat(fd, &s)) == 0))
		*size = (gsize) s.st_size;
	errno = old_errno;
	return ret ? TRUE : FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-12-20 -	This is under the hood of the fut_dir_size() function. */
static gint recurse_dir_size(MainInfo *min, const gchar *path, guint64 *bytes, FUCount *fc, gboolean with_progress)
{
	gchar		old_dir[PATH_MAX];
	DIR		*dir;
	struct dirent	*de;
	struct stat	stat;

	if(path == NULL)
		return 0;

	if(fut_cd(path, old_dir, sizeof old_dir))
	{
		if((dir = opendir(".")) != NULL)
		{
			err_clear(min);
			while((de = readdir(dir)) != NULL)
			{
				if(with_progress)
				{
					pgs_progress_item_begin(min, de->d_name, 0);
					if(pgs_progress_item_update(min, 0) == PGS_CANCEL)
					{
						errno = EINTR;
						break;
					}
				}
				if(!min->cfg.dir_filter(de->d_name))
					continue;
				if(lstat(de->d_name, &stat) == 0)
				{
					if(bytes != NULL)
						(*bytes) += stat.st_size;
					if(fc != NULL)
					{
						fc->num_bytes  += (guint64) stat.st_size;
						fc->num_blocks += (guint64) stat.st_blocks;
					}
					if(S_ISDIR(stat.st_mode))
					{
						if(fc != NULL)
							fc->num_dirs++;
						recurse_dir_size(min, de->d_name, bytes, fc, with_progress);
					}
					else if(!S_ISDIR(stat.st_mode) && fc != NULL)
					{
						if(S_ISREG(stat.st_mode))
							fc->num_files++;
						else if(S_ISLNK(stat.st_mode))
							fc->num_links++;
						else if(S_ISCHR(stat.st_mode))
							fc->num_cdevs++;
						else if(S_ISBLK(stat.st_mode))
							fc->num_bdevs++;
						else if(S_ISFIFO(stat.st_mode))
							fc->num_fifos++;
						else if(S_ISSOCK(stat.st_mode))
							fc->num_socks++;
					}
				}
				if(with_progress)
					pgs_progress_item_end(min);
			}
			closedir(dir);
		}
		fut_cd(old_dir, NULL, 0);
	}
	return errno == 0;
}

static gint do_dir_size(MainInfo *min, const gchar *path, guint64 *bytes, FUCount *fc, gboolean with_progress)
{
	gint	ret;

	if(fc != NULL)		/* If there is a fc present, wipe it clean. */
	{
		fc->num_dirs  = 0U;
		fc->num_files = 0U;
		fc->num_links = 0U;
		fc->num_cdevs = 0U;
		fc->num_bdevs = 0U;
		fc->num_fifos = 0U;
		fc->num_socks = 0U;

		fc->num_total = 0U;

		fc->num_bytes  = 0ULL;
		fc->num_blocks = 0ULL;
	}
	if((ret = recurse_dir_size(min, path, bytes, fc, with_progress)) != 0 && fc != NULL)
		fc->num_total = fc->num_dirs + fc->num_files + fc->num_links + fc->num_cdevs +
				fc->num_bdevs + fc->num_fifos + fc->num_socks;

	return ret;
}

/* 1998-12-19 -	Scan recursively at <path>, accumulating total number of bytes seen, as well as counting
**		how many different "objects" (dirs, files, links etc) were seen. Will never follow a link
**		to a dir or count it as anything but a soft link. <fc> can be NULL if you just don't care.
**		Returns 1 on success, 0 on failure (such as permission denied).
** 1999-03-22 -	Now zeros the <bytes> value before beginning. This fixes a bug in the GetSize command.
*/
gint fut_dir_size(MainInfo *min, const gchar *path, guint64 *bytes, FUCount *fc)
{
	return do_dir_size(min, path, bytes, fc, FALSE);
}

/* 1999-03-28 -	Size up a directory, with GUI progress indication. */
gint fut_dir_size_progress(MainInfo *min, const gchar *path, guint64 *bytes, FUCount *fc)
{
	return do_dir_size(min, path, bytes, fc, TRUE);
}

/* ----------------------------------------------------------------------------------------- */

/* Transversal state for a single directory when doing asynchronous dir sizing. */
typedef struct {
	DIR 	*dir;
	gchar	*path;
} DirSizeState;

/* An "operation" is the result of a single call to fut_dir_size_start(). */
typedef struct {
	SizeFunc	func;
	gpointer	user;
	FUCount	count;
	DirSizeState	*current;
	GSList	*stack;
} DirSizeOp;

/* Global state for asynchronous directory size computation. */
static struct {
	guint	idle;
	GList	*ops;
} dir_size_info = { 0U };

/* 2003-10-16 -	Open given path and return new dir size state struct describing it. */
static DirSizeState * dir_size_open(const gchar *path)
{
	DirSizeState	*st;

	st = g_malloc(sizeof *st);
	if((st->dir = opendir(path)) != NULL)
	{
		st->path = g_strdup(path);
		return st;
	}
	g_free(st);
	return NULL;
}

/* 2003-10-16 -	Free a directory state. */
static void dir_size_free(DirSizeState *state)
{
	if(state->dir)
		closedir(state->dir);
	g_free(state->path);
	g_free(state);
}

/* 2003-10-16 -	Update asynchronous directory sizing. */
static gboolean dir_size_update(DirSizeOp *op)
{
	int	i;
	struct dirent	*de;
	struct stat	stbuf;

	for(i = 0; op->current && i < 32; i++)
	{
		if((de = readdir(op->current->dir)) != NULL)
		{
			gchar	full[PATH_MAX];

			if(strcmp(de->d_name, ".") == 0 || strcmp(de->d_name, "..") == 0)
				continue;
			/* Don't CD for asynchronous reasons. Build full paths, it's fairly cheap. */
			g_snprintf(full, sizeof full, "%s%s%s", op->current->path, G_DIR_SEPARATOR_S, de->d_name);
			if(lstat(full, &stbuf) == 0)
			{
				op->count.num_bytes  += (guint64) stbuf.st_size;
				op->count.num_blocks += (guint64) stbuf.st_blocks;
				if(S_ISDIR(stbuf.st_mode))
				{
					DirSizeState	*nd;

					if((nd = dir_size_open(full)) != NULL)
					{
						op->stack   = g_slist_prepend(op->stack, op->current);
						op->current = nd;
						op->count.num_dirs++;
					}
				}
				else
				{
					if(S_ISREG(stbuf.st_mode))
						op->count.num_files++;
					else if(S_ISLNK(stbuf.st_mode))
						op->count.num_links++;
					else if(S_ISCHR(stbuf.st_mode))
						op->count.num_cdevs++;
					else if(S_ISBLK(stbuf.st_mode))
						op->count.num_bdevs++;
					else if(S_ISFIFO(stbuf.st_mode))
						op->count.num_fifos++;
					else if(S_ISSOCK(stbuf.st_mode))
						op->count.num_socks++;
				}
			}
		}

		if(de == NULL)
		{
			dir_size_free(op->current);
			op->current = NULL;
			if(op->stack != NULL)
			{
				GSList	*top = op->stack;

				op->stack = g_slist_remove_link(op->stack, top);
				op->current = top->data;
				g_slist_free_1(top);
			}
		}
	}
	/* After looping around for a while, tell op owner new numbers. */
	op->func(&op->count, op->user);
	return op->current != NULL;
}

/* 2003-10-16 -	GTK+ idle callback for asynchronous directory sizing. Update all on-going operations. */
static gint idle_dir_size(gpointer user)
{
	GList	*iter, *next;

	for(iter = dir_size_info.ops; iter != NULL; iter = next)
	{
		next = g_list_next(iter);
		if(!dir_size_update(iter->data))
			fut_dir_size_stop(iter->data);
	}
	return TRUE;
}

/* 2003-10-16 -	Start off a new operation, by allocating, initializing it, and adding it to list. */
gpointer fut_dir_size_start(const gchar *path, SizeFunc func, gpointer user)
{
	DirSizeOp	*op;

	op = g_malloc(sizeof *op);
	op->func = func;
	op->user = user;
	memset(&op->count, 0, sizeof op->count);
	op->current = dir_size_open(path);
	op->stack = NULL;

	dir_size_info.ops = g_list_append(dir_size_info.ops, op);

	/* If the idle function isn't active, start it up. */
	if(dir_size_info.idle == 0U)
		dir_size_info.idle = gtk_idle_add_priority(G_PRIORITY_LOW, idle_dir_size, NULL);
	return op;
}

/* 2003-10-16 -	Stop a directory sizing operation. */
void fut_dir_size_stop(gpointer handle)
{
	DirSizeOp	*op = handle;
	GList	*link;

	if((link = g_list_find(dir_size_info.ops, op)) != NULL)
	{
		GSList	*iter;

		dir_size_info.ops = g_list_remove(dir_size_info.ops, op);

		op->func(NULL, op->user);

		if(op->current)
			dir_size_free(op->current);
		for(iter = op->stack; iter != NULL; iter = g_slist_next(iter))
			dir_size_free(iter->data);
		g_slist_free(op->stack);
		g_free(op);

		/* If no more operations active, stop the idle function too. */
		if(dir_size_info.ops == NULL)
		{
			gtk_idle_remove(dir_size_info.idle);
			dir_size_info.idle = 0U;
		}
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-29 -	Here's a strtok()-workalike. Almost. The arg <paths> should be either a string
**		of colon-separated path components (e.g. "/usr/local:/home/emil:/root"), or NULL.
**		The function fills in <component> with either the first component, or, when <paths>
**		is NULL, with the _next_ component (relative to the last one). If this sounds confusing,
**		read the man page for strtok(3) and associate freely from there. The main difference
**		between this and strtok(3) is that the latter modifies the string being tokenized,
**		while this function does not. The <component> buffer is assumed to have room for
**		PATH_MAX bytes. Returns 0 if no more components were found, 1 otherwise.
*/
int fut_path_component(const gchar *paths, gchar *component)
{
	static const gchar	*src = NULL;
	gchar			here, *dst = component;

	if(paths != NULL)
		src = paths;
	else if(src == NULL)
		return 0;

	/* First, find the component, if processing a full multi-directory path. */
	while(*src == ':')
		src++;
	for(; (here = *src) != '\0'; src++)
	{
		if(here == ':' && src[1] == G_DIR_SEPARATOR)	/* Colon followed by separator ends a path. */
		{
			src += 1;	/* Leave src at final separator for next time. */
			break;
		}
		*dst++ = here;
	}
	*dst = '\0';
	if(here == '\0')		/* End of path list reached? */
		src = NULL;		/* Makes us stop next time. */
	if(dst > component)
		fut_interpolate(component);

	return dst > component;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-22 -	Evaluate (look up) any environment variable(s) used in <component>. Replaces
**		the entire string with a new version, where all environment variable names
**		have been replaced by the corresponding variable's value. Undefined variables
**		are treated as if their value were "" (empty string). Considers anything with
**		letters, digits and underscores to be a valid variable name. You can also use
**		braces {around} any combination of characters to force a look-up on that.
*/
static void eval_env(gchar *component)
{ 
	GString	*temp = NULL;
	gchar	*ptr, vname[256], *vptr, *vval;

	for(ptr = component; *ptr != '\0'; ptr++)
	{
		if(*ptr == '$')		/* Environment variable? */
		{
			const gchar	*start = ptr;

			ptr++;
			if(*ptr == '{')		/* Braced name? */
			{
				for(vptr = vname, ptr++; *ptr && *ptr != '}' &&
					(gsize) (vptr - vname) < sizeof vname - 1;)
				{
					*vptr++ = *ptr++;
				}
				ptr++;
			}
			else
			{
				for(vptr = vname; *ptr && (isalnum((guchar) *ptr) || *ptr == '_') &&
					(gsize) (vptr - vname) < sizeof vname - 1;)
				{
					*vptr++ = *ptr++;
				}
			}
			*vptr = '\0';
			if((vval = g_getenv(vname)) != NULL)
			{
				if(temp == NULL)
				{
					temp = g_string_new(component);
					g_string_truncate(temp, start - component);
				}
				if(*vval == G_DIR_SEPARATOR)
					g_string_truncate(temp, 0);
				g_string_append(temp, vval);
			}
			ptr--;
		}
		else
		{
			if(temp)
				g_string_append_c(temp, *ptr);
		}
	}
	if(temp)
	{
		stu_strncpy(component, temp->str, PATH_MAX);
		g_string_free(temp, TRUE);
	}
}

static void eval_home(gchar *component)
{
	GString		*temp;
	gchar		ubuf[USER_NAME_SIZE], *uptr;
	const gchar	*ptr, *uname = ubuf, *dir;

	if((temp = g_string_new(NULL)) != NULL)
	{
		for(ptr = component + 1, uptr = ubuf; (*ptr && *ptr != G_DIR_SEPARATOR) &&
			(gsize) (uptr - ubuf) < sizeof ubuf - 1;)
		{
			*uptr++ = *ptr++;
		}
		*uptr = '\0';
		if((dir = usr_lookup_uhome(uname)) != NULL)
			g_string_assign(temp, dir);
		else
			g_string_assign(temp, "");
		g_string_append(temp, ptr);
		if(*temp->str)
			stu_strncpy(component, temp->str, PATH_MAX);
		else
			g_snprintf(component, PATH_MAX, "~%s", uname);
		g_string_free(temp, TRUE);
	}
}

/* 2003-11-29 -	Interpolate special things (like $VARIABLES and ~homedirs) into a path. */
void fut_interpolate(gchar *buffer)
{
	eval_env(buffer);
	if(buffer[0] == '~')
		eval_home(buffer);
	else if(buffer[0] != G_DIR_SEPARATOR && base_path[0] == G_DIR_SEPARATOR)
	{
		gint	len = strlen(buffer);

		/* Make relative path absolute, if buffer space allows it. */
		if(len + base_len + 2 < PATH_MAX)
		{
			memmove(buffer + base_len + 1, buffer, len + 1);
			memcpy(buffer, base_path, base_len);
			buffer[base_len] = G_DIR_SEPARATOR;
		}
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-15 -	Attempt to locate file called <name> in any of the colon-separated locations
**		specified by <paths>. When found, returns a pointer to the complete name. If
**		nothing has been found when all paths have been tried, returns a sullen NULL.
**		Very useful for loading icons.
** 1998-08-23 -	Now uses the fut_path_component() function. Feels more robust.
** 1998-12-01 -	Now uses the fut_exists() call rather than a kludgy open().
** 1999-06-12 -	Now handles absolute <name>s, too.
** BUG BUG BUG	Calls to this function DO NOT nest very well, due to the static buffer pointer
**		being returned. Beware.
*/
const gchar * fut_locate(const gchar *paths, const gchar *name)
{
	static gchar	buf[PATH_MAX];
	const gchar	*ptr = paths;
	gsize		len;

	if(*name == G_DIR_SEPARATOR)		/* Absolute name? */
	{
		if(fut_exists(name))
			return name;
		return NULL;
	}

	while(fut_path_component(ptr, buf))	/* No, iterate paths. */
	{
		len = strlen(buf);
		g_snprintf(buf + len, sizeof buf - len, "%s%s", G_DIR_SEPARATOR_S, name);
		if(fut_exists(buf))
			return buf;
		ptr = NULL;
	}
	return NULL;
}

/* ----------------------------------------------------------------------------------------- */

/* Just free a path. */
static gboolean kill_path(gpointer key, gpointer data, gpointer user)
{
	g_free(key);

	return TRUE;
}

/* 1998-08-23 -	Scan through a list of colon-separated paths, and return a (possibly HUGE)
**		sorted list of all _files_ found. Very useful for selecting icon pixmaps... The
**		scan is non-recursive; we don't want to go nuts here.
** 1998-09-05 -	Now actually sorts the list, too. That bit had been mysteriously forgotten.
** 1998-12-23 -	Now ignores directories already visited. Uses a string comparison rather than a
**		file-system level check, so you can still fool it. It's a bit harder, though.
*/
GList * fut_scan_path(const gchar *paths, gint (*filter)(const gchar *path, const gchar *name))
{
	gchar		buf[PATH_MAX], *item;
	const gchar	*ptr = paths;
	DIR		*dir;
	struct dirent	*de;
	GList		*list = NULL;
	GHashTable	*set;
	GString		*temp;

	if((paths == NULL) || (filter == NULL))
		return NULL;

	if((temp = g_string_new("")) == NULL)
		return NULL;

	if((set = g_hash_table_new(g_str_hash, g_str_equal)) == NULL)
	{
		g_string_free(temp, TRUE);
		return NULL;
	}

	for(ptr = paths; fut_path_component(ptr, buf); ptr = NULL)
	{
		if(g_hash_table_lookup(set, buf))
			continue;
		g_hash_table_insert(set, g_strdup(buf), GINT_TO_POINTER(TRUE));
		if((dir = opendir(buf)) != NULL)
		{
			while((de = readdir(dir)) != NULL)
			{
				if(filter(buf, de->d_name))
				{
					item = g_strdup(de->d_name);
					list = g_list_insert_sorted(list, item, (GCompareFunc) strcmp);
				}
			}
			closedir(dir);
		}
	}
	g_hash_table_foreach_remove(set, kill_path, NULL);
	g_hash_table_destroy(set);
	g_string_free(temp, TRUE);

	return list;
}

/* 1998-08-23 -	Free the list created during a previous call to fut_scan_paths(). */
void fut_free_path(GList *list)
{
	GList	*head = list;

	if(list != NULL)
	{
		for(; list != NULL; list = g_list_next(list))
		{
			if(list->data != NULL)
				g_free(list->data);		/* Free the file name. */
		}
		g_list_free(head);		/* Free all list items. */
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-11-13 -	Basic low-level file copying routine. Reads from <in>, writes into <out>,
**		both of which must be open with the appropriate modes. Copies by reading
**		<chunk_size> bytes at a time. Will handle files having 0 as their size, like
**		most of the fun things in the /proc directory (on Linux systems).
**		Returns the number of bytes successfully written to destination. Note that
**		for empty files, a return value of 0 indicates success.
** 2000-07-16 -	Bug fix: the interruption check was done after the write(), so small files
**		that fit into the chunk_size could not be interrupted. Oops.
*/
off_t fut_copy(gint in, gint out, gsize chunk_size)
{
	gpointer	buf;
	gssize		got, wrote;
	off_t		put = 0U;

	if((in < 0) || (out < 0))
		return 0;

	chunk_size = MAX(chunk_size, MIN_COPY_CHUNK);

	buf = g_malloc(chunk_size);
	while((got = read(in, buf, chunk_size)) > 0)
	{
		if(!pgs_progress_item_update(NULL, put))
		{
			errno = EINTR;
			break;
		}
		wrote = write(out, buf, got);
		if(wrote < 0)
			break;
		put += wrote;
	}
	g_free(buf);

	return put;
}

/* 1999-11-13 -	Copy just <length> bytes from <in> to <out>. */
off_t fut_copy_partial(gint in, gint out, gsize chunk_size, gsize length)
{
	gpointer	buf;
	gssize		got, wrote, block;
	off_t		put = 0U;

	chunk_size = MAX(MIN_COPY_CHUNK, MIN(chunk_size, length));

	buf = g_malloc(chunk_size);
	while(put < length)
	{
		block = (put + chunk_size <= length) ? chunk_size : length - put;
		got = read(in, buf, block);
		if(got <= 0)
			break;
		wrote = write(out, buf, block);
		if(wrote < 0)
			break;
		put += block;
	}
	g_free(buf);

	return put;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-07 -	Check if the user described by (<uid>,<gid>), which is the user currently running
**		gentoo, is allowed to READ from a file described by <stat>. Note that this is a
**		complex check.
** 1998-09-22 -	Moved this function (and friends) into the fileutil module.
*/
gboolean fut_can_read(const struct stat *stat, uid_t uid, gid_t gid)
{
	if(uid == 0)			/* Root rules! */
		return TRUE;

	if(stat->st_uid == uid)		/* Current user's file? */
		return (stat->st_mode & S_IRUSR) ? TRUE : FALSE;
	if(stat->st_gid == gid)		/* User belongs to file's group? */
		return (stat->st_mode & S_IRGRP) ? TRUE : FALSE;

	/* User doesn't own file or belong to file owner's group. Check others. */
	return (stat->st_mode & S_IROTH) ? TRUE : FALSE;
}

/* 1998-09-23 -	Just a convenience interface for the above. Name must be absolute, or you
**		must be very lucky (the current dir while running gentoo is a highly
**		stochastic variable).
*/
gboolean fut_can_read_named(const gchar *name)
{
	struct stat	s;

	if(stat(name, &s) == 0)
		return fut_can_read(&s, geteuid(), getegid());
	return FALSE;
}

/* 1998-09-07 -	Do complex WRITE permission check. */
gboolean fut_can_write(const struct stat *stat, uid_t uid, gid_t gid)
{
	if(uid == 0)
		return TRUE;

	if(stat->st_uid == uid)
		return (stat->st_mode & S_IWUSR) ? TRUE : FALSE;
	if(stat->st_gid == gid)
		return (stat->st_mode & S_IWGRP) ? TRUE : FALSE;

	return (stat->st_mode & S_IWOTH) ? TRUE : FALSE;
}

/* 1998-09-07 -	Do complex EXECUTE permission check. */
gboolean fut_can_execute(const struct stat *stat, uid_t uid, gid_t gid)
{
	if(stat->st_uid == uid)
		return (stat->st_mode & S_IXUSR) ? TRUE : FALSE;
	if(stat->st_gid == gid)
		return (stat->st_mode & S_IXGRP) ? TRUE : FALSE;

	return (stat->st_mode & S_IXOTH) ? TRUE : FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-05-29 -	Check if there is a file named <name> in <path>. If such a file exists, check
**		if the current user can execute it. This is handy as a prelude to actually doing
**		a fork()/exec() pair on <name>, since it allows simpler error handling. If execution
**		is not possible, set errno to something suitable and return FALSE. If everything
**		looks fins (not a guarantee exec() will succeed), clear errno and return TRUE.
*/
gboolean fut_locate_executable(const gchar *path, const gchar *name)
{
	errno = EINVAL;		/* Assume the worst. */

	if((path != NULL) && (name != NULL))
	{
		const gchar	*pname;

		if((pname = fut_locate(path, name)) != NULL)
		{
			if(access(pname, X_OK) == 0)
				return TRUE;
		}
		else
			errno = ENOENT;
	}
	return FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-10-26 -	Check if the user doesn't want to see <name>. Returns TRUE if the file is
**		NOT hidden, FALSE if it is. If <name> is NULL, we free any allocated resources
**		and then return TRUE. This is very useful when the RE source is changing.
** 1998-12-14 -	The check is now done with respect to pane <dp>. This has relevance, since
**		hiding can now be enabled/disabled on a per-pane basis. If <name> is NULL,
**		any pane can be used (but it must not be NULL!).
*/
gboolean fut_check_hide(DirPane *dp, const gchar *name)
{
	MainInfo	*min = dp->main;
	HideInfo	*hi = &min->cfg.path.hideinfo;
	guint		cflags = REG_EXTENDED | REG_NOSUB;

	if(name == NULL)	/* The NULL-name special case is pane-independant. */
	{
		if(hi->hide_re != NULL)
		{
			g_free(hi->hide_re);
			hi->hide_re = NULL;
		}
		return TRUE;
	}
	if(!(min->cfg.dp_format[dp->index].hide_allowed))	/* Hiding disabled? */
		return TRUE;
	switch(hi->mode)
	{
		case HIDE_NONE:
			return TRUE;
		case HIDE_DOT:
			return name[0] != '.';
		case HIDE_REGEXP:
			if(hi->hide_re == NULL)			/* RE not compiled yet? */
			{
				if((hi->hide_re = g_malloc(sizeof *hi->hide_re)) != NULL)
				{
					if(hi->no_case)
						cflags |= REG_ICASE;
					if(regcomp(hi->hide_re, hi->hide_re_src, cflags) != 0)
					{
						g_free(hi->hide_re);
						hi->hide_re = NULL;
					}
				}
			}
			if(hi->hide_re != NULL)
				return regexec(hi->hide_re, name, 0, NULL, 0) == REG_NOMATCH;
			break;
	}
	return TRUE;					/* When in doubt, hide nothing. */
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-30 -	Initialize the file utility module. Grabs the current working directory for
**		use as the global 'base' for paths. Run this early!
*/
void fut_initialize(void)
{
	if(getcwd(base_path, sizeof base_path) == NULL)
	{
		fprintf(stderr, "FILEUTIL: Couldn't get current working dir - defaulting to /\n");
		stu_strncpy(base_path, "/", sizeof base_path);
	}
	base_len = strlen(base_path);
}

/* 2003-11-10 -	Change back to the directory saved in fut_initialize(). Might help profiling. */
void fut_finalize(void)
{
	fut_cd(base_path, NULL, 0U);
}
