/*
    Glurp - A GTK+ client for Music Player Daemon
    Copyright (C) 2004, 2005 Andrej Kacian

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    http://musicpd.org/glurp.shtml

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <glib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "structs.h"
#include "support.h"
#include "comm.h"
#include "gui.h"

extern GladeXML *guixml, *addxml;
extern GlurpState *glurp;

void debug_real (const char *file, const int line, const char *function, const char *msg, ...) {

  char str[4097];
  va_list va;

  va_start(va, msg);
  g_vsnprintf(str, sizeof(str)-1, msg, va);
  fprintf(stderr, "%s:%d, %s(): %s\n", file, line, function, str);
  va_end(va);
  fflush(stderr);

}

void statusbar_print(const char *string, ...) {
  char str[129];
  va_list va;
  GtkWidget *sb = glade_xml_get_widget(guixml, "statusbar_main");

  va_start(va, string);
  g_vsnprintf(str, sizeof(str)-1, string, va);

  gtk_statusbar_push(GTK_STATUSBAR(sb), 0, str);

  glurp->statusbar_status++;

  g_timeout_add(2000, statusbar_reset, (gpointer)&glurp->statusbar_status);
}

gboolean statusbar_reset(gpointer msg) {
  GtkWidget *sb;

  if( glurp->progress_dragging ) return FALSE;

  if( *(gint*)msg == glurp->statusbar_status ) {
    sb = glade_xml_get_widget(guixml, "statusbar_main");
    gtk_statusbar_push(GTK_STATUSBAR(sb), 0, status_string());
    glurp->statusbar_status = 0;
  }

  return FALSE;
}

void title_print(GtkWidget *window, const char *string, ...) {
  char str[129-strlen(GLURP_TITLE_PREFIX)], *str2;
  va_list va;

  if( !string ) string = "";

  va_start(va, string);
  g_vsnprintf(str, sizeof(str)-1, string, va);

  if(!string || !str || !strlen(str)) str2 = g_strdup_printf("%s", GLURP_TITLE_PREFIX);
  else str2 = g_strdup_printf("%s :: %s", GLURP_TITLE_PREFIX, str);

  gtk_window_set_title(GTK_WINDOW(window), str2);
  g_free(str2);
}

gchar *status_string() {
  mpd_Status *mpdstatus;

  if(!CONNECTED) return "Not connected to server";

  mpdstatus = get_status(TRUE);

  if(mpdstatus->state == MPD_STATUS_STATE_PLAY) return "Connected to server, playing";
  if(mpdstatus->state == MPD_STATUS_STATE_PAUSE) return "Connected to server, paused";
  if(mpdstatus->state == MPD_STATUS_STATE_STOP) return "Connected to server, stopped";
  return "Connected to server, unknown state";
}

GlurpSong *get_nth_song_from_playlist(gint pos) {
  gint i;
  GlurpSong *s;

  for( i = 0, s = glurp->playlist; s && i<pos; i++, s = s->next ) {}

  return s;
}

/* this will return a pointer to the first character after last '/' in path */
gchar *strip_dirs(gchar *path) {
  gchar *c = path, *s = path;

  while( c && *c != '\0' ) {
    if( *c == '/' ) s = c;
    c++;
  }
  return s+1;
}

gchar *glade_path() {
  gchar *path;

  path = g_strdup_printf(DATADIR"glurp.glade");
  if(!g_file_test(path, G_FILE_TEST_EXISTS)) {
    debug("file '%s' does not exist", path);
    g_free(path);
    path = g_strdup("glurp.glade");
    if(!g_file_test(path, G_FILE_TEST_EXISTS)) {
      g_free(path);
      return NULL;
    } else {
      debug("found file '%s'", path);
      return path;
    }
  } else {
    debug("found file '%s'", path);
    return path;
  }
}


gchar *glurp_filename(gchar *path) {
  gchar **dirs = g_strsplit(path, "/", 0);
  gint i = 0;

  while( dirs[i+1] ) i++;
  return dirs[i];
}

GlurpSong *get_song_by_pos(gint pos) {
  gint i;
  GlurpSong *s = NULL;

  for( i = 0, s = glurp->playlist; i < pos && s; i++, s = s->next ) {}

  return (s ? s : NULL);
}

GlurpSong *get_song_by_id(gint id) {
  GlurpSong *s = NULL;

  for( s = glurp->playlist; s; s = s->next ) {
    if( s->id == id ) {
      debug("Found it, returning");
      return s;
    }
  }

  return NULL;
}

gint get_song_pos(GlurpSong *song) {
  GlurpSong *s = NULL;
  gint i;

  for( i = 0, s = glurp->playlist; s; s = s->next, i++ ) {
    if( s == song ) {
      debug("Found song position: %d", i);
      return i;
    }
  }

  debug("Couldn't find song position");
  return -1;
}

gint get_num_songs() {
  GlurpSong *s = NULL;
  gint i;

  for( i = 0, s = glurp->playlist; s; s = s->next, i++ ) {}
  return i;
}

void update_song(mpd_Song *song) {
  debug("calling glurp_update_song()");
  glurp_update_song(song);
  debug("calling gui_update_song()");
  gui_update_song(song->id);
}

void add_song(mpd_Song *song) {
  GlurpSong *gsong = NULL, *s = NULL;

  if( get_song_by_id(song->id) ) {
    debug("Already added, ignoring");
    return;
  }

  gsong = malloc(sizeof(GlurpSong));
  gsong->file = g_strdup(song->file);
  gsong->artist = g_strdup(song->artist);
  gsong->title = g_strdup(song->title);
  gsong->album = g_strdup(song->album);
  gsong->track = g_strdup(song->track);
  gsong->name = g_strdup(song->name);
  gsong->time = song->time;
  gsong->pos = song->pos;
  gsong->id = song->id;
  gsong->next = NULL;

  for(s = glurp->playlist; s && s->next; s = s->next) {}

  if(s) s->next = gsong;
  else glurp->playlist = gsong;
}

void debug_print_playlist() {
  GlurpSong *s = NULL;

  debug("-----------------------------------------------");
  for( s = glurp->playlist; s; s = s->next )
    debug("(%d) %s", s->id, s->file);
  debug("-----------------------------------------------");
}

GlurpStream *get_stream_history(gchar *urls) {
  gchar **streamh = NULL;
  gint i;
  GlurpStream *s = NULL, *os = NULL, *ss = NULL;

  if( !urls || !strlen(urls) ) {
    debug("No stream history");
    return NULL;
  }

  streamh = g_strsplit(urls, " ", 0);

  for( i = 0; streamh[i] && i < MAX_STREAM_HISTORY; i++ ) {
    s = malloc(sizeof(GlurpStream));
    s->url = g_strdup(streamh[i]);
    s->next = NULL;
    if( !i ) ss = s;
    if( os ) os->next = s;
    os = s;
  }

  g_strfreev(streamh);

  return ss;
}

void print_stream_history() {
  GlurpStream *s;

  for( s = glurp->stream_history; s; s = s->next ) debug("<%s>", s->url);
}

GlurpStream *get_stream_by_url(gchar *url) {
  GlurpStream *s = NULL;

  for( s = glurp->stream_history; s; s = s->next )
    if( !strcmp(url, s->url) ) return s;

  return NULL;
}

gchar *dump_stream_history() {
  GlurpStream *s;
  gchar *h = g_strdup("");

  for( s = glurp->stream_history; s; s = s->next ) h = g_strconcat(h, (strlen(h) ? " " : ""), s->url, NULL);
  return h;
}

void prepend_stream(gchar *url) {
  GlurpStream *s = NULL;
  gint i = 0;

  debug("Prepending '%s'", url);
  if( !(s = malloc(sizeof(GlurpStream))) ) {
    debug("Couldn't allocate memory for stream history item");
    return;
  }
  s->url = g_strdup(url);
  s->next = NULL;

  if( !glurp->stream_history ) {
    debug("First stream history item");
    glurp->stream_history = s;
    return;
  }

  debug("Not a first item");
  s->next = glurp->stream_history;
  glurp->stream_history = s;

  while( i < (MAX_STREAM_HISTORY - 1) && s ) {
    s = s->next;
    i++;
  }

  if( s && s->next ) {
    g_free(s->next);
    s->next = NULL;
  }
}

void push_stream(gchar *url) {
  GlurpStream *s = NULL;

  if( !(s = get_stream_by_url(url)) )
    prepend_stream(url);
//  else lift_stream(s);
}
